/*	mkdisktab.c
 *
 * History
 * -------
 * 09-Aug-89	Mike DeMoney at NeXT
 *	Created
 */
 
#include <sys/file.h>
#include <nextdev/disk.h>
#include <nextdev/scsireg.h>

#ifndef DEVTYPE_RWOPTICAL
#define	DEVTYPE_RWOPTICAL	0x07
#endif

#define	FPORCH	160
#define	BPORCH	0

#define	BLKSIZE	8192
#define	FRGSIZE	1024
#define	CYLPGRP	32
#define	BYTPERINO 4096
#define	MINFREE	10

#include <stdio.h>

#define	MS_DASD		3		/* Direct access device mode page */
#define	MS_RDG		4		/* Rigid geometry mode page */

struct mode_sense_cmd {
	u_int	msc_opcode:8,
		msc_lun:3,
		msc_mbz1:5,
		msc_pcf:2,
		msc_page:6,
		msc_mbz2:8;
	u_char	msc_len;
	u_char	msc_ctrl;
};

struct param_list_header {
	u_char	plh_len;
	u_char	plh_medium;
	u_char	plh_wp:1,
		plh_reserved:7;
	u_char	plh_blkdesclen;
};

struct block_descriptor {
	u_int	bd_density:8,
		bd_nblk:24;
	u_int	bd_reserved:8,
		bd_blklen:24;
};

struct device_format_params {
	u_char	dfp_savable:1,
		dfp_reserved:1,
		dfp_pagecode:6;
	u_char	dfp_pagelen;
	u_short	dfp_trkszone;
	u_short	dfp_altsecszone;
	u_short	dfp_alttrkszone;
	u_short	dfp_alttrksvol;
	u_short	dfp_sectors;
	u_short	dfp_bytessector;
	u_short	dfp_interleave;
	u_short	dfp_trkskew;
	u_short	dfp_cylskew;
	u_char	dfp_ssec:1,
		dfp_hsec:1,
		dfp_rmb:1,
		dfp_surf:1,
		dfp_reserved2:4;
	u_char	dfp_reserved3;
	u_char	dfp_reserved4;
	u_char	dfp_reserved5;
};

struct rigid_drive_params {
	u_char	rdp_savable:1,
		rdp_reserved:1,
		rdp_pagecode:6;
	u_char	rdp_pagelen;

	u_char	rdp_maxcylmsb;
	u_char	rdp_maxcylinb;
	u_char	rdp_maxcyllsb;

	u_char	rdp_maxheads;

	u_char	rdp_wpstartmsb;
	u_char	rdp_wpstartinb;
	u_char	rdp_wpstartlsb;

	u_char	rdp_rwcstartmsb;
	u_char	rdp_rwcstartinb;
	u_char	rdp_rwcstartlsb;

	u_char	rdp_stepratemsb;
	u_char	rdp_stepratelsb;

	u_char	rdp_landcylmsb;
	u_char	rdp_landcylinb;
	u_char	rdp_landcyllsb;

	u_char	rdp_reserved2;
	u_char	rdp_reserved3;
	u_char	rdp_reserved4;
};

#define	THREE_BYTE(x)	\
		(((x##msb)<<16)|((x##inb)<<8)|(x##lsb))

#define	TWO_BYTE(x)	\
		(((x##msb)<<8)|(x##lsb))

struct mode_sense_reply {
	struct param_list_header msr_plh;
	struct block_descriptor msr_bd;
	union {
		struct device_format_params u_msr_dfp;
		struct rigid_drive_params u_msr_rdp;
	}u;
};

#define msr_dfp	u.u_msr_dfp
#define msr_rdp	u.u_msr_rdp

char *progname;

static int
do_inquiry(int fd, struct inquiry_reply *irp)
{
	struct scsi_req sr;
	struct cdb_6 *c6p;

	bzero((char *)&sr, sizeof(sr));

	c6p = (struct cdb_6 *)&sr.sr_cdb;
	c6p->c6_opcode = C6OP_INQUIRY;
	c6p->c6_len = sizeof(*irp);

	sr.sr_addr = (char *)irp;
	sr.sr_dma_max = sizeof(*irp);
	sr.sr_ioto = 1;
	sr.sr_dma_dir = SR_DMA_RD;

	return ioctl(fd, SDIOCSRQ, &sr);
}

static int
do_modesense(int fd, struct mode_sense_reply *msrp, int page)
{
	struct scsi_req sr;
	struct mode_sense_cmd *mscp;

	bzero((char *)&sr, sizeof(sr));

	mscp = (struct mode_sense_cmd *)&sr.sr_cdb;
	mscp->msc_opcode = C6OP_MODESENSE;
	mscp->msc_pcf = 0;	/* report current values */
	mscp->msc_page = page;
	mscp->msc_len = sizeof(*msrp);

	sr.sr_addr = (char *)msrp;
	sr.sr_dma_max = sizeof(*msrp);
	sr.sr_ioto = 1;
	sr.sr_dma_dir = SR_DMA_RD;

	return ioctl(fd, SDIOCSRQ, &sr);
}

static int
do_readcapacity(int fd, struct capacity_reply *crp)
{
	struct scsi_req sr;
	struct cdb_10 *c10p;

	bzero((char *)&sr, sizeof(sr.sr_cdb));

	c10p = (struct cdb_10 *)&sr.sr_cdb;
	c10p->c10_opcode = C10OP_READCAPACITY;

	sr.sr_addr = (char *)crp;
	sr.sr_dma_max = sizeof(*crp);
	sr.sr_ioto = 1;
	sr.sr_dma_dir = SR_DMA_RD;

	return ioctl(fd, SDIOCSRQ, &sr);
}

main(int argc, char **argv)
{
	int fd;
	struct inquiry_reply ir;
	struct mode_sense_reply dasd, rd;
	struct capacity_reply cr;
	struct drive_info di;
	int sfactor, ncyl, ntracks, nsectors, nblocks;

	progname = *argv++;
	argc--;

	for (; argc > 1 && **argv == '-'; argv++, argc--) {
		switch ((*argv)[1]) {
		default:
			fatal("Unknown option: %s\n", *argv);
		}
	}

	if (argc != 1)
		fatal("Usage: %s RAW_DEV", progname);

	if ((fd = open(*argv, O_RDONLY, 0)) < 0)
		fatal("Can't open %s", *argv);

	if (do_inquiry(fd, &ir))
		fatal("inquiry cmd failed");

	if (do_modesense(fd, &dasd, MS_DASD))
		fatal("mode sense for dasd params failed");

	if (do_modesense(fd, &rd, MS_RDG))
		fatal("mode sense for rigid drive params failed");

	if (do_readcapacity(fd, &cr))
		fatal("read capacity failed");

	if (ioctl(fd, DKIOCINFO, &di))
		fatal("Couldn't get drive info");

	if (ir.ir_devicetype != DEVTYPE_DISK
	    && ir.ir_devicetype != DEVTYPE_RWOPTICAL)
		fatal("Unknown device type");

	sfactor = 1024/(cr.cr_blklen);
	if (sfactor != 1 && sfactor != 2 && sfactor != 4)
		fatal("Can't determine device sector size");

	if (dasd.msr_dfp.dfp_sectors == 0) {
		dasd.msr_dfp.dfp_sectors =
		   (cr.cr_lastlba+1)/(THREE_BYTE(rd.msr_rdp.rdp_maxcyl)
		   * rd.msr_rdp.rdp_maxheads);
		fprintf(stderr,
		    "Zone sectored device, guessing sectors at %d\n",
		    dasd.msr_dfp.dfp_sectors);
	}

	ntracks = rd.msr_rdp.rdp_maxheads;
	nsectors = dasd.msr_dfp.dfp_sectors/sfactor;
	nblocks = (cr.cr_lastlba + 1)/sfactor;

	ncyl = nblocks / (ntracks * nsectors);


	printf("%.*s-%d:\\\n", MAXDNMLEN, di.di_name, cr.cr_blklen);
	printf("\t:ty=%s_rw_%s:nc#%d:nt#%d:ns#%d:ss#1024:rm#3600:\\\n",
	    ir.ir_removable ? "removable" : "fixed",
	    ir.ir_devicetype == DEVTYPE_DISK ? "scsi" : "optical",
	    ncyl, ntracks, nsectors);
	printf("\t:fp#%d:bp#%d:ng#0:gs#0:ga#0:ao#0:\\\n",
	    FPORCH, BPORCH);
	printf("\t:os=sdmach:z0#32:z1#96:ro=a:\\\n");
	printf("\t:pa#0:sa#%d:ba#%d:fa#%d:ca#%d:da#%d:ra#%d:oa=time:\\\n",
	    nblocks - FPORCH - BPORCH, BLKSIZE, FRGSIZE, CYLPGRP,
	    BYTPERINO, MINFREE);
	printf("\t\t:ia:ta=4.3BSD:\n");

	close(fd);
	exit(0);
}

fatal(msg, arg)
char *msg;
{
	fprintf(stderr, "%s: ", progname);
	fprintf(stderr, msg, arg);
	fprintf(stderr, "\n");
	exit(1);
}
