/*** File Classes Test Harness: filetest.c ***/

#undef NDEBUG

#include "file.h"
#include "stdfile.h"
#include "unixfile.h"


#define TEST_SIZE 32767

void init(const char *iname)
{
    puts("generate test file with standard io");
    {   FILE *f = fopen(iname,"wb");
        long i;
        for (i=0; i < TEST_SIZE; i++)
            fputc(i,f);
        fclose(f);
    }

    puts("traverse test file with standard io");
    {   FILE *f = fopen(iname,"rb");
        long i;
        char c;
        for (i=0; i < TEST_SIZE; i++) {
            assert(fread(&c,1,1,f) == 1);
            assert(c == (char)i);
        }
        fclose(f);
    }

}

void test(File *input, File *output)
{   long i, nb;
    char cin, cout;

    puts("  bytewise copy");
    assert(FileSeek(input,0,SEEK_SET) == 0);
    for (i = 0; i < TEST_SIZE; i++) {
        assert(FileRead(input,&cin,1) == 1);
        assert(FileWrite(output,&cin,1) == 1);
    }

    puts("  random recopy");
    for (i = 0; i < TEST_SIZE; i += nb) {
        char buf[512];
        int n = rand();
        nb = rand() % sizeof buf;
        if (!nb || n + nb > TEST_SIZE)
            continue;
        i += nb;
        assert(FileSeek(input,n,SEEK_SET) == n);
        assert(FileRead(input,&buf,nb) == nb);
        assert(FileSeek(output,n,SEEK_SET) == n);
        assert(FileWrite(output,&buf,nb) == nb);
    }

    puts("  bytewise comparison");
    assert(FileSeek(input,0,SEEK_SET) == 0);
    assert(FileSeek(output,0,SEEK_SET) == 0);
    for (i = 0; i < TEST_SIZE; i++) {
        assert(FileRead(input,&cin,1) == 1);
        assert(FileRead(output,&cout,1) == 1);
        assert(cin == cout);
    }
    assert(FileSeek(input,0,SEEK_CUR) == TEST_SIZE );
    assert(FileSeek(output,0,SEEK_CUR) == TEST_SIZE);
}

main()
{
    char iname[L_tmpnam];
    const char *oname;
    tmpnam(iname);

    puts("initialize classes");
    USE(File);
    USE(StdFile);
    USE(UnixFile);

    init(iname);

    puts("test with input as StdFile, output as StdFile");
    oname = tmpnam(0);
    {   PUSH(fin,StdFile,(iname,"rb"));
        PUSH(fout,StdFile,(oname,"w+b"));
        test(fin,fout);
        POP(fout);
        POP(fin);
    }
    remove(oname);

    puts("test with input as UnixFile, output as UnixFile");
    oname = tmpnam(0);
    {   PUSH(fin,UnixFile,(iname,O_RDWR|O_BINARY,0));
        PUSH(fout,UnixFile,(oname,O_RDWR|O_BINARY|O_CREAT,
                                  S_IREAD|S_IWRITE));
        test(fin,fout);
        POP(fout);
        POP(fin);
    }
    remove(oname);

    puts("test with input as StdFile, output as UnixFile");
    oname = tmpnam(0);
    {   PUSH(fin,StdFile,(iname,"rb"));
        PUSH(fout,UnixFile,(oname,O_RDWR|O_BINARY|O_CREAT,
                                  S_IREAD|S_IWRITE));
        test(fin,fout);
        POP(fout);
        POP(fin);
    }
    remove(oname);

    puts("test with input as UnixFile, output as StdFile");
    oname = tmpnam(0);
    {   PUSH(fin,UnixFile,(iname,O_RDWR|O_BINARY,0));
        PUSH(fout,StdFile,(oname,"w+b"));
        test(fin,fout);
        POP(fout);
        POP(fin);
    }
    remove(oname);

    puts("remove test file");
    remove(iname);
}
