//CORDIC.HPP  a header file for the CORDIC class which provides integer
//					sine and cosine approximations for graphics processing.
//					These routines are a C++ adaptation of routines written in C in
//					"The CORDIC Method for Faster sin and cos Calculations"
//					by Michael Bertrand, in _The C User's Journal_, Vol. 10,
//					#11, pp 57 - 64.

//					Principal changes from the C implementation are: making a lot of
//					global int's into private static members, and establishing
//					a predefined instance of the cordic object: cord.

//					The work is done by the cordic::sinCos() member functions.
//					The results are returned as integers between 0 and 16K,
//					and the long input is in CORDIC Angle units which divide
//					the circle into 64K parts, instead of degrees or radians.

//					The version which uses an unsigned long as an input is the
//					fastest, but the inputs must be 0 <= theta <= 64K.
//					The version which takes a long as an input is the next
//					fastest:  it moves the input theta into the above range,
//					then calls the unsigned long version to do the work.  The
//					float version is the slowest:  it takes a radian input and
//					converts it to CAU's before calling the second version to
//					massage the value for the first function to work on.

//					The sine cosine values can be normalized, but the real value
//					of these approximations is best found by designing routines
//					to use the results as they are, to improve speed.  For the
//					same reason, it is better to use the unsigned long value
//					as input to the routines, to avoid the time overhead
//					associated with changing from radians to CAU's.

#ifndef CORDIC_HPP
#define CORDIC_HPP

#include <math.h>


enum {Quad1, Quad2, Quad3, Quad4, NBits = 14};

enum {cordbase = 16384};


class cordic
{
private:
	static int ArcTan[NBits];  	//array of integer arcTan's
	static int xInit;          	//length of initial vector
	static int instance;			 	//previous instance of CORDIC?
	static long HalfBase;         //an eighth of a circle in CAU's
	static long Quad2Boundary;    //half a circle in CAU's
	static long Quad3Boundary;    //3/4 circle in CAU's
	static long CordicBase;       //a quarter of a circle in CAU's

public:
	cordic();
	~cordic();

	long cordicbase(void);			//provides the value of CordicBase to users

	//these routines should be used by placing the value of the angle for
	//which the sine and/or cosine is to be evaluated into theta, and
	//looking for the return values in the sin and cos variables.
	//The unsigned long version will accept only positive angle values
	//less than 64K, while the long version accepts any long input and
	//the radian version accepts most reasonable float inputs
	void sinCos(unsigned long theta, int &sin, int &cos); //input in CAU's
	void sinCos(long theta, int &sin, int &cos);
	void sinCos(float theta, int &sin, int &cos);   // input in Radians

};// Class CORDIC

extern cordic cord;		//predefined instance

#endif //CORDIC_HPP
