/**************************************************************************
*  File Name     : SPEED_UP.C
*  Description   : Preprocessor utility for the Xlate System.
*                  Creates Binary versions of the Translate File.
*  Author        : R. Scott Guthrie  /  All Rights Reserved
*  Compile\Link  : Link this source code with XLATE.OBJ
**************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <alloc.h>
#include <assert.h>
#include "xlate.h"

/* Local Prototypes */
void instructions(void);
int open_files(const char *filename, FILE **tf, FILE **bf);

/* Global Variables */
extern XLATE *XlateBase;
extern int XlateLines;
extern char *XlateFile;

/*=======================*/
  void instructions(void)
/*=======================*/
/* 'instructions()' is called if the user does  */
/* not specify filename to be converted.        */
{
  /* Print instructions for the user. */
  printf(" ##############\n #  SPEED_UP  #\n ##############\n");
  printf("The 'SPEED_UP' program builds a version of XLATE's translate\n");
  printf("file that loads faster then the text version.\n\n");
  printf("    Syntax:   SPEED_UP filename\n\n");
  printf("The parameter 'filename' is the name of the Text Based\n");
  printf("('.TRN') Translate File.  (The filename extension of '.TRN'\n");
  printf("MUST NOT be entered as part of the filename.)\n\n");
  printf("    Example:  SPEED_UP AMERICAN\n\n");
  printf("The example will use the file 'AMERICAN.TRN' to build a faster\n");
  printf("loading version that will have an extension of '.TRB'.\n\n");
  printf("If both the '.TRN' and '.TRB' (speed-up) versions of the\n");
  printf("translate file exist, XLATE will use the speed-up version\n");
  printf("at run time.\n");
  return;
}

/*==========================================================*/
  int open_files(const char *filename, FILE **tf, FILE **bf)
/*==========================================================*/
/* Open the Text File and the new Speed-up file */
{
  char tfname[128];     /* Name of the Source File */
  char bfname[128];   /* Name of the Destination File */

  /* Open the Text (Source) File for Reading. */
  sprintf(tfname, "%s.TRN", filename);
  if((*tf = fopen(tfname, "r")) == NULL)
  {
    printf("Speed_up: Error opening file %s for reading.\n", tfname);
    return 0;  /* fail */
  }

  /* Open the Binary File for Writing. (Overwrite if it exists) */
  sprintf(bfname, "%s.TRB", filename);
  if((*bf = fopen(bfname, "wb")) == NULL)
  {
    printf("Speed_up: Error creating or opening file %s for writing.\n",
	    bfname);
    return 0;  /* fail */
  }
  return 1;    /* success */
}

/*================================*/
  void main(int argc, char **argv)
/*================================*/
{
  FILE  *tf;                    /* Text File handle              */
  FILE  *bf;                    /* Binary File handle            */
  int   i;                      /* Local Index Variable          */
  XLATE *node;                  /* Translate Node                */
  int   length;                 /* strlen variable.              */
  int   retval;                 /* Return value variable.        */

  /* Check the parameters.  There should be 2. The Program name  */
  /* and the File Name (without the extension).                  */
  if(argc != 2)
  {
    instructions();
    exit(0);
  }

  /* Open (or Create) the Binary File for Writing. */
  if(open_files(argv[1], &tf, &bf))
  {
    printf("\nSpeed_up: Begin Scan...\n");

    /* Build Translation table */
    retval = XlateLoadText(tf);
    if(retval == -1)
    {
      /* memory allocation error */
      fclose(tf);     /* Close Text Translate File (input)    */
      fclose(bf);     /* Close Binary Translate File (output) */
      printf("Speed_up: Memory Allocation Error.\n");
      exit(0);  /* stop processing */
    }
    fclose(tf);       /* Finished with Text File */

    /* Report progress... */
    printf("Speed_up: %d Translate lines found.\n", retval);
    printf("Speed_up: Generating Binary File...\n");

    /* In-Memory table generated.  Read data from it into Binary File */

    /* Format of the Binary Version of the Translate File:               */
    /*  File signiture  (char[4] = ".TRB")                               */
    /*  Number of translate file entries.  (int)                         */
    /*  The number of translate file entries of:                         */
    /*    Length of Null Terminated Key String (including NULL) (int)    */
    /*    Null Terminated Key String                                     */
    /*    Length of Null Terminated Result String (including NULL) (int) */
    /*    Null Terminated Result String                                  */
    /*  <eof>                                                            */

    node = XlateBase;  /* Set node to XlateBase */

    /* Write the data to the Binary Output file */
    /* Signature */
    if(fwrite(".TRB", 4, 1, bf) != 1)
      printf("Speed_up: Error Writing Signature!\n");
    if(fwrite(&XlateLines, sizeof(XlateLines), 1, bf) != 1)
      printf("Speed_up: Error Writing Number of Key/Result Pairs!\n");

    for(i = 0; i < XlateLines ; i++)
    {
      length = strlen(node->translate_key) + 1;
      if(fwrite(&length, sizeof(length), 1, bf) != 1)
        printf("Speed_up: Error Writing Length of Key Value.\n");

      if(fwrite(node->translate_key,
                strlen(node->translate_key) + 1, 1, bf) != 1)
        printf("Speed_up: Error Writing Key Value number %d.\n", i);

      length = strlen(node->translated_value) + 1;
      if(fwrite(&length, sizeof(length), 1, bf) != 1)
        printf("Speed_up: Error Writing Length of Result Value.\n");
      if(fwrite(node->translated_value,
                strlen(node->translated_value) + 1, 1, bf) != 1)
        printf("Speed_up: Error Writing Result Value number %d.\n", i);

      node++;                   /* advance to next entry. */
    }
    /* Done, close destination file */
    fclose(bf);
  }
  printf("Speed_up: Finished.\n");
}
/* end 'main()' */
