/* ----------------------------------------------------
 *  block.h
 *
 *  IOCTL for block devices
 * ------------------------------------------------- */

#ifndef  __BLOCK_H
#define  __BLOCK_H

#include <ctype.h>      //toupper()
#include "iostruct.h"
#include "iobase.h"

//Converts ASCII drive letter designator ('A') to
//IOCTL "1-Based" drive numbers, which maps drive A:
//to 1, B = 2, etc. and uses 0 as the default drive.
//
#define DRIVE_CODE(d)   ((toupper(d)-'A')+1)
#define DEFAULT_DRIVE   0

class IoctlBlock : public IoctlBase  {

protected:
    unsigned drive_info( int drive );
    int block_ioctl( block_cmd, void * );
    unsigned ioctl_data( ioctl_cmd, unsigned, void *);
    int format_track( int, int, int );
    IoctlBlock(int drive);      //constructor
    void IoctlError( int );
    int _drive;     //current drive (1-based)
    unsigned _info;    //current drive info word

public:
    static IoctlBlock *Init( int drive );
    ~IoctlBlock() { }

    enum localDrive   {
    //bit settings for local drives, i.e. bit 12=0....
    _32bit = 0x0002,    //b1=1, drive uses 32-bit
                        //sector addressing
    genBlockIoctl = 0x0040, //b6=1, can use GIOCTL for
			    //block devices (440dh) and
			    //and get/set logical drive
			    //map (440eh and 440fh)
    queryIoctl = 0x0080, //b7=1, can use Query IOCTL
			 //Device (4411h)
    isShared = 0x0200,   //b9=1, drive local but shared
			 //by others on network
    rmvMedia = 0x0800,   //b11=1, can use Removable
			 //media check (4408h)
    drvRemote = 0x1000,  //b12 = 1 if drive is remote
    mdFat = 0x2000,      //b13=1, needs FAT media descr
    xmitIoctl = 0x4000,  //b14=1, can send/recv
                         //Ioctl data to/from block
                         //device (4404h and 4405h)
    substDrive = 0x8000  //b15=1, substitution drive
                         //set by 'subst' command
    };

    //Return values for checkFunction() ......
    enum query_return_t   {
        is_supported,      //OK!
        invalid_function,  //no IOCTL support at all
        access_denied,     //function not supported
        invalid_drive,
        unknown            //unable to comply, or
                           //_dos_version < 5.0
        };

    unsigned driveInfo( void ) { return _info; }
    int currentDrive(void ) { return _drive; }
    int sendIoctl( unsigned *count, void *buffer );
    int readIoctl( unsigned *count, void *buffer );
    int isRemote(void) { return _info & drvRemote; }
    int isRemovable(void);
    int checkAlias( void);
    int nextAlias( int next );
    int getParams( struct DEVICEPARAMS *dpms );
    int setParams( struct DEVICEPARAMS *dpms );
    int readSectors( struct RWBLOCK *tpms );
    int writeSectors( struct RWBLOCK *tpms );
    int getAccessFlag(void);
    int setAccessFlag(int flag);
    int formatTrack( int head, int cyl );
    int checkFormat( int head, int cyl )
        { return format_track( 1, head, cyl );  }
    int verifyTrack( int head, int cyl );
    int getMediaID( struct MID *mid )     {
        return block_ioctl( get_media_id,
                            (void *) mid );  }
    int setMediaID( struct MID *mid )     {
        return block_ioctl( set_media_id,
                            (void *) mid );  }
    int senseMediaType( struct MTYPE *mtype )   {
        return block_ioctl( sense_media_type,
                            (void *) mtype   );  }
    int checkFunction( block_cmd cmd );

};  //.... end class IoctlBlock

#endif   //__BLOCK_H

/* ----- End of File ------------------------------- */
