Listing 1:

/******************************************************
*  FMEXT.C - Generic File Manager Extension           *
*  Copyright (c) 1994                                 *
*  Trevor Harmon, Microdyne Development Technologies  *
******************************************************/

/* Use strict type casting for safety. */
#define STRICT

/* Standard Windows definitions */
#include <windows.h> 
/* File Manager Extensions header */
#include <wfext.h>   
/* Local program definitions */
#include "fmext.h"    

/* Function prototypes */
void FillSelectionsListBox(HWND);
void FillDriveInfoDlg(HWND);

/* Global constants */
char szExtensionName[] =
    "Generic File Manager Extension";
char szMenuName[] = "&Extension";

/* Global variables */
HINSTANCE hInst;    /* Handle to instance of DLL */
WORD wFMMenuDelta;  /* Menu-item delta value */
HWND hwndFileMan;   /* Handle to File Manager */
/* Tells if the Checkmark Test menu item is checked */
BOOL fItemChecked;

/* Starting point for the DLL */
int FAR PASCAL LibMain(HINSTANCE hInstance,
     WORD wDataSeg, WORD cbHeapSize, LPSTR lpCmdLine)
{
  /* Save the hInstance handle for later. */
  hInst = hInstance;

  /* The menu item is initially unchecked. */ 
  fItemChecked = FALSE;

  /* Perform any DLL initialization routines here. */

  /* Return success. */
  return 1;
}

/* Exit point for the DLL */
int FAR PASCAL WEP(int nParameter)
{
  /* Perform any DLL cleanup routines here. */

  /* Return success. */
  return 1;
}

/******************************************************
* Fill the list box in the given dialog with the File *
* Manager selections.                                 *
******************************************************/
void FillSelectionsListBox(HWND hWnd)
{
  WORD wSelectedItems;    /* Number of selections */
  FMS_GETFILESEL fmsgfs;  /* File information */

  /* Retrieve the number of selected items. */
  wSelectedItems = (WORD) SendMessage(hwndFileMan,
                               FM_GETSELCOUNT, 0, 0L);

  /* If one or more items are selected... */
  if (wSelectedItems > 0) {

    /* Loop through each selected file, retrieve */ 
    /* its name, and add it to the list box. */
    while (wSelectedItems > 0) {

      wSelectedItems--;
      /* Send a File Manager message to fill the */
      /* FMS_GETFILESEL structure. */
      SendMessage(hwndFileMan, FM_GETFILESEL,
           wSelectedItems, (LPARAM) &fmsgfs);
      /* Transfer the szName field in the structure */
      /* over to the list box. */
      SendDlgItemMessage(hWnd, IDC_SELLIST,
           LB_ADDSTRING, 0, (LPARAM) fmsgfs.szName);

    }

  }
  else {

    /* If no items are selected, inform the user */
    /* and abort. */
    MessageBox(hWnd, "Please select at least one " \
       "file or directory from the list on the right.",
       "Show Selections", MB_ICONSTOP | MB_OK);
    EndDialog(hWnd, 0);

  }

}

/* Dialog box procedure for Show Selections */
BOOL FAR PASCAL ShowSelDlgProc(HWND hWnd, WORD wMsg,
                     WPARAM wParam, LPARAM lParam)
{
  switch (wMsg) {

    case WM_INITDIALOG:
      /* Fill list box with File Manager selections. */
      FillSelectionsListBox(hWnd);
      break;

    case WM_COMMAND:
      /* End the dialog if OK is pressed or the */
      /* box is closed. */
      switch (wParam) {
	case IDOK:
        case IDCANCEL:
          EndDialog(hWnd, 0);
	  break;
      }

  }

  return FALSE;
}

/******************************************************
* Query the File Manager for information on the       *
* current drive, then fill the dialog's static text   *
* controls accordingly.                               *
******************************************************/
void FillDriveInfoDlg(HWND hWnd)
{
  FMS_GETDRIVEINFO fmsgdi;  /* Info on current drive */
  char s[10]; /* Buffer for number-string conversion */

  /* Retrieve info on current File Manager drive. */
  SendMessage(hwndFileMan, FM_GETDRIVEINFO, 0,
       (LPARAM) &fmsgdi);

  /* Convert the dwTotalSpace field to a string. */
  wsprintf(s, "%lu", fmsgdi.dwTotalSpace);
  /* Set the dialog's Total Space text control. */
  SendDlgItemMessage(hWnd, IDC_TOTALSPACE, WM_SETTEXT,
       0, (LPARAM) s);

  /* Convert the dwFreeSpace field to a string. */
  wsprintf(s, "%lu", fmsgdi.dwFreeSpace);
  /* Set the dialog's Free Space text control. */
  SendDlgItemMessage(hWnd, IDC_FREESPACE, WM_SETTEXT,
       0, (LPARAM) s);

  /* Set the dialog's Current Path text control. */
  SendDlgItemMessage(hWnd, IDC_CURRENTPATH, WM_SETTEXT,
       0, (LPARAM) fmsgdi.szPath);
  /* Set the dialog's Volume Label text control. */
  SendDlgItemMessage(hWnd, IDC_VOLUMELABEL, WM_SETTEXT,
       0, (LPARAM) fmsgdi.szVolume);

  if ( lstrlen(fmsgdi.szShare) > 0 )
    /* If the network sharepoint exists, set the */
    /* dialog's text control. */
    SendDlgItemMessage(hWnd, IDC_SHAREPOINT,
         WM_SETTEXT, 0, (LPARAM) fmsgdi.szShare);
  else
    /* If the sharepoint is a null string, set the */
    /* text control to "None". */
    SendDlgItemMessage(hWnd, IDC_SHAREPOINT,
         WM_SETTEXT, 0, (LPARAM) "None");
}

/* Dialog box procedure for Drive Information */
BOOL FAR PASCAL DriveInfoDlgProc(HWND hWnd, WORD wMsg,
      WPARAM wParam, LPARAM lParam)
{
  switch (wMsg) {

    case WM_INITDIALOG:
      /* Fill the dialog's text controls with */
      /* information on the current drive. */
      FillDriveInfoDlg(hWnd);
      break;

    case WM_COMMAND:
      /* End the dialog if OK is pressed or the */
      /* dialog box is closed. */
      switch (wParam) {
	case IDOK:
        case IDCANCEL:
          EndDialog(hWnd, 0);
	  break;
      }

  }

  return FALSE;
}

/******************************************************
* Entry point for FMEVENT_xxxx messages sent by the   *
* File Manager to the Extension.                      *
******************************************************/
HMENU FAR PASCAL FMExtensionProc(HWND hWnd, WORD wMsg,
                      LONG lParam)
{
  FARPROC lpDlgProcInst; /* Points to a dlg function */
  HMENU hMenu;  /* Handle to a menu */

  /* Store the File Manager's window handle. */
  hwndFileMan = hWnd;

  switch (wMsg) {

    case FMEVENT_LOAD:
    {
      LPFMS_LOAD lpfmsl; /* Holds info on Extension */

      /* Convert the lParam parameter to an FMS_LOAD */
      /* pointer to make reading the code easier. */
      lpfmsl = (LPFMS_LOAD) lParam;

      /* Initialize the FMS_LOAD structure... */
      /* Size of structure */
      lpfmsl->dwSize = sizeof(FMS_LOAD);     
      /* Name of Extension menu */
      lstrcpy(lpfmsl->szMenuName, szMenuName);
      /* Load Extension menu from .RC into memory. */
      hMenu = LoadMenu(hInst, "FMExtMenu");     
      /* Store Extension menu. */
      lpfmsl->hMenu = hMenu;               

      /* Get the File Manager's menu-item delta for */
      /* this Extension. */
      wFMMenuDelta = lpfmsl->wMenuDelta;

      /* Important: non-zero value must be returned. */
      return hMenu;
    }

    case FMEVENT_INITMENU:
      /* Convert the lParam parameter to an hMenu */
      /* handle to make reading the code easier. */
      hMenu = (HMENU) HIWORD(lParam);
      /* If the Checkmark Test menu item is checked, */
      /* place a checkmark on the item.  Otherwise, */
      /* remove the checkmark. */
      if (fItemChecked)
        CheckMenuItem(hMenu,
             wFMMenuDelta + IDM_INITMENUTEST,
             MF_CHECKED | MF_BYCOMMAND);
      else
        CheckMenuItem(hMenu,
             wFMMenuDelta + IDM_INITMENUTEST,
             MF_UNCHECKED | MF_BYCOMMAND);
      break;

    case FMEVENT_SELCHANGE:
      break;

    case FMEVENT_USER_REFRESH:
      MessageBox(hWnd, "The user refreshed the " \
           "File Manager windows.", "User Refresh",
           MB_OK | MB_ICONINFORMATION);
      break;

    case FMEVENT_UNLOAD:
      break;

    case IDM_SHOWSEL:
      /* Display the Show Selections dialog box. */
      lpDlgProcInst = MakeProcInstance( (FARPROC)
           ShowSelDlgProc, hInst);
      DialogBox(hInst, "ShowSel", hwndFileMan,
           (DLGPROC) lpDlgProcInst);
      FreeProcInstance(lpDlgProcInst);
      break;

    case IDM_DRIVEINFO:
      /* Display the Drive Information dialog box. */
      lpDlgProcInst = MakeProcInstance( (FARPROC)
           DriveInfoDlgProc, hInst);
      DialogBox(hInst, "DriveInfo", hwndFileMan,
           (DLGPROC) lpDlgProcInst);
      FreeProcInstance(lpDlgProcInst);
      break;

    case IDM_INITMENUTEST:
      /* Invert the global variable fItemChecked. */
      fItemChecked = ~fItemChecked;
      break;

    case IDM_REFRESH:
      /* Tell File Manager to refresh its windows. */
      SendMessage(hwndFileMan, FM_REFRESH_WINDOWS,
           TRUE, 0);
      break;

    case IDM_UNINSTALL:
      /* Check to see if the user really wants to */
      /* remove the Extension. */ 
      if ( MessageBox(hWnd, "Are you sure you want " \
        "to remove the extension?", "Remove Extension",
        MB_YESNO | MB_ICONQUESTION) == IDYES ) {

        /* Remove Extension string from WINFILE.INI. */
	WritePrivateProfileString("AddOns",
             szExtensionName, "", "winfile.ini");
        /* Tell File Manager to reload Extensions. */
	PostMessage(hwndFileMan, FM_RELOAD_EXTENSIONS,
             0, 0L);

      }
      break;

  }

  return (HMENU) 0;
}

