//	cmostr.h
//	modification of Microsoft Foundation Class code supporting CString, to make use
//	of CMemoryObject base class.

// Microsoft Foundation Classes C++ library.
// Copyright (C) 1992 Microsoft Corporation,
// All rights reserved.

// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and Microsoft
// QuickHelp and/or WinHelp documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

#ifndef _CMOSTR_H_
#define _CMOSTR_H_

#ifndef __cplusplus
#error Microsoft Foundation Classes require C++ compilation (use a .cpp suffix)
#endif

#include	<iostream.h>
#include	<string.h>

#if !defined (_MOBJECT_H_)
	#include	"mobject.h"
#endif

/////////////////////////////////////////////////////////////////////////////
// Strings

typedef const char FAR * lp_cc;		//	To fix compiler bug in declaring
									//	conversion functions, as advised
									//	by MSKB article Q103789

class FAR CMOString : public MemoryObject
{
public:

// Constructors
	CMOString();
	CMOString(const CMOString& stringSrc);
	CMOString(char ch, int nRepeat = 1);
	CMOString(const char FAR * psz);
	CMOString(const char FAR * pch, int nLength);
#ifdef _NEARDATA
	// Additional versions for far string data
//	CMOString(LPCSTR lpsz);
//	CMOString(LPCSTR lpch, int nLength);
#endif
	~CMOString();

// Attributes & Operations

	// as an array of characters
	int GetLength() const;
	BOOL IsEmpty() const;
	void Empty();                       // free up the data

	char GetAt(int nIndex) const;       // 0 based
	char operator[](int nIndex) const;  // same as GetAt
	void SetAt(int nIndex, char ch);
	operator lp_cc () const;       // as a C string

	// overloaded assignment
	const CMOString& operator=(const CMOString& stringSrc);
	const CMOString& operator=(char ch);
	const CMOString& operator=(const char FAR * psz);

	// string concatenation
	const CMOString& operator+=(const CMOString& string);
	const CMOString& operator+=(char ch);
	const CMOString& operator+=(const char FAR * psz);

	friend CMOString AFXAPI operator+(const CMOString& string1,
			const CMOString& string2);
	friend CMOString AFXAPI operator+(const CMOString& string, char ch);
	friend CMOString AFXAPI operator+(char ch, const CMOString& string);
	friend CMOString AFXAPI operator+(const CMOString& string, const char FAR * psz);
	friend CMOString AFXAPI operator+(const char FAR * psz, const CMOString& string);
    friend ostream & operator << (ostream & os, const CMOString & str);
    
	// string comparison
	int Compare(const char FAR * psz) const;         // straight character
	int CompareNoCase(const char FAR * psz) const;   // ignore case
	int Collate(const char FAR * psz) const;         // NLS aware

	// simple sub-string extraction
	CMOString Mid(int nFirst, int nCount) const;
	CMOString Mid(int nFirst) const;
	CMOString Left(int nCount) const;
	CMOString Right(int nCount) const;

	CMOString SpanIncluding(const char FAR * pszCharSet) const;
	CMOString SpanExcluding(const char FAR * pszCharSet) const;

	// upper/lower/reverse conversion
	void MakeUpper();
	void MakeLower();
	void MakeReverse();

	// searching (return starting index, or -1 if not found)
	// look for a single character match
	int Find(char ch) const;                    // like "C" strchr
	int ReverseFind(char ch) const;
	int FindOneOf(const char FAR * pszCharSet) const;

	// look for a specific sub-string
	int Find(const char FAR * pszSub) const;         // like "C" strstr

	// input and output
#ifdef _DEBUG
	friend CDumpContext& AFXAPI operator<<(CDumpContext& dc,
				const CMOString& string);
#endif
	friend CArchive& AFXAPI operator<<(CArchive& ar, const CMOString& string);
	friend CArchive& AFXAPI operator>>(CArchive& ar, CMOString& string);

	// Windows support
#ifdef _WINDOWS
	BOOL LoadString(UINT nID);          // load from string resource
										// 255 chars max
	// ANSI<->OEM support (convert string in place)
	void AnsiToOem();
	void OemToAnsi();
#endif //_WINDOWS

	// Access to string implementation buffer as "C" character array
	char FAR * GetBuffer(int nMinBufLength);
	void ReleaseBuffer(int nNewLength = -1);
	char FAR * GetBufferSetLength(int nNewLength);

// Implementation
protected:
	// lengths/sizes in characters
	//  (note: an extra character is always allocated)
	char FAR * m_pchData;       // actual string (zero terminated)
	int m_nDataLength;          // does not include terminating 0
	int m_nAllocLength;         // does not include terminating 0

	// implementation helpers
	void Init();
	void AllocCopy(CMOString& dest, int nCopyLen, int nCopyIndex, int nExtraLen) const;
	void AllocBuffer(int nLen);
	void AssignCopy(int nSrcLen, const char FAR * pszSrcData);
	void ConcatCopy(int nSrc1Len, const char FAR * pszSrc1Data, int nSrc2Len, const char FAR * pszSrc2Data);
	void ConcatInPlace(int nSrcLen, const char FAR * pszSrcData);
};


// Compare helpers
BOOL AFXAPI operator==(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator==(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator==(const char FAR * s1, const CMOString& s2);
BOOL AFXAPI operator!=(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator!=(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator!=(const char FAR * s1, const CMOString& s2);
BOOL AFXAPI operator<(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator<(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator<(const char FAR * s1, const CMOString& s2);
BOOL AFXAPI operator>(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator>(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator>(const char FAR * s1, const CMOString& s2);
BOOL AFXAPI operator<=(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator<=(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator<=(const char FAR * s1, const CMOString& s2);
BOOL AFXAPI operator>=(const CMOString& s1, const CMOString& s2);
BOOL AFXAPI operator>=(const CMOString& s1, const char FAR * s2);
BOOL AFXAPI operator>=(const char FAR * s1, const CMOString& s2);

#define _AFX_INLINE inline

_AFX_INLINE int CMOString::GetLength() const
	{ return m_nDataLength; }
_AFX_INLINE BOOL CMOString::IsEmpty() const
	{ return m_nDataLength == 0; }
_AFX_INLINE CMOString::operator lp_cc () const
	{ return (const char FAR *)m_pchData; }
_AFX_INLINE int CMOString::Compare(const char FAR * psz) const
	{ return _fstrcmp(m_pchData, psz); }
_AFX_INLINE int CMOString::CompareNoCase(const char FAR * psz) const
	{ return _fstricmp(m_pchData, psz); }
//	_AFX_INLINE int CMOString::Collate(const char FAR * psz) const
//		{ return strcoll(m_pchData, psz); }		//	Need a model-independent version
_AFX_INLINE void CMOString::MakeUpper()
	{ _fstrupr(m_pchData); }
_AFX_INLINE void CMOString::MakeLower()
	{ _fstrlwr(m_pchData); }
// Windows version in AFXWIN.H
_AFX_INLINE void CMOString::MakeReverse()
	{ _fstrrev(m_pchData); }
_AFX_INLINE char CMOString::GetAt(int nIndex) const
	{
		ASSERT(nIndex >= 0);
		ASSERT(nIndex < m_nDataLength);

		return m_pchData[nIndex];
	}
_AFX_INLINE char CMOString::operator[](int nIndex) const
	{
		// same as GetAt

		ASSERT(nIndex >= 0);
		ASSERT(nIndex < m_nDataLength);

		return m_pchData[nIndex];
	}
_AFX_INLINE void CMOString::SetAt(int nIndex, char ch)
	{
		ASSERT(nIndex >= 0);
		ASSERT(nIndex < m_nDataLength);
		ASSERT(ch != 0);

		m_pchData[nIndex] = ch;
	}
_AFX_INLINE BOOL AFXAPI operator==(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) == 0; }
_AFX_INLINE BOOL AFXAPI operator==(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) == 0; }
_AFX_INLINE BOOL AFXAPI operator==(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) == 0; }
_AFX_INLINE BOOL AFXAPI operator!=(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) != 0; }
_AFX_INLINE BOOL AFXAPI operator!=(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) != 0; }
_AFX_INLINE BOOL AFXAPI operator!=(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) != 0; }
_AFX_INLINE BOOL AFXAPI operator<(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) < 0; }
_AFX_INLINE BOOL AFXAPI operator<(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) < 0; }
_AFX_INLINE BOOL AFXAPI operator<(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) > 0; }
_AFX_INLINE BOOL AFXAPI operator>(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) > 0; }
_AFX_INLINE BOOL AFXAPI operator>(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) > 0; }
_AFX_INLINE BOOL AFXAPI operator>(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) < 0; }
_AFX_INLINE BOOL AFXAPI operator<=(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) <= 0; }
_AFX_INLINE BOOL AFXAPI operator<=(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) <= 0; }
_AFX_INLINE BOOL AFXAPI operator<=(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) >= 0; }
_AFX_INLINE BOOL AFXAPI operator>=(const CMOString& s1, const CMOString& s2)
	{ return s1.Compare(s2) >= 0; }
_AFX_INLINE BOOL AFXAPI operator>=(const CMOString& s1, const char FAR * s2)
	{ return s1.Compare(s2) >= 0; }
_AFX_INLINE BOOL AFXAPI operator>=(const char FAR * s1, const CMOString& s2)
	{ return s2.Compare(s1) <= 0; }


inline ostream & operator << (ostream & os, 
                              const CMOString & str)
	{	//	for some reason, operator << does not like FAR pointers
		//	so give it a near pointer.  Inefficient, but it works!
	char buffer [256];
	ASSERT (str.m_nDataLength < 256);
	_fstrcpy (buffer, str.m_pchData);
	return os << buffer;
	}

#endif // _CMOSTR_H_
