/* errmgr.hpp - error manager declarations  */

/* this is a mechanism for handling error messages that is transparent to */
/* the environment under which the system is running. */

/* there are three classes of error: absolutely fatal, very serious, and */
/* warning only. if a fatal error is triggered (err_mgr.fail()) then the */
/* message is printed and program execution halts. if a serious error is */
/* triggered then the user is asked whether to continue execution. */
/* execution always continues after a warning. */

/* each function prints formatted output in the style of vprintf(). the */
/* message is responsible for any '\n' to be printed; the handler will */
/* not do so. messages may be "keyed" so that their text may easily be */
/* replaced by a user (e.g. internationalization). */

/* there is also a "post" routine for non-error messages. */

#ifndef ERRMGR_HPP                      /* in case of double inclusion */
#define ERRMGR_HPP

#include <stdarg.h>                     /* varargs */
#include "utils.hpp"                    /* ptr_stack */

/* an instance of the failure_handler class is responsible for actually */
/* printing the messages. it doesn't need to look up the message; that's */
/* handled by the manager. a default handler is provided by the manager; */
/* new handlers are stacked. note that any trailing '\n' is the */
/* responsibility of the caller; these routines do not supply them. */

/* the handler may NOT save the pointer fmt, because the data it points to */
/* is liable to be overwritten at any moment without warning. */

/* handlers may not be copied. */

class failure_handler {                 /* failure handling utility */
    public:
        failure_handler(void) {}
        virtual void fail(const char *fmt,va_list ap);  /* exits */
        virtual void error(const char *fmt,va_list ap);
        virtual void warn(const char *fmt,va_list ap);
        virtual void post(const char *fmt,va_list ap);
    private:
        failure_handler(const failure_handler &other);  /* unimplemented */
        failure_handler &operator =(const failure_handler &other);
};  /* end of class failure_handler() */

/* define_handler() pushes the current handler onto a stack and installs the */
/* argument as the new handler. restore_handler() restores the previous */
/* handler (or returns to the default handler if no more remain). both */
/* routines return the handler just replaced. */

/* the routines fail(), vfail(), error(), verror(), warn(), vwarn(), post(), */
/* vpost(), and message() are like fprintf() but have special semantics: if */
/* the first character of the format string is '$', then the string is */
/* split into a key and a default format string. each error dictionary is */
/* searched for the key (in reverse order of definition). if it is found, */
/* it is compared with the default string to ensure that all of the '%' */
/* format specifiers are in the proper order. if they are, then the */
/* dictionary string is passed to the failure handler. otherwise, a warning */
/* is printed and the default string is passed to the handler instead. */

/* format of a keyed error message: */
/* '$' <identifier> ':' <default format string> */

/* fail(), vfail(), error(), verror(), warn(), vwarn(), post(), and */
/* vpost() print their messages; message() simply returns the text after */
/* the key has been removed and any replacement has been performed. */

/* the system thus runs even if error dictionaries are not provided or are */
/* ill-formed. */

/* all error messages go through a single global handler via these routines. */
/* the idea is that a graphical environment, for example, could have all */
/* text go to a window. it would probably be a good idea to have them go to */
/* a log file as well, regardless of the environment. */

/* since there is only one error_mgr allowed at any one time, copying is */
/* not allowed. */

class error_dict_list;                  /* internal to errmgr.cpp */

class error_mgr {
    public:
        error_mgr(void);
        ~error_mgr(void);

        /* error dictionary management - add a new file to the list of key */
        /* lookups. returns 0 if any errors are detected; the dictionary */
        /* is invalid and will not be referenced. */

        int define_dictionary(const char *filename);

        /* push and pop failure handlers. */

        failure_handler *define_handler(failure_handler *new_handler);
        failure_handler *restore_handler(void);

        /* these call fail(), error(), and warn(), respectively, of the */
        /* current handler. note that they, rather than the handler's */
        /* routines, have the "..." because they'll be calling the handler. */

        /* we have fail() and vfail() rather than two fail() routines */
        /* because of potential amibiguities when va_list is a (char *) */
        /* pointer. a call with two (char *) arguments would go to the */
        /* va_list function by mistake. thanks to Turbo C++ for pointing */
        /* this out - "..." is the last possible match. */

        void fail(const char *fmt,...);
        void vfail(const char *fmt,va_list ap);
        void error(const char *fmt,...);
        void verror(const char *fmt,va_list ap);
        void warn(const char *fmt,...);
        void vwarn(const char *fmt,va_list ap);
        void post(const char *fmt,...);
        void vpost(const char *fmt,va_list ap);

        /* get the replacement text for the keyed message. this is useful */
        /* for returning string-valued messages that will then be used in */
        /* something sent to the user. the result is written into msg_line */
        /* and a pointer to msg_line.buf() is returned so that this function */
        /* can be used in a printf() statement. */

        const char *message(const char *fmt,char *msg_line,int linelen);

        /* turn assertions on or off at runtime. they're always compiled */
        /* into the code; setting "assertions" true or false in the boot */
        /* file enables them or disables them, respectively. assertions are */
        /* on by default unless the code is compiled with NDEBUG set. */
        /* returns the previous value of the assertion flag. */

        int set_assert_flag(int asserts_on);

        /* for assertion checking: returns true, short-circuiting */
        /* assertion evaluation, when assertions are off. */

        static int assertions_off(void) { return asserts_off; }

        /* a special routine for assertion failures. don't call it directly; */
        /* use ASSERT (defined below). */

        int assert_failed(const char *exp,const char *fname,unsigned linenum);

    private:
        void setup(void);               /* takes place of constructor */

        /* the private variables are all static because there is no */
        /* constructor to ensure they are built before we use them. */
        /* they're pointers, allocated off the heap, for the same */
        /* reason. */

        static failure_handler *curr_handler,*default_handler;
        static int is_set_up,asserts_off;
        static error_dict_list *error_dicts;
        static ptr_stack *handler_stack;
        int find_replacement(const char *key,char *msg_line,int linelen);
        error_mgr(const error_mgr &other);  /* unimplemented */
        error_mgr &operator =(const error_mgr &other);
};  /* end of class error_mgr */


/* this is the only instance of error_mgr which may be defined. all error */
/* and warning messages go through it. */

extern error_mgr err_mgr;


/* ASSERT() works just like assert() except that it calls err_mgr.error() */
/* through err_mgr.assert_failed() instead of just printing to stderr and */
/* exiting. this of course gives the user the opportunity of continuing at */
/* his/her own risk. */

/* it is considered bad form to call err_mgr.assert_failed() directly. */
/* ASSERT() is a single string that needs to be searched for and it has */
/* well-defined semantics. the semantics of err_mgr.assert_failed() are */
/* intentionally undefined. */

/* in this system assertions are always compiled into the code; they may */
/* be turned off at run time if desired. */

#undef ASSERT

#define ASSERT(e)  ((void)(err_mgr.assertions_off() || (e) || \
                           err_mgr.assert_failed(#e,__FILE__,__LINE__)))

#endif  /* ERRMGR_HPP */
