//file: BitArr.h ---------------------
#ifndef BITARR_H
#define BITARR_H
#include "Boolean.h"
#include <assert.h>
#include <iostream.h>

// Class of temporary objects to reference bit fields
class BitField{
public:
	BitField(unsigned long *data, unsigned int pos, unsigned width):
	  _data(data), _pos(pos), _width(width)
	{
	  assert(_pos+_width<=32);
	}
	operator unsigned long(){
	  unsigned long mask = 0xffffffff >> (32 - _width);
	  unsigned long result = (*_data >> _pos) & mask;
	  return result;
	}

	//Let the compiler supply these as necessary:
	//  BitField(const BitField& src);
	//  BitField& operator=(const BitField& rhs);
	//  ~BitField();

	// Assignment operator.  This is the code that deposits
	//   an integer value into the packed array of bits.
	BitField& operator=(unsigned long rhs){
	  unsigned long mask = 0xffffffff >> (32 - _width);
	  rhs &= mask;
	  rhs <<= _pos;
	  mask <<= _pos;
	  *_data &= ~mask;
	  *_data |= rhs;
	  return *this;
	}
private:
	unsigned long *_data;
	int _pos;
	int _width;
};

class BitArray{
public:
	// Default Constructor.
	//  Optional arguments:
	//   width:  number of bits in an array element
	//   val:    initial value of representation of bit array
	BitArray(unsigned int width=1, unsigned long val=0):
	 _len(32), _data(val),_width(width){
	 assert(_width <= _len);
	}

	//Let the compiler supply these as necessary:
	//  BitArray(const BitArray& src);
	//  BitArray& operator=(const BitArray& rhs);
	//  ~BitArray();

	// Return the number of bits in an array element
	unsigned int width() const {
	  return _width;
	}

	// Return the index of the first element in the array
	unsigned int first() const {
	  return 0;
	}

	// Return the index of the last element in the array
	unsigned int last() const {
	  return _len/_width-1;
	}

	// Return the number of bits in the array
	unsigned int length()const{
	  return _len;
	}

	// Fetch an array element
	unsigned long operator[](unsigned int s) const {
	  s*=_width;
	  assert(s+_width<=_len);
	  unsigned long mask = 0xffffffff >> (_len - _width);
	  unsigned long result = _data;
	  result >>= s;
	  result &= mask;
	  return result;
	}

	// Possibly deposit a value into an array element.
	//  Defer actual deposit operation to BitField::operator=
	BitField operator[](unsigned int s) {
	  return BitField(&_data, s*_width, _width);
	}

private:
	unsigned long _data;	// Actual representation of BitArray
	unsigned int _len;  	// number of bits in representation
	unsigned int _width;	// number of bits in a single array
		//     element
};

#endif

