/************** Associative Array Module *************/
/*   Entry Points: aa_create, aa_resize, aa_addr,    */
/*                 aa_keys, aa_delete, aa_defined    */
/*   Test Platforms: HP-UX V9.01 on HP 9000/715      */
/*                   PC-DOS 3.1 with Turbo C++       */
/*****************************************************/
#include "aa.h"

#define DELETED_KEY 1
#define MAX_KEY_SIZE 1024
#define NULLKEY(addr) (addr == NULL)
#define MAXFULL(size) (size * 0.75)
#define TOO_FULL(count,max) \
  ((count > MAXFULL(max)) ? TRUE : FALSE)
#define KEY_MATCH(type,key,ptr,size) \
  (type!=STRING_KEY && memcmp(key,*ptr,size)==0)
#define STRING_KEY_MATCH(t,k,p) \
  (t==STRING_KEY && strcmp((char *)k,*(char **)p)==0)
#define VALID_KEY(addr) \
  (addr != (void *)NULL && addr != (void *)DELETED_KEY)
#define INC_SIZE(size) \
  ((size_t)(size * 1.5))
#define HASH_FUNCTION(a,key,size) \
  (*(a->hash_function))(key,a->key_size,size,a->type)
#define KEYSIZE(aa,key) \
  (aa->type==STRING_KEY ? strlen(key)+1 : aa->key_size)
#define KEYCOPY(aa,dest,key) (aa->type==STRING_KEY ? \
  strcpy(*dest,key) : memcpy(*dest,key,aa->key_size))
#define TRUE 1
#define FALSE 0

typedef unsigned char BYTE;
typedef unsigned char BOOLEAN;

static int hash_function();
static size_t prime_size();

/*>>>> Create associative array (AA) definition <<<<*/
AA *aa_create (type,key_size,data_size,size,user_hashf)
enum AA_KEYTYPE type; /* input -- key type */
size_t key_size;      /* input -- max binary key size */
size_t data_size;     /* input -- max data size */
size_t size;          /* input -- initial size */
int (*user_hashf)();  /* input -- optional hash func */
{
  AA * aa;

  /* all AA information stored in structure */
  if (aa=(AA *)malloc(sizeof(AA))) {
     size=prime_size(size);
     aa->keys=(void **)calloc(size, sizeof(void *));
     aa->data=(void **)calloc(size, sizeof(void *));
     if (aa->keys && aa->data) { /*if allocation ok*/
	aa->type=type;
	aa->key_size=key_size;
	aa->max_elements=size;
	aa->current_elements=0;
	aa->data_size=data_size;
	if ((aa->hash_function=user_hashf) == NULL)
	   aa->hash_function=&hash_function;
     } else { /* failure, release memory */
	if (aa->keys) free((void *)aa->keys);
	if (aa->data) free((void *)aa->data);
	free((void *)aa);
	aa=NULL;  /* return NULL on failure */
     }
  }
  return(aa); /* return AA definition */
}

/*>>>> Get 1st prime larger than request size <<<<*/
static size_t prime_size(size)
size_t size;  /* start search for prime at size */
{
  int divisor,i=(size<5 ? 5 : size);
  i=(i % 2) ? i : i + 1; /* start on odd number */
  do {
     /* try each number which could be exact divisor */
     for (divisor=3; divisor <= i/divisor; divisor+=2)
	if ((i % divisor) == 0) break; /* if !prime */
     if (divisor > (i/divisor)) break; /* if prime */
  } while (i+=2);  /* try next odd number */
  return(i);  /* return prime number */
}

/*>>>>>>>>>>>> Default hash function <<<<<<<<<<<<<<<*/
static int hash_function(key,keysize,maxhash,type)
void *key;        /* input -- key to be hashed */
int keysize;      /* input -- size of key */
int maxhash;      /* input -- max legal hash value */
enum AA_KEYTYPE type; /* input -- type of input key */
{
  int i;
  static shifts_initialized=FALSE;
  static BYTE svalues[]={0,24,3,21,5,19,7,17,9,15,11};
  static BYTE sconstants[AA_MAX_KEY_SIZE];
  register unsigned long count=0;
  register BYTE *kp=(BYTE *)key, *end;
  register BYTE *shifts=sconstants;

  /* on 1st call init shifts; empirically determined */
  if (!shifts_initialized) {
     for(i=0; i < AA_MAX_KEY_SIZE; i++)
	sconstants[i]=svalues[i % sizeof(svalues)];
     shifts_initialized=TRUE;
  }

  /* the loops are high runners; need to be fast */
  if (type == STRING_KEY)
     for ( ; *kp != (char) 0; kp++,shifts++)
	count ^= *kp << *shifts;
  else
     for (end=kp+keysize; kp < end; kp++,shifts++)
	count ^= *kp << *shifts;
  return (count % maxhash); /* return hash value */
}

/*>>> Find index for key; Use hash algorithm.  <<<*/
/*>>> Method: Linear Probe with Double Hashing <<<*/
static int hashindex (aa,key)
AA * aa;        /* input -- AA definition */
void *key; 	/* input -- lookup key */
{
  int found=FALSE, index;
  void **ptr;
  BOOLEAN first_collision=TRUE;

  /* hash function determines where to start search */
  index=HASH_FUNCTION(aa,key,aa->max_elements);

  do {  /* do until matching key or empty slot found */
     ptr=&aa->keys[index];  /* get key pointer */

     if (NULLKEY(*ptr)) 
	found=TRUE; /* empty slot */
     /* keys comparisons must consider the key type */
     else if (STRING_KEY_MATCH(aa->type,key,ptr))
        found=TRUE; /* string key found */
     else if (KEY_MATCH(aa->type,key,ptr,aa->key_size))
	found=TRUE; /* non-string key found */
     /* wrong key was found so collision occurred */
     else if (first_collision) {
	/* try double hash on first collision only */
	index=HASH_FUNCTION(aa,key,aa->max_elements-4);
	first_collision=FALSE;
     } else /* collision -- use linear search/probe */
	if (++index == aa->max_elements) /* if end */
	   index=0; /* restart search at beginning */
  } while (!found);

  return(index);  /* return key and data index */
}

/*>>>>>>>>>>>>>>>>>> Resize the AA <<<<<<<<<<<<<<<<<*/
BOOLEAN aa_resize (aa,newsize)
AA * aa;         /* input -- AA definition */
size_t newsize;  /* input -- new size for AA */
{
  BOOLEAN rc=TRUE;
  void **newkeys, **newdata, **oldkeys, **olddata;
  int i, oldsize, index;

  newsize=prime_size(newsize); /* insure prime size */
  /* alloc new/enlarged key and data tables */
  newkeys=(void **)calloc(newsize, sizeof(void *));
  newdata=(void **)calloc(newsize, sizeof(void *));
  if ((aa->current_elements < MAXFULL(newsize)) &&
      (newkeys && newdata)) {
     oldkeys=aa->keys;  /* save old table ptrs */
     olddata=aa->data;
     oldsize=aa->max_elements;
     aa->keys=newkeys;  /* put new tables in AA def */
     aa->data=newdata;
     aa->max_elements=newsize;

     /* now add/rehash all keys/data into new table */
     for (i=0; i<oldsize; i++) /*scan for valid keys*/
	if (VALID_KEY(oldkeys[i])) {
	   index=hashindex(aa,oldkeys[i]);
	   aa->keys[index]=oldkeys[i];
	   aa->data[index]=olddata[i];
	}
     free((void *)oldkeys); free((void *)olddata);
  } else {  /* cleanup on error */
     if (newkeys) free((void *)newkeys);
     if (newdata) free((void *)newdata);
     rc=FALSE;
  }
  return(rc);  /* return status, TRUE if successful */
}

/*>>>> Return data pointer for specified key <<<<*/
/* Main interface to store and retrieve AA data */
void *aa_addr(aa,key)
AA * aa;        /* input -- AA definition */
void *key;      /* input -- lookup key */
{
  int index;
  void **keyptr, **dataptr;

  if (TOO_FULL(aa->current_elements,aa->max_elements))
     if (!aa_resize(aa,INC_SIZE(aa->max_elements)))
	fprintf(stderr,"AA: resize failed\n");

  index=hashindex(aa,key); /* index for key & data */
  keyptr=&aa->keys[index];
  dataptr=&aa->data[index];
  if (!VALID_KEY(*keyptr)) { /* if key not in table */
     /* allocate key & data memory; copy/define key */
     *keyptr=(void *)malloc(KEYSIZE(aa,key));  
     *dataptr=(void *)calloc(aa->data_size,1);
     if (keyptr && dataptr)  { /* if allocation ok */
        KEYCOPY(aa,keyptr,key);
        aa->current_elements++;
     } else { /* allocation failed */
	fprintf(stderr,"AA: key allocation failed\n");
	fflush(stderr); /* probably will crash soon */
     }
  }
  return(*dataptr);  /* return pointer to data area */
}

/*>>>>>>>>>>>>> Return all keys in AA <<<<<<<<<<<<<<<*/
void aa_keys(aa,keys,num)
AA * aa;        /* input -- AA definition */
void ***keys;   /* output -- array of key pointers */
int *num;       /* output -- number of keys returned */
{
  register int i;
  register void **ptr;
  size_t buf_size=aa->current_elements*sizeof(void **);

  if (*keys=(void **)malloc(buf_size)) {
     *num=0; ptr=aa->keys;
     for (i=0; i<aa->max_elements; i++,ptr++)
	if (VALID_KEY(*ptr))
	   (*keys)[(*num)++]=*ptr;
  } else /* allocation failed */
     *num=-1;
}

/*<<<<<<<<<< Delete an entry from the AA <<<<<<<<<<<*/
void aa_delete(aa,key)
AA * aa;      /* input -- AA definition */
void * key;   /* input -- key to delete, with data */
{
   int index=hashindex(aa,key);
   if (VALID_KEY(aa->keys[index])) {
	free ((void *)aa->keys[index]);
	aa->keys[index]=(void *)DELETED_KEY;
	free ((void *)aa->data[index]);
	aa->data[index]=NULL;
	aa->current_elements--;
   }
}

/*>>>>>> Return TRUE if key defined in the AA <<<<<<<*/
BOOLEAN aa_defined(aa,key)
AA * aa;        /* input -- pointer to AA definition */
void * key;     /* input -- key to check if defined */
{
   return(VALID_KEY(aa->keys[hashindex(aa,key)]));
}
