/*==========================================================================*\

   stdio.c  (With apologies to the real stdio.c)

      Provides (extremely) minimal I/O capabilities via a serial port;
      useful for debugging in an embedded environment.

   Donated to the public domain by Jeff D. Pipkins, who of course is
   not liable for damages...

\*--------------------------------------------------------------------------*/

#define PUBLIC
#define PRIVATE static

/*--------------------------------------------------------------------------*\
   82050-compatible UART
\*--------------------------------------------------------------------------*/

PRIVATE unsigned int uart      = 0x3F8;      /* base address of UART */
PRIVATE unsigned int baud_rate =  1200;

#define UART_DATA       (uart)
#define UART_IER        (uart+1)    /* Interrupt Enable Reg.         */
#define UART_IIR        (uart+2)    /* Interrupt Identification Reg. */
#define UART_LCR        (uart+3)    /* Line Control Reg.             */
#define UART_MCR        (uart+4)    /* Modem Control Reg.            */
#define UART_LSR        (uart+5)    /* Line Status Reg.              */
#define UART_MSR        (uart+6)    /* Modem Status Reg.             */
#define UART_SCR        (uart+7)    /* Scratchpad Reg.               */
#define UART_BAL        UART_DATA
#define UART_BAH        UART_IER

#define UART_CLOCK      (115200uL)

/*==========================================================================*\

   init_stdio()

      Must be called before any other function in this module!

\*--------------------------------------------------------------------------*/

PUBLIC void init_stdio(void)
{
   unsigned int baud_divisor;

   /* Compute divisor from baud rate and clock. */

   baud_divisor = (unsigned) (UART_CLOCK / (unsigned long)baud_rate);

   /* initialize UART */

   outp(UART_LCR, 0x83);   /* get ready to set baud rate... */
   outp(UART_BAL, baud_divisor & 0x00FFu);
   outp(UART_BAH, baud_divisor >> 8);
   outp(UART_LCR, 0x03);   /* 8 data bits, 1 stop bit, no parity */
   outp(UART_IER, 0x00);   /* no interrupts for any reason */
   outp(UART_MCR, 0x03);   /* raise RTS & DTR */
   inp(UART_LSR);          /* clear any pending receiver errors */

}


/*==========================================================================*\

   putchar()

\*--------------------------------------------------------------------------*/

PUBLIC void putchar(int ch)
{
   if (ch == '\n') {       /* if sending a newline (LF), send a CR first */
      putchar('\r');
   }

   while ((inp(UART_LSR) & 0x20) == 0) {
      /* Wait for tranmission holding register to be empty */
   }

   outp(UART_DATA, ch);    /* transmit one character */
}


/*==========================================================================*\

   puts()

\*--------------------------------------------------------------------------*/

PUBLIC void puts(char *s)
{
   while (*s) {
      putchar(*s);
      s++;
   }
   putchar('\n');
}

/*==========================================================================*\
   End of source file.
\*==========================================================================*/


