/* pagefuncs.c: HTML `functions' can be executed with page_process_page (). */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Tue Jul 18 17:50:42 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#define COMPILING_PAGEFUNCS_C 1
#define LANGUAGE_DEFINITIONS_FILE 1

#if defined (HAVE_CONFIG_H)
#  include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <regex.h>
#include <setjmp.h>
#include <sys/types.h>
#if defined (Solaris)
#  include <ucbinclude/sys/fcntl.h>
#  include <ucbinclude/sys/file.h>
#else
#  include <sys/file.h>
#endif /* !Solaris */
#include <time.h>
#include <math.h>

#include <bprintf/bprintf.h>
#include <xmalloc/xmalloc.h>
#include <wisper/wisp.h>
#include "database.h"
#include "session_data.h"
#include "pages.h"
#include "parser.h"

#if defined (MHTML_STREAMS)
#  include "streamfuncs.h"
#endif /* MHTML_STREAMS */

#include <locking.h>

#if !defined (HAVE_SRANDOM)
#  define srandom(seed) srand (seed)
#  define random() rand()
#endif

extern void initialize_external_functions (Package *p);

#if defined (macintosh)
extern char *strdup (const char *string);
#  define os_open(name, flags, mode) open (name, flags)
#else
#  define os_open(name, flags, mode) open (name, flags, mode)
#endif

char *metahtml_copyright_string = "
Copyright (C) 1995, 1996, Brian J. Fox\n\
Copyright (C) 1996, Universal Access Inc.";

/* The list of variables that have been set. */
static Package *PageVars = (Package *)NULL;

/* Globally known variable holds onto to the reserved words. */
Package *pagefunc_function_package = (Package *)NULL;

/* Macro writing and processing. */
Package *mhtml_user_keywords = (Package *)NULL;

/* How to lookup the user function object associated with NAME. */
static UserFunction *find_user_function (char *name);

/* Here is how to execute a subst or macro. */
static void execute_user_function (UserFunction *uf, PFunArgs, char *attr);

/* If you want to delete a package, you should probably call this function
   rather than calling symbol_destroy_package () from symbols.c.  This 
   allows the engine to reset a bunch of internal variables if necessary. */
void
pagefunc_destroy_package (char *package_name)
{
  Package *package = symbol_lookup_package (package_name);

  if (package != (Package *)NULL)
    {
      if (package == PageVars)
	PageVars = (Package *)NULL;
      else if (package == pagefunc_function_package)
	pagefunc_function_package = (Package *)NULL;
      else if (package == mhtml_user_keywords)
	mhtml_user_keywords = (Package *)NULL;

      symbol_destroy_package (package);
    }
}

/* For internal use only.  Returns the zeroith value element for
   NAME in PACKAGE. */
char *
get_value (Package *package, char *name)
{
  char *value = (char *)NULL;

  if (package != (Package *)NULL)
    {
      Symbol *sym = symbol_lookup_in_package (package, name);

      if (sym && sym->values_index)
	value = sym->values[0];
    }
  return (value);
}

/* Return the values list of *pvars* in PACKAGE. */
char **
get_vars_names (Package *package)
{
  char **names = (char **)NULL;

  if (package != (Package *)NULL)
    {
      Symbol *sym = symbol_lookup_in_package (package, "*pvars*");

      if (sym != (Symbol *)NULL)
	names = sym->values;
    }

  return (names);
}

/* Return the values list of *pvals* in PACKAGE. */
char **
get_vars_vals (Package *package)
{
  char **vals = (char **)NULL;

  if (package != (Package *)NULL)
    {
      Symbol *sym = symbol_lookup_in_package (package, "*pvals*");

      if (sym != (Symbol *)NULL)
	vals = sym->values;
    }

  return (vals);
}

void
pagefunc_set_variable (char *tag, char *value)
{
  Package *orig_package = CurrentPackage;

  if (PageVars == (Package *)NULL)
    PageVars = symbol_get_package (DEFAULT_PACKAGE_NAME);

  if (CurrentPackage == (Package *)NULL)
    symbol_set_default_package (PageVars);

  forms_set_tag_value (tag, value);
  symbol_set_default_package (orig_package);
}

void
pagefunc_set_variable_readonly (char *tag, char *value)
{
  Package *orig_package = CurrentPackage;
  Symbol *sym;

  if (PageVars == (Package *)NULL)
    PageVars = symbol_get_package (DEFAULT_PACKAGE_NAME);

  if (CurrentPackage == (Package *)NULL)
    symbol_set_default_package (PageVars);

  forms_set_tag_value (tag, value);
  sym = symbol_lookup (tag);
  if (sym != (Symbol *)NULL)
    symbol_set_flag (sym, sym_READONLY);

  symbol_set_default_package (orig_package);
}

char *
pagefunc_get_variable (char *tag)
{
  Package *orig_package = CurrentPackage;
  char *value;

  if (PageVars == (Package *)NULL)
    PageVars = symbol_get_package ("default");

  if (CurrentPackage == (Package *)NULL)
    symbol_set_default_package (PageVars);

  value = forms_get_tag_value (tag);

  symbol_set_default_package (orig_package);
  return (value);
}

char *
get_one_of (Package *package, char *tag, ...)
{
  char *value = (char *)NULL;
  va_list args;

  va_start (args, tag);

  while (tag)
    {
      value = forms_get_tag_value_in_package (package, tag);

      if (value)
	break;

      tag = va_arg (args, char *);
    }

  va_end (args);
  return (value);
}

char *
get_positional_arg (Package *package, int position)
{
  char *result = (char *)NULL;
  int pos = 0;

  if (package != (Package *)NULL)
    {
      Symbol *pvars = symbol_lookup_in_package (package, "*pvars*");

      if (pvars != (Symbol *)NULL)
	{
	  register int i;
	  Symbol *sym;

	  for (i = 0; i < pvars->values_index; i++)
	    {
	      sym = symbol_lookup_in_package (package, pvars->values[i]);

	      if ((sym != (Symbol *)NULL) && (sym->values_index == 0))
		{
		  if (position == pos)
		    {
		      result = pvars->values[i];
		      break;
		    }
		  else
		    pos++;
		}
	    }
	}
    }
  return (result);
}

char *
read_sexp_1 (char *string, int *start, int stop_at_equals_p, int one_list)
{
  static char *workspace = (char *)NULL;
  static int wsize = 0;
  int expr_present = 0;
  char *result = (char *)NULL;

  if (string != (char *)NULL)
    {
      register int i = *start;
      register int string_len = strlen (string);
      int windex, gobbled, quoted, depth;

      windex = gobbled = quoted = depth = 0;

      if (string_len >= wsize)
	workspace = (char *)xrealloc (workspace, (wsize = 10 + string_len));

      workspace[0] = '\0';

      /* Skip leading whitespace. */
      while (whitespace (string[i])) i++;

      gobbled = 0;
      while (!gobbled)
	{
	  register int c = string[i++];

	  switch (c)
	    {
	    case '\\':
	      c = string[i++];

	      if (depth == 0)
		{
		  switch (c)
		    {
		    case 'n':
		      workspace[windex++] = '\n';
		      break;

		    case 't':
		      workspace[windex++] = '\t';
		      break;

		    case 'r':
		      workspace[windex++] = '\r';
		      break;

		    case 'f':
		      workspace[windex++] = '\f';
		      break;

		    case '\0':
		      workspace[windex] = '\\';
		      gobbled++;
		      break;

		    default:
		      workspace[windex++] = c;
		      break;
		    }
		}
	      else
		{
		  /* Skip the backslash, and the character which follows it.
		     We have to do this for the case of bizarre constructs,
		     such as <get-var <get-var \>>>. */
		  if (c != '\0')
		    {
		      workspace[windex++] = '\\';
		      workspace[windex++] = c;
		    }
		  else
		    {
		      workspace[windex] = '\\';
		      gobbled++;
		    }
		}
	      break;

	    case '<':
	      workspace[windex++] = '<';
	      if (!quoted)
		depth++;
	      break;

	    case '>':
	      workspace[windex++] = '>';
	      if (!quoted)
		{
		  depth--;
		  if (one_list && (depth == 0))
		    {
		      workspace[windex] = '\0';
		      gobbled++;
		    }
		}
	      break;

	    case '"':
	      quoted = !quoted;
	      if (depth)
		workspace[windex++] = '"';
	      else
		expr_present++;
	      break;

	    case ' ':
	    case '\t':
	    case '\n':
	      if (!quoted && depth <= 0)
		{
		  workspace[windex] = '\0';
		  gobbled++;
		}
	      else
		workspace[windex++] = c;
	      break;

	    case '=':
	      if (stop_at_equals_p && !quoted && depth <= 0)
		{
		  workspace[windex] = '\0';
		  gobbled++;
		  i--;
		}
	      else
		workspace[windex++] = c;
	      break;

	    case '\0':
	      workspace[windex] = '\0';
	      gobbled++;
	      i--;
	      break;

	    default:
	      workspace[windex++] = c;
	      break;
	    }
	}

      if (windex || expr_present)
	result = strdup (workspace);

      *start = i;
    }

  return (result);
}

char *
read_sexp (char *string, int *start, int stop_at_equals_p)
{
  return (read_sexp_1 (string, start, stop_at_equals_p, 0));
}

/* A misnomer.  This is acutally the argument gatherer for function calls. */
static Package *
pagefunc_snarf_vars (char *string, int allow_assignments_p)
{
  Package *package = (Package *)NULL;
  int offset = 0;
  int string_len;

  if (string == (char *)NULL)
    return (package);
  else
    string_len = strlen (string);

  /* Gobble name and value pairs. */
  while (offset < string_len)
    {
      char *name = (char *)NULL;
      char *value = (char *)NULL;

      name = read_sexp (string, &offset, allow_assignments_p);

      /* Skip any whitespace between the name and the '='
	 starting the value. */
      while (whitespace (string[offset])) offset++;

      /* If there is an equals sign here, get the value string. */
      if (string[offset] == '=')
	{
	  offset++;
	  if (name)
	    value = read_sexp (string, &offset, 0);
	}

      if (!name)
	continue;

      /* Add this pair to our list. */
      if (package == (Package *)NULL)
	package = symbol_get_package ((char *)NULL);

      if (value == (char *)NULL)
	symbol_intern_in_package (package, name);
      else
	forms_set_tag_value_in_package (package, name, value);

      /* Add the name and value to the list of ordered variables. */
      {
	Symbol *symbol = symbol_intern_in_package (package, "*pvars*");
	symbol_add_value (symbol, name);
	symbol = symbol_intern_in_package (package, "*pvals*");
	symbol_add_value (symbol, value ? value : "");
      }

      free (name);
      if (value) free (value);
    }

  return (package);
}

PFunDesc *
pagefunc_get_descriptor (char *tag)
{
  PFunDesc *desc = (PFunDesc *)NULL;
  Symbol *sym;

  if (pagefunc_function_package == (Package *)NULL)
    {
      register int i;

      pagefunc_function_package = symbol_get_package ("*meta-html*");

      for (i = 0; pagefunc_table[i].tag != (char *)NULL; i++)
	{
	  sym = symbol_intern_in_package
	    (pagefunc_function_package, pagefunc_table[i].tag);
	  sym->type = symtype_FUNCTION;
	  sym->values = (char **)(&pagefunc_table[i]);
	}
      initialize_external_functions (pagefunc_function_package);
    }

  sym = symbol_lookup_in_package (pagefunc_function_package, tag);

  if (sym != (Symbol *)NULL)
    desc = (PFunDesc *)(sym->values);

  return (desc);
}

/* Special case code can throw out from multiple levels deep in order to
   immediately return some HTTP.  You should call the following function
   page_return_this_page (page) in order to make that happen. */
static PAGE *ImmediatePage = (PAGE *)NULL;
jmp_buf page_jmp_buffer;

void
page_return_this_page (PAGE *page)
{
  ImmediatePage = page_copy_page (page);
  longjmp (page_jmp_buffer, 1);
}

/* Sequentialy process PAGE. */
static PAGE *ThePage = (PAGE *)NULL;
static int TheOffset = 0;

PageEnv *
pagefunc_save_environment (void)
{
  PageEnv *env = (PageEnv *)xmalloc (sizeof (PageEnv));

  memcpy (&(env->env), &page_jmp_buffer, sizeof (jmp_buf));
  env->page = ThePage;
  env->offset = TheOffset;

  return (env);
}

void
pagefunc_restore_environment (PageEnv *env)
{
  memcpy (&page_jmp_buffer, &(env->env), sizeof (jmp_buf));
  ThePage = env->page;
  TheOffset = env->offset;

  free (env);
}

/* Gets 1 when mhtml::inhibit-comment-parsing has a value, 0 otherwise. */
static int mhtml_inhibit_comment_parsing = 0;

/* Gets 1 when mhtml::decimal-places has a value, 0 otherwise. */
static int mhtml_decimal_notify = 0;
static int mhtml_decimal_places = 2;

void
pagefunc_initialize_notifiers (void)
{
  Symbol *sym;

  sym = symbol_intern ("mhtml::inhibit-comment-parsing");
  symbol_notify_value (sym, &mhtml_inhibit_comment_parsing);
  sym = symbol_intern ("mhtml::decimal-places");
  symbol_notify_value (sym, &mhtml_decimal_notify);
}

static int syntax_checking = 0;
static int syntax_failure = 0;

int
page_check_syntax (PAGE *page)
{
  int syntax_ok;

  syntax_checking = 1;
  syntax_failure = 0;
  page_process_page_internal (page);
  syntax_ok = !syntax_failure;
  syntax_checking = 0;
  syntax_failure = 0;
  return (syntax_ok);
}

void
page_process_page (volatile PAGE *page)
{
  static int notifiers_initialized = 0;

  if (!notifiers_initialized)
    {
      pagefunc_initialize_notifiers ();
      notifiers_initialized++;
    }
  else
    {
      forms_gc_pointers ();
    }

  ImmediatePage = (PAGE *)NULL;
  ThePage = (PAGE *)page;
  TheOffset = 0;

  /* The ugliest hack in the world.  Please shoot me. */
  if (setjmp (page_jmp_buffer) != 0)
    {
      page->buffer = ImmediatePage->buffer;
      page->bindex = ImmediatePage->bindex;
      page->bsize  = ImmediatePage->bsize;
    }
  else
    page_process_page_internal ((PAGE *)page);
}

void
page_process_page_internal (PAGE *page)
{
  register int i, c;
  int search_start = 0;
  int done = 0;
  int semicolon_comments;
  static char *fname = (char *)NULL;
  static int fname_size = 0;

  if (page == (PAGE *)NULL)
    return;

  while (!done)
    {
      PFunDesc *desc = (PFunDesc *)NULL;
      static PFunDesc uf_desc;
      UserFunction *uf = (UserFunction *)NULL;

      semicolon_comments = !mhtml_inhibit_comment_parsing;

      for (i = search_start; i < page->bindex; i++)
	{
	  if (page->buffer[i] == '<')
	    break;

	  /* If there is a semicolon comment here, ignore it now. */
	  if (semicolon_comments && page->buffer[i] == ';')
	    {
	      if (((i + 2) < page->bindex) &&
		  (page->buffer[i + 1] == ';') &&
		  (page->buffer[i + 2] == ';'))
		{
		  int marker = i;
		  i += 3;
		  while (i < page->bindex && page->buffer[i] != '\n') i++;
		  bprintf_delete_range (page, marker, i + 1);
		  i = marker - 1;
		}
	    }
	}

      if (i >= page->bindex)
	{
	  done = 1;
	  continue;
	}
      else
	{
	  int fname_beg;
	  int fname_end;
	  int fname_len;

	  search_start = i;
	  fname_beg = ++i;

	  for (; (c = page->buffer[i]) != '\0'; i++)
	    if ((c == ' ') || (c == '>') || (c == '\t') || (c == '\n'))
	      break;

	  if (!c)
	    {
	      search_start++;
	      continue;
	    }

	  fname_end = i;
	  fname_len = fname_end - fname_beg;

	  if (fname_len + 4 > fname_size)
	    fname = (char *)xrealloc (fname, fname_size += (20 + fname_len));

	  strncpy (fname, page->buffer + fname_beg, fname_len);
	  fname[fname_len] = '\0';

	  /* Look for a user-defined command before a static one. */
	  uf = find_user_function (fname);
	  if (uf)
	    {
	      desc = &uf_desc;
	      desc->tag = uf->name;
	      desc->complexp = (uf->type == user_MACRO);
	      desc->debug_level = uf->debug_level;
	      desc->fun = (PFunHandler *)NULL;
	    }

	  /* Find the description of this function, so we know how to find
	     it in the page. */
	  if (!desc)
	    desc = pagefunc_get_descriptor (fname);

	  if (!desc)
	    {
	      search_start++;
	      continue;
	    }
	  else
	    {
	      int start, end;
	      int found;

	      start = search_start;

	      if (desc->complexp)
		found = page_complex_tag_bounds (page, fname, &start, &end);
	      else
		{
		  end = page_find_tag_end (page, start);
		  found = end != -1;
		}

	      if (!found)
		{
		  /* The MTHML programmer didn't close the opener correctly.
		     Ignore the opener, and move on. */
		  page_debug ("Closing tag missing for <%s ...>", desc->tag);
		  if (syntax_checking)
		    {
		      syntax_failure = 1;
		      done = 1;
		    }
		  search_start += fname_len;
		  continue;
		}
	      else
		{
		  char *open_body = (char *)NULL;
		  char *strbody = (char *)NULL;
		  Package *vars = (Package *)NULL;
		  int open_start, open_end, open_body_len;

		  /* For simple and complex tags alike, we want to eat the
		     variables which appear in the opener. */
		  open_start = start;
		  if (desc->complexp)
		    open_end = page_find_tag_end (page, start);
		  else
		    open_end = end;

		  open_body_len = open_end - open_start;
		  open_body = (char *)xmalloc (1 + open_body_len);
		  strncpy (open_body, page->buffer + start, open_body_len);
		  open_body[open_body_len] = '\0';

		  /* Kill the closing '>'. */
		  open_body[open_body_len - 1] = '\0';
		  memmove (open_body, open_body + 1 + fname_len,
			   (open_body_len - (1 + fname_len)));

		  vars = pagefunc_snarf_vars (open_body, uf ? 0 : 1);

		  if (!desc->complexp)
		    strbody = open_body;
		  else
		    {
		      int open_len = open_end - open_start;
		      int body_len, body_end;
		      char *closer = (char *)xmalloc (3 + fname_len);

		      closer[0] = '<';
		      closer[1] = '/';
		      strcpy (closer + 2, desc->tag);

		      strbody = page_complex_tag_extract
			(page, desc->tag, &open_start, &end);

		      /* Get rid of the opening tag. */
		      {
			int extra = 0;
			if (strbody[open_len] == '\n')
			  extra++;

			memmove (strbody, strbody + open_len + extra,
				 (1 + strlen (strbody) - (open_len + extra)));
		      }

		      /* Get rid of the closing tag. */
		      body_len = strlen (strbody);
		      body_end = body_len - (1 + fname_len);

		      while (strncasecmp
			     (strbody + body_end, closer, 1 + fname_len) != 0)
			body_end--;

		      if (body_end > 0 && strbody[body_end - 1] == '\n')
			body_end--;

		      strbody[body_end] = '\0';
		      free (closer);
		    }

		  /* Call the handler function. */
		  if (syntax_checking)
		    {
		      search_start = end;
		    }
		  else
		    {
		      char *display_body = (char *)NULL;
		      PAGE *body;
		      body = page_create_page ();
		      page_set_contents (body, strbody);

		      /* This text is no longer in the page. */
#if defined (BREAK_SEMANTICS)
		      if (page->buffer[end + 1] == '\n')
			bprintf_delete_range (page, start, end + 1);
		      else
#endif
			bprintf_delete_range (page, start, end);

		      if (desc->debug_level > 5)
			{
			  display_body = strdup (open_body ? open_body : "");
			  if (strlen (display_body) > 33)
			    strcpy (display_body + 30, "...");

			  page_debug ("Entering <%s %s>",
				      desc->tag, display_body);
			}
		    
		      if (uf)
			execute_user_function
			  (uf, page, body, vars, start, end, &search_start,
			   desc->debug_level, open_body);
		      else
			(*desc->fun)
			  (page, body, vars, start, end, &search_start,
			   desc->debug_level);

		      if (search_start < 0)
			{
			  page_debug ("PPI: `%s' bashed SEARCH_START!",
				      desc->tag);
			  search_start = page->bindex;
			}

		      if (desc->debug_level > 5)
			{
			  page_debug ("Leaving <%s %s>",
				      desc->tag, display_body);
			  free (display_body);
			}

		      page_free_page (body);
		    }

		  /* Free up the variables and the body. */
		  if (strbody != open_body) free (open_body);
		  symbol_destroy_package (vars);
		  free (strbody);
		}
	    }
	}
    }
}

char *
page_evaluate_string (char *body)
{
  PAGE *evaluated;
  char *result = (char *)NULL;
  int clear_whitespace_p = 0;

  if (!body)
    return ((char *)NULL);

#if defined (NOT_DEF)
  clear_whitespace_p = var_present_p (PageVars, "mhtml::ignore-whitespace");
#endif

  evaluated = page_create_page ();
  page_set_contents (evaluated, body);
  page_process_page_internal (evaluated);
  result = evaluated->buffer;
  free (evaluated);

  /* Strip leading and trailing whitespace from the string.  Yes? */
  if (clear_whitespace_p && (result != (char *)NULL))
    {
      register int i;
      char *temp = result;

      /* Strip leading. */
      while (whitespace (*temp)) temp++;

      if (temp != result)
	memmove (result, temp, 1 + strlen (temp));

      /* Strip trailing. */
      for (i = strlen (result) - 1; i > -1 && whitespace (result[i]); i--);
      if (i > -1)
	{
	  i++;
	  result[i] = '\0';
	}

      /* If there was nothing but whitespace, return the NULL string. */
      if (*result == '\0')
	{
	  free (result);
	  result = (char *)NULL;
	}
    }

  return (result);
}

/* Control debugging on a per-function basis. */
static void
pf_debugging_on (PFunArgs)
{
  if (vars)
    {
      register int i;
      Symbol **symbols = symbols_of_package (vars);
      Symbol *sym;

      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
	{
	  UserFunction *uf = find_user_function (sym->name);
	  PFunDesc *desc = pagefunc_get_descriptor (sym->name);

	  if (uf || desc)
	    {
	      int new_debug_level = 1;

	      if (sym->values && sym->values[0])
		new_debug_level = atoi (sym->values[0]);

	      if (uf)
		uf->debug_level = new_debug_level;
	      else
		desc->debug_level = new_debug_level;
	    }
	}
    }
}

/* Evaluate BODY, and add the result to the debugging output. */
static void
pf_page_debug (PFunArgs)
{
  char *value;

  value = page_evaluate_string (body->buffer ? body->buffer : "");
  if (!empty_string_p (value))
    page_debug ("%s", value);
  if (value) free (value);
}

static UserFunction *
find_user_function (char *name)
{
  Symbol *sym = symbol_lookup_in_package (mhtml_user_keywords, name);

  return (sym ? (UserFunction *)(sym->values) : (UserFunction *)NULL);
}

static void
add_user_function (int type, char *name, char *body, Package *vars)
{
  UserFunction *uf = find_user_function (name);
  char *body_whitespace = get_value (vars, "whitespace");
  char *debug_level = get_value (vars, "debug");
  char *wrapper_packname = page_evaluate_string (get_value (vars, "package"));
  char **named_parameters = (char **)NULL;
  int np_size = 0;
  int np_index = 0;

  if (type == user_DEFUN)
    {
      body_whitespace = "delete";
      if (!wrapper_packname)
	wrapper_packname = strdup ("local");
    }

  /* Gather named arguments if present. */
  {
    register int i;
    char *param;

    for (i = 1; (param = get_positional_arg (vars, i)) != (char *)NULL; i++)
      {
	if (np_index + 2 > np_size)
	  named_parameters = (char **) xrealloc
	  (named_parameters, (np_size += 10) * sizeof (char *));

	named_parameters[np_index++] = strdup (param);
	named_parameters[np_index] = (char *)NULL;
      }
  }

  if (empty_string_p (wrapper_packname))
    {
      if (wrapper_packname) free (wrapper_packname);
      wrapper_packname = (char *)NULL;
    }

  if (uf == (UserFunction *)NULL)
    {
      Symbol *sym;

      uf = (UserFunction *)xmalloc (sizeof (UserFunction));
      uf->type = type;
      uf->debug_level = debug_level ? atoi (debug_level) : 0;
      uf->name = strdup (name);
      uf->packname = wrapper_packname;
      uf->named_parameters = named_parameters;
      uf->body = strdup (body ? body : "");

      if (mhtml_user_keywords == (Package *)NULL)
	mhtml_user_keywords = symbol_get_package ("*user-functions*");
      sym = symbol_intern_in_package (mhtml_user_keywords, name);
      sym->values = (char **)uf;
      sym->type = symtype_USERFUN;
    }
  else
    {
      uf->type = type;
      if (uf->packname) free (uf->packname);
      uf->packname = wrapper_packname;
      if (uf->named_parameters)
	{
	  register int i;

	  for (i = 0; uf->named_parameters[i] != (char *)NULL; i++)
	    free (uf->named_parameters[i]);
	  free (uf->named_parameters);
	}
      uf->named_parameters = named_parameters;

      free (uf->body);
      uf->body = strdup (body ? body : "");
    }

  uf->flags = 0;

  /* If the user wants special behaviour for the whitespace present in
     the macro body, then handle it now. */
  if (body_whitespace != (char *)NULL)
    {
      char *b = uf->body;

      if (strcasecmp (body_whitespace, "delete") == 0)
	{
	  register int i, c, l, start;
	  int brace_level = 0;
	  int quote_level = 0;

	  uf->flags |= user_WHITESPACE_DELETED;

	  l = strlen (b);

	  /* Delete all occurences of whitespace outside of 
	     `< ...>' and `" ... "'. */
	  i = 0;
	  while ((c = b[i]) != '\0')
	    {
	      switch (c)
		{
		case '"':
		  quote_level = !quote_level;
		  break;

		case '<':
		  if (!quote_level)
		    brace_level++;
		  break;

		case '>':
		  if (!quote_level)
		    brace_level--;
		  break;

		case '\\':
		  if (b[i + 1])
		    i++;
		  break;

		  /* Handle comments. */
		case ';':
		  if (!quote_level && !mhtml_inhibit_comment_parsing &&
		      ((i + 2) < l) && (b[i + 1] == ';' && b[i + 2] == ';'))
		    {
		      start = i;
		      while (b[i] && b[i] != '\n') i++;
		      memmove (b + start, b + i, 1 + strlen (b + i));
		      i = start - 1;
		    }
		  break;

		case ' ':
		case '\t':
		case '\n':
		  if (((c == ' ' || c == '\t') && (i == 0)) ||
		      (c == '\n' && !quote_level && !brace_level))
		    {
		      start = i;
		      while (whitespace (b[i])) i++;
		      memmove (b + start, b + i, 1 + strlen (b + i));
		      i = start - 1;
		    }
		}
	      i++;
	    }
	}
    }
}

static void
pf_defsubst (PFunArgs)
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = page_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    add_user_function (user_SUBST, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

static void
pf_defmacro (PFunArgs)
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = page_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    add_user_function (user_MACRO, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

static void
pf_defun (PFunArgs)
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = page_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    add_user_function (user_DEFUN, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

static void
pf_undef (PFunArgs)
{
  register int i;
  char *name;

  if (!mhtml_user_keywords)
    return;

  for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
    {
      char *varname = name;

      varname = page_evaluate_string (name);

      if (varname)
	{
	  UserFunction *uf = find_user_function (varname);
	  Symbol *sym;

	  if (uf)
	    {
	      free (uf->body);
	      free (uf->name);
	      free (uf);
	    }

	  sym = symbol_remove_in_package (mhtml_user_keywords, varname);

	  if (sym)
	    {
	      sym->values= (char **)NULL;
	      sym->values_index = 0;
	      symbol_free (sym);
	    }

	  free (varname);
	}
    }
}

/* Get the body of the macro, verbatim. */
static void
pf_function_def (PFunArgs)
{
  if (mhtml_user_keywords == (Package *)NULL)
    return;
  else
    {
      register int i;
      char *name;
    
      for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
	{
	  char *varname = page_evaluate_string (name);

	  if (varname != (char *)NULL)
	    {
	      UserFunction *uf = find_user_function (varname);

	      if (uf)
		{
		  BPRINTF_BUFFER *insertion = bprintf_create_buffer ();

		  if (uf->type == user_MACRO)
		    bprintf (insertion, "<defmacro %s", uf->name);
		  else if (uf->type == user_DEFUN)
		    bprintf (insertion, "<defun %s", uf->name);
		  else if (uf->type == user_SUBST)
		    bprintf (insertion, "<defsubst %s", uf->name);

		  /* Any named parameters? */
		  if (uf->named_parameters != (char **)NULL)
		    {
		      register int j;

		      for (j = 0; uf->named_parameters[j] != (char *)NULL; j++)
			bprintf (insertion, " %s", uf->named_parameters[j]);
		    }


		  /* Any special package name? */
		  if (uf->packname != (char *)NULL)
		    bprintf (insertion, " package=%s", uf->packname);


		  /* Any CR's in body? */
		  if (uf->flags & user_WHITESPACE_DELETED)
		    bprintf (insertion, " whitespace=delete>");
		  else
		    bprintf (insertion, ">\n");

		  bprintf (insertion, "%s", uf->body);

		  if ((uf->flags & user_WHITESPACE_DELETED) == 0)
		    bprintf (insertion, "\n");

		  if (uf->type == user_MACRO)
		    bprintf (insertion, "</defmacro>");
		  else if (uf->type == user_DEFUN)
		    bprintf (insertion, "</defun>");
		  else if (uf->type == user_SUBST)
		    bprintf (insertion, "</defsubst>");

		  bprintf_insert (page, start, "%s\n", insertion->buffer);
		  start += insertion->bindex + 1;
		  bprintf_free_buffer (insertion);
		}

	      free (varname);
	    }
	}

      *newstart = start;
    }
}

/* Here is how to execute a subst or macro. */
static void
execute_user_function (UserFunction *uf, PFunArgs, char *attr)
{
  PAGE *subber = (PAGE *)NULL;

  if (!empty_string_p (uf->body))
    {
      register int i = 0;

      subber = page_create_page ();
      page_set_contents (subber, uf->body);

      /* Process the body. */
      while (i < subber->bindex)
	{
	  for (; (i < subber->bindex) && (subber->buffer[i] != '%'); i++);

	  i++;
	  if (i < subber->bindex)
	    {
	      if (isdigit (subber->buffer[i]))
		{
		  int which = subber->buffer[i] - '0';
		  char *arg = get_positional_arg (vars, which);

		  i--;
		  bprintf_delete_range (subber, i, i + 2);

		  if (!empty_string_p (arg))
		    {
		      char *insertion = page_evaluate_string (arg);

		      if (!empty_string_p (insertion))
			{
			  bprintf_insert (subber, i, "%s", insertion);
			  i += strlen (insertion);
			}

		      if (insertion) free (insertion);
		    }
		}
	      else if (subber->buffer[i] == '\\')
		{
		  bprintf_delete_range (subber, i, i + 1);
		  continue;
		}
	      else if (((subber->bindex - i) > 3) &&
		       (strncasecmp (subber->buffer + i, "BODY", 4) == 0))
		{
		  i--;
		  bprintf_delete_range (subber, i, i + 5);

		  if (!empty_string_p (body->buffer))
		    {
		      bprintf_insert (subber, i, "%s", body->buffer);
		      i += strlen (body->buffer);
		    }
		}
	      else if (((subber->bindex - i) > 9) &&
		       (strncasecmp (subber->buffer + i, "ATTRIBUTES", 10)
			== 0))
		{
		  i--;
		  bprintf_delete_range (subber, i, i + 11);

		  if (!empty_string_p (attr))
		    {
		      bprintf_insert (subber, i, "%s", attr);
		      i += strlen (attr);
		    }
		}
	      else
		i++;
	    }
	}

      if (!empty_string_p (subber->buffer))
	{
	  char *packname = uf->packname;
	  char *parameter_setter = (char *)NULL;

	  if (uf->named_parameters != (char **)NULL)
	    {
	      BPRINTF_BUFFER *wrapper = bprintf_create_buffer ();
	      bprintf (wrapper, "<set-var");
	      for (i = 0; uf->named_parameters[i] != (char *)NULL; i++)
		{
		  char *value = get_positional_arg (vars, i);

		  if (value != (char *)NULL)
		    {
		      register int j, k;
		      char *setval;

		      if (uf->type == user_DEFUN)
			value = page_evaluate_string (value);

		      setval = (char *)xmalloc ((2 * strlen (value)) + 4);

		      setval[0] = '"';

		      for (j = 0, k = 1; value[j] != '\0'; j++)
			{
			  if (value[j] == '"')
			    setval[k++] = '\\';

			  setval[k++] = value[j];
			}

		      setval[k++] = '"';
		      setval[k] = '\0';

		      bprintf (wrapper, " %s=%s",
			       uf->named_parameters[i], setval);

		      if (uf->type == user_DEFUN)
			free (value);

		      free (setval);
		    }
		}
	      bprintf (wrapper, ">");
	      parameter_setter = wrapper->buffer;
	      free (wrapper);
	    }

	  if (packname)
	    bprintf_insert (page, start, "<in-package %s>%s%s</in-package>",
			    packname, parameter_setter ? parameter_setter : "",
			    subber->buffer);
	  else
	    bprintf_insert (page, start, "%s%s", 
			    parameter_setter ? parameter_setter : "",
			    subber->buffer);
	}

      page_free_page (subber);
    }
}

static void
pf_symbol_info (PFunArgs)
{
  char *name = page_evaluate_string (get_positional_arg (vars, 0));

  if (name)
    {
      Symbol *sym = symbol_lookup (name);

      if (sym != (Symbol *)NULL)
	{
	  switch (sym->type)
	    {
	    case symtype_STRING:
	      bprintf_insert (page, start, "STRING\n%d", sym->values_index);
	      break;

	    case symtype_FUNCTION:
	      bprintf_insert (page, start, "FUNCTION\n0");
	      break;

	    case symtype_BINARY:
	      bprintf_insert (page, start, "BINARY\n%d",
			      ((Datablock *)sym->values)->length);
	      break;
	    }
	}
      free (name);
    }
}

static void
pf_set_variable (PFunArgs)
{
  if (vars)
    {
      char **names = get_vars_names (vars);
      char **vals = get_vars_vals (vars);

      if (names != (char **)NULL)
	{
	  register int i;
	  char *sym_name;

	  for (i = 0; (sym_name = names[i]) != (char *)NULL; i++)
	    {
	      char *name = sym_name;
	      char *value = vals[i];
	      int free_value = 0;

	      name = page_evaluate_string (sym_name);

	      if (debug_level >= 5)
		{
		  if (value)
		    page_debug ("<set-var \"%s\"=\"%s\">", sym_name, value);
		  else
		    page_debug ("<set-var \"%s\">");
		}

	      if (value == (char *)NULL)
		{
		  if (debug_level)
		    page_debug ("<set-var %s ...> missing `='", sym_name);
		}
	      else
		{
		  value = page_evaluate_string (value);
		  if (value) free_value++;
		}

	      if (debug_level >= 6)
		page_debug ("--> <set-var \"%s\"=\"%s\">",
			    name ? name : "", value ? value : "");

	      if (name)
		pagefunc_set_variable (name, value);

	      if (free_value) free (value);
	      if (name != sym_name) free (name);
	    }
	}
    }
}

static void
pf_get_variable (PFunArgs)
{
  register int i;
  char *name;

  for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
    {
      char *insertion;
      char *value;

      insertion = page_evaluate_string (name);

      value = pagefunc_get_variable (insertion);

      if (debug_level > 5)
	page_debug ("<get-var \"%s\">", insertion ? insertion : "");

      if (value)
	{
	  bprintf_insert (page, start, "%s", value);
	  start += strlen (value);
	}
      else
	{
	  if (debug_level > 10)
	    page_debug ("<get-var \"%s\">: Unbound Variable \"%s\"!",
			insertion, insertion);
	}

      if (debug_level > 5)
	page_debug ("--> `%s'", value ? value : "");

      if (insertion)
	free (insertion);
    }
}

static void
pf_get_variable_once (PFunArgs)
{
  register int i;
  char *name;

  for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
    {
      char *insertion;
      char *value;

      insertion = page_evaluate_string (name);

      value = pagefunc_get_variable (insertion);

      if (debug_level > 5)
	page_debug ("<get-var \"%s\">", insertion ? insertion : "");

      if (value)
	{
	  bprintf_insert (page, start, "%s", value);
	  start += strlen (value);
	}
      else
	{
	  if (debug_level > 10)
	    page_debug ("<get-var \"%s\">: Unbound Variable \"%s\"!",
			insertion, insertion);
	}

      if (debug_level > 5)
	page_debug ("--> `%s'", value ? value : "");

      if (insertion)
	free (insertion);

      *newstart = start;
    }
}

static void
pf_unset_variable (PFunArgs)
{
  register int i;
  char *name;

  for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
    {
      char *varname = name;
      Symbol *sym;

      varname = page_evaluate_string (name);

      sym = symbol_lookup (varname);

      if (sym)
	{
	  /* Don't really remove this symbol if it has a notifier
	     attached to it, simply zap the contents. */
	  if (sym->notifier)
	    {
	      register int j;

	      *(sym->notifier) = 0;

	      for (j = 0; j < sym->values_index; j++)
		free (sym->values[j]);

	      if (sym->values_index)
		sym->values[0] = (char *)NULL;

	      sym->values_index = 0;
	    }
	  else
	    sym = symbol_remove (varname);

	  if (sym) symbol_free (sym);
	}

      if (varname != name) free (varname);
    }
}

static void
pf_package_names (PFunArgs)
{
  if (AllPackages)
    {
      register int i;
      Package *pack;

      for (i = 0; (pack = AllPackages[i]) != (Package *)NULL; i++)
	if (pack->name)
	  {
	    bprintf_insert (page, start, "%s\n", pack->name);
	    start += strlen (pack->name) + 1;
	  }

      *newstart = start;
    }
}

static void
pf_package_vars (PFunArgs)
{
  register int pos = 0;
  char *strip = get_value (vars, "STRIP");
  char *name;

  if ((CurrentPackage != (Package *)NULL) &&
      (get_positional_arg (vars, 0) == (char *)NULL))
    {
      Symbol **symbols = symbols_of_package (CurrentPackage);

      if (symbols != (Symbol **)NULL)
	{
	  register int i;

	  for (i = 0; symbols[i] != (Symbol *)NULL; i++)
	    {
	      bprintf_insert (page, start, "%s\n", symbols[i]->name);
	      start += symbols[i]->name_len + 1;
	    }

	  free (symbols);
	}
    }

  while ((name = get_positional_arg (vars, pos)) != (char *)NULL)
    {
      Package *pack = (Package *)NULL;

      pos++;

      name = page_evaluate_string (name);

      if (!empty_string_p (name))
	pack = symbol_lookup_package (name);

      if (pack)
	{
	  Symbol **symbols = symbols_of_package (pack);

	  if (symbols != (Symbol **)NULL)
	    {
	      register int i;

	      for (i = 0; symbols[i] != (Symbol *)NULL; i++)
		{
		  if ((pack->name[0] != '\0') && (strip == (char *)NULL))
		    {
		      bprintf_insert (page, start, "%s::%s\n",
				      pack->name, symbols[i]->name);
		      start += pack->name_len + 3 + symbols[i]->name_len;
		    }
		  else
		    {
		      bprintf_insert (page, start, "%s\n", symbols[i]->name);
		      start += symbols[i]->name_len + 1;
		    }
		}

	      free (symbols);
	    }
	}

      if (name) free (name);
    }

  *newstart = start;
}

static void
pf_package_delete (PFunArgs)
{
  char **names = get_vars_names (vars);

  if (names != (char **)NULL)
    {
      register int i;

      for (i = 0; names[i] != (char *)NULL; i++)
	{
	  char *name = names[i];

	  name = page_evaluate_string (name);

	  if (name)
	    {
	      pagefunc_destroy_package (name);
	      free (name);
	    }
	}
    }
}

static void
pf_if (PFunArgs)
{
  char *test_clause;
  char *then_clause;
  char *else_clause;
  char *consequence;

  test_clause = get_positional_arg (vars, 0);
  then_clause = get_positional_arg (vars, 1);
  else_clause = get_positional_arg (vars, 2);

  test_clause = page_evaluate_string (test_clause);

  if (!empty_string_p (test_clause))
    consequence = then_clause;
  else
    consequence = else_clause;

  if (!empty_string_p (consequence))
    {
      consequence = page_evaluate_string (consequence);
      if (consequence != (char *)NULL)
	{
	  bprintf_insert (page, start, "%s", consequence);
	  free (consequence);
	}
    }

  if (test_clause) free (test_clause);
}

static void
pf_ifeq (PFunArgs)
{
  char *left_clause;
  char *right_clause;
  char *then_clause;
  char *else_clause;
  char *consequence;
  int caseless_p = var_present_p (vars, "CASELESS");

  left_clause = get_positional_arg (vars, 0);
  right_clause = get_positional_arg (vars, 1);
  then_clause = get_positional_arg (vars, 2);
  else_clause = get_positional_arg (vars, 3);

  left_clause = page_evaluate_string (left_clause);
  right_clause = page_evaluate_string (right_clause);

  if (((empty_string_p (left_clause)) && (empty_string_p (right_clause))) ||
      ((left_clause && right_clause) &&
       (((!caseless_p) && (strcmp (left_clause, right_clause) == 0)) ||
	((caseless_p) && (strcasecmp (left_clause, right_clause) == 0)))))
    consequence = then_clause;
  else
    consequence = else_clause;

  if (consequence)
    {
      consequence = page_evaluate_string (consequence);

      if (consequence)
	{
	  bprintf_insert (page, start, "%s", consequence);
	  free (consequence);
	}
    }

  if (left_clause) free (left_clause);
  if (right_clause) free (right_clause);
}

#if defined (MHTML_ARITHMETIC)

/* Arithmetic operations.  This is pretty ugly. */
/* <gt  12 10> --> "true"
   <lt  10 12> --> "true"
   <eq  10 10> --> "true"
   <add 10 10> --> "20"
   <sub 10 10> --> "0"
   <mul 10 10> --> "100"
   <div 12 10> --> "1"
   <mod 12 10> --> "2" */
#define pf_GT	1
#define pf_LT	2
#define pf_EQ	3
#define pf_ADD	4
#define pf_SUB	5
#define pf_MUL	6
#define pf_DIV	7
#define pf_MOD	8

typedef struct { int op; char *name; } OP_ALIST;
static OP_ALIST op_alist[] = {
  { pf_GT, "GT" },
  { pf_LT, "LT" },
  { pf_EQ, "EQ" },
  { pf_ADD, "ADD" },
  { pf_SUB, "SUB" },
  { pf_MUL, "MUL" },
  { pf_DIV, "DIV" },
  { pf_MOD, "MOD" },

  { 0, (char *)NULL }
};

static char *
operator_name (int op)
{
  register int i;

  for (i = 0; op_alist[i].name != (char *)NULL; i++)
    if (op == op_alist[i].op)
      return (op_alist[i].name);

  return ("*invalid-op*");
}

static int
number_p (char *string)
{
  register int i;
  int result = 0;

  if (!string)
    return (0);

  /* Skip leading whitespace. */
  for (i = 0; whitespace (string[i]); i++);

  if (string[i])
    {
      int decimal_seen = 0;

      result = 1;

      if ((string[i] == '-') || (string[i] == '+'))
	i++;

      for (; string[i]; i++)
	{
	  if (string[i] == '.')
	    {
	      if (decimal_seen)
		{
		  result = 0;
		  break;
		}
	      else
		decimal_seen++;
	    }
	  else if (!isdigit (string[i]))
	    {
	      result = 0;
	      break;
	    }
	}
    }

  return (result);
}

#if !defined (macintosh)
/* Fucking SunOS doesn't declare this, result is assumed to be INT. */
extern double strtod (const char *, char **);
#endif

static char *
arithmetic_operate (int op, char *arg1, char *arg2)
{
  double val1 = arg1 ? strtod (arg1, (char **)NULL) : 0.0;
  double val2 = arg2 ? strtod (arg2, (char **)NULL) : 0.0;
  static char resbuf[40];
  static int orig_mhtml_decimal_notify = 0;
  char *result = resbuf;

  result[0] = '\0';

  switch (op)
    {
    case pf_GT:
      if (val1 > val2) result = "true";
      break;

    case pf_LT:
      if (val1 < val2) result = "true";
      break;

    case pf_EQ:
      if (!arg1) arg1 = "";
      if (!arg2) arg2 = "";

      if (number_p (arg1) && number_p (arg2))
	{
	  if (val1 == val2)
	    result = "true";
	}
      else
	{
	  if (strcasecmp (arg1, arg2) == 0)
	    result = "true";
	}
      break;

    default:
      {
	double arith_result = 0.0;
	int dot_present = ((arg1 ? (strchr (arg1, '.') != (char *)NULL) : 0) ||
			   (arg2 ? (strchr (arg2, '.') != (char *)NULL) : 0));
    
	switch (op)
	  {
	  case pf_ADD:
	    arith_result = val1 + val2;
	    break;

	  case pf_SUB:
	    arith_result = val1 - val2;
	    break;

	  case pf_MUL:
	    arith_result = val1 * val2;
	    break;

	  case pf_DIV:
	    arith_result = val2 ? val1 / val2 : 0.0;
	    break;

	  case pf_MOD:
	    arith_result = val2 ? (double)((int)val1 % (int)val2) : 0.0;
	    break;
	  }

	if (mhtml_decimal_notify != orig_mhtml_decimal_notify)
	  {
	    orig_mhtml_decimal_notify = mhtml_decimal_notify;

	    if (mhtml_decimal_notify)
	      {
		char *temp = pagefunc_get_variable ("mhtml::decimal-places");

		if (temp)
		  mhtml_decimal_places = atoi (temp);
		else
		  mhtml_decimal_places = 0;
	      }
	    else
	      mhtml_decimal_places = 2;
	  }

	if (mhtml_decimal_notify)
	  sprintf (result, "%.*f", mhtml_decimal_places, arith_result);
	else if (!dot_present /* || (arith_result == (int)arith_result) */)
	  sprintf (result, "%d", (int) arith_result);
	else
	  sprintf (result, "%.*f", mhtml_decimal_places, arith_result);
      }
    }

  return (result);
}

static void
arithmetic_process (int op, PFunArgs)
{
  char *_arg1 = get_positional_arg (vars, 0);
  char *_arg2 = get_positional_arg (vars, 1);
  char *arg1, *arg2;
  int error_found = 0;
  char *result;

  if (!_arg1)
    {
      error_found++;
      page_debug ("<%s ...> seen with zero args", operator_name (op));
      _arg1 = "";
    }

  if (!_arg2)
    {
      if (!error_found)
	page_debug ("<%s %s ?> seen with one arg", operator_name (op), _arg1);

      _arg2 = "";
    }

  arg1 = _arg1; arg2 = _arg2;

  /* Evaluate the args as variable names if they are not
     already all digits. */
  if (!number_p (arg1) || !number_p (arg2))
    {
      register int i;
      char *rarg1 = (char *)NULL;
      char *rarg2 = (char *)NULL;

      if (strchr (arg1, '<'))
	arg1 = page_evaluate_string (arg1);


      if (strchr (arg2, '<'))
	arg2 = page_evaluate_string (arg2);

      rarg1 = arg1;
      rarg2 = arg2;

      /* If still not all digits, lookup arg as variable name.
	 Only do this when the operation is not pf_EQ, or if
	 the variables were not page evaluated. */
      if (!number_p (arg1))
	{
	  if ((op != pf_EQ) || (arg1 == _arg1))
	    {
	      for (i = 0; whitespace (arg1[i]); i++);
	      rarg1 = pagefunc_get_variable (arg1 + i);
	    }
	}

      if (!number_p (arg2))
	{
	  if ((op != pf_EQ) || (arg2 == _arg2))
	    {
	      for (i = 0; whitespace (arg2[i]); i++);
	      rarg2 = pagefunc_get_variable (arg2 + i);
	    }
	}

      result = arithmetic_operate (op, rarg1 ? rarg1 : "", rarg2 ? rarg2 : "");
    }
  else
    result = arithmetic_operate (op, arg1 ? arg1 : "", arg2 ? arg2 : "");

  if (arg1 && arg1 != _arg1) free (arg1);
  if (arg2 && arg2 != _arg2) free (arg2);

  if (*result)
    bprintf_insert (page, start, "%s", result);
}

static void pf_gt (PFunArgs)
{
  arithmetic_process
    (pf_GT, page, body, vars, start, end, newstart, debug_level);
}

static void pf_lt (PFunArgs)
{
  arithmetic_process
    (pf_LT, page, body, vars, start, end, newstart, debug_level);
}

static void pf_eq (PFunArgs)
{
  arithmetic_process
    (pf_EQ, page, body, vars, start, end, newstart, debug_level);
}

static void pf_add (PFunArgs)
{
  arithmetic_process
    (pf_ADD, page, body, vars, start, end, newstart, debug_level);
}

static void pf_sub (PFunArgs)
{
  arithmetic_process
    (pf_SUB, page, body, vars, start, end, newstart, debug_level);
}

static void pf_mul (PFunArgs)
{
  arithmetic_process
    (pf_MUL, page, body, vars, start, end, newstart, debug_level);
}

static void pf_div (PFunArgs)
{
  arithmetic_process
    (pf_DIV, page, body, vars, start, end, newstart, debug_level);
}

static void pf_mod (PFunArgs)
{
  arithmetic_process
    (pf_MOD, page, body, vars, start, end, newstart, debug_level);
}

#endif /* MHTML_ARITHMETIC */

static int randomize_called = 0;
static void
pf_randomize (PFunArgs)
{
  unsigned int seed = (unsigned int)getpid ();
  char *user_seed = page_evaluate_string (get_positional_arg (vars, 0));

  if (user_seed)
    {
      seed = (unsigned int)atoi (user_seed);
      free (user_seed);
    }

  if (seed == 0) seed = 1;
  srandom (seed);
  randomize_called = 1;
}

static void
pf_random (PFunArgs)
{
  char *max_arg = page_evaluate_string (get_positional_arg (vars, 0));
  int max_value = max_arg ? atoi (max_arg) : 0;
  int result;

  if (!randomize_called)
    {
      srandom ((unsigned int)getpid ());
      randomize_called++;
    }

  result = random ();

  if (max_value)
    result = result % max_value;

  if (max_arg) free (max_arg);

  bprintf_insert (page, start, "%d", result);
}

int
empty_string_p (char *string)
{
  int result = 1;

  if (string != (char *)NULL)
    {
      while (whitespace (*string)) string++;

      if (*string != '\0')
	result = 0;
    }

  return (result);
}

static void
pf_when (PFunArgs)
{
  char *test = page_evaluate_string (get_positional_arg (vars, 0));

  if (!empty_string_p (test))
    bprintf_insert (page, start, "%s", body->buffer);

  if (test) free (test);
}

static void
pf_not (PFunArgs)
{
  int offset = 0;
  char *sexp = read_sexp (body->buffer, &offset, 0);
  char *test = page_evaluate_string (sexp);

  if (empty_string_p (test))
    bprintf_insert (page, start, "true");

  if (test) free (test);
  if (sexp) free (sexp);
}

static void
pf_and (PFunArgs)
{
  register int i = 0;
  char *result = strdup ("true");
  char *temp;

  while ((temp = get_positional_arg (vars, i++)) != (char *)NULL)
    {
      char *value = page_evaluate_string (temp);

      if (!empty_string_p (value))
	{
	  free (result);
	  result = value;
	}
      else
	{
	  if (value) free (value);
	  free (result);
	  result = (char *)NULL;
	  break;
	}
    }

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      free (result);
    }
}

static void
pf_or (PFunArgs)
{
  register int i = 0;
  char *result = (char *)NULL;
  char *temp;

  while ((temp = get_positional_arg (vars, i++)) != (char *)NULL)
    {
      char *value = page_evaluate_string (temp);
      if (!empty_string_p (value))
	{
	  result = value;
	  break;
	}
      else if (value) free (value);
    }

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      free (result);
    }
}

static void
pf_while (PFunArgs)
{
  char *test = get_positional_arg (vars, 0);
  int verbatim_p = var_present_p (vars, "VERBATIM");
  int iteration_count = 0;
  char *limit_string = pagefunc_get_variable ("mhtml::iteration-limit");
  int limit = limit_string ? atoi (limit_string) : PAGE_ITERATOR_MAX_COUNT;
  char *result;

  while (((result = page_evaluate_string (test)) != (char *)NULL) &&
	 (result[0] != '\0'))
    {
      PAGE *code;

      iteration_count++;

      if (iteration_count > limit)
	break;

      code = page_copy_page (body);
      page_process_page_internal (code);

      if ((code != (PAGE *)NULL) && (code->bindex != 0))
	{
	  bprintf_insert (page, start, "%s", code->buffer);
	  start += (code->bindex);

	  if (verbatim_p)
	    *newstart = start;
	}

      if (code != (PAGE *)NULL)
	page_free_page (code);

      free (result);
    }

  if (result) free (result);
}

static void
pf_var_case (PFunArgs)
{
  register int i = 0;
  char **names = get_vars_names (vars);
  char **vals = get_vars_vals (vars);
  static char *nullval = "";

  if (names != (char **)NULL)
    {
      while (1)
	{
	  char *name = (char *)NULL;
	  char *case_value = (char *)NULL;
	  char *page_value = (char *)NULL;
	  char *action = (char *)NULL;

	  if ((names[i] == (char *)NULL) || (names[i + 1] == (char *)NULL))
	    break;

	  name = page_evaluate_string (names[i]);
	  case_value = page_evaluate_string (vals[i]);
	  page_value = pagefunc_get_variable (name);
	  if (name) free (name);
	  action = names[i + 1];

	  i += 2;

	  /* Check the value against the page value. */
	  if (empty_string_p (page_value))
	    {
	      page_value = nullval;
	    }

	  if (empty_string_p (case_value))
	    {
	      if (case_value) free (case_value);
	      case_value = nullval;
	    }

	  if ((page_value == case_value) ||
	      (strcasecmp (page_value, case_value) == 0))
	    {
	      if (action != (char *)NULL)
		bprintf_insert (page, start, "%s", action);

	      if (case_value != nullval) free (case_value);
	      break;
	    }

	  if (case_value != nullval) free (case_value);
	}
    }
}

static void
change_increment (PFunArgs, int default_amount)
{
  char *var_name = page_evaluate_string (get_positional_arg (vars, 0));

  if (!empty_string_p (var_name))
    {
      char *var_value = pagefunc_get_variable (var_name);
      char *incr = get_one_of (vars, "BY", "AMOUNT", (char *)NULL);
      int value = 0;
      int amount = default_amount;
      static char number[40];

      if (var_value != (char *)NULL)
	value = atoi (var_value);

      if (!empty_string_p (incr))
	{
	  incr = page_evaluate_string (incr);
	  if (incr)
	    {
	      amount = atoi (incr);
	      free (incr);
	    }
	}

      value += amount;
      sprintf (number, "%d", value);

      pagefunc_set_variable (var_name, number);
    }
  if (var_name) free (var_name);
}

static void
pf_increment (PFunArgs)
{
  change_increment (page, body, vars, start, end, newstart, debug_level, 1);
}

static void
pf_decrement (PFunArgs)
{
  change_increment (page, body, vars, start, end, newstart, debug_level, -1);
}

#define MAX_SUBEXPS 10
static void
pf_match (PFunArgs)
{
  char *_string = get_positional_arg (vars, 0);
  char *_regex = get_positional_arg (vars, 1);
  char *result = (char *)NULL;

  if (_string && _regex)
    {
      char *string = page_evaluate_string (_string);
      char *regex = page_evaluate_string (_regex);
      int caseless = var_present_p (vars, "caseless");
      char *action = "report";

      if ((string != (char *)NULL) && (regex != (char *)NULL))
	{
	  /* Only up to MAX_SUBEXPS subexpressions kept. */
	  regex_t re;
	  regmatch_t offsets[MAX_SUBEXPS];
	  int slen = strlen (string);
	  int matched;
	  int so = 0, eo = 0, len = 0;
	  char *temp = get_value (vars, "action");
	  char *packname = page_evaluate_string (get_value (vars, "package"));

	  if (temp) action = temp;

	  regcomp (&re, regex, REG_EXTENDED | (caseless ? REG_ICASE : 0));

	  matched = (regexec (&re, string, MAX_SUBEXPS, offsets, 0) == 0);

	  if (matched)
	    {
	      so = offsets[0].rm_so;
	      eo = offsets[0].rm_eo;
	      len = eo - so;
	    }

	  /* If the caller has specified a package to receive the detailed
	     results of the match, put the information there now. */
	  if (matched && packname)
	    {
	      register int i, limit;
	      Package *p = symbol_get_package (packname);
	      Symbol *starts, *ends, *lengths;
	      Symbol *matches = (Symbol *)NULL;
	      char digitbuff[40];

	      forms_set_tag_value_in_package (p, "expr", regex);
	      starts = symbol_intern_in_package (p, "start");
	      ends = symbol_intern_in_package (p, "end");
	      lengths = symbol_intern_in_package (p, "length");
	      if (strcasecmp (action, "extract") == 0)
		matches = symbol_intern_in_package (p, "matches");

	      for (limit = MAX_SUBEXPS; limit; limit--)
		if (offsets[limit - 1].rm_so != -1)
		  break;

	      sprintf (digitbuff, "%d", limit - 1);
	      forms_set_tag_value_in_package (p, "matched", digitbuff);

	      for (i = 0; i < limit; i++)
		{
		  int sublen = offsets[i].rm_eo - offsets[i].rm_so;

		  sprintf (digitbuff, "%d", offsets[i].rm_so);
		  symbol_add_value (starts, digitbuff);
		  sprintf (digitbuff, "%d", offsets[i].rm_eo);
		  symbol_add_value (ends, digitbuff);
		  sprintf (digitbuff, "%d", sublen);
		  symbol_add_value (lengths, digitbuff);

		  if (matches != (Symbol *)NULL)
		    {
		      char *substring = (char *)xmalloc (1 + sublen);
		      strncpy (substring, string + offsets[i].rm_so, sublen);
		      substring[sublen] = '\0';
		      symbol_add_value (matches, substring);
		      free (substring);
		    }
		}
	    }

	  if (packname != (char *)NULL) free (packname);
	      
	  if (matched && strcasecmp (action, "report") == 0)
	    {
	      result = strdup ("true");
	    }
	  else if (matched && (strcasecmp (action, "extract") == 0))
	    {
	      result = (char *)xmalloc (1 + len);
	      strncpy (result, string + so, len);
	      result[len] = '\0';
	    }
	  else if (strcasecmp (action, "delete") == 0)
	    {
	      result = strdup (string);
	      if (matched)
		memmove (result + so, result + eo, (slen + 1) - eo);
	    }
	  else if ((strcasecmp (action, "startpos") == 0) ||
		   (strcasecmp (action, "endpos") == 0) ||
		   (strcasecmp (action, "length") == 0))
	    {
	      result = (char *)xmalloc (20);
	      result[0] = '0';
	      result[1] = '\0';

	      if (matched)
		{
		  if (strcasecmp (action, "startpos") == 0)
		    sprintf (result, "%d", so);
		  else if (strcasecmp (action, "endpos") == 0)
		    sprintf (result, "%d", eo);
		  else
		    sprintf (result, "%d", len);
		}
	    }
	  regfree (&re);
	}

      if (string) free (string);
      if (regex) free (regex);
    }

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      free (result);
    }
}

/* <substring string start [end]> */
static void
pf_substring (PFunArgs)
{
  char *str_arg = page_evaluate_string (get_positional_arg (vars, 0));
  char *beg_arg = page_evaluate_string (get_positional_arg (vars, 1));
  char *end_arg = page_evaluate_string (get_positional_arg (vars, 2));

  if (str_arg != (char *)NULL)
    {
      register int i;
      char *temp;
      int len = strlen (str_arg);
      int beg_index = 0;
      int end_index = len;

      /* If not all digits, lookup arg as variable name. */
      if (!empty_string_p (beg_arg))
	{
	  if (!number_p (beg_arg))
	    {
	      for (i = 0; whitespace (beg_arg[i]); i++);
	      temp = pagefunc_get_variable (beg_arg + i);
	      if (temp != (char *)NULL)
		beg_index = atoi (temp);
	    }
	  else
	    beg_index = atoi (beg_arg);
	}

      if (!empty_string_p (end_arg))
	{
	  if (!number_p (end_arg))
	    {
	      for (i = 0; whitespace (end_arg[i]); i++);
	      temp = pagefunc_get_variable (end_arg + i);
	      if (temp != (char *)NULL)
		end_index = atoi (temp);
	    }
	  else
	    end_index = atoi (end_arg);
	}

      if (beg_index > end_index)
	{ i = beg_index; beg_index = end_index; end_index = i; }

      if (end_index > len) end_index = len;

      if ((beg_index != end_index) && (beg_index < len))
	{
	  if ((end_index - beg_index) < 100)
	    {
	      char buffer[100];

	      strncpy (buffer, str_arg + beg_index, end_index - beg_index);
	      buffer[end_index - beg_index] = '\0';
	      bprintf_insert (page, start, "%s", buffer);
	      *newstart += (end_index - beg_index);
	    }
	  else
	    {
	      temp = (char *)xmalloc (1 + (end_index - beg_index));
	      strncpy (temp, str_arg + beg_index, end_index - beg_index);
	      temp[end_index - beg_index] = '\0';
	      bprintf_insert (page, start, "%s", temp);
	      *newstart += (end_index - beg_index);
	      free (temp);
	    }
	}
    }

  if (str_arg) free (str_arg);
  if (beg_arg) free (beg_arg);
  if (end_arg) free (end_arg);
}

static void
pf_downcase (PFunArgs)
{
  char *value = page_evaluate_string (get_positional_arg (vars, 0));

  if (value != (char *)NULL)
    {
      register int i;

      for (i = 0; value[i] != '\0'; i++)
	if (isupper (value[i]))
	  value[i] = tolower (value[i]);

      bprintf_insert (page, start, "%s", value);
      free (value);
    }
}

static void
pf_upcase (PFunArgs)
{
  char *value = page_evaluate_string (get_positional_arg (vars, 0));

  if (value != (char *)NULL)
    {
      register int i;

      for (i = 0; value[i] != '\0'; i++)
	if (islower (value[i]))
	  value[i] = toupper (value[i]);

      bprintf_insert (page, start, "%s", value);
      free (value);
    }
}

static void
pf_capitalize (PFunArgs)
{
  char *value = page_evaluate_string (get_positional_arg (vars, 0));

  if (value != (char *)NULL)
    {
      register int i;
      int capnext = 1;

      for (i = 0; value[i] != '\0'; i++)
	{
	  if (!isalpha (value[i]))
	    capnext = 1;
	  else
	    {
	      if (capnext)
		{
		  if (islower (value[i]))
		    value[i] = toupper (value[i]);

		  capnext = 0;
		}
	      else
		{
		  if (isupper (value[i]))
		    value[i] = tolower (value[i]);
		}
	    }
	}

      bprintf_insert (page, start, "%s", value);
      free (value);
    }
}

static void
pf_string_compare (PFunArgs)
{
  char *string_1 = page_evaluate_string (get_positional_arg (vars, 0));
  char *string_2 = page_evaluate_string (get_positional_arg (vars, 1));
  int caseless_p = get_value (vars, "caseless") != (char *)NULL;
  char *result = (char *)NULL;

  /* Both strings empty? */
  if (string_1 == string_2)
    result = "equal";
  else if (string_1 == (char *)NULL)
    result = "less";
  else if (string_2 == (char *)NULL)
    result = "greater";
  else
    {
      int temp;

      if (caseless_p)
	temp = strcasecmp (string_1, string_2);
      else
	temp = strcmp (string_1, string_2);

    switch (temp)
      {
      case 0: result = "equal"; break;
      case 1: result = "greater"; break;
      case -1: result = "less"; break;
      }
    }

  if (string_1 != (char *)NULL) free (string_1);
  if (string_2 != (char *)NULL) free (string_2);

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart = start + strlen (result);
    }
}

static void
pf_word_wrap (PFunArgs)
{
  char *width_spec = page_evaluate_string (get_value (vars, "width"));
  int width = (width_spec != (char *)NULL) ? atoi (width_spec) : 60;
  char *text = page_evaluate_string (get_positional_arg (vars, 0));

  if (width == 0) width = 60;

  if (!empty_string_p (text))
    {
      BPRINTF_BUFFER *temp = bprintf_create_buffer ();

      bprintf (temp, "%s", text);
      bprintf_word_wrap (temp, width);

      bprintf_insert (page, start, "%s", temp->buffer);
      bprintf_free_buffer (temp);
    }

  if (text != (char *)NULL) free (text);
  if (width_spec != (char *)NULL) free (width_spec);
}

static void
pf_comment (PFunArgs)
{
  /* Contents already deleted by caller. */
}

/* Does modifications to the plain text in BODY.  Usually, this simply
   inserts paragraph breaks where they appear, and optionally operates
   on the first character of paragraphs.  The text starts with a <P>,
   unless the variable NOBR is set.*/
static void
pf_plain_text (PFunArgs)
{
  register int i;
  char *first_char;
  char *nobr = page_evaluate_string (get_value (vars, "NOBR"));
  char *nolower = page_evaluate_string (get_value (vars, "NOLOWER"));

  first_char = page_evaluate_string (get_value (vars, "FIRST-CHAR"));

  /* Remove all comments from BODY. */
  page_subst_in_page (body, ";;;[^\n]*(\n|$)", "");

  /* Insert one blank line in the front of BODY. */
  bprintf_insert (body, 0, "<p>");

  /* Modify blank lines in BODY such that they contain <p> instead. */
  page_subst_in_page (body, "\n[ \t]*\n", "<p>");

  /* Modify the first character of every paragraph by inserting the
     open tag before it, and inserting a matching close tag after it. */
  if (first_char)
    {
      register int begin;
      char *closer = (char *)NULL;
      int o_len = strlen (first_char);
      int c_len = 0;

      if (*first_char == '<')
	{
	  register int c;

	  for (i = 1; whitespace (first_char[i]); i++);

	  begin = i;

	  for (i = begin; (c = first_char[i]) != '\0'; i++)
	    if ((c == '>') || (whitespace (c)))
	      break;

	  closer = (char *)xmalloc (4 + (i - begin));
	  closer[0] = '<';
	  closer[1] = '/';
	  strncpy (closer + 2, first_char + begin, i - begin);
	  closer[(i - begin) + 2] = '>';
	  closer[(i - begin) + 3] = '\0';
	  c_len = strlen (closer);
	}

      /* Now quickly find occurences of "<p>" in BODY. */
      begin = 0;

      while ((begin = page_search (body, "<p>", begin)) != -1)
	{
	  begin += 3;

	  while (begin < body->bindex && whitespace (body->buffer[begin]))
	    begin++;

	  if ((begin < body->bindex) && (isalnum (body->buffer[begin])) &&
	      ((empty_string_p (nolower)) || (isupper (body->buffer[begin]))))
	    {
	      /* Insert closer first, if present. */
	      if (closer)
		bprintf_insert (body, begin + 1, "%s", closer);

	      /* Now insert the opener. */
	      bprintf_insert (body, begin, "%s", first_char);

	      /* Bump BEGIN past just inserted text. */
	      begin += o_len + c_len;
	    }
	}
      if (closer) free (closer);
    }

  /* Insert the modified body. */
  bprintf_insert (page, start, "%s", body->buffer + (nobr ? 3 : 0));
  if (nobr) free (nobr);
  if (nolower) free (nolower);
  if (first_char) free (first_char);
}

#if defined (DEPRECATED)
/* Something to make table building easier on the eyes. */
static void
pf_input_item (PFunArgs)
{
  char *label = page_evaluate_string (get_positional_arg (vars, 0));

  if (label)
    {
      char *name = page_evaluate_string (get_value (vars, "NAME"));
      char *type = page_evaluate_string (get_value (vars, "TYPE"));
      char *value= page_evaluate_string (get_value (vars, "VALUE"));
      char *size = page_evaluate_string (get_value (vars, "SIZE"));
      BPRINTF_BUFFER *output = bprintf_create_buffer ();

      bprintf (output, "<td align=right>%s:</td>", label);
      bprintf (output, "<td align=left><INPUT ");

      bprintf (output, "NAME=\"%s\" ", name ? name : label);
      bprintf (output, "TYPE=\"%s\" ", type ? type : "TEXT");
      bprintf (output, "VALUE=\"%s\" ", value ? value : "" );
      if (size)	bprintf (output, "SIZE=\"%s\" ", size);

      bprintf (output, ">");
      if (var_present_p (vars, "br")) bprintf (output, "<br>");
      bprintf (output, "</td>");

      bprintf_insert (page, start, "%s", output->buffer);
      bprintf_free_buffer (output);

      if (name) free (name);
      if (type) free (type);
      if (size) free (size);
      if (value) free (value);

      free (label);
    }
}
#endif /* DEPRECATED */

#define INCLUDE_RECURSION_LIMIT 10
static int include_recursive_calls = 0;

/* Canonicalize the filename given such that it is a complete path to a
   file. */
char *
pagefunc_canonicalize_file_name (char *input)
{
  char *include_prefix = pagefunc_get_variable ("MHTML::INCLUDE-PREFIX");
  char *relative_prefix = pagefunc_get_variable ("MHTML::RELATIVE-PREFIX");
  char *result = (char *)NULL;

  if (input == (char *)NULL)
    return (input);

  /* Ignore leading and trailing whitespace. */
  {
    register int i;

    input = strdup (input);
    for (i = 0; input[i] && whitespace (input[i]); i++);

    if (i != 0)
      memmove (input, input + i, strlen (input + i) + 1);

    for (i = strlen (input) - 1; i > 0 && whitespace (input[i]); i--);
    if (input[i])
      input[i + 1] = '\0';
  }
  
  result = input;

  if (include_prefix != (char *)NULL)
    {
      BPRINTF_BUFFER *file_name = bprintf_create_buffer ();
      int offset = 0;

      /* If the include prefix and the beginning of the user
	 supplied filename are the same, don't double them. */
      if (strncmp (include_prefix, input, strlen (include_prefix)) == 0)
	include_prefix = "";

      bprintf (file_name, "%s", include_prefix);

      /* The semantics of INCLUDE are similar to the semantics of web-space.
	 This means that "<include /header.mhtml>" gets `header.mhtml' from
	 the root directory, and not from the local directory. */
      if (*input == '/')
	offset++;
      else
	{
	  /* This is a relative pathname.  Try to get the relative
	     prefix from the caller. */
	  if (relative_prefix != (char *)NULL)
	    bprintf (file_name, "%s/", relative_prefix);
	}

      if (!file_name->bindex ||
	  file_name->buffer[file_name->bindex - 1] != '/')
	{
	  bprintf (file_name, "/");
	}

      bprintf (file_name, "%s", input + offset);
      result = file_name->buffer;
      free (file_name);
      free (input);
    }

#if defined (macintosh) || defined (__WINNT__)
  /* Fix pathname separators. */
  if (result)
    {
      register int i;

#if defined (__WINNT__)
      if ((result[0] != '\0') && (result[1] == ':'))
	memmove (result, result + 2, strlen (result) - 2);
#endif /* __WINNT__ */

      for (i = 0; result[i] != '\0'; i++)
	{
#if defined (macintosh)
	  if (result[i] == '/')
	    result[i] = ':';
#endif /* macintosh */
#if defined (__WINNT__)
	  if (result[i] == '\\')
	    result[i] = '/';
#endif /* __WINNT__ */
	}
    }
#endif /* mac || NT */

  return (result);
}

static void
pf_include (PFunArgs)
{
  int verbatim_p = var_present_p (vars, "VERBATIM");

  include_recursive_calls++;
  if (include_recursive_calls < INCLUDE_RECURSION_LIMIT)
    {
      char *arg = page_evaluate_string (get_positional_arg (vars, 0));

      if (arg != (char *)NULL)
	{
	  char *pathname = pagefunc_canonicalize_file_name (arg);
	  PAGE *file_contents = (PAGE *)NULL;

	  if (pathname)
	    {
	      file_contents = page_read_template (pathname);
	      free (pathname);
	    }

	  /* Did the user specify some alternate HTML if the file
	     couldn't be found? */
	  if (!file_contents)
	    {
	      char *alt = page_evaluate_string
		(get_one_of (vars, "ALT", "ALTERNATE", (char *)0));

	      if (alt != (char *)NULL)
		{
		  verbatim_p = 0;
		  file_contents = page_create_page ();
		  bprintf (file_contents, "%s", alt);
		  free (alt);
		}
	    }

	  if (file_contents)
	    {
#if defined (NOT_BINARY_COMPATIBLE)
	      bprintf_insert (page, start, "%s", file_contents->buffer);
#else
	      /* Manually insert the file instead of letting bprintf
		 do it for us.  This is because the file could contain
		 binary data, and then file->bindex wouldn't necessarily
		 reflect the length of what was inserted. */
	      if ((file_contents->bindex + page->bindex) >= page->bsize)
		page->buffer = (char *)xrealloc
		(page->buffer, (page->bsize += (file_contents->bindex + 100)));

	      memmove (page->buffer + start + file_contents->bindex,
		       page->buffer + start,
		       (page->bindex + 1) - start);

	      memcpy (page->buffer + start, file_contents->buffer,
		      file_contents->bindex);
	      page->bindex += file_contents->bindex;

#endif /* BINARY_COMPATIBLE */

	      if (verbatim_p)
		*newstart += file_contents->bindex;

	      page_free_page (file_contents);
	    }
	  free (arg);
	}
    }
  include_recursive_calls--;
}

static void
pf_replace_page (PFunArgs)
{
  PAGE *newpage = page_create_page ();
  int newpage_start = 0;

  pf_include (newpage, body, vars, 0, 0, &newpage_start, debug_level);
  ThePage = newpage;
  page_process_page_internal (newpage);
  page_return_this_page (newpage);
}

#define OTHER 1
#define UPPER 2
#define LOWER 3

#define CLOSE_STATE \
  switch (state) \
    { \
    case OTHER: bprintf (buffer, "%s", other_close); break; \
    case UPPER: bprintf (buffer, "%s", upper_close); break; \
    case LOWER: bprintf (buffer, "%s", lower_close); break; \
    }

static char *
wrap_by_character_class (char *string, int small_caps_p, int leave_braces,
			 char *upper_open, char *upper_close,
			 char *lower_open, char *lower_close,
			 char *other_open, char *other_close)
{
  register int i, c, state;
  char *result;
  BPRINTF_BUFFER *buffer;

  /* Handle easiest case first. */
  if (!string)
    return ((char *)NULL);

  if (!upper_open) upper_open = "";
  if (!upper_close) upper_close = "";
  if (!lower_open) lower_open = "";
  if (!lower_close) lower_close = "";
  if (!other_open) other_open = "";
  if (!other_close) other_close = "";

  buffer = bprintf_create_buffer ();

  state = 0;

  for (i = 0; (c = string[i]) != '\0'; i++)
    {
      if (isupper (c) && state != UPPER)
	{
	  CLOSE_STATE;
	  state = UPPER;
	  bprintf (buffer, "%s", upper_open);
	}
      else if (islower (c) && state != LOWER)
	{
	  CLOSE_STATE;
	  state = LOWER;
	  bprintf (buffer, "%s", lower_open);
	}
      else if (isspace (c))
	{
	}
      else if (leave_braces && ((c == '<') || (c == '>')))
	{
	  int point = i;
	  char *sexp;

	  CLOSE_STATE;
	  state = 0;
	  sexp = read_sexp_1 (string, &point, 0, 1);
	  if (sexp != (char *)NULL)
	    {
	      bprintf (buffer, "%s", sexp);
	      free (sexp);
	      c = '\0';
	      i = point - 1;
	    }
	}
      else if (!(isupper (c) || islower (c)) && state != OTHER)
	{
	  CLOSE_STATE;
	  state = OTHER;
	  bprintf (buffer, "%s", other_open);
	}

      if (small_caps_p && islower (c))
	c = toupper (c);

      if (c)
	bprintf (buffer, "%c", c);
    }

  CLOSE_STATE;

  result = buffer->buffer;
  free (buffer);

  return (result);
}

static void
pf_small_caps (PFunArgs)
{
  char *string = page_evaluate_string (body->buffer);

  if (string)
    {
      char *upper_size = get_one_of (vars, "upper", "upper-size", (char *)0);
      char *lower_size = get_one_of (vars, "lower", "lower-size", (char *)0);
      char *other_size = get_one_of (vars, "other", "other-size", (char *)0);
      char uo[100], lo[100], oo[1000], *cl = "</FONT>";
      char *result;

      if (!upper_size) upper_size = "+0";
      if (!lower_size) lower_size = "-1";
      if (!other_size) other_size = "+0";

      sprintf (uo, "<FONT SIZE=\"%s\">", upper_size);
      sprintf (lo, "<FONT SIZE=\"%s\">", lower_size);
      sprintf (oo, "<FONT SIZE=\"%s\">", other_size);

      result = wrap_by_character_class (string, 1, 1, uo, cl, lo, cl, oo, cl);
      bprintf_insert (page, start, "%s", result);
      free (string);
      free (result);
    }
}

char *
html_quote_string (char *string)
{
  BPRINTF_BUFFER *newstring = bprintf_create_buffer ();
  char *result;

  if (string != (char *)NULL)
    {
      register int i;

      for (i = 0; string[i] != '\0'; i++)
	{
	  if (string[i] == '<')
	    bprintf (newstring, "&lt;");
	  else if (string[i] == '>')
	    bprintf (newstring, "&gt;");
	  else if (string[i] == '&')
	    bprintf (newstring, "&amp;");
	  else
	    bprintf (newstring, "%c", string[i]);
	}
    }

  result = newstring->buffer;
  free (newstring);

  return (result);
}

#if defined (DEPRECATED)
static void
pf_html_quote (PFunArgs)
{
  char *string = body->buffer;
  char *evalled = page_evaluate_string (string);
  char *result = html_quote_string (evalled);

  if (evalled) free (evalled);

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      free (result);
    }
}
#endif /* DEPRECATED */

/* Pad the input text.  ALIGN can be one of LEFT, MIDDLE, or RIGHT.
   Insert the correct number of spaces to make the input argument
   take the desired number of spaces (presumably for use in a
   <pre> ... </pre> statement).  ALIGN defaults to "RIGHT".
   Optional arg "TRUNCATE" says to force the string to be the specified
   length.  Calling sequence:

      <pad <get-var foo> 23 align=middle truncate>
*/

#define align_RIGHT  0
#define align_LEFT   1
#define align_MIDDLE 2

static void
pf_pad (PFunArgs)
{
  register int i;
  char *input = page_evaluate_string (get_positional_arg (vars, 0));
  char *wtext = page_evaluate_string (get_positional_arg (vars, 1));
  char *align = page_evaluate_string (get_value (vars, "ALIGN"));
  int truncation = var_present_p (vars, "TRUNCATE");
  int width = wtext ? atoi (wtext) : 15;
  int alignment = align_RIGHT;
  int input_len = input ? strlen (input) : 0;

  if (align)
    {
       if (strcasecmp (align, "left") == 0)
	 alignment = align_LEFT;
       else if ((strcasecmp (align, "middle") == 0) ||
		(strcasecmp (align, "center") == 0))
	 alignment = align_MIDDLE;

       free (align);
     }

  if (wtext) free (wtext);

  if (!input)
    return;

  /* Strip leading and trailing whitespace from the input. */
  if (input_len)
    {
      for (i = 0; whitespace (input[i]); i++);
      if (i)
	memmove (input, input + i, (input_len - i) + 1);

      for (i = strlen (input) - 1; i > -1; i--)
	if (!whitespace (input[i]))
	  break;

      input[i + 1] = '\0';
      input_len = i + 1;
    }

  /* Handle truncation. */
  if (input_len > width)
    {
      if (truncation)
	input[width] = '\0';
    }
  else
    {
      int offset = 0;
      int left_pad = 0;
      int right_pad = 0;
      char *string = (char *)xmalloc (2 + width);

      /* Get the amount to pad on the left and right. */
      switch (alignment)
	{
	case align_LEFT:
	  right_pad = width - input_len;
	  break;

	case align_RIGHT:
	  left_pad = width - input_len;
	  break;

	case align_MIDDLE:
	  left_pad = (width - input_len) ? (width - input_len) / 2 : 0;
	  right_pad = width - (input_len + left_pad);
	  break;
	}

      /* Put the left-hand spaces in place. */
      for (offset = 0; offset < left_pad; offset++)
	string[offset] = ' ';

      /* Drop the input string down. */
      for (i = 0; (string[offset] = input[i]) != '\0'; i++, offset++);

      /* Put the right-hand spaces in place. */
      for (i = 0; i < right_pad; i++)
	string[offset++] = ' ';

      /* Terminate the string. */
      string[offset] = '\0';

      free (input);
      input = string;
    }

  if (input)
    {
      bprintf_insert (page, start, "%s", input);
      free (input);
    }
}

static void
pf_subst_in_page (PFunArgs)
{
  int changed = 0;
  int arg = 0, done = 0;

  while (!done)
    {
      char *this, *that;

      this = get_positional_arg (vars, arg++);
      that = get_positional_arg (vars, arg++);

      if (this == (char *)NULL)
	done = 1;
      else
	{
	  this = page_evaluate_string (this);
	  that = page_evaluate_string (that);

	  if (this)
	    changed += page_subst_in_page (page, this, that);


	  if (this) free (this);
	  if (that) free (that);
	}
    }
  *newstart = start + changed;
}

static char *
subst_in_string_internal (char *contents, Package *vars, int debug_level)
{
  char *result = (char *)NULL;

  if (contents != (char *)NULL)
    {
      int done = 0;
      int arg = 1;
      PAGE *temp = page_create_page ();
      page_set_contents (temp, contents);

      while (!done)
	{
	  char *this = get_positional_arg (vars, arg++);
	  char *that = get_positional_arg (vars, arg++);

	  if (this == (char *)NULL)
	    done = 1;
	  else
	    {
	      this = page_evaluate_string (this);
	      that = page_evaluate_string (that);

	      if (debug_level > 5)
		page_debug
		  ("<subst-in-var \"%s\" \"%s\" \"%s\">",
		   contents, this, that ? that : "");

	      if (this)
		page_subst_in_page (temp, this, that);

	      if (debug_level > 5)
		page_debug ("--> `%s'", temp->buffer ? temp->buffer : "");

	      if (this) free (this);
	      if (that) free (that);
	    }
	}

      result = temp->buffer;
      free (temp);
    }

  return (result);
}

static void
pf_subst_in_var (PFunArgs)
{
  char *varname = page_evaluate_string (get_positional_arg (vars, 0));

  if (!empty_string_p (varname))
    {
      char *contents = pagefunc_get_variable (varname);
      char *result = subst_in_string_internal (contents, vars, debug_level);

      pagefunc_set_variable (varname, result);
      if (result) free (result);
    }

  if (varname != (char *)NULL) free (varname);
}

static void
pf_subst_in_string (PFunArgs)
{
  char *contents = page_evaluate_string (get_positional_arg (vars, 0));

  if (contents != (char *)NULL)
    {
      char *result = subst_in_string_internal (contents, vars, debug_level);

      free (contents);

      if (result)
	{
	  bprintf_insert (page, start, "%s", result);
	  free (result);
	}
    }
}

static void
pf_with_local_package (PFunArgs)
{
  int jump_again = 0;
  Package *current_package = CurrentPackage;
  volatile char *result = (char *)NULL;

  {
    PageEnv *page_environ = pagefunc_save_environment ();
    Package *temp = symbol_get_package ((char *)NULL);

    CurrentPackage = temp;

    if ((jump_again = setjmp (page_jmp_buffer)) == 0)
      result = page_evaluate_string (body->buffer);

    CurrentPackage = current_package;
    symbol_destroy_package (temp);
    pagefunc_restore_environment (page_environ);
  }

  if (result != (char *)NULL)
    {
      if (jump_again == 0)
	bprintf_insert (page, start, "%s", (char *)result);
      free ((char *)result);
    }
  if (jump_again) longjmp (page_jmp_buffer, 1);
}

static void
pf_in_package (PFunArgs)
{
  int jump_again = 0;
  volatile char *packname;
  volatile char *result;

  packname = page_evaluate_string (get_positional_arg (vars, 0));
  result = (char *)NULL;

  if (empty_string_p (packname))
    {
      if (packname) free (packname);
      packname = strdup ("DEFAULT");
    }

  if (strcasecmp (packname, "local") == 0)
    {
      free ((char *)packname);
      packname = (char *)NULL;
    }

  {
    PageEnv *page_environ = pagefunc_save_environment ();
    Package *current_package = CurrentPackage;

    CurrentPackage = symbol_get_package (packname);

    if ((jump_again = setjmp (page_jmp_buffer)) == 0)
      result = page_evaluate_string (body->buffer);

    CurrentPackage = current_package;

    pagefunc_restore_environment (page_environ);
  }

  if (result != (char *)NULL)
    {
      if (jump_again == 0)
	bprintf_insert (page, start, "%s", (char *)result);
      free ((char *)result);
    }

  if (packname) free (packname);
  if (jump_again) longjmp (page_jmp_buffer, 1);
}

static void
pf_page_search (PFunArgs)
{
  char *search_start = page_evaluate_string (get_positional_arg (vars, 0));
  char *search_string = page_evaluate_string (get_positional_arg (vars, 1));

  if ((!empty_string_p (search_start)) && (!empty_string_p (search_string)) &&
      (ThePage != (PAGE *)NULL))
    {
      int loc = atoi (search_start);

      if (loc < ThePage->bindex)
	{
	  int end_point, beg_point;

	  beg_point =
	    page_search_boundaries (ThePage, search_string, loc, &end_point);

	  if (beg_point != -1)
	    {
	      int use_end_p = var_present_p (vars, "end");

	      bprintf_insert (page, start, "%d",
			      use_end_p ? end_point : beg_point);
	    }
	}
    }
}

static void
pf_page_insert (PFunArgs)
{
  char *insert_loc = page_evaluate_string (get_positional_arg (vars, 0));
  char *insertion = page_evaluate_string (get_positional_arg (vars, 1));

  if ((!empty_string_p (insert_loc)) && (!empty_string_p (insertion)))
    {
      int loc = atoi (insert_loc);

      if ((loc > -1) && (loc < ThePage->bindex))
	{
	  bprintf_insert (ThePage, loc, "%s", insertion);
	  if ((loc < start) && (ThePage == page))
	    *newstart += strlen (insertion);
	}
    }
}

static void
pf_set_form_input_values (PFunArgs)
{
  Symbol **symbols = symbols_of_package (PageVars);

  if (symbols != (Symbol **)NULL)
    {
      register int i;
      Symbol *sym;

      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
	{
	  char *value = sym->values ? sym->values[0] : "";

	  page_set_form_input_value (page, sym->name, value);
	}

      free (symbols);
    }
}

static void
pf_redirect (PFunArgs)
{
  PAGE *new_page;
  char *arg = (char *)NULL;

  if ((body != (PAGE *)NULL) && (body->buffer != (char *)NULL))
    {
      int offset = 0;
      char *sexp = read_sexp (body->buffer, &offset, 0);

      if (!empty_string_p (sexp))
	arg =  page_evaluate_string (sexp);

      if (sexp != (char *)NULL)
	free (sexp);
    }

  new_page = page_create_page ();

  /* If there is something to redirect to, then do it now.
     Otherwise, return a null response code, indicating that the
     browser should retain the old view. */
  if (!empty_string_p (arg))
    {
      register int i;

      for (i = 0; whitespace (arg[i]); i++);

      /* Fully qualify ARG if it isn't already. */
      if ((strncasecmp (arg + i, "http://", 7) != 0) &&
	  (strncasecmp (arg + i, "ftp://", 6) != 0) &&
	  (strncasecmp (arg + i, "gopher://", 9) != 0))
	{
	  BPRINTF_BUFFER *newarg = bprintf_create_buffer ();
	  char *temp = pagefunc_get_variable ("mhtml::relative-prefix");

	  bprintf (newarg, "%s", pagefunc_get_variable ("mhtml::http-prefix"));
	  if (arg[i] != '/')
	    bprintf (newarg, "%s/%s", temp ? temp : "", arg + i);
	  else
	    bprintf (newarg, "%s", arg + i);

	  free (arg);
	  arg = newarg->buffer;
	  free (newarg);
	  i = 0;
	}

#if defined (macintosh)
      bprintf (new_page,  "HTTP/1.0 302 Found\nLocation: %s\n\n", arg + i);
#else
      if (pagefunc_get_variable ("mhtml::unparsed-headers"))
	bprintf (new_page, "HTTP/1.0 302 Found\nLocation: %s\n\n", arg + i);
      else
	bprintf (new_page, "Location: %s\n\n", arg + i);
#endif /* !macintosh */
    }
  else
    bprintf (new_page, "HTTP/1.0 204 No Response\n\n");

  if (arg) free (arg);
  page_return_this_page (new_page);
}

static void
pf_cgi_encode (PFunArgs)
{
  char **names = get_vars_names (vars);

  if (names != (char **)NULL)
    {
      register int i;
      char *name;
      char *result = (char *)NULL;
      Package *cgivars = symbol_get_package ((char *)NULL);
      Symbol **symbols = (Symbol **)NULL;

      for (i = 0; (name = names[i]) != (char *)NULL; i++)
	{
	  name = page_evaluate_string (name);

	  if (!empty_string_p (name))
	    {
	      Symbol *sym = symbol_lookup (name);
	      if ((sym != (Symbol *)NULL) && (sym->type == symtype_STRING))
		{
		  register int j;
		  Symbol *newsym = symbol_intern_in_package (cgivars, name);

		  for (j = 0; j < sym->values_index; j++)
		    symbol_add_value (newsym, sym->values[j]);
		}
	    }

	  if (name) free (name);
	}

      symbols = symbols_of_package (cgivars);
      result = forms_unparse_items (symbols);

      if (!empty_string_p (result))
	{
	  bprintf_insert (page, start, "%s", result);
	  *newstart = start + strlen (result);
	}

      if (result) free (result);
      if (symbols) free (symbols);
      symbol_destroy_package (cgivars);
    }
}

/* <cgi-decode string [package]> Decode STRING into PACKAGE.
   If PACKAGE is not specified the current package is used. */
static void
pf_cgi_decode (PFunArgs)
{
  char *string, *packname = (char *)NULL;
  char *temp;
  Package *package = CurrentPackage;
  int offset = 0;

  string = read_sexp (body->buffer, &offset, 0);
  packname = read_sexp (body->buffer, &offset, 0);

  if (string != (char *)NULL)
    {
      temp = page_evaluate_string (string);
      free (string);
      string = temp;
    }

  if (!empty_string_p (string))
    {
      if (packname != (char *)NULL)
	{
	  temp = page_evaluate_string (packname);
	  free (packname);
	  packname = temp;

	  if (!empty_string_p (packname))
	    package = symbol_get_package (packname);

	  if (packname != (char *)NULL)
	    free (packname);
	}

      forms_parse_data_string (string, package);
    }

  if (string != (char *)NULL)
    free (string);
}

/* Insert the contents of the body verbatim.
   With optional arg "quote", quote the string being inserted. */
static void
pf_verbatim (PFunArgs)
{
  int quote_p = var_present_p (vars, "quote");

  /* Insert the contents, and then skip past them. */
  if (body && body->buffer)
    {
      char *string = body->buffer;

      if (quote_p)
	string = html_quote_string (string);

      if (string != (char *)NULL)
	{
	  bprintf_insert (page, start, "%s", string);
	  *newstart = start + strlen (string);

	  if (quote_p)
	    free (string);
	}
    }
}

Package *
alist_to_package (char *string)
{
  WispObject *list = wisp_from_string (string);
  Package *package = (Package *)NULL;

  if (!CONS_P (list))
    return (package);

  while (list != NIL)
    {
      WispObject *pair;

      pair = CAR (list);
      list = CDR (list);

      if (CONS_P (pair) & STRING_P (CAR (pair)))
	{
	  char *tag;
	  Symbol *sym;

	  tag = STRING_VALUE (CAR (pair));

	  if (!package)
	    {
	      int old_prime = symbol_small_prime;
	      symbol_small_prime = 23;
	      package = symbol_get_package ((char *)NULL);
	      symbol_small_prime = old_prime;
	    }

	  if (STRING_P (CDR (pair)))
	    {
	      sym = symbol_intern_in_package (package, tag);
	      symbol_add_value (sym, STRING_VALUE (CDR (pair)));
	    }
	  else
	    {
	      WispObject *values = CDR (pair);

	      sym = symbol_intern_in_package (package, tag);

	      while (CONS_P (values) && STRING_P (CAR (values)))
		{
		  symbol_add_value (sym, STRING_VALUE (CAR (values)));
		  values = CDR (values);
		}
	    }
	}
    }
  gc_wisp_objects ();
  return (package);
}

char *
package_to_alist (Package *package, int strip)
{
  char *result = (char *)NULL;
  Symbol **symbols = symbols_of_package (package);

  if (symbols != (Symbol **)NULL)
    {
      register int i;
      BPRINTF_BUFFER *buffer = bprintf_create_buffer ();
      char *packname = package->name;
      Symbol *sym;

      bprintf (buffer, "(");

      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
	{
	  static char *fullname = (char *)NULL;
	  static int fn_size = 0;
	  int name_len = package->name_len + sym->name_len + 3;
	  char *item_name;

	  if (name_len >= fn_size)
	    fullname = (char *)xrealloc (fullname, (fn_size = name_len + 20));

	  if (package->name_len && !strip)
	    sprintf (fullname, "%s::%s", packname, sym->name);
	  else
	    strcpy (fullname, sym->name);

	  item_name = strdup (wisp_readable (fullname));

	  switch (sym->values_index)
	    {
	    case 0:
	      bprintf (buffer, "(%s)", item_name);
	      break;

	    case 1:
	      bprintf (buffer, "(%s . %s)",
		       item_name, wisp_readable (sym->values[0]));
	      break;

	    default:
	      {
		register int j;

		bprintf (buffer, "(%s", item_name);
		for (j = 0; j < sym->values_index; j++)
		  bprintf (buffer, " %s", wisp_readable (sym->values[j]));
		bprintf (buffer, ")");
	      }
	    }
	  free (item_name);
	}

      free (symbols);
      bprintf (buffer, ")");
      result = buffer->buffer;
      free (buffer);
    }
  return (result);
}

static void
pf_package_to_alist (PFunArgs)
{
  char *packname = page_evaluate_string (get_positional_arg (vars, 0));
  char *strip = get_value (vars, "STRIP");
  char *result = (char *)NULL;
  Package *package = (Package *)NULL;

  if (!empty_string_p (packname))
    package = symbol_lookup_package (packname);
  else
    package = CurrentPackage;

  if (packname != (char *)NULL) free (packname);

  if (package != (Package *)NULL)
    result = package_to_alist (package, (strip != (char *)NULL));

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      free (result);
    }
}

static void
pf_alist_to_package (PFunArgs)
{
  char *alist = page_evaluate_string (get_positional_arg (vars, 0));
  char *packname = page_evaluate_string (get_positional_arg (vars, 1));

  if (!empty_string_p (alist))
    {
      Package *from = (Package *)NULL;
      Package *to = (Package *)NULL;

      from = alist_to_package (alist);

      if (from)
	{
	  Symbol **symbols = symbols_of_package (from);

	  if (packname == (char *)NULL)
	    packname = strdup (DEFAULT_PACKAGE_NAME);

	  if (symbols != (Symbol **)NULL)
	    {
	      register int i;
	      Symbol *sym, *copy;

	      to = symbol_get_package (packname);

	      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
		{
		  char *sym_name = sym->name;
		  char *temp = strstr (sym_name, "::");

		  if (temp)
		    sym_name = temp + 2;

		  copy = symbol_copy (sym, to);
		  if (temp)
		    symbol_rename (copy, sym_name);
		}
	      free (symbols);
	    }

	  symbol_destroy_package (from);
	}
    }
  if (alist) free (alist);
  if (packname) free (packname);
}

static void
pf_time (PFunArgs)
{
  unsigned long ticks = (unsigned long)time ((time_t *)0);

  bprintf_insert (page, start, "%ld", ticks);
}

static void
pf_pid (PFunArgs)
{
  pid_t pid = getpid ();

  bprintf_insert (page, start, "%ld", (unsigned long)pid);
}

/* Here's a wild one.

   <server-push>
     <html>
     <head><title>Just a Moment, please</title></head>
     <body>
     <h3>Please wait a moment, we are searching the entire Web...</h3>
     </body>
     </html>
   </server-push>

   Immediately sends this stuff down the line, but doesn't affect
   processesing or the current page. */
static void
pf_server_push (PFunArgs)
{
  PAGE *text = (PAGE *)NULL;
  static int called_yet = 0;
  static int output_fd = 0;
  char *type = page_evaluate_string (get_value (vars, "type"));

  if (body && body->buffer)
    {
      text = page_create_page ();
      page_set_contents (text, body->buffer);
      page_process_page_internal (text);

      if (text && text->buffer)
	{
	  PAGE *pushed = page_create_page ();
	  char *boundary = "Meta-HTML-server-push-boundary";

	  /* Only do this for the first time through. */
	  if (!called_yet)
	    {
	      char *nph = pagefunc_get_variable ("mhtml::unparsed-headers");
	      Symbol *sym = symbol_remove ("mhtml::unparsed-headers");

	      called_yet++;

#if defined (MHTML_STREAMS)
	      output_fd = mhtml_stdout_fileno;
#else
	      output_fd = fileno (stdout);
#endif
	      if (nph)
		bprintf (pushed, "HTTP/1.0 200\n");

	      symbol_free (sym);
	      pagefunc_set_variable ("mhtml::server-pushed", "true");
	      bprintf (pushed, "Content-type: multipart/x-mixed-replace;");
	      bprintf (pushed, "boundary=%s\n\n", boundary);
	      bprintf (pushed, "%s\n", boundary);
	    }

	  if (type == (char *)NULL)
	    type = strdup ("text/html");

	  bprintf (pushed, "Content-type: %s\n", type);
	  bprintf (pushed, "Content-length: %d\n\n", text->bindex);
	  write (output_fd, pushed->buffer, pushed->bindex);
	  write (output_fd, text->buffer, text->bindex);
	  page_free_page (text);
	  write (output_fd, "\n", 1);
	  write (output_fd, boundary, strlen (boundary));
	  write (output_fd, "\n", 1);
	  free (type);
	  page_free_page (pushed);
	}
    }
}

static char *mhtml_sort_function_name = (char *)NULL;
static int mhtml_sort_is_caseless = 0;
static int mhtml_sort_is_descending = 0;

static int
sort_with_function (const void *item1, const void *item2)
{
  char *string1 = *(char **)item1;
  char *string2 = *(char **)item2;
  int should_free = 0;
  int result;

  if (mhtml_sort_function_name)
    {
      PAGE *page = page_create_page ();

      bprintf (page, "<%s %s>", mhtml_sort_function_name, string1);
      string1 = page_evaluate_string (page->buffer);
      page->bindex = 0;
      bprintf (page, "<%s %s>", mhtml_sort_function_name, string2);
      page->buffer[page->bindex] = '\0';
      string2 = page_evaluate_string (page->buffer);

      page_free_page (page);
      should_free++;
    }

  if (string1 && !string2)
    result = 1;
  else if (string2 && !string1)
    result = -1;
  else if (!string1 && !string2)
    result = 0;
  else if (mhtml_sort_is_caseless)
    result = strcasecmp (string1, string2);
  else
    result = strcmp (string1, string2);

  if (should_free && string1) free (string1);
  if (should_free && string2) free (string2);

  if (result && mhtml_sort_is_descending)
    result = -result;

  return (result);
}

static void
pf_sort (PFunArgs)
{
  char *sortvar = page_evaluate_string (get_positional_arg (vars, 0));

  if (!empty_string_p (sortvar))
    {
      Symbol *sym = symbol_lookup (sortvar);

      /* If there is anything to sort... */
      if ((sym != (Symbol *)NULL) && (sym->values_index != 0))
	{
	  char *sorter = page_evaluate_string (get_positional_arg (vars, 1));
	  int caseless = (get_value (vars, "caseless") != (char *)NULL);
	  int descending = (get_value (vars, "descending") != (char *)NULL);

	  mhtml_sort_function_name = (char *)NULL;

	  if (!empty_string_p (sorter))
	    mhtml_sort_function_name = sorter;

	  mhtml_sort_is_caseless = caseless;
	  mhtml_sort_is_descending = descending;

	  qsort ((void *)sym->values, sym->values_index, sizeof (char *),
		 sort_with_function);

	  if (sorter) free (sorter);
	}
    }

  if (sortvar) free (sortvar);
}

static void
pf_date (PFunArgs)
{
  /* We can handle GMT and friends later. */
  char *tstring = page_evaluate_string (get_positional_arg (vars, 0));
  time_t ticks = tstring ? (time_t)atol (tstring) : (time_t)time ((time_t *)0);
  char *time_string = ctime (&ticks);
  char *temp = strchr (time_string, '\n');

  if (temp) *temp = '\0';

  bprintf_insert (page, start, "%s", time_string);
  *newstart += strlen (time_string);
  if (tstring) free (tstring);
}
