/*
 Copyright (C) 1998, 1999 Gerald L. Gay
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.
 
 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.
 
 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

#include <Xarm/DragDrop.h>

#ifdef XARM_HAS_CDE

#include <vector>
#include <Xm/XmAll.h>

/*
 * The Drag Threshold is the distance, measured in pixels, over which the
 * pointer must travel while the BSelect button (first mouse button) is held
 * down in order to start a drag. CDE defines this to be 10 pixels.
 */

#define DRAG_THRESHOLD 10

/*
 * Absolute value macro
 */

#ifndef ABS
#define ABS(x) (((x) > 0) ? (x) : (-(x)))
#endif

class FakeDragObject { };

#if defined(XARM_HAS_NAMESPACES)
typedef std::vector< XarmDragInfo<FakeDragObject> * > dragVector;
typedef std::vector< XarmDropSiteInfo<FakeDragObject> * > dropVector;
#else
typedef vector< XarmDragInfo<FakeDragObject> * > dragVector;
typedef vector< XarmDropSiteInfo<FakeDragObject> * > dropVector;
#endif

typedef XarmDragInfo<FakeDragObject> dragInfo;
typedef XarmStartDragInfo<FakeDragObject> startDragInfo;
typedef XarmDropSiteInfo<FakeDragObject>  dropSiteInfo;
typedef bool (FakeDragObject::*p_dragStart)(Widget, XEvent *, XtPointer);
typedef void (FakeDragObject::*p_call)(Widget, XtPointer, XtPointer);

static dragVector theDragList;
static dropVector theDropList;

static bool       doingDrag = false;

static void xarmTransferProc(Widget w, XtPointer clientData, XtPointer callData)
{
    dropSiteInfo *dsi = (dropSiteInfo *)clientData;

    FakeDragObject *obj = dsi->obj;
    p_call transferProc = dsi->transferProc;

    (obj->*(transferProc))(w, dsi->transferClosure, callData);

}

static void xarmDragSourceDestroyed(Widget w, XtPointer, XtPointer)
{
    // A widget that was registered as a drag source has been destroyed.

    dragVector::iterator dit = theDragList.begin();

    while ( dit != theDragList.end() ) {
        if ((*dit)->w == w) {
	    delete *dit;
	    theDragList.erase(dit);
	    return;
	}
	++dit;
    }
}

static void xarmDropSiteDestroyed(Widget w, XtPointer, XtPointer)
{

    bool foundOne;

    do {

      foundOne = false;

      dropVector::iterator dit = theDropList.begin();

      while (dit != theDropList.end()) {
	  if ((*dit)->w == w) {
	      foundOne = true;
	      delete *dit;
	      theDropList.erase(dit);
	      break;
	  }
      }

    } while (foundOne);
}

static void xarmDragMotionHandler(Widget w, XtPointer clientData, XEvent *event)
{

    static int initialX = -1;
    static int initialY = -1;

    int diffX, diffY;

    if (!doingDrag) {

        if ((initialX == -1) && (initialY == -1)) {
	    initialX = event->xmotion.x;
	    initialY = event->xmotion.y;
	}

	diffX = initialX - event->xmotion.x;
	diffY = initialY - event->xmotion.y;

	if ((ABS(diffX) >= DRAG_THRESHOLD) || (ABS(diffY) >= DRAG_THRESHOLD)) {
	    doingDrag = true;

	    // It looks like a drag! Call the object's DragStart function

	    initialX = initialY = -1;

	    dragInfo *di = (dragInfo *)clientData;
	    FakeDragObject *fdi = di->obj;
	    p_dragStart dsp = di->dragStartProc;

	    bool result = (fdi->*(dsp))(w, event, di->closure);

	    if (result == false) doingDrag = false;
	}
    }
}

static void xarmDndConvertProc(Widget w, XtPointer clientData, XtPointer callData)
{
    startDragInfo *sdi = (startDragInfo *)clientData;

    FakeDragObject *obj = sdi->obj;
    p_call convertProc = sdi->convertProc;

    (obj->*(convertProc))(w, sdi->convClosure, callData);
}

static void xarmDndFinishProc(Widget w, XtPointer clientData, XtPointer callData)
{
    startDragInfo *sdi = (startDragInfo *)clientData;

    FakeDragObject *obj = sdi->obj;
    p_call finishProc = sdi->finishProc;

    (obj->*(finishProc))(w, sdi->finishClosure, callData);

    doingDrag = false;
    delete sdi;
}

void RegisterDropSite(void *data,
		      Widget w,
		      XarmDndProtocol proto,
		      unsigned char ops,
		      ArgList argList,
		      Cardinal argCount)
{

    static XtCallbackRec transferCBRec[] = { { xarmTransferProc, NULL },
					     { NULL,             NULL } };

    transferCBRec[0].closure = data;

    DtDndDropRegister(w,
		      proto,
		      ops,
		      transferCBRec,
		      argList,
		      argCount);

    XarmDropSiteInfo<FakeDragObject> *xdsi = (XarmDropSiteInfo<FakeDragObject> *)data;

    theDropList.push_back(xdsi);

    XtAddCallback(w, XtNdestroyCallback, xarmDropSiteDestroyed, NULL);
}


void RegisterDragSource(void *data)
{
    dragInfo *di = (dragInfo *)data;

    // Make sure this widget isn't already registered

    for (dragVector::iterator dit = theDragList.begin(); dit != theDragList.end(); ++dit) {
        if ((*dit)->w == di->w) {
	    delete di;
	    return;
	}
    }

    theDragList.push_back(di);

    // Add a left mouse button handler for this widget.
    // Since we have access to client data, we can pass dragInfo * to it.

    XtAddEventHandler(di->w,
		      Button1MotionMask,
		      False,
		      (XtEventHandler)xarmDragMotionHandler,
		      (XtPointer)di);

    // Add a destroy handler to keep our vector clean

    XtAddCallback(di->w, XtNdestroyCallback, xarmDragSourceDestroyed, NULL);

}

Widget XarmBeginDragOps(void           *data,
			Widget          w,
			XEvent         *event,
			XarmDndProtocol proto,
			Cardinal        numItems,
			unsigned char   operations,
			ArgList         argList,
			Cardinal        argCount)
{

    static XtCallbackRec convertCBRec[] = { { xarmDndConvertProc, NULL },
                                            { NULL,               NULL } };

    static XtCallbackRec dragFinishCBRec[] =  { { xarmDndFinishProc, NULL },
                                                { NULL,              NULL } };

    startDragInfo *sdi = (startDragInfo *)data;

    convertCBRec[0].closure    = (XtPointer)sdi;
    dragFinishCBRec[0].closure = (XtPointer)sdi;

    return DtDndDragStart(w,
			  event,
			  proto,
			  numItems,
			  operations,
			  convertCBRec,
			  dragFinishCBRec,
			  argList,
			  argCount);
}

#endif    /* XARM_HAS_CDE */
