/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains the text input widgets
   
   Designed and implemented by Brad Myers
*/

#include <am_inc.h>

#include AM_IO__H

#include WIDGETS_ADVANCED__H
#include STANDARD_SLOTS__H
#include OPAL_ADVANCED__H  // for Am_DRAWONABLE, Am_Window_Coordinate
#include INTER_ADVANCED__H  // for Am_Set_Old_Owner_To_Me,
			    //     Am_INTER_PRIORITY_DIFF

#include WIDGETS__H
#include GEM__H
#include INTER__H
#include OPAL__H
#include REGISTRY__H
#include TEXT_FNS__H  //for Am_Set_Pending_Delete

///////////////////////////////////////////////////////////////////////////
// Text Input Box
///////////////////////////////////////////////////////////////////////////
// Uses the label of the command as the label of the box, the value of
// the command as the initial (default) value

void draw_motif_text_input (Am_Object /* self */,
			       int left, int top, int width, int height,
			       const char* string, Am_Object obj,
			       Am_Object string_obj,
			       bool active, bool key_selected,
			       Am_Font label_font,
			       const Computed_Colors_Record& rec,
			       Am_Drawonable* draw) {
  int box_left = 2; //where the text input area starts
  int box_width = width-4;
  if (key_selected) // draw box showing that the keyboard is over this item
    draw->Draw_Rectangle (Am_Key_Border_Line, Am_No_Style, left, top,
			  width, height);
  // first, draw the label as a string or object:
  if(string) {
    int str_width, ascent, descent, a, b, str_left, str_top;
    draw->Get_String_Extents (label_font, string, strlen (string), str_width,
			      ascent, descent, a, b);
    // text goes at left, centered vertically
    str_left = left + 2;
    str_top = top + (height - ascent - descent) / 2;
    // set a clip region in case string bigger than the button
    draw->Push_Clip (str_left, top+2, width-4, height-4);
    draw->Draw_Text (Am_Black, string, strlen (string), label_font,
		     str_left, str_top);
    draw->Pop_Clip ();
    box_left = left + 2 + str_width;
    box_width = width - 4 - str_width;
  }
  else if (obj.Valid ()) {
    // since a part of the button, will be offset from
    // buttons' left and top automatically.
    int obj_left = obj.Get(Am_LEFT);
    int obj_top = obj.Get(Am_TOP);
    int obj_width = obj.Get(Am_WIDTH);
    int obj_height = obj.Get(Am_HEIGHT);
    int x_offset = left - obj_left + 2;
    int y_offset = top - obj_top + (height - obj_height) / 2;
    // call the object's draw method to draw the component
    Am_Draw (obj, draw, x_offset, y_offset);
    box_left = left + 2 + obj_width;
    box_width = width - 4 - obj_width;
  }
  
  // draw the external box, leave 2 pixel border for the key-selected box
  // selected and interim selected look the same in Motif
  Am_Draw_Motif_Box(box_left+2, top+2, box_width, height-4,
		    true, rec, draw);
  
  //now draw the input string
  Am_Style text_style;
  if (active)
    text_style = Am_Black;
  else
    text_style = Am_Motif_Inactive_Stipple;

  // call the object's draw method to draw the component
  Am_Draw (string_obj, draw, left, top);
}

Am_Define_Method(Am_Draw_Method, void, text_input_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  Am_Font label_font;
  label_font = self.Get (Am_LABEL_FONT);
  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else Am_Error("String slot of widget should have string or object value");

  Am_Object string_obj = self.Get_Part(Am_TEXT_WIDGET_TEXT_OBJ);

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK)
    draw_motif_text_input(self, left, top, width, height, string,
			     obj, string_obj, active, key_selected,
			     label_font, rec, drawonable);
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

//in the height slot of the text input widget
Am_Define_Formula (int, get_text_input_height) {
  int height = 8;
  Am_Object window;
  Am_Font label_font, font;
  window = self.GV (Am_WINDOW);
  if (!window.Valid()) return 10;
  Am_Drawonable* draw = GV_a_drawonable (window, cc);
  if (!draw) return 10;
  label_font = self.GV(Am_FONT);
  //first, check the label's height
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  // contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING) string = value;
  else if (value.type == Am_OBJECT) obj = value;
  if ((const char*)string) {
    font = self.GV(Am_FONT);
    int str_width, ascent, descent, a, b, height;
    draw->Get_String_Extents (font, string, strlen (string), str_width,
			      ascent, descent, a, b);
    height = ascent + descent + 8;
  }
  else if (obj.Valid ())
    return height = (int)obj.GV(Am_HEIGHT) + 8;

  //now see if text input area is bigger
  Am_Object text = self.GV_Part(Am_TEXT_WIDGET_TEXT_OBJ);
  if (text.Valid()) {
    int text_height = (int)text.GV(Am_HEIGHT) + 8;
    if (text_height > height) height = text_height;
  }
  return height;
}

//Am_Where_Function:  For the where test of interactor.
// Return the Am_Text text_value part if click anywhere in the box region
Am_Define_Method(Am_Where_Method, Am_Object, text_input_in,
		 (Am_Object /* inter */,
		  Am_Object object, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  //object is the text input widget
  // cout  << "Testing widget " << object << endl << flush;
  if (Am_Point_In_All_Owners(object, x, y, event_window) &&
      (Am_Point_In_Obj (object, x, y, event_window).Valid())) {
    //make sure is inside the box area
    int box_left = object.Get(Am_TEXT_WIDGET_BOX_LEFT);
    int outx, outy;
    // get the coordinates with respect to the widget
    Am_Translate_Coordinates(event_window, x, y, object, outx, outy);
    if (outx > box_left) {
      Am_Object text = object.Get_Part(Am_TEXT_WIDGET_TEXT_OBJ);
      return text;
    }
  }
  return Am_No_Object;
}

Am_Define_Method(Am_Object_Method, void, text_input_command_inter_do,
		 (Am_Object inter_command)) {
  // put a copy of the finally edited text into Am_VALUE slot of
  // command obj of the widget.  Widget value itself is set by the constraint
  Am_Object text, widget, widget_command;
  Am_String edited_text;
  text = inter_command.Get(Am_OBJECT_MODIFIED);
  edited_text = text.Get(Am_TEXT);
  widget = text.Get_Owner();
  // widget.Set(Am_VALUE, Am_Narrow (edited_text)); -->set by constraint
  widget_command = widget.Get_Part(Am_COMMAND);
  if (widget_command.Valid()) {
    // set up for undo
    Am_Value old_value;
    widget_command.Get(Am_VALUE, old_value);
    widget_command.Set(Am_OLD_VALUE, old_value);
    //set the new value
    widget_command.Set(Am_VALUE, (Am_Wrapper*)edited_text);
  }
  else cout << "**No widget for " << inter_command << " in " << widget
	    << endl << flush;
}

Am_Define_String_Formula(get_text_part_value) {
  return self.GV_Part(Am_TEXT_WIDGET_TEXT_OBJ).GV(Am_TEXT);
}

// put into the text slot of the text object that is edited.  This
// sets up a circular constraint so setting the Value of the widget
// will affect the current string.
Am_Define_String_Formula(get_text_widget_value) {
  return self.GV_Owner().GV(Am_VALUE);
}

//self is the Am_Text_Input_Widget
Am_Define_Formula (int, text_widget_box_left) {
  Am_String string;
  Am_Object obj, window;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  window = self.GV(Am_WINDOW);
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  Am_Drawonable* draw = NULL;
  if (window.Valid())
    draw = GV_a_drawonable (window, cc);
  if (!draw) return 10;
  if(string.Valid()) {
    Am_Font label_font;
    label_font = self.Get(Am_LABEL_FONT);
    int str_width, ascent, descent, a, b;
    draw->Get_String_Extents (label_font, string, strlen (string), str_width,
			      ascent, descent, a, b);
    // text goes at left, centered vertically
    return 2 + str_width;
  }
  else if (obj.Valid ()) {
    int obj_width = obj.GV(Am_WIDTH);
    return 2 + obj_width;
  }
  else return 10;
}

Am_Define_Formula (int, text_widget_value_left) {
  return (int)(self.GV_Owner().GV(Am_TEXT_WIDGET_BOX_LEFT)) + 6;
}

Am_Define_Formula (int, text_widget_value_width) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    int box_left = owner.GV(Am_TEXT_WIDGET_BOX_LEFT);
    int owner_width = owner.GV(Am_WIDTH);
    return owner_width - box_left - 10;
  }
  else return 100;
}

Am_Define_Formula (int, text_widget_value_top) {
  int ret = 0;
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    int owner_height = owner.GV(Am_HEIGHT);
    int my_height = self.GV(Am_HEIGHT);
    ret = (owner_height - my_height) / 2;
  }
  return ret;
}

Am_Define_Style_Formula (text_widget_value_style) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    if ((bool)owner.GV(Am_ACTIVE)) return Am_Black;
    else return Am_Motif_Inactive_Stipple;
  }
  else return Am_Black;
}


//need special method because must pass the text_obj to start_interactor
Am_Define_Method(Am_Explicit_Widget_Run_Method, void, text_widget_start_method,
		 (Am_Object widget, Am_Value initial_value)) {
  if (initial_value.Valid()) widget.Set(Am_VALUE, initial_value);
  Am_Object inter = widget.Get_Part(Am_INTERACTOR);
  Am_Object text_obj = widget.Get_Part(Am_TEXT_WIDGET_TEXT_OBJ);
  Am_Start_Interactor(inter, text_obj);
  //need to set pending delete after starting the interactor, since
  //starting the interactor moves the cursor which turns pending
  //delete off.
  if ((bool)widget.Get(Am_WANT_PENDING_DELETE)) 
    Am_Set_Pending_Delete (text_obj, true);  //start off with pending delete
}


///////////////////////////////////////////////////////////////////////////
// Tabbing from widget to widget
///////////////////////////////////////////////////////////////////////////

//default formula for the Am_LIST_OF_TEXT_WIDGETS slot of the interactor
Am_Define_Value_List_Formula(generate_list_of_text_widgets) {
  Am_Object group = self.GV_Owner();
  Am_Value_List all_text_widgets;
  if (group.Valid()) {
    Am_Value v;
    group.GVM(Am_GRAPHICAL_PARTS, v);
    if (v.Valid()) {
      Am_Value_List all_parts = v;
      Am_Object part;
      for (all_parts.Start(); !all_parts.Last(); all_parts.Next()) {
	part = all_parts.Get();
	if (part.Is_Instance_Of(Am_Text_Input_Widget))
	  all_text_widgets.Add(part);
      }
    }
  }
  return all_text_widgets;
}


Am_Object get_next_from(Am_Value_List &all_text_widgets, bool forward) {
  if (forward) {
    all_text_widgets.Next();
    if (all_text_widgets.Last()) all_text_widgets.Start(); //wrap around
  }
  else {
    all_text_widgets.Prev();
    if (all_text_widgets.First()) all_text_widgets.End(); //wrap around
  }
  return all_text_widgets.Get();
}

Am_Define_Method(Am_Object_Method, void, tab_to_next_widget, (Am_Object cmd)) {
  Am_Object inter = cmd.Get_Owner();
  if (inter.Valid()) {
    Am_Value_List all_text_widgets = inter.Get(Am_LIST_OF_TEXT_WIDGETS);
    Am_Object last_widget, cur_widget, next_widget;
    bool forward;
    bool running = false;
    Am_Value v;
    if (all_text_widgets.Empty()) return; //nothing to do
    cmd.Get(Am_LAST_TEXT_WIDGET, v);
    if (v.Valid()) last_widget = v;
    // see if going forward or backwards
    inter = cmd.Get_Owner();
    Am_Input_Char ic = Am_Input_Char::Narrow(inter.Get(Am_START_CHAR));
    //go forward if not shifted
    forward = !ic.shift;
    // search to see which part is active, if any
    for (all_text_widgets.Start(); !all_text_widgets.Last();
	 all_text_widgets.Next()) {
      cur_widget = all_text_widgets.Get();
      inter = cur_widget.Get_Part(Am_INTERACTOR);
      inter.Get(Am_CURRENT_STATE, v);
      if (v.Valid()) { //then this is it
	running = true;
	next_widget = get_next_from(all_text_widgets, forward);
	break;
      }
    }
    if (!next_widget.Valid()) {
      //nothing running, see if have current
      if (last_widget.Valid()) {
	all_text_widgets.Start();
	if (all_text_widgets.Member(last_widget)) {
	  next_widget = get_next_from(all_text_widgets, forward);
	}
      }
      if (!next_widget.Valid()) { // just use first in the list
	all_text_widgets.Start();
	next_widget = all_text_widgets.Get();
      }
    }
    if (last_widget.Valid()) {
      last_widget.Set(Am_KEY_SELECTED, false);
    }
    if (running) {
      Am_Stop_Widget(cur_widget);
      cur_widget.Set(Am_KEY_SELECTED, false);
    }
    if (next_widget.Valid()) {
      cmd.Set(Am_LAST_TEXT_WIDGET, next_widget);
      next_widget.Set(Am_KEY_SELECTED, true);
      Am_Start_Widget(next_widget);
    }
  }
}

  ///////////////////////////////////////////////////////////////////////////
  // Initialization
  ///////////////////////////////////////////////////////////////////////////

// exported objects

Am_Object Am_Text_Input_Widget;

Am_Object Am_Tab_To_Next_Widget_Command;
Am_Object Am_Tab_To_Next_Widget_Interactor;


void Am_Text_Widgets_Initialize () {
  Am_Object inter; // interactor in the widget
  Am_Object command_obj; 
  Am_Object_Advanced obj_adv; // to get at advanced features like
			       // local-only and demons.

  //////////// Command Object /////////////



  ///////////////////////////////////////////////////////////////////////////
  // Text Input
  ///////////////////////////////////////////////////////////////////////////

  Am_Font bold_font(Am_FONT_FIXED, true);
  
  Am_Text_Input_Widget = Am_Aggregate.Create("Text_Input_Widget")
    .Set (Am_VALUE, "") //empty string is initial value
    .Set (Am_VALUE, get_text_part_value) //in case UNDO
    .Set (Am_ACTIVE, Am_Active_From_Command)
    .Set (Am_ACTIVE_2, true) // used by interactive tools
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_KEY_SELECTED, false)
    .Set (Am_FONT, Am_Default_Font)
    .Set (Am_LABEL_FONT, bold_font)
    .Set (Am_FILL_STYLE, Am_Amulet_Purple)
    .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
    .Set (Am_DRAW_METHOD, text_input_draw)
    .Set (Am_WIDTH, 150)
    .Set (Am_HEIGHT, get_text_input_height)
    .Set (Am_WANT_PENDING_DELETE, true)
    .Set (Am_TEXT_WIDGET_BOX_LEFT, text_widget_box_left)
    .Set (Am_SET_COMMAND_OLD_OWNER, Am_Set_Old_Owner_To_Me)
    .Set (Am_WIDGET_START_METHOD, text_widget_start_method)
    .Set (Am_WIDGET_ABORT_METHOD, Am_Standard_Widget_Abort_Method)
    .Set (Am_WIDGET_STOP_METHOD, Am_Standard_Widget_Stop_Method)
    .Add_Part (Am_COMMAND, Am_Command.Create("Text_Input_Command")
	       .Set (Am_LABEL, "Text_Input")
	       .Set (Am_VALUE, ""))
    .Add_Part (Am_TEXT_WIDGET_TEXT_OBJ,
	       Am_Text.Create("Value Text in Text_Widget")
	       .Set(Am_LEFT, text_widget_value_left)
	       .Set(Am_TOP,  text_widget_value_top)
	       .Set(Am_WIDTH, text_widget_value_width)
	       .Set(Am_FONT, Am_Font_From_Owner)
	       .Set(Am_LINE_STYLE, text_widget_value_style)
	       .Set(Am_TEXT, get_text_widget_value)
	       )
    .Add_Part (Am_INTERACTOR,
	 	inter = Am_Text_Edit_Interactor.Create("inter_in_text_input")
	        .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
		.Set (Am_START_WHERE_TEST, text_input_in)
	        .Set (Am_WANT_PENDING_DELETE,
		      Am_From_Owner(Am_WANT_PENDING_DELETE))
		.Set (Am_ACTIVE, Am_Active_And_Active2)
		)
    .Set (Am_REAL_STRING_OR_OBJ, Am_Get_Real_String_Or_Obj)
    ; 
  inter.Get_Part(Am_COMMAND)
    .Set(Am_IMPLEMENTATION_PARENT, Am_Get_Owners_Command)
    .Set(Am_DO_METHOD, text_input_command_inter_do)
    .Set_Name("Command_In_Text_Input_Widget")
    ;

  obj_adv = (Am_Object_Advanced&)Am_Text_Input_Widget;
  obj_adv.Get_Slot(Am_VALUE).Set_Single_Constraint_Mode(false);


  Am_Object obj = Am_Text_Input_Widget.Get_Part(Am_TEXT_WIDGET_TEXT_OBJ);
  obj_adv = (Am_Object_Advanced&)obj;
  obj_adv.Get_Slot(Am_TEXT).Set_Single_Constraint_Mode(false);
 
  obj_adv = (Am_Object_Advanced&)Am_Text_Input_Widget;

  obj_adv.Get_Slot (Am_REAL_STRING_OR_OBJ) 
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_KEY_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_ACTIVE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_WIDGET_LOOK)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FONT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  
  // all the next slots should not be inherited
  obj_adv.Get_Slot(Am_KEY_SELECTED).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_ACTIVE).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_ACTIVE_2).Set_Inherit_Rule(Am_COPY);

  ///////////////////////////////////////////////////////////////////////////
  // Moving to next widget
  ///////////////////////////////////////////////////////////////////////////

  Am_Tab_To_Next_Widget_Command = Am_Command.Create("Tab_To_Next_Widget_Cmd")
    .Set(Am_LABEL, "To Next Widget")
    .Set(Am_ACTIVE, true)
    .Set(Am_ACCELERATOR, NULL)
    .Set(Am_DO_METHOD, tab_to_next_widget)
    .Set(Am_IMPLEMENTATION_PARENT, Am_NOT_USUALLY_UNDONE) //not undo-able
    ;
  Am_Tab_To_Next_Widget_Interactor =
    Am_One_Shot_Interactor.Create("Tab_To_Next_Widget_Interactor") 
    .Set(Am_START_WHEN, Am_Input_Char("ANY_TAB"))
    .Set(Am_PRIORITY, Am_INTER_PRIORITY_DIFF+5 ) //must be higher than running
    // fill next slot with a list of the tab widgets, or a formula to
    // compute it
    .Set(Am_LIST_OF_TEXT_WIDGETS, generate_list_of_text_widgets)
    .Add_Part(Am_COMMAND, Am_Tab_To_Next_Widget_Command.Create())
    ;

}
