/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef TPen
#define TPen TPen

#include <toad/font.hh>
#include <toad/color.hh>
#include <toad/bitmap.hh>

class TRegion;
class TRectangle;
class TScreenPencil;


class TPen:
	public TOADBase
{
		friend TWindow;
		friend TBitmap;
		friend TColor;
		friend TScreenPencil;

		int _dx, _dy;			// translation

	public:
	
		// rename this into EMode
		enum EPenMode	{
			NORMAL=3,
			XOR=6,
			INVERT=10
		};

		// rename this into ELineStyle
		enum EPenLineStyle {
			SOLID=1,
			DASH,
			DOT,
			DASHDOT,
			DASHDOTDOT
		};

		TPen(TBitmap*);
		TPen(TWindow*);
		virtual ~TPen();

		// origin
		//-----------------------
		//. Set the origin to (x, y).
		void SetOrigin(int x,int y){_dx=x; _dy=y;}
		//. Move the origin by (dx, dy).
		void Translate(int dx,int dy){_dx+=dx; _dy+=dy;}
		int OriginX() { return _dx; }
		int OriginY() { return _dy; }

		// color & pattern
		//-----------------------
		void SetBitmap(TBitmap*);
		void SetColor(const TColor&);
		void SetColor(byte r,byte g,byte b) {
			SetColor(TColor(r,g,b));
		}
		void SetColor(const TRGB &c) {
			SetColor(TColor(c.r,c.g,c.b));
		}
		void SetColor(TColor::ESystemColor c) {
			SetColor(TColor(c));
		}
		void SetColor(TColor::EColor16 c) {
			SetColor(TColor(c));
		}

		void SetFillColor(const TColor&);
		void SetFillColor(byte r, byte g, byte b) {
			SetFillColor(TColor(r,g,b));
		}
		void SetFillColor(const TRGB &c) {
			SetFillColor(TColor(c.r,c.g,c.b));
		}

		void SetLineColor(const TColor&);
		void SetLineColor(byte r, byte g, byte b) {
			SetLineColor(TColor(r,g,b));
		}
		void SetLineColor(const TRGB &c) {
			SetLineColor(TColor(c.r,c.g,c.b));
		}

		void SetBackColor(const TColor&);
		void SetBackColor(byte r, byte g, byte b) {
			SetBackColor(TColor(r,g,b));
		}

		// more parameters
		//-----------------------
		TFont* SetFont(TFont*);
		void SetMode(EPenMode);
		void SetLineWidth(int);
		void SetLineStyle(EPenLineStyle);
		void SetColorMode(TColor::EDitherMode);
		void SetClipChildren(bool);

		// clipping
		//-----------------------
		void SetClipRegion(TRegion*);
		void SetClipRect(const TRectangle&);
		
		void ClrClipBox();
		void GetClipBox(TRectangle*) const;

		void operator&=(TRectangle&);
		void operator|=(TRectangle&);

		// point
		//-----------------------
		void DrawPoint(int x,int y) const;
		void DrawPoint(const TPoint &p) const { DrawPoint(p.x, p.y); }
		
		// line
		//-----------------------
		void DrawLine(int x1,int y1,int x2,int y2) const;
		void DrawLine(const TPoint &a, const TPoint &b) const {
			DrawLine(a.x, a.y, b.x, b.y);
		}
		void DrawLines(const TPoint *points, int n) const;

		// rectangle
		//-----------------------
		void DrawRectangle(int x,int y,int w,int h) const;
		void DrawRectangle(const TRectangle &r) const {DrawRectangle(r.x,r.y,r.w,r.h);}
		void DrawRectangle(const TRectangle *r) const {DrawRectangle(r->x,r->y,r->w,r->h);}

		void DrawRectanglePC(int x,int y,int w,int h) const;
		void DrawRectanglePC(const TRectangle &r) const {DrawRectanglePC(r.x,r.y,r.w,r.h);}
		void DrawRectanglePC(const TRectangle *r) const {DrawRectanglePC(r->x,r->y,r->w,r->h);}

		void DrawRectangle(const TPoint &a, const TPoint &b) const {
			DrawRectanglePC(a.x, a.y, b.x-a.x, b.y-a.y);
		}

		void FillRectangle(int x,int y,int w,int h) const;
		void FillRectangle(const TRectangle &r) const {FillRectangle(r.x,r.y,r.w,r.h);}
		void FillRectangle(const TRectangle *r) const {FillRectangle(r->x,r->y,r->w,r->h);}

		void FillRectanglePC(int x,int y,int w,int h) const;
		void FillRectanglePC(const TRectangle &r) const {FillRectanglePC(r.x,r.y,r.w,r.h);}
		void FillRectanglePC(const TRectangle *r) const {FillRectanglePC(r->x,r->y,r->w,r->h);}

		void FillRectangle(const TPoint &a, const TPoint &b) const {
			FillRectanglePC(a.x, a.y, b.x-a.x, b.y-a.y);
		}

		// circle
		//-----------------------
		void DrawCircle(int x,int y,int w,int h) const;
		void DrawCircle(const TRectangle &r) const {DrawCircle(r.x,r.y,r.w,r.h);}
		void DrawCircle(const TRectangle *r) const {DrawCircle(r->x,r->y,r->w,r->h);}

		void DrawCirclePC(int x,int y,int w,int h) const;
		void DrawCirclePC(const TRectangle &r) const {DrawCirclePC(r.x,r.y,r.w,r.h);}
		void DrawCirclePC(const TRectangle *r) const {DrawCirclePC(r->x,r->y,r->w,r->h);}
		
		void DrawCircle(const TPoint &a, const TPoint &b) const {
			DrawCirclePC(a.x, a.y, b.x-a.x, b.y-a.y);
		}

		void FillCircle(int x,int y,int w,int h) const;
		void FillCircle(const TRectangle &r) const {FillCircle(r.x,r.y,r.w,r.h);}
		void FillCircle(const TRectangle *r) const {FillCircle(r->x,r->y,r->w,r->h);}

		void FillCirclePC(int x,int y,int w,int h) const;
		void FillCirclePC(const TRectangle &r) const {FillCirclePC(r.x,r.y,r.w,r.h);}
		void FillCirclePC(const TRectangle *r) const {FillCirclePC(r->x,r->y,r->w,r->h);}
		
		void FillCircle(const TPoint &a, const TPoint &b) const {
			FillCirclePC(a.x, a.y, b.x-a.x, b.y-a.y);
		}

		// arc
		//-----------------------		
		void DrawArc(int x,int y,int w,int h, double r1, double r2) const;
		void DrawArc(const TRectangle &r, double r1, double r2) const {
			DrawArc(r.x,r.y,r.w,r.h, r1, r2);
		}
		void DrawArc(const TRectangle *r, double r1, double r2) const {
			DrawArc(r->x,r->y,r->w,r->h, r1, r2);
		}

		void DrawArcPC(int x,int y,int w,int h, double r1, double r2) const;
		void DrawArcPC(const TRectangle &r, double r1, double r2) const {
			DrawArcPC(r.x,r.y,r.w,r.h, r1, r2);
		}
		void DrawArcPC(const TRectangle *r, double r1, double r2) const {
			DrawArcPC(r->x,r->y,r->w,r->h, r1, r2);
		}

		void DrawArc(const TPoint &a, const TPoint &b, double r1, double r2) const {
			DrawArcPC(a.x, a.y, b.x-a.x, b.y-a.y, r1, r2);
		}

		void FillArc(int x,int y,int w,int h, double r1, double r2) const;
		void FillArc(const TRectangle &r, double r1, double r2) const {
			FillArc(r.x,r.y,r.w,r.h, r1, r2);
		}
		void FillArc(const TRectangle *r, double r1, double r2) const {
			DrawArc(r->x,r->y,r->w,r->h, r1, r2);
		}

		void FillArcPC(int x,int y,int w,int h, double r1, double r2) const;
		void FillArcPC(const TRectangle &r, double r1, double r2) const {
			FillArcPC(r.x,r.y,r.w,r.h, r1, r2);
		}
		void FillArcPC(const TRectangle *r, double r1, double r2) const {
			FillArcPC(r->x,r->y,r->w,r->h, r1, r2);
		}

		void FillArc(const TPoint &a, const TPoint &b, double r1, double r2) const {
			FillArcPC(a.x, a.y, b.x-a.x, b.y-a.y, r1, r2);
		}

		// polygon
		//-----------------------
		void FillPolygon(const TPoint *points, int n) const;
		void DrawPolygon(const TPoint *points, int n) const;

		// bezier curves
		//-----------------------
		void DrawBezier(int x1,int y1, int x2,int y2, int x3,int y3, int x4,int y4);
		void DrawBezier(double x1,double y1, double x2,double y2, double x3,double y3, double x4,double y4);
		void DrawBezier(const TPoint *points){ DrawPolyBezier(points,4); }
		void DrawBezier(const TDPoint *points){ DrawPolyBezier(points,4); }

		void DrawPolyBezier(const TPoint *points, int n);
		void DrawPolyBezier(const TPolygon &p) { DrawPolyBezier(p.begin(), p.size()); }
		void DrawPolyBezier(const TDPoint *points, int n);
		void DrawPolyBezier(const TDPolygon &p) { DrawPolyBezier(p.begin(), p.size()); }

		void FillPolyBezier(const TPoint *points, int n);
		void FillPolyBezier(const TPolygon &p) { FillPolyBezier(p.begin(), p.size()); }
		void FillPolyBezier(const TDPoint *points, int n);
		void FillPolyBezier(const TDPolygon &p) { FillPolyBezier(p.begin(), p.size()); }

		static void Poly2Bezier(const TPoint *src, int n, TPolygon &dst);
		static void Poly2Bezier(const TPolygon &p, TPolygon &d) { Poly2Bezier(p.begin(), p.size(), d); }
		static void Poly2Bezier(const TDPoint *src, int n, TDPolygon &dst);
		static void Poly2Bezier(const TDPolygon &p, TDPolygon &d) { Poly2Bezier(p.begin(), p.size(), d); }

		// 3D rectangle
		//-----------------------
		void Draw3DRectangle(int x, int y, int w, int h);
		void Draw3DRectangle(const TRectangle &r) {Draw3DRectangle(r.x,r.y,r.w,r.h);}
		void Draw3DRectangle(const TRectangle *r) {Draw3DRectangle(r->x,r->y,r->w,r->h);}

		// text string
		//-----------------------
		int TextWidth(const char* string) const;
		int TextWidth(const char* string, int len) const;
		int TextWidth(const string&) const;
		int Ascent() const;
		int Descent() const;
		int Height() const;
		void DrawString(int x, int y, const char*, int len) const;
		void DrawString(int x, int y, const string&) const;
		void FillString(int x, int y, const char*, int len) const;
		void FillString(int x, int y, const string&) const;
		int DrawTextWidth(int x, int y, const string &text, unsigned width) const;
		// void DrawTextAspect(int x,int y,const char* text,double xa,double ya);

		// bitmap
		//-----------------------
		void DrawBitmap(int x,int y, const TBitmap*) const;
		void DrawBitmap(int x,int y, const TBitmap&) const;
		void DrawBitmap(int,int,const TBitmap*, int,int,int,int) const;
		void DrawBitmap(int,int,const TBitmap&, int,int,int,int) const;

		// ops with cursor
		//-----------------------
		void MoveTo(int x, int y) { _pos.Set(x,y); }
		void MoveTo(const TPoint &p) { _pos = p; }
		void LineTo(int x, int y) { DrawLine(_pos.x, _pos.y, x,y); _pos.Set(x,y); }
		void LineTo(const TPoint &p) { DrawLine(_pos.x, _pos.y, p.x,p.y); _pos = p; }
		void CurveTo(int x2, int y2, int x3, int y3, int x4, int y4) {
			DrawBezier(_pos.x, _pos.y, x2,y2, x3,y3, x4,y4);
			_pos.Set(x4,y4);
		}
		void CurveTo(const TPoint &p2, const TPoint &p3, const TPoint &p4) {
			DrawBezier(_pos.x, _pos.y, p2.x,p2.y, p3.x,p3.y, p4.x,p4.y);
			_pos=p4;
		}
		
	private:
		TPoint _pos;
	
		TColor o_color, 			// outline/only color
					 f_color; 			// fill color
		bool two_colors:1;		// `true' when fill- and outline color are different
		bool using_bitmap:1;	// `true' after `SetBitmap'
		bool bDeleteRegion:1;	// delete `region' in destructor
		TColor::EDitherMode cmode;
		int width;
		EPenLineStyle style;
		TWindow *wnd;
		TBitmap *bmp;
		TFont	*font;
	public:
//#warning "'TRegion *region' is public"
		TRegion	*region;
	private:
		void _SetLineAttributes();
		void _init();
		
		#ifdef _TOAD_PRIVATE
		public:
		#endif

		_TOAD_GC o_gc, f_gc;
		_TOAD_DRAWABLE x11drawable; // either window or pixmap
};

#endif
