/*
 * File:      wb_ps.cc
 * Purpose:     Device context implementation (PostScript)
 * Author:      Julian Smart
 * Created:     1993
 * Updated:	August 1994
 * RCS_ID:      $Id: wb_ps.cc,v 1.5 1994/08/14 21:34:01 edz Exp $
 * Copyright:   (c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "@(#)wb_ps.cc	1.2 5/9/94";

#ifdef __GNUG__
#pragma implementation "wx_dcps.h"
#pragma implementation
#pragma interface
#endif

#include "common.h"
#include "wx_frame.h"
#include "wx_dcps.h"
#include "wx_utils.h"
#include "wx_dialg.h"
#include "wx_main.h"
#include "wx_lbox.h"
#include "wx_rbox.h"
#include "wx_buttn.h"
#include "wx_choic.h"
#include "wx_check.h"
#include "wx_messg.h"
#include "wx_txt.h"
#include "wx_mtxt.h"

// If not MS C++, don't include wx.h: we'll just include
// the minimum set of files.
// If MS C++, we'll use a precompiled header instead.
#if !defined(_MSC_VER) && !defined(wx_wxh)
#define wx_wxh
#endif

#include "wx.h"

#if USE_POSTSCRIPT

class wxCanvas;

#include "wx_privt.h"

#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

// Declarations local to this file
#define YSCALE(y) (yorigin - (y))

#ifdef wx_x
Bool XPrinterDialog (void);
#endif

// Determine the Default Postscript Previewer
// available on the platform
#if defined(sun) && defined(wx_xview)
// OpenWindow/NeWS's Postscript Previewer
# define PS_VIEWER_PROG "pageview"
#elif defined(__sgi)
// SGI's Display Postscript Previewer
# define PS_VIEWER_PROG "dps"
#elif defined(wx_x)
// Front-end to ghostscript 
# define PS_VIEWER_PROG "ghostview"
#else
// Windows ghostscript/ghostview
# define PS_VIEWER_PROG NULL
#endif

static char *wx_preview_command = copystring(PS_VIEWER_PROG);
static char *wx_printer_command = copystring("lpr");
static char *wx_printer_flags = NULL;
static Bool wx_printer_orientation = PS_PORTRAIT;
static int wx_printer_mode = PS_PREVIEW;

#define _MAXPATHLEN 500

/* See "wxspline.cc" and "xfspline.cc" */
#if USE_XFIG_SPLINE_CODE
static const char *wxPostScriptHeaderSpline = " \
/DrawSplineSection {\n\
	/y3 exch def\n\
	/x3 exch def\n\
	/y2 exch def\n\
	/x2 exch def\n\
	/y1 exch def\n\
	/x1 exch def\n\
	/xa x1 x2 x1 sub 0.666667 mul add def\n\
	/ya y1 y2 y1 sub 0.666667 mul add def\n\
	/xb x3 x2 x3 sub 0.666667 mul add def\n\
	/yb y3 y2 y3 sub 0.666667 mul add def\n\
	x1 y1 lineto\n\
	xa ya xb yb x3 y3 curveto\n\
	} def\n\
";
#else
// No extra PS header for this spline implementation.
static const char *wxPostScriptHeaderSpline = NULL;

#endif /* USE_XFIG_SPLINE_CODE */


wxPostScriptDC::wxPostScriptDC (wxPostScriptDC * old_dc)
{
  __type = wxTYPE_DC_POSTSCRIPT;
  device = wxDEVICE_EPS;
  min_x = 0;
  min_y = 0;
  max_x = 0;
  max_y = 0;
  clipping = FALSE;

  ok = FALSE;
  pstream = NULL;
  title = NULL;
  filename = NULL;

  current_logical_function = -1;
  font = NULL;
  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  current_pen = NULL;
  current_brush = NULL;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = *wxBLACK;
//  current_text_background = NULL;

  ok = TRUE;

  currentRed = 255;
  currentGreen = 255;
  currentBlue = 0;

  Colour = wxColourDisplay ();
  SetBrush (wxWHITE_BRUSH);
  SetPen (wxBLACK_PEN);
}



wxPostScriptDC::wxPostScriptDC (char *file, Bool interactive)
{
  __type = wxTYPE_DC_POSTSCRIPT;
  wx_interactive = interactive;
  font = NULL;
  device = wxDEVICE_EPS;
  clipping = FALSE;

  logical_origin_x = 0;
  logical_origin_y = 0;

  device_origin_x = 0;
  device_origin_y = 0;

  logical_scale_x = 1.0;
  logical_scale_y = 1.0;

  user_scale_x = 1.0;
  user_scale_y = 1.0;

  mapping_mode = MM_TEXT;

  yorigin = 792;		// For EPS output

  min_x = 1000.0;
  min_y = 1000.0;
  max_x = -1000.0;
  max_y = -1000.0;
  title = NULL;
  if (file)
    filename = copystring (file);
  else
    filename = NULL;

  pstream = NULL;

#ifdef wx_x	/* Why not let Windows use this? */
  if (interactive)
    if ((ok = XPrinterDialog () ) == FALSE) return;
  else
    ok = TRUE;

  if (interactive && (wx_printer_mode == PS_PREVIEW  || (!file && wx_printer_mode == PS_PRINTER)))
    {
      // For PS_PRINTER action this depends on a Unix-style print spooler
      // since the wx_printer_file can be destroyed during a session
      // @@@ TODO: a Windows-style answer for non-Unix
      char userId[256];
      wxGetUserId (userId, sizeof (userId) / sizeof (char));
      strcpy (wx_printer_file, "/tmp/preview_");
      strcat (wx_printer_file, userId);
      strcat (wx_printer_file, ".ps");
      file = wx_printer_file;
    }
#endif

  if (device == wxDEVICE_EPS)
    {
      if (!file)
	{
	  file = wxSaveFileSelector ("PostScript", "ps");
	  if (!file)
	    {
	      ok = FALSE;
	      return;
	    }
	  strcpy (wx_printer_file, file);
	  ok = TRUE;
	}
      else
	strcpy (wx_printer_file, file);

      pstream = new ofstream (wx_printer_file);
      if (!pstream)
	{
	  wxMessageBox (wxSTR_ERROR, "Cannot open file!", wxOK);
	  ok = FALSE;
	  return;
	}
      ok = TRUE;
    }
  current_logical_function = -1;
  current_pen = NULL;
  current_brush = NULL;
  current_background_brush = wxWHITE_BRUSH;
  current_text_foreground = *wxBLACK;
//  current_text_background = NULL;
  Colour = wxColourDisplay ();

  currentRed = 255;
  currentGreen = 255;
  currentBlue = 0;

  SetBrush (wxBLACK_BRUSH);
  SetPen (wxBLACK_PEN);
}

wxPostScriptDC::~wxPostScriptDC (void)
{
  if (pstream)
    delete pstream;
  if (filename)
    delete[]filename;
}

void wxPostScriptDC::SetClippingRegion (float cx, float cy, float cw, float ch)
{
  if (clipping)
    return;

  clipping = TRUE;
  *pstream << "gsave\n";
  *pstream << "newpath\n";
  *pstream << cx << " " << YSCALE (cy) << " moveto\n";
  *pstream << cx + cw << " " << YSCALE (cy) << " lineto\n";
  *pstream << cx + cw << " " << YSCALE (cy + ch) << " lineto\n";
  *pstream << cx << " " << YSCALE (cy + ch) << " lineto\n";
  *pstream << "closepath clip newpath\n";
}

void wxPostScriptDC::DestroyClippingRegion (void)
{
  if (clipping)
    {
      clipping = FALSE;
      *pstream << "grestore\n";
    }
}

void wxPostScriptDC::Clear (void)
{
}

void wxPostScriptDC::FloodFill (float x, float y, wxColour * col, int style)
{
}

Bool wxPostScriptDC::GetPixel (float x, float y, wxColour * col)
{
  return FALSE;
}

void wxPostScriptDC::IntDrawLine (int x1, int y1, int x2, int y2)
{
  DrawLine ((float) x1, (float) y1, (float) x2, (float) y2);
}

void wxPostScriptDC::CrossHair (int x, int y)
{
}

void wxPostScriptDC::DrawLine (float x1, float y1, float x2, float y2)
{
  if (current_pen)
    SetPen (current_pen);
  *pstream << "newpath\n";
  *pstream << x1 << " " << YSCALE (y1) << " moveto\n";
  *pstream << x2 << " " << YSCALE (y2) << " lineto\n";
  *pstream << "stroke\n";
  CalcBoundingBox (x1, YSCALE (y1));
  CalcBoundingBox (x2, YSCALE (y2));
}

void wxPostScriptDC::DrawArc (float x1, float y1, float x2, float y2, float xc, float yc)
{
  if (current_pen)
    SetPen (current_pen);
}

void wxPostScriptDC::DrawPoint (float x, float y)
{
  if (current_pen)
    SetPen (current_pen);
  *pstream << "newpath\n";
  *pstream << x << " " << YSCALE (y) << " moveto\n";
  *pstream << (x+1) << " " << YSCALE (y) << " lineto\n";
  *pstream << "stroke\n";
  CalcBoundingBox (x, YSCALE (y));
}

void wxPostScriptDC::DrawPolygon (int n, wxPoint points[], float xoffset, float yoffset, int fillStyle)
{
  if (n > 0)
    {
      if (current_brush && current_brush->GetStyle () != wxTRANSPARENT)
	{
	  SetBrush (current_brush);
	  *pstream << "newpath\n";

	  float xx = points[0].x + xoffset;
	  float yy = YSCALE (points[0].y + yoffset);
	  *pstream << xx << " " << yy << " moveto\n";
	  CalcBoundingBox (xx, yy);

	  int i;
	  for (i = 1; i < n; i++)
	    {
	      xx = points[i].x + xoffset;
	      yy = YSCALE (points[i].y + yoffset);
	      *pstream << xx << " " << yy << " lineto\n";
	      CalcBoundingBox (xx, yy);
	    }
	  *pstream << "fill\n";
	}

      if (current_pen && current_pen->GetStyle () != wxTRANSPARENT)
	{
	  SetPen (current_pen);
	  *pstream << "newpath\n";

	  float xx = points[0].x + xoffset;
	  float yy = YSCALE (points[0].y + yoffset);
	  *pstream << xx << " " << yy << " moveto\n";
	  CalcBoundingBox (xx, yy);

	  int i;
	  for (i = 1; i < n; i++)
	    {
	      xx = points[i].x + xoffset;
	      yy = YSCALE (points[i].y + yoffset);
	      *pstream << xx << " " << yy << " lineto\n";
	      CalcBoundingBox (xx, yy);
	    }

	  // Close the polygon
	  xx = points[0].x + xoffset;
	  yy = YSCALE (points[0].y + yoffset);
	  *pstream << xx << " " << yy << " lineto\n";

	  // Output the line
	  *pstream << "stroke\n";
	}
    }
}

void wxPostScriptDC::DrawLines (int n, wxIntPoint points[], int xoffset, int yoffset)
{
  if (n > 0)
    {
      if (current_pen)
	SetPen (current_pen);

      *pstream << "newpath\n";

      float xx = (float) (points[0].x + xoffset);
      float yy = (float) (YSCALE (points[0].y + yoffset));
      *pstream << xx << " " << yy << " moveto\n";
      CalcBoundingBox (xx, yy);

      int i;
      for (i = 1; i < n; i++)
	{
	  xx = (float) (points[i].x + xoffset);
	  yy = (float) (YSCALE (points[i].y + yoffset));
	  *pstream << xx << " " << yy << " lineto\n";
	  CalcBoundingBox (xx, yy);
	}
      *pstream << "stroke\n";
    }
}

void wxPostScriptDC::DrawLines (int n, wxPoint points[], float xoffset, float yoffset)
{
  if (n > 0)
    {
      if (current_pen)
	SetPen (current_pen);

      *pstream << "newpath\n";

      float xx = points[0].x + xoffset;
      float yy = YSCALE (points[0].y + yoffset);
      *pstream << xx << " " << yy << " moveto\n";
      CalcBoundingBox (xx, yy);

      int i;
      for (i = 1; i < n; i++)
	{
	  xx = points[i].x + xoffset;
	  yy = YSCALE (points[i].y + yoffset);
	  *pstream << xx << " " << yy << " lineto\n";
	  CalcBoundingBox (xx, yy);
	}
      *pstream << "stroke\n";
    }
}

void wxPostScriptDC::DrawRectangle (float x, float y, float width, float height)
{
  if (current_brush && current_brush->GetStyle () != wxTRANSPARENT)
    {
      SetBrush (current_brush);

      *pstream << "newpath\n";
      *pstream << x << " " << YSCALE (y) << " moveto\n";
      *pstream << x + width << " " << YSCALE (y) << " lineto\n";
      *pstream << x + width << " " << YSCALE (y + height) << " lineto\n";
      *pstream << x << " " << YSCALE (y + height) << " lineto\n";
      *pstream << "closepath\n";
      *pstream << "fill\n";

      CalcBoundingBox (x, YSCALE (y));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
  if (current_pen && current_pen->GetStyle () != wxTRANSPARENT)
    {
      SetPen (current_pen);

      *pstream << "newpath\n";
      *pstream << x << " " << YSCALE (y) << " moveto\n";
      *pstream << x + width << " " << YSCALE (y) << " lineto\n";
      *pstream << x + width << " " << YSCALE (y + height) << " lineto\n";
      *pstream << x << " " << YSCALE (y + height) << " lineto\n";
      *pstream << "closepath\n";
      *pstream << "stroke\n";

      CalcBoundingBox (x, YSCALE (y));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
}

void wxPostScriptDC::DrawRoundedRectangle (float x, float y, float width, float height, float radius)
{
  if (current_brush && current_brush->GetStyle () != wxTRANSPARENT)
    {
      SetBrush (current_brush);
      // Draw rectangle anticlockwise
      *pstream << "newpath\n";
      *pstream << x + radius << " " << YSCALE (y + radius) << " " << radius << " 90 180 arc\n";

      *pstream << x << " " << YSCALE (y + radius) << " moveto\n";

      *pstream << x + radius << " " << YSCALE (y + height - radius) << " " << radius << " 180 270 arc\n";
      *pstream << x + width - radius << " " << YSCALE (y + height) << " lineto\n";

      *pstream << x + width - radius << " " << YSCALE (y + height - radius) << " " << radius << " 270 0 arc\n";
      *pstream << x + width << " " << YSCALE (y + radius) << " lineto\n";

      *pstream << x + width - radius << " " << YSCALE (y + radius) << " " << radius << " 0 90 arc\n";

      *pstream << x + radius << " " << YSCALE (y) << " lineto\n";

      *pstream << "closepath\n";

      *pstream << "fill\n";

      CalcBoundingBox (x, YSCALE (y));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
  if (current_pen && current_pen->GetStyle () != wxTRANSPARENT)
    {
      SetPen (current_pen);
      // Draw rectangle anticlockwise
      *pstream << "newpath\n";
      *pstream << x + radius << " " << YSCALE (y + radius) << " " << radius << " 90 180 arc\n";

      *pstream << x << " " << YSCALE (y + height - radius) << " lineto\n";

      *pstream << x + radius << " " << YSCALE (y + height - radius) << " " << radius << " 180 270 arc\n";
      *pstream << x + width - radius << " " << YSCALE (y + height) << " lineto\n";

      *pstream << x + width - radius << " " << YSCALE (y + height - radius) << " " << radius << " 270 0 arc\n";
      *pstream << x + width << " " << YSCALE (y + radius) << " lineto\n";

      *pstream << x + width - radius << " " << YSCALE (y + radius) << " " << radius << " 0 90 arc\n";

      *pstream << x + radius << " " << YSCALE (y) << " lineto\n";

      *pstream << "closepath\n";

      *pstream << "stroke\n";

      CalcBoundingBox (x, YSCALE (y));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
}

void wxPostScriptDC::DrawEllipse (float x, float y, float width, float height)
{
  if (current_brush && current_brush->GetStyle () != wxTRANSPARENT)
    {
      SetBrush (current_brush);

      *pstream << "newpath\n";
      *pstream << x + width / 2 << " " << YSCALE (y + height / 2) << " ";
      *pstream << width / 2 << " " << height / 2 << " 0 360 ellipse\n";
      *pstream << "fill\n";

      CalcBoundingBox (x - width, YSCALE (y - height));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
  if (current_pen && current_pen->GetStyle () != wxTRANSPARENT)
    {
      SetPen (current_pen);

      *pstream << "newpath\n";
      *pstream << x + width / 2 << " " << YSCALE (y + height / 2) << " ";
      *pstream << width / 2 << " " << height / 2 << " 0 360 ellipse\n";
      *pstream << "stroke\n";

      CalcBoundingBox (x - width, YSCALE (y - height));
      CalcBoundingBox (x + width, YSCALE (y + height));
    }
}

void wxPostScriptDC::DrawIcon (wxIcon * icon, float x, float y)
{
}


void wxPostScriptDC::SetFont (wxFont * the_font)
{
  if (font == the_font)
    return;

  font = the_font;
  char buf[100];
  char *name;
  char *style = "";
  int Style = font->GetStyle ();
  int Weight = font->GetWeight ();

  switch (font->GetFamily ())
    {
    case wxMODERN:
      name = "/Courier";
      break;
    case wxSWISS:
      name = "/Helvetica";
      break;
    default:
    case wxDEFAULT:
    case wxROMAN:
      name = "/Times-Roman";
      break;
    }

  if (Style == wxNORMAL && (Weight == wxNORMAL || Weight == wxLIGHT))
    {
      if (font->GetFamily () == wxROMAN)
	style = "-Roman";
      else
	style = "";
    }
  else if (Style == wxNORMAL && Weight == wxBOLD)
    style = "-Bold";

  else if (Style == wxITALIC && (Weight == wxNORMAL || Weight == wxLIGHT))
    {
      if (font->GetFamily () == wxROMAN)
	style = "-Italic";
      else
	style = "-Oblique";
    }
  else if (Style == wxITALIC && Weight == wxBOLD)
    {
      if (font->GetFamily () == wxROMAN)
	style = "-BoldItalic";
      else
	style = "-BoldOblique";
    }
  else if (Style == wxSLANT && (Weight == wxNORMAL || Weight == wxLIGHT))
    {
      if (font->GetFamily () == wxROMAN)
	style = "-Italic";
      else
	style = "-Oblique";
    }
  else if (Style == wxSLANT && Weight == wxBOLD)
    {
      if (font->GetFamily () == wxROMAN)
	style = "-BoldItalic";
      else
	style = "-BoldOblique";
    }
  else
    style = "";

  strcpy (buf, name);
  strcat (buf, style);
  *pstream << buf << " findfont\n";
  *pstream << font->GetPointSize () << " scalefont setfont\n";
}

void wxPostScriptDC::SetPen (wxPen * pen)
{
  wxPen *oldPen = current_pen;

  if ((current_pen = pen) == NULL)
    return;			/* NIL */

  // Line width
  *pstream << pen->GetWidth () << " setlinewidth\n";

  // Line style - WRONG: 2nd arg is OFFSET
  /*
     Here, I'm afraid you do not conceive meaning of parameters of 'setdash'
     operator correctly. You should look-up this in the Red Book: the 2nd parame-
     ter is not number of values in the array of the first one, but an offset
     into this description of the pattern. I mean a real *offset* not index
     into array. I.e. If the command is [3 4] 1 setdash   is used, then there
     will be first black line *2* units long, then space 4 units, then the
     pattern of *3* units black, 4 units space will be repeated.
   */
  static char *dotted = "[2 5] 2";
  static char *short_dashed = "[4 4] 2";
  static char *long_dashed = "[4 8] 2";
  static char *dotted_dashed = "[6 6 2 6] 4";

  char *psdash = NULL;
  switch (pen->GetStyle ())
    {
    case wxDOT:
      psdash = dotted;
      break;
    case wxSHORT_DASH:
      psdash = short_dashed;
      break;
    case wxLONG_DASH:
      psdash = long_dashed;
      break;
    case wxDOT_DASH:
      psdash = dotted_dashed;
      break;
    case wxSOLID:
    case wxTRANSPARENT:
    default:
      psdash = "[] 0";
      break;
    }
  if (oldPen != pen)
    *pstream << psdash << " setdash\n";

  // Line colour
  unsigned char red = pen->GetColour ().Red ();
  unsigned char blue = pen->GetColour ().Blue ();
  unsigned char green = pen->GetColour ().Green ();

  if (!Colour)
    {
      // Anything not white is black
      if (!(red == (unsigned char) 255 && blue == (unsigned char) 255
	    && green == (unsigned char) 255))
	{
	  red = (unsigned char) 0;
	  green = (unsigned char) 0;
	  blue = (unsigned char) 0;
	}
    }

  if (!(red == currentRed && green == currentGreen && blue == currentBlue))
  {
    float redPS = (float) (((int) red) / 255.0);
    float bluePS = (float) (((int) blue) / 255.0);
    float greenPS = (float) (((int) green) / 255.0);

    *pstream << redPS << " " << greenPS << " " << bluePS << " setrgbcolor\n";
    
    currentRed = red;
    currentBlue = blue;
    currentGreen = green;
  }
}

void wxPostScriptDC::SetBrush (wxBrush * brush)
{
//  wxBrush *oldBrush = current_brush;

  if ((current_brush = brush) == NULL)
    return;			/* NIL */

  // Brush colour
  unsigned char red = brush->GetColour ().Red ();
  unsigned char blue = brush->GetColour ().Blue ();
  unsigned char green = brush->GetColour ().Green ();

  if (!Colour)
    {
      // Anything not black is white
      if (!(red == (unsigned char) 0 && blue == (unsigned char) 0
	    && green == (unsigned char) 0))
	{
	  red = (unsigned char) 255;
	  green = (unsigned char) 255;
	  blue = (unsigned char) 255;
	}
    }

  if (!(red == currentRed && green == currentGreen && blue == currentBlue))
  {
    float redPS = (float) (((int) red) / 255.0);
    float bluePS = (float) (((int) blue) / 255.0);
    float greenPS = (float) (((int) green) / 255.0);
    *pstream << redPS << " " << greenPS << " " << bluePS << " setrgbcolor\n";
    currentRed = red;
    currentBlue = blue;
    currentGreen = green;
  }
}

void wxPostScriptDC::DrawText (const char *text, float x, float y)
{
  if (font)
    SetFont (font);

  if (current_text_foreground.Ok ())
    {
      unsigned char red = current_text_foreground.Red ();
      unsigned char blue = current_text_foreground.Blue ();
      unsigned char green = current_text_foreground.Green ();

      if (!Colour)
	{
	  // Anything not white is black
	  if (!(red == (unsigned char) 255 && blue == (unsigned char) 255
		&& green == (unsigned char) 255))
	    {
	      red = (unsigned char) 0;
	      green = (unsigned char) 0;
	      blue = (unsigned char) 0;
	    }
	}
      if (!(red == currentRed && green == currentGreen && blue == currentBlue))
      {
        float redPS = (float) (((int) red) / 255.0);
        float bluePS = (float) (((int) blue) / 255.0);
        float greenPS = (float) (((int) green) / 255.0);
        *pstream << redPS << " " << greenPS << " " << bluePS << " setrgbcolor\n";

        currentRed = red;
        currentBlue = blue;
        currentGreen = green;
      }
    }

  int size = 10;
  if (font)
    size = font->GetPointSize ();

  *pstream << x << " " << YSCALE (y + size) << " moveto\n";

//  *pstream << "(" << text << ")" << " show\n";
  *pstream << "(";
  int len = strlen (text);
  for (int i = 0; i < len; i++)
    {
      char ch = text[i];
      if (ch == ')' || ch == '(' || ch == '\\')
	*pstream << "\\";
      *pstream << ch;
    }

  *pstream << ")" << " show\n";

  CalcBoundingBox (x, YSCALE (y + size));
  CalcBoundingBox (x + size * strlen (text), YSCALE (y));
}


void wxPostScriptDC::SetBackground (wxBrush * brush)
{
  current_background_brush = brush;
}

void wxPostScriptDC::SetLogicalFunction (int function)
{
}

static const char *wxPostScriptHeaderEllipse = "\
/ellipsedict 8 dict def\n\
ellipsedict /mtrx matrix put\n\
/ellipse\n\
{ ellipsedict begin\n\
  /endangle exch def\n\
  /startangle exch def\n\
  /yrad exch def\n\
  /xrad exch def\n\
  /y exch def\n\
  /x exch def\n\
  /savematrix mtrx currentmatrix def\n\
  x y translate\n\
  xrad yrad scale\n\
  0 0 1 startangle endangle arc\n\
  savematrix setmatrix\n\
  end\n\
  } def\n\
";

Bool wxPostScriptDC::StartDoc (char *message)
{
  wxPageNumber = 1;
  if (message)
    title = copystring (message);
  return TRUE;
}



void wxPostScriptDC::EndDoc (void)
{
  if (clipping)
    {
      clipping = FALSE;
      *pstream << "grestore\n";
    }

  // Will reuse pstream for header
  if (pstream)
    {
      delete pstream;
      pstream = NULL;
    }

  // Write header now
  char *header_file = wxGetTempFileName("ps");

  pstream = new ofstream (header_file);

  *pstream << "%!PS-Adobe-2.0 EPSF-2.0\n";	/* PostScript magic strings */
  if (title)
    *pstream << "%%Title: " << title << "\n";
  *pstream << "%%Creator: " << wxTheApp->argv[0] << "\n";
  time_t when; time (&when);
  *pstream << "%%CreationDate: " << ctime (&when);

  // User Id information
  {
    char userID[256];
    if ( wxGetEmailAddress(userID, sizeof(userID)) ) {
	*pstream << "%%For: " << userID;
	char userName[245];
	if (wxGetUserName(userName, sizeof(userName)))
	  *pstream << " (" << userName << ")";
	*pstream << "\n";
    } else if ( wxGetUserName(userID, sizeof(userID)) ) {
	*pstream << "%%For: " << userID << "\n";
    }
  }

  // In landscape mode, what should bounding box be changed to????
  // Bounding box is currently wrong in landscape mode.

  *pstream << "%%BoundingBox: " << min_x << " " << min_y << " " << max_x << " " << max_y << "\n";
  *pstream << "%%Pages: " << wxPageNumber - 1 << "\n";
  *pstream << "%%EndComments\n\n";

  // Output scaling
  float real_translate_y = wx_printer_translate_y;
  if (wx_printer_orientation == PS_LANDSCAPE)
    {
      real_translate_y -= max_y;
      *pstream << "90 rotate\n";
    }

  *pstream << wx_printer_scale_x << " " << wx_printer_scale_y << " scale\n";
  *pstream << wx_printer_translate_x << " " << real_translate_y << " translate\n";

  *pstream << wxPostScriptHeaderEllipse;

  if (wxPostScriptHeaderSpline)
    *pstream << wxPostScriptHeaderSpline;

  delete pstream;
  pstream = NULL;

  // Paste header Before wx_printer_file
  wxConcatFiles (header_file, wx_printer_file, wx_printer_file);
  wxRemoveFile (header_file);
  delete[]header_file;

#ifdef wx_x
  if (ok && wx_interactive)
    {
      switch (wx_printer_mode) {
	case PS_PREVIEW:
	{
/* HP compiler complains
	  char *argv[] = {
	     wx_preview_command,
	     wx_printer_file,
	     NULL};
*/
          char *argv[3];
          argv[0] = wx_preview_command;
          argv[1] = wx_printer_file;
          argv[2] = NULL;
	  wxExecute (argv);
	}
	break;

	case PS_PRINTER:
	{
/* HP compiler complains
	  char *argv[] = {
	     wx_printer_command,
	     wx_printer_flags,
	     wx_printer_file,
	     NULL};
*/
          char *argv[4];
          argv[0] = wx_preview_command;
          argv[1] = wx_printer_flags;
          argv[2] = wx_printer_file;
          argv[3] = NULL;
	  wxExecute (argv);
	}
	break;

	case PS_FILE:
	  break;
	}
    }
#endif
}

void wxPostScriptDC::StartPage (void)
{
  *pstream << "%%Page: " << wxPageNumber++ << "\n";
}

void wxPostScriptDC::EndPage (void)
{
  *pstream << "showpage\n";
}

Bool wxPostScriptDC::
Blit (float xdest, float ydest, float width, float height,
      wxCanvasDC * source, float xsrc, float ysrc, int rop)
{
  return FALSE;
}

float wxPostScriptDC::GetCharHeight (void)
{
  if (font)
    return (float) font->GetPointSize ();
  else
    return 12.0;
}


float wxPostScriptDC::GetCharWidth (void)
{
  return 0;
}

void wxPostScriptDC::GetTextExtent (const char *string, float *x, float *y,
	       float *descent, float *externalLeading)
{
  // Provide a VERY rough estimate (avoid using it)
  int width = 12;
  int height = 12;

  if (font)
    {
      height = font->GetPointSize ();
      width = height;
    }
  *x = (float) strlen (string) * width;
  *y = (float) height;
  if (descent)
    *descent = 0.0;
  if (externalLeading)
    *externalLeading = 0.0;
}

void wxPostScriptDC::SetMapMode (int mode)
{
  mapping_mode = mode;
  return;
}

void wxPostScriptDC::SetUserScale (float x, float y)
{
  user_scale_x = x;
  user_scale_y = y;
}

float wxPostScriptDC::DeviceToLogicalX (int x)
{
  return (float) x;
}

float wxPostScriptDC::DeviceToLogicalXRel (int x)
{
  return (float) x;
}

float wxPostScriptDC::DeviceToLogicalY (int y)
{
  return (float) y;
}

float wxPostScriptDC::DeviceToLogicalYRel (int y)
{
  return (float) y;
}

int wxPostScriptDC::LogicalToDeviceX (float x)
{
  return (int) x;
}

int wxPostScriptDC::LogicalToDeviceXRel (float x)
{
  return (int) x;
}

int wxPostScriptDC::LogicalToDeviceY (float y)
{
  return (int) y;
}

int wxPostScriptDC::LogicalToDeviceYRel (float y)
{
  return (int) y;
}

class wxPrinterDialogBox:public wxDialogBox
{
  public:
  wxPrinterDialogBox (wxFrame * frame, char *title, Bool modal = FALSE,
		      int x = -1, int y = -1, int
		      width = -1, int height = -1);
};

#ifdef wx_x
wxPrinterDialogBox::wxPrinterDialogBox (wxFrame * frame, char *title, Bool modal,
		    int x, int y, int width, int height):
wxDialogBox (frame, title, modal, x, y, width, height)
{
}

Bool wxPrinterDialogAnswer = TRUE;

void 
wxPrinterDialogOk (wxButton & button, wxEvent & event)
{
  wxPrinterDialogAnswer = TRUE;
  wxPrinterDialogBox *dialog = (wxPrinterDialogBox *) button.GetParent ();
  dialog->Show (FALSE);
}

void 
wxPrinterDialogCancel (wxButton & button, wxEvent & event)
{
  wxPrinterDialogAnswer = FALSE;
  wxPrinterDialogBox *dialog = (wxPrinterDialogBox *) button.GetParent ();
  dialog->Show (FALSE);
}

Bool 
XPrinterDialog (void)
{
  wxBeginBusyCursor();
  char buf[100];
  wxPrinterDialogBox dialog (NULL, "Printer Settings", TRUE, 150, 150, 400, 400);
  dialog.SetLabelPosition(wxVERTICAL);

  wxButton *okBut = new wxButton (&dialog, (wxFunction) wxPrinterDialogOk, wxSTR_BUTTON_OK);
  (void) new wxButton (&dialog, (wxFunction) wxPrinterDialogCancel, wxSTR_BUTTON_CANCEL);
  dialog.NewLine ();
  dialog.NewLine ();
  okBut->SetDefault();

#ifdef wx_x
  wxText text_prt (&dialog, (wxFunction) NULL, "Printer Command: ", wx_printer_command, -1, -1, 100, -1);

  wxText text0 (&dialog, (wxFunction) NULL, "Printer Options: ", wx_printer_flags, -1, -1, 150, -1);
  dialog.NewLine ();
  dialog.NewLine ();
#endif

//  char *orientation[] = {"Portrait", "Landscape"};  // HP compiler complains
  char *orientation[2];
  orientation[0] = "Portrait";
  orientation[1] = "Landscape";
  wxRadioBox radio0 (&dialog, (wxFunction)NULL, "Orientation: ",-1,-1,-1,-1,2,orientation,2,wxFLAT);
  radio0.SetSelection((int)wx_printer_orientation);

  // @@@ Configuration hook
  if (wx_preview_command == NULL)
    wx_preview_command = copystring(PS_VIEWER_PROG);
  wxGetResource ("wxWindows", "PSView", &wx_preview_command);

//  char *print_modes[] = {"Send to Printer", "Print to File", "Preview Only"}; 
  char *print_modes[3];
  print_modes[0] = "Send to Printer";
  print_modes[1] = "Print to File";
  print_modes[2] = "Preview Only";
  int features = (wx_preview_command && *wx_preview_command) ? 3 : 2;
  wxRadioBox radio1 (&dialog, (wxFunction)NULL, "PostScript:"
    ,-1,-1,-1,-1, features, print_modes, features, wxFLAT);
  radio1.SetSelection((int)wx_printer_mode);

  sprintf (buf, "%.2f", wx_printer_scale_x);
  dialog.NewLine ();
  dialog.NewLine ();

  wxText text1 (&dialog, (wxFunction) NULL, "X Scaling: ", buf, -1, -1, 100, -1);

  sprintf (buf, "%.2f", wx_printer_scale_y);
  wxText text2 (&dialog, (wxFunction) NULL, "Y Scaling: ", buf, -1, -1, 100, -1);

  dialog.NewLine ();

  sprintf (buf, "%.2f", wx_printer_translate_x);
  wxText text3 (&dialog, (wxFunction) NULL, "X Translation: ", buf, -1, -1, 100, -1);

  sprintf (buf, "%.2f", wx_printer_translate_y);
  wxText text4 (&dialog, (wxFunction) NULL, "Y Translation: ", buf, -1, -1, 100, -1);

  dialog.NewLine ();
  dialog.NewLine ();

  dialog.Fit ();

  wxEndBusyCursor();
  dialog.Show (TRUE);

  if (wxPrinterDialogAnswer)
    {
      StringToFloat (text1.GetValue (), &wx_printer_scale_x);
      StringToFloat (text2.GetValue (), &wx_printer_scale_y);
      StringToFloat (text3.GetValue (), &wx_printer_translate_x);
      StringToFloat (text4.GetValue (), &wx_printer_translate_y);

#ifdef wx_x
      wx_printer_flags = copystring (text0.GetValue ());
      wx_printer_command = copystring (text_prt.GetValue ());
#endif

      wx_printer_orientation = (radio0.GetSelection() == PS_LANDSCAPE ? PS_LANDSCAPE : PS_PORTRAIT);

      // C++ wants this
      switch ( radio1.GetSelection() ) {
	case PS_PREVIEW: wx_printer_mode = PS_PREVIEW; break;
	case PS_FILE:    wx_printer_mode = PS_FILE;    break;
	case PS_PRINTER: wx_printer_mode = PS_PRINTER; break;
     }

    }

  return wxPrinterDialogAnswer;
}
#endif		/* wx_x */

// PostScript printer settings
void wxSetPrinterCommand(char *cmd)
{
  if (wx_printer_command)
    delete[] wx_printer_command;
  if (cmd)
    wx_printer_command = copystring(cmd);
  else
    wx_printer_command = NULL;
}

void wxSetPrintPreviewCommand(char *cmd)
{
  if (wx_preview_command)
    delete[] wx_preview_command;
  if (cmd)
    wx_preview_command = copystring(cmd);
  else
    wx_preview_command = NULL;
}

void wxSetPrinterOptions(char *flags)
{
  if (wx_printer_flags)
    delete[] wx_printer_flags;
  if (flags)
    wx_printer_flags = copystring(flags);
  else
    wx_printer_flags = NULL;
}

void wxSetPrinterFile(char *f)
{
  if (f)
    strcpy(wx_printer_file, f);
}

void wxSetPrinterOrientation(int orient)
{
  wx_printer_orientation = orient;
}

void wxSetPrinterScaling(float x, float y)
{
  wx_printer_scale_x = x;
  wx_printer_scale_y = x;
}

void wxSetPrinterTranslation(float x, float y)
{
  wx_printer_translate_x = x;
  wx_printer_translate_y = x;
}

// 1 = Preview, 2 = print to file, 3 = send to printer
void wxSetPrinterMode(int mode)
{
  wx_printer_mode = mode;
}

// Get current values
char *wxGetPrinterCommand(void)
{
  return wx_printer_command;
}

char *wxGetPrintPreviewCommand(void)
{
  return wx_preview_command;
}

char *wxGetPrinterOptions(void)
{
  return wx_printer_flags;
}

char *wxGetPrinterFile(void)
{
  return wx_printer_file;
}

int wxGetPrinterOrientation(void)
{
  return wx_printer_orientation;
}

void wxGetPrinterScaling(float *x, float *y)
{
  *x = wx_printer_scale_x;
  *y = wx_printer_scale_y;
}

void wxGetPrinterTranslation(float *x, float *y)
{
  *x = wx_printer_translate_x;
  *y = wx_printer_translate_y;
}

int wxGetPrinterMode(void)
{
  return wx_printer_mode;
}

#endif
