/****************************************************************************/
/*   xcObject.C                                                             */
/****************************************************************************/
/*                                                                          */
/*   Copyright (c) 1992, 1993 Bernhard Strassl                              */
/*       Vienna User Interface Group                                        */
/*       Institute for Applied Computer Science and Information Systems     */
/*       University of Vienna, Austria                                      */
/*                                                                          */
/*   See file COPYRIGHT in this directory for details. If this file is      */
/*   missing please mail to xmpp@ani.univie.ac.at                           */
/****************************************************************************/

// #include <stdlib.h>
// #include <unistd.h>
// #include <sys/wait.h>

#include <stdarg.h>
#include <stream.h>
#include <strstream.h>
#include <ctype.h>

#include "xmObject.h"

extern "C"
{
int atoi(const char*);
void exit(int);
}

#undef cb
#undef display
#undef screen
#undef mainWindow

#ifdef ASCII

int cwXfactor = 8, cwYfactor = 15;

#endif


// Application /////////////////////////////////////////////

XmApp* App;

void (*noX )() = NULL;

XmApp::XmApp(int argc, char** argv)
{
	App = this;
	appArgc = argc;
	appArgv = argv;
	appShell = NULL;
#ifdef ASCII
	if(argc > 1)
	{	string scalearg(argv[1]);
		if(scalearg.includes("scale") && scalearg.includes("x"))
		{	scalearg = scalearg.after("scale");
			string xsc, ysc;
			xsc = scalearg.before("x");
			ysc = scalearg.after("x");
			int xfactor = atoi(xsc);
			int yfactor = atoi(ysc);
			if(xfactor > 3 && yfactor > 5)
			{	cwXfactor = xfactor;
				cwYfactor = yfactor;
			}
			else
				errout << "xmApp (cw++ interface): invalid scaling " << xfactor << "x" << yfactor << " - ignored.\n";
		}
	}
#endif
	appContext = 0;
	display = screen = NULL;
	rscDb = NULL;
	unitType = 0;	// unused...
	useWindows = FALSE;
	active = FALSE;
#ifdef TCL
	xmTclInitApp();
#endif
	initialize();
	run();
}

XmApp::~XmApp()
{
}

void XmApp::run()
{
	if(appShell)
	{	appShell->realize();
		active = TRUE;
#ifdef ASCII
		Cw_App.run();
#else
		xmTclMainLoop();
#endif
		return;
	}
	cerr << "xmApp: Error - no application window created, exiting...\n";
}

char* XmApp::getCmdArg(int n)
{
	return(n < appArgc ? appArgv[n] : NULL);
}

XmObject* XmApp::findObject(char*)
{
	return(NULL);
}

Widget XmApp::getMainWindow()
{
	return(appShell ? appShell->handle() : NULL);
}

void XmApp::setScreenUnitType(unsigned char u)
{
	if(!active)
		unitType = u;
	else
		cerr << "xmApp: Warning - cannot change unit type of running application!\n";
}

Widget XmApp::newMainWindow(XmObject* anObj, char* name, int width, int height, xmStyle s)
{
	Widget w = NULL;
#ifdef ASCII
	int cwWidth = x2cwX(width);
	int cwHeight = x2cwY(height);

	width = cwWidth > 80 ? 80 : cwWidth;
	height = cwHeight > 25 ? 25 : cwHeight;
	int nx = cwWidth > 0 && cwWidth < 80 ? (80 - cwWidth) / 2 : 0;
	int ny = cwHeight > 0 && cwHeight < 25 ? (25 - cwHeight) / 2 : 0;
	w = new CwDialog(name, NULL, nx, ny, cwWidth, cwHeight);
#else
	w = anObj->handle();
	if(!appShell)
		w->setRoot();
	if(appShell || !tclRootApp)
	{	w->cmd() << "toplevel " << w->getPath();
		if(width && height)
			w->cmd() << " -width " << width << " -height " << height;
		w->cmd() << "\n";
	}
#endif
	if(!appShell)
		appShell = anObj;
	return(w);
}

void XmApp::removeMainWindow(XmObject* obj)
{
	if(obj == appShell)		// exit, if it is the default window (does this make sense??)
	{
#ifdef TCL
		if(tclRootApp)
			tclOutput("destroy .\n");
		else
			xmTclQuitApp();
#endif
		exit(1);
	}
}


main(int argc, char** argv)
{
	new XmApp(argc, argv);
}

// Manager //////////////////////////////////////////////////

// will provide support for a more sufficient application
// architecture, not really implemented yet....

XmManager::~XmManager()
{
	if(objects)
	{	if(objectCount)
			cerr << "xmManager: Warning - deleting manager with " << objectCount << " references!\n";
	}
}

bool XmManager::addObject(XmObject* anObject, objType t)
{
	XmObject** newArray;

	if(t == WindowObj || t == UserDialog || t == DialogWindow)
	{	if(objectCount == maxObjects)
		{	newArray = new XmObject*[objectCount ? objectCount * 2 : 5];
			for(int i = 0; i < objectCount; i++)
				newArray[i] = objects[i];
			delete objects;
			objects = newArray;
		}
		objects[objectCount++] = anObject;
		return(TRUE);
	}
	return(FALSE);
}

bool XmManager::removeObject(XmObject* anObject)
{
	for(int i = 0; i < objectCount; i++)
	{	if(objects[i] == anObject)
		{	--objectCount;
			for(int j = i; j < objectCount; j++)
				objects[j] = objects[j + 1];
			return(TRUE);
		}
	}
	return(FALSE);
}

XmObject* XmManager::findObject(char* n, objType t)
{
	for(int i = 0; i < objectCount; i++)
	{	if(!strcmp(objects[i]->getName(), n) && objects[i]->objectType() == t)
			return(objects[i]);
	}
	return(NULL);
}

void XmManager::changed(char* txt, XmObject* except)
{
	for(int i = 0; i < objectCount; i++)
	{	// cerr << objects[i]->getName() << ", ";
		if(objects[i] != except)
			objects[i]->update(txt);
	}
	if(manager)
		manager->changed(txt);
}


// Object //////////////////////////////////////////////////


void XmObject::init(char* n, XmObject* p)
{
	magic = MAGIC_OBJECT_CONST;
	basePtr = this;
	parent = p; 
#ifdef TCL
	wid = new XmTclController(this, n, (p ? p->handle() : NULL));
#else
	wid = (Widget )NULL;
#endif
	win = (Window )NULL;
	if(!n)
		n = "unnamedObject";
	wname =  strcpy(new char[strlen(n) + 1], n);
	wargcount = 0;
	gc = (GC )NULL;
	inputMask = 0;
	destroyWidget = FALSE;	// avoids overhead in all cases where a widget tree is destroyed
}

XmObject::~XmObject()
{
	if(magic)
	{	magic = 0L;
		if(destroyWidget)
			XmObject::destroy();	// can't call virtual here...
	}
	else
		cerr << "xmObject: Warning - deleting invalid object pointer!\n";
}

void genericCallback(Widget w, XtPointer cl_data, XtPointer ca_data)
{
	cb_data* d = (cb_data* )cl_data;

//	if(d->object->valid())			// ensure that object has not been deleted...
	{	void* answer = NULL;

		if(d->client_data == CB_XM_DATA)
			answer = ca_data;
		else if(d->client_data == CB_XM_HANDLE)
			answer = w;
		else if(d->client_data == CB_OBJ_NAME)
			answer = getObjName(w);
		else if(d->client_data > CB_OBJ_NAME)
			answer = d->client_data;

		((d->object)->*(d->callback))(answer);
	}
}

#ifdef ASCII

void cw_genericCallback(CwObject* obj, int reason, void* cl_data)
{
	genericCallback(obj, cl_data, (XtPointer)reason);
}

#endif

char XmObject::parse(char* str)		// caution: parses string in place!
{
	char c;
#ifdef ASCII
	for(int i = 0; c = str[i]; i++)
	{	if(c == '&')
		{	for(int j = i; str[j]; j++)
				str[j] = str[j + 1];
			return(str[i]);
		}
	}
	return(str[0]);
#else
	for(int i = 0; c = str[i]; i++)
	{	if(c == '&')
		{	c = i;
			for(int j = i; str[j]; j++)
				str[j] = str[j + 1];
			return(c);
		}
	}
	return(0);
#endif
}

bool XmObject::setCbFor(Widget w, char* evtName, XmObject* obj, cbProc cb, bool set, void* data)
{
	cb_data* d;

	if(!wid)
		return(FALSE);

	if(set)
	{	d = new cb_data;
		d->callback = cb;
		d->object = obj->getBase();
		d->client_data = (XtPointer )data;
#ifdef ASCII
		w->setCallback(cw_genericCallback, d);
#else
		w->addCbk(evtName, d);
#endif
		return(TRUE);
	}
	else
	{
#ifdef ASCII
		if(w->matchCallback(cw_genericCallback, d))
		{	w->setCallback(NULL);
#else
		if(w->removeCbk(evtName, d))
		{
#endif
			delete d;
		}
	}
	return(FALSE);
}

// A comment to the usage of the XmObject::XXXarg() functions: use them to collect
// a unknown number of Args in several functions, avoid them in one shot operations
// (use a local arg array instead, there may be some pending things in the wargs
// array which should take affect later).

int XmObject::argIndex(String n)
{
	for(int i = 0; i < wargcount; i++)
	{	if(!strcmp(wargs[i].name, n))
			return(i);
	}
	return(-1);
}

void XmObject::setArg(String n, XtArgVal  v)
{
	addArg(n, v);
	if(wid)
	{	//XtSetValues(wid, wargs, wargcount);
		wargcount = 0;
	}
}

void XmObject::changeArg(String n, XtArgVal  v)
{
	int ndx;

	if((ndx = argIndex(n)) != -1)
		wargs[ndx].value = v;			// dont use this func with in-place-args!!
	else
		addArg(n, v);
}

void XmObject::forceArg(String n, XtArgVal  v)
{
	Arg a;

	if(wid)
	{	if(n)
		{	; //XtSetArg(a, n, v);
			//XtSetValues(wid, &a, 1);
		}
		else
		{	//XtSetValues(wid, wargs, wargcount);
			wargcount = 0;
		}
	}
	else
		cerr << "xmObject: Warning - forceArg " << n << " ignored: object not created.\n";
}

void XmObject::syscall(char*)	// callback name, not implemented yet...
{
}

char RPB[500];

char* XmObject::getResourcePath(bool append)
{
	int olen, alen, clen;

	olen = append ? strlen(RPB) : 0;
	alen = parent ? 0 : strlen(App->getName());
	clen = strlen(wname) + alen;

	char* tmp = strcpy(new char[strlen(RPB) + 1], RPB);
	strncpy(&RPB[clen + 1], tmp, olen);
	delete tmp;
	RPB[clen + olen + 1] = '\0';
	RPB[alen] = '*';
	strncpy(&RPB[alen + 1], wname, strlen(wname));

	if(parent)
		return(parent->getResourcePath(TRUE));
	if(alen)
		strncpy(RPB, App->getName(), alen);
	return(RPB);
}

bool XmObject::checkResource(char* aName)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr;

	rstr << getResourcePath() << "." << aName; rstr.put('\0');
	//return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
	return(FALSE);
}

bool XmObject::realize()
{
	if(!wid)
	{	cerr << "xmObject: error - cannot realize object (no widget created).\n";
		return(FALSE);
	}
	if(App->isActive())
	{	if(parent && !isObjVisible(parent->handle()))
		{	cerr << "xmObject: error - cannot realize object (parent not realized).\n";
			return(FALSE);
		}
#ifdef ASCII
		wid->show();
#else
		widcmd 	<< wid->processArgs(wargs, wargcount);
		wid->getMaster()->realize();
		wargcount = 0;
#endif
		return(TRUE);
	}
	if(!parent)
	{
#ifdef ASCII
		wid->show();	// cw dialogs need no parent...
		return(TRUE);
#endif
	}
	return(FALSE);
}

bool XmObject::destroy()
{
	if(wid)				// may have been destroyed already in subclass
	{	if(!parent)
			App->removeMainWindow(this);	// app window...
#ifdef ASCII
		delete wid;
#else
		Widget w = wid->getMaster();
		w->resetCmd();
		w->cmd() << "destroy " << w->getPath() << "\n";
		w->execCmd();
		delete w;
#endif
		wid = NULL;
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::hide()
{
	if(isVisible())
	{
#ifdef ASCII
		wid->show(FALSE);
		return(TRUE);
#endif
	}
	return(FALSE);
}

bool XmObject::show()
{
	if(!isVisible())
	{
#ifdef ASCII
		wid->show(TRUE);
		return(TRUE);
#endif
	}
	return(FALSE);
}

bool XmObject::isVisible()
{
	return(wid && isObjVisible(wid) ? TRUE : FALSE);
}

bool XmObject::move(int nx, int ny)
{
	if(wid)
	{
#ifdef ASCII
		wid->moveObj(x2cwX(nx), x2cwY(ny));
#else
		Widget w = wid->getMaster();
		if(!w->isReframeable())
			return(FALSE);
		w->cmd() << "place " << w->getPath()
			 << " -x " << nx << " -y " << ny << "\n";
		w->execIfVisible();
#endif
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::resize(int nw, int nh)
{
	if(wid)
	{
#ifdef ASCII
		wid->resizeObj(x2cwX(nw), x2cwY(nh));
#else
		Widget w = wid->getMaster();
		if(!w->isReframeable())
			return(FALSE);
		w->cmd() << w->getPath() << " configure "
			 << w->resizeCmd(x2tclX(this, nw), x2tclY(this, nh)) << "\n";
		w->execIfVisible();
#endif
		return(TRUE);
	}
	return(FALSE);
}

bool XmObject::reframe(int nx, int ny, int nw, int nh)
{
	if(wid)
		return(move(nx, ny) && resize(nw, nh) ? TRUE : FALSE);
	return(FALSE);
}

bool XmObject::getFrame(int& fx, int& fy, int& fw, int& fh)
{
	if(wid)
	{	
#ifdef ASCII
		wid->getDimensions(fx, fy, fw, fh);
		fx = cw2xX(fx); fy = cw2xY(fy);
		fw = cw2xX(fw); fh = cw2xY(fh);
#else
		; // seems to be compliciated in Tcl....
#endif
		return(TRUE);
	}
	return(FALSE);
}


// Menu base class //////////////////////////////////////////////////////////////


XmMenu::XmMenu(char* name, XmObject* par) : XmObject(name, par)
{
	nextItemPos = -1;
	numSubmenues = 0;
}

XmMenu::~XmMenu()
{
}

XmObject* XmMenu::parentOfMenu()	// find the root in a submenue hierarchy
{
	if(parent->objectType() == Menu)
		return(((XmMenu* )parent)->parentOfMenu());
	return(parent);
}

bool XmMenu::checkMenuResource(char* anItem)
{
	XrmValue dum;
	char* ddum;
	ostrstream rstr;

	rstr << getResourcePath() << "*" << anItem << ".labelString"; rstr.put('\0');
	//return(XrmGetResource(App->getResourceDatabase(), rstr.str(), rstr.str(), &ddum, &dum) ? TRUE : FALSE);
	return(FALSE);
}

#define menuResourcePath(x)  x->getPath()

bool XmMenu::addSeparator(char* name)
{
	Widget w;
	
	if(!wid || !(w = getCurrentLabel()))
		return(FALSE);
#ifdef ASCII
	if(!name)
		name = "separator";
	CwButton* aButton = new CwButton((CwDialog*)w, name, "-", 0, 0, 0, 1, FALSE, CW_MENU_ENTRY);
	aButton->setSensitive(FALSE);
#else
	w->addName(name);
	w->cmd() << menuResourcePath(w) << " add separator\n";
	w->execIfVisible();
#endif
	return(TRUE);
}

bool XmMenu::removeSeparator(char* name)
{
	char* n = name;
	Widget w;

	if(!n)
		n = "Separator";
	if(!(w = getObjChild(wid, n)))
		return(FALSE);
#ifdef ASCII
	delete w;
#else
	int ndx;
	if(!(ndx = w->nameIndex(name)) > 0)
		return(FALSE);
	w->cmd() << menuResourcePath(w) << " delete " << ndx << "\n";
	w->execIfVisible();
#endif
	return(TRUE);
}

void menuCallback(Widget w, XtPointer cl_data, XtPointer p)
{
	cb_data* d = (cb_data* )cl_data;
	char* aStr = NULL;
#ifdef ASCII
	if(d->client_data)
		aStr = getObjText(w);
	else
		aStr = getObjName(w);
#else
	aStr = (char*)p;
#endif
	((d->object)->*(d->callback))(aStr);
}

#ifdef ASCII

void cw_menuCallback(CwObject* obj, int reason, void* cl_data)
{
	menuCallback(obj, cl_data, (XtPointer)reason);
}

#endif

#ifdef NO_VA_OBJS
VA_OBJ_SUBST_IMPL
#define NO_VARARGS
#endif

#define MAXITEMS 20

bool XmMenu::addItems(Entry entry, ...)
{
	Arg a;
	int argno = 0;
	char* names[MAXITEMS], * aName;
	char* items[MAXITEMS], * anItem;
	XmManager* receivers[MAXITEMS];
	cbProc procs[MAXITEMS];
	void* data[MAXITEMS];
	bool flags1[MAXITEMS];
	bool flags2[MAXITEMS];
	Entry anEntry;
	bool locate = FALSE;

	names[argno] = entry.name;
	items[argno] = entry.text;
	receivers[argno] = entry.receiver;
	procs[argno] = entry.callback;
	data[argno] = entry.client_data;
	flags1[argno] = entry.checkable;
	flags2[argno++] = entry.radio;
#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, entry);
	while(1)
	{	anEntry = va_arg(ap, Entry);
		if(!(names[argno] = anEntry.name))
			break;
		items[argno] = anEntry.text;
		receivers[argno] = anEntry.receiver;
		procs[argno] = anEntry.callback;
		data[argno] = anEntry.client_data;
		flags1[argno] = anEntry.checkable;
		flags2[argno++] = anEntry.radio;
	}
	va_end(ap);
#else
	names[argno] = NULL;
#endif

	for(int i = 0; aName = names[i]; i++)
	{	ostrstream tmpstrm;
		tmpstrm << items[i]; tmpstrm.put('\0');
		anItem = tmpstrm.str();
#ifndef ASCII
		char mm = parse(anItem);
#endif
		if(names[i] == items[i])  // name and text are identical...
			aName = anItem;
		Widget w;
		cb_data* d = new cb_data;
#ifdef ASCII
		if(flags1[i]) // checkable...
		{	cwButtonStyle bstyle = CW_CHECK_BOX;
			if(flags2[i]) // radio...
				bstyle = CW_RADIO_BUTTON;
			w = new CwToggleButton((CwDialog*)getCurrentLabel(), aName, anItem, 0, 0, 0, 1, FALSE, bstyle);
		}
		else
			w = new CwButton((CwDialog*)getCurrentLabel(), aName, anItem, 0, 0, 0, 1, FALSE, CW_MENU_ENTRY);
#else
		w = getCurrentLabel();
		w->cmd() << menuResourcePath(w) << " add "
			<< (flags1[i] ? (flags2[i] ? " radiobutton " : " checkbutton ") : " command ")
			<< " -label \"" << anItem << "\" -underline " << int(mm) << " ";
		if(flags1[i])
			w->cmd() << " -variable " << w->getVarName() << aName;
		w->cmd() << " -command " << w->callbackString(aName) << "\n";
		w->addName(aName);
		w->execIfVisible();
#endif
		wargcount = 0;
		d->callback = procs[i];
		d->object = receivers[i] ? receivers[i] : parentOfMenu()->getBase();
		d->client_data = data[i];
#ifdef ASCII
		w->setCallback(cw_menuCallback, d);
#else
		w->addCbk(ANY_CALLBACK, d, aName);
#endif
	}
	return(TRUE);
}

#ifdef NO_VA_OBJS
#undef NO_VARARGS
#endif

bool XmMenu::removeItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;
#ifndef NO_VARARGS
	char* item = first;
	va_list ap;

	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
#ifdef ASCII
		delete w;
#else
		w->cmd() << menuResourcePath(w) << " delete " << w->nameIndex(item) << "\n";
		w->execIfVisible();
#endif
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
#ifdef ASCII
		delete w;
#else
	int ndx;
	if(!(ndx = w->nameIndex(name)) > 0)
		return(FALSE);
	w->cmd() << menuResourcePath(w) << " delete " << ndx << "\n";
	w->execIfVisible();
#endif
#endif
	return(ret);

}

bool XmMenu::enableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;

#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
#ifdef ASCII
		w->setSensitive(TRUE);
#else
		int ndx;
		if(!(ndx = w->nameIndex(item)) > 0)
		{	ret = FALSE;
			continue;
		}
		w->cmd() << menuResourcePath(w) << " enable " << ndx << "\n";
		w->execIfVisible();
#endif
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
#ifdef ASCII
	w->setSensitive(TRUE);
#else
	int ndx;
	if(!(ndx = w->nameIndex(name)) > 0)
		return(FALSE);
	w->cmd() << menuResourcePath(w) << " enable " << ndx << "\n";
	w->execIfVisible();
#endif
#endif
	return(ret);
}

bool XmMenu::disableItems(char* first, ...)
{
	bool ret = TRUE;
	Widget w;

#ifndef NO_VARARGS
	char* item = first;
	va_list ap;
	va_start(ap, first);
	do
	{	if(!(w = getObjChild(getCurrentLabel(), item)))
		{	ret = FALSE;
			continue;
		}
#ifdef ASCII
		w->setSensitive(FALSE);
#else
		int ndx;
		if(!(ndx = w->nameIndex(item)) > 0)
		{	ret = FALSE;
			continue;
		}
		w->cmd() << menuResourcePath(w) << " disable " << ndx << "\n";
		w->execIfVisible();
#endif
	} while(item = va_arg(ap, char*));
	va_end(ap);
#else
	if(!(w = getObjChild(getCurrentLabel(), first)))
		return(FALSE);
#ifdef ASCII
	w->setSensitive(FALSE);
#else
	int ndx;
	if(!(ndx = w->nameIndex(name)) > 0)
		return(FALSE);
	w->cmd() << menuResourcePath(w) << " disable " << ndx << "\n";
	w->execIfVisible();
#endif
#endif
	return(ret);
}

XmSubMenu* XmMenu::addSubmenu(char* xn) // not implemented yet...
{
	XmSubMenu* aMenu = NULL;
	return(aMenu);
}

XmSubMenu* XmMenu::submenuAt(char* xn)
{
	XmSubMenu* aMenu = NULL;
	return(aMenu);
}

bool XmMenu::removeSubmenu(char* n)
{
	return(FALSE);
}

bool XmMenu::changeItemText(char* itemName, char* xnewText)
{
	Widget w;
	Arg a[2];
	ostrstream tmpstrm;
	tmpstrm << xnewText; tmpstrm.put('\0');
	char* newText = tmpstrm.str();
#ifndef ASCII
	char mm = parse(newText);
#endif		
	if(!(w = getObjChild(getCurrentLabel(), itemName)))
		return(FALSE);
#ifdef ASCII
	w->setText(newText);
#else
	int ndx;
	if(!(ndx = w->nameIndex(itemName)) > 0)
		return(FALSE);
	w->cmd() << menuResourcePath(w) << " entryconfigure "
		 << ndx << " -label " << xnewText << "\n";
	w->execIfVisible();
#endif
	return(TRUE);
}

bool XmMenu::getItemStatus(char* itemName)
{
	Widget w;

	if(!(w = getObjChild(getCurrentLabel(), itemName)) ||
#ifdef ASCII
	    !w->isA("ToggleButton"))
#else
		1)
#endif
		return(FALSE);
#ifdef ASCII
	return(((CwToggleButton*)w)->getState());
#else
	return(FALSE);
#endif
}

bool XmMenu::setItemStatus(char* itemName, bool newStatus)
{
	Widget w;

	if(!(w = getObjChild(getCurrentLabel(), itemName)) ||
#ifdef ASCII
	    !w->isA("ToggleButton"))
#else
		1)
#endif
		return(FALSE);
#ifdef ASCII
	return(((CwToggleButton*)w)->setState(newStatus));
#else
	return(FALSE);
#endif
}

// Popup menu /////////////////////////////////////////////////////////////////


XmPopupMenu::XmPopupMenu(char* xname, XmObject* par, bool defaultMenu) : XmMenu(xname, par)
{
}

void XmPopupMenu::makeDefault(bool sr)
{
}

bool XmPopupMenu::setLabel(char* l)
{
	return(FALSE);
}

bool XmPopupMenu::popup()
{
	return(FALSE);
}


// Dropdown menu /////////////////////////////////////////////////////////////////

XmDropdownMenu::XmDropdownMenu(char* name, XmObject* par) : XmMenu(name, par)
{
	numDropdowns = 0;
	nextLabelPos = -1;
	currentLabel = (Widget )NULL;

#ifdef ASCII
	wid = new CwMenuBar(name, (CwDialog*)par->handle());
#else
	widcmd << "frame " << widpath << " -relief raised -borderwidth 1\n";
	wid->setPackingOptions("-side top -fill x");
#endif
}

bool XmDropdownMenu::addLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;

	for(int i = 0; xitem = items[i]; i++)
	{	ostrstream tmpstrm;
		tmpstrm << xitem; tmpstrm.put('\0');
		item = tmpstrm.str();
		char mm = parse(item);
		Widget w;

#ifdef ASCII
		new CwButton((CwMenuBar*)wid, item, xitem, 0, 0, 0, 1, FALSE, CW_MENU_LABEL);
		currentLabel = ((CwMenuBar*)wid)->menu(item);
#else
		ostrstream bnbuf;
		bnbuf << item << "-mb"; bnbuf.put('\0');
		w = new XmTclController(this, bnbuf.str(), wid);
		currentLabel = new XmTclController(this, item, w);

		w->cmd() << "menubutton " << w->getPath()
			<< " -text " << item << " -underline " << int(mm);
		w->cmd() << " -menu " << currentLabel->getPath() << "\n";
		w->setPackingOptions("-side left");
		currentLabel->cmd() << "menu " << currentLabel->getPath() << "\n";
		currentLabel->dontRealize();
		wid->execIfVisible();
#endif
	}
	return(TRUE);
}

bool XmDropdownMenu::removeLabels(char* first, ...)
{
	int argno = 0;
	char* items[MAX_DROPDOWN_LABELS], * xitem, * item;
	bool ret = TRUE;

	items[argno++] = first;

#ifndef NO_VARARGS
	va_list ap;
	va_start(ap, first);
	while((items[argno++] = va_arg(ap, char *)) != (char *)0);
	va_end(ap);
#endif
	items[argno] = (char* )NULL;

	for(int i = 0; xitem = items[i]; i++)
	{	Widget w;
		if(!(w = getObjChild(wid, xitem)))
		{	ret = FALSE;
			continue;
		}
#ifdef ASCII
		delete w;
#else
		w = w->getParent();	// get the button...
		wid->removeChild(w);
		widcmd << "destroy " << w->getPath() << "\n";
		delete w;
#endif		
	}
	return(ret);
}

bool XmDropdownMenu::setCurrentLabel(char* n)
{
	Widget lastLabel = currentLabel;
#ifdef ASCII
	if(currentLabel = ((CwMenuBar*)wid)->menu(n))
		return(TRUE);
#else
	if(currentLabel = wid->getChild(n))
		return(TRUE);	
#endif
	cerr << "xmDropdownMenu: Warning - cannot find label " << n << " using: " << hex(long(lastLabel)) << "\n";
	currentLabel = lastLabel;
	return(FALSE);
}

bool XmDropdownMenu::changeLabelText(char*, char*)
{
	return(FALSE);
}

// subMenu /////////////////////////////////////////////////////////////////////

XmSubMenu::XmSubMenu(char* n, XmMenu* par) : XmMenu(n, par)
{
}

XmSubMenu::~XmSubMenu()
{
}


// Dialog base class //////////////////////////////////////////////////////////

XmDialog::XmDialog(char* n, XmObject* par, xmStyle s) : XmObject(n, par)
{
	basePtr = (XmObject* )((void* )this);
	dlgStyle = (s & XmSdlgWinModal || s & XmSdlgAppModal || s & XmSdlgSysModal || s & XmSpopup) ? s : XmSdlgModeless;
#ifdef TCL
	wid->setToplevel();
#endif
}

XmDialog::~XmDialog()
{
}


bool XmDialog::setLabelAndMode(char* label)
{
#ifdef ASCII
	if(!wid)
		changeArg("Label", label);
	else
		wid->setText(label);
#else
	widcmd << "wm title " << widpath << " \"" << label << "\"\n";
	wid->execIfVisible();
#endif
	return(TRUE);
}

bool XmDialog::run()
{
	if(dlgStyle & XmSdlgModeless)
	{	cerr << "xmDialog: Error - cannot run a modeless dialog.\n";
		return(FALSE);
	}
	completed = returned = FALSE;
	while(!completed)
#ifdef ASCII
		Cw_App.processNextEvent();
#else
		; // xmTclNextEvent();
#endif
	return(returned);
}

void XmDialog::ok(void*)
{
	returned = completed = TRUE;
}

void XmDialog::cancel(void*)
{
	completed = TRUE;
}

void XmDialog::close(void*)
{
	delete this;
}

// Predefined system dialogs //////////////////////////////////////////////////

XmSystemDialog::XmSystemDialog(char* n, XmObject* par, xmStyle s) : XmDialog(n, par, s)
{
	destroyOnDelete();
}

XmMsgBox::XmMsgBox(char* text, char* label, xmStyle s, XmObject* rec, cbProc okCb) : XmSystemDialog("msg", NULL, s ? s : XmSmsgOkCancel | XmSdlgAppModal)
{
#ifdef ASCII
	CwDialog* aDlg = new CwDialog("mbox-dialog", rec->handle()->getRoot(), 20, 8, 35, 12);
	CwButton* aButton;

	CwText* aText = new CwText(aDlg, "mbox-text", 2, 2, 31, 6);
	aText->setEditable(FALSE);
	aText->setText(text);
	aButton = new CwButton(aDlg, "OK", "&OK", 5, 9);
	aButton = new CwButton(aDlg, "Cancel", "&Cancel", 15, 9);
#else
	widcmd 	<< "toplevel .sorrywin\n"
		<< "label .sorrywin.text -text \"Sorry, message boxes are\nstill not implemented.\""
		<< "button .sorrywin.cmd -text OK -command { destroy .sorrywin }\n";
	wid->realize();
#endif
	setDefaultButtonProcs(rec, okCb);
	setLabelAndMode(label);
}


XmMsgBox::XmMsgBox(char* n, XmObject* par, xmStyle s) : XmSystemDialog(n, par, (s ? s : XmSmsgOkCancel | XmSdlgAppModal))
{
}

void XmMsgBox::setDefaultButtonProcs(XmObject* rec, cbProc okCb)
{
	if(!(dlgStyle & XmSdlgModeless) && validProc(okCb))
	{	cerr << "xmMsgBox: Warning - cannot use callback in modal dialog.\n";
		rec = NULL;
	}
#ifdef ASCII
	Widget dlgWid = wid;
	wid = getObjChild(dlgWid, "OK");
	if(wid || (wid = getObjChild(dlgWid, "Yes")))
	{	if(rec && validProc(okCb))
			setCallback(XtNcallback, rec, okCb);
		else
			setCallback(XtNcallback, this, CBK(XmDialog, ok));
	}
	else
		cerr << "XmMsgBox: Error - no OK or Yes button found...\n";
	wid = getObjChild(dlgWid, "Cancel");
	if(wid || (wid = getObjChild(dlgWid, "No")))
		setCallback(XtNcallback, this, CBK(XmDialog, cancel));
	wid = dlgWid;
#endif
}

bool XmMsgBox::showMsg()
{
	bool rcode = run();
	delete this;
	return(rcode);
}

bool XmMsgBox::setButtonText(msgButton b, char* t)
{
	if(!wid)
		return(FALSE);
#ifdef ASCII
	return(FALSE);
#endif
	return(FALSE);
}


XmPrompter::XmPrompter(char* n, XmObject* par, xmStyle s) : XmMsgBox(n, par, s)
{
}

#ifdef ASCII
XmPrompter::XmPrompter(char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) :XmMsgBox(prompt, label, s, rec, okCb)
{
}
#else
XmPrompter::XmPrompter(char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmMsgBox("prompter", (XmObject* )NULL, s)
{
}
#endif

char* XmPrompter::prompt()
{
	char* txt = NULL;

	if(run())
		txt = getText();
	delete this;
	return(txt);
}

bool XmPrompter::setText(char* txt)
{
#ifdef ASCII
	;
#endif
	return(TRUE);
}

char promptBuf[500];

char* XmPrompter::getText()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
#ifdef ASCII
	txt = NULL;
#endif
	return(txt);
}

#ifdef ASCII
XmListPrompter::XmListPrompter(char** list, char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("Sorry, list selection not implemented.", label, list[0], s, rec, okCb)
{
}
#else
XmListPrompter::XmListPrompter(char** list, char* prompt, char* label, char* def, xmStyle s, XmObject* rec, cbProc okCb) : XmPrompter("listPrompter", NULL, s)
{
}
#endif

int XmListPrompter::promptIndex()
{
	int ndx = -1;

	if(run())
		ndx = getIndex();
	delete this;
	return(ndx);
}

XmListPrompter* XmListPrompter::selectText(char*)
{
	return(this);
}

XmListPrompter* XmListPrompter::selectIndex(int)
{
	return(this);
}

int XmListPrompter::getIndex()
{
	return(0);
}

#ifdef XAW
XmFileSelector::XmFileSelector(char* path, char* deflt, xmStyle s, XmObject*rec, cbProc okCb) : XmPrompter("Enter filename:", "File Selector", path, s, rec, okCb)
{
}
#else
XmFileSelector::XmFileSelector(char* path, char* deflt, xmStyle s, XmObject*rec, cbProc okCb) : XmPrompter("fileSelector", NULL, s)
{
}
#endif

char* XmFileSelector::promptFile()
{
	char* txt;

	if(!run())
	{	delete this;
		return(NULL);
	}
	txt = getPath();
	delete this;
	return(txt);
}

bool XmFileSelector::setPath(char* txt)
{
#ifdef ASCII
	return(FALSE);
#else
	return(FALSE);
#endif
}

char* XmFileSelector::getPath()
{
	char* txt;

	if(!wid)
		return((char* )NULL);
	return(getText());
}


// Window classes /////////////////////////////////////////////////////////////


XmDialogWindow::XmDialogWindow(char* n, XmObject* par, xmStyle s) : XmUserDialog(n, par, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::XmDialogWindow(char* n, XmObject* par, XmManager* mgr, xmStyle s) : XmUserDialog(n, par, mgr, s)
{
	style = s;
	dropdown = NULL;
	popup = NULL;
}

XmDialogWindow::~XmDialogWindow()
{
//	if(manager)
//		manager->removeObject(this);
}


bool XmDialogWindow::realize()
{
	return(XmUserDialog::realize());
}

XmControl* XmDialogWindow::findControl(ctrlType t, char* n, bool w)
{
	return(XmDialogPane::findControl(t, n, w));
}

bool XmDialogWindow::add(XmControl* c, XmObject* rec, cbProc cb, void* cd)	
{
	return(XmDialogPane::add(c, rec, cb, cd));
}

void XmWindow::initWindowContents(char* n)
{
	ostrstream pn;

	pn << n << "Panes"; pn.put('\0');

	panes = new XmPaneArea(pn.str(), 0, 0, 1, 1);
	add(panes);
#ifdef TCL
	Widget w = panes->handle();
	w->setConstrained(TRUE);
	w->setPackingOptions("-side bottom -expand yes -fill both");
#endif
	toolbars[0] = toolbars[1] = toolbars[2] = toolbars[3] = NULL;
	statusbar = NULL;
}

void XmWindow::menuSet()
{
	checkLayout();
}

bool XmWindow::realize()
{
#ifdef ASCII
	checkLayout();
#endif
	return(XmUserDialog::realize());
}

inline int orNdx(xmStyle s)
{
	return(s == XmSleft ? 0 : (s == XmSright ? 1 : (s == XmSbottom ? 3 : 2)));
}

#ifdef ASCII

/*inline*/ int objDimension(XmObject* o, int which)
{
	int dims[4];
	o->getFrame(dims[0], dims[1], dims[2], dims[3]);
	return(dims[which]);
}
#define objX(o) objDimension(o, 0)
#define objY(o) objDimension(o, 1)
#define objW(o) objDimension(o, 2)
#define objH(o) objDimension(o, 3)
#define menuBarHeight 20
	
void XmWindow::checkLayout()
{
	int ox, oy, ow, oh;

	if(toolbars[2]) // top
	{	ox = 0;
		oy = dropdown ? menuBarHeight : 0;
		ow = objW(this);
		oh = objH(toolbars[2]);
		toolbars[2]->reframe(ox, oy, ow, oh);
	}
	if(toolbars[3]) // bottom
	{	ox = 0;
		oy = objH(this) - (statusbar ? objH(statusbar) : 0) - objH(toolbars[3]);
		ow = objW(this);
		oh = objH(toolbars[3]);
		toolbars[3]->reframe(ox, oy, ow, oh);
	}
	if(toolbars[0]) // left
	{	ox = 0;
		oy = (dropdown ? 2 : 0) + (toolbars[2] ? objH(toolbars[2]) : 0);
		ow = objW(this);
		oh = objH(this) - (dropdown ? menuBarHeight : 0)
						- (toolbars[2] ? objH(toolbars[2]) : 0)
						- (toolbars[3] ? objH(toolbars[3]) : 0)
						- (statusbar ? objH(statusbar) : 0);
		toolbars[0]->reframe(ox, oy, ow, oh);
	}
	if(toolbars[1]) // right
	{	ox = objW(this) - objW(toolbars[1]);
		oy = (dropdown ? 2 : 0) + (toolbars[2] ? objH(toolbars[2]) : 0);
		ow = objW(toolbars[1]);
		oh = objH(this) - (dropdown ? menuBarHeight : 0)
						- (toolbars[2] ? objH(toolbars[2]) : 0)
						- (toolbars[3] ? objH(toolbars[3]) : 0)
						- (statusbar ? objH(statusbar) : 0);
		toolbars[1]->reframe(ox, oy, ow, oh);
	}
	if(statusbar)
	{	ox = 0;
		oy = objH(this) - objH(statusbar);
		ow = objW(this);
		oh = objH(statusbar);
		statusbar->reframe(ox, oy, ow, oh);
	}
	ox = toolbars[0] ? objW(toolbars[0]) : 0;
	oy = (dropdown ? menuBarHeight : 0) + (toolbars[2] ? objH(toolbars[2]) : 0);
	ow = objW(this) - (toolbars[0] ? objW(toolbars[0]) : 0)
					- (toolbars[1] ? objW(toolbars[1]) : 0);
	oh = objH(this) - (dropdown ? 2 : 0)
					- (toolbars[2] ? objH(toolbars[2]) : 0)
					- (toolbars[3] ? objH(toolbars[3]) : 0)
					- (statusbar ? objH(statusbar) : 0);
	panes->reframe(ox, oy, ow, oh);
}

#else

void XmWindow::checkLayout()
{
/*
	if(toolbars[2]) // top
	if(toolbars[3]) // bottom
	if(toolbars[0]) // left
	if(toolbars[1]) // right
*/
}

#endif

bool XmWindow::addSubpane(ctrlType ct, char* n, char* l, int y, int h, XmObject* rec, cbProc cb)
{
	XmControl* aControl;

	switch(ct)
	{	case Edit:
		aControl = new XmEdit(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case ListBox:
		aControl = new XmListBox(n, 0, y, 10, h, XmSautovscroll | XmSautohscroll);
		break;
		case Drawing:
		case GroupBox: case StaticText: case StaticImage: case PushButton:
		case CheckBox: case RadioButton: case ScrollBar: case ComboBox: case Pane: 
		case PaneArea: case ToolBar:
		cerr << "xmWindow: Warning - control type " << ct << " not supported as subpane, ignoring...\n"; 
		return(FALSE);
	}
	return(addSubpane(aControl, l, rec, cb));
}

bool XmWindow::addSubpane(XmControl* c, char* l, XmObject* rec, cbProc cbk)
{
	if(!panes->numSubpanes())
		checkLayout();

	return(panes->addSubpane(c, l, rec, cbk));
}

XmToolBar* XmWindow::addToolbar(xmStyle s)
{
	ostrstream tbName;
	int ndx = orNdx(s);
	XmToolBar* aToolbar;

	if(toolbars[ndx])
		return(NULL);
	tbName << wname << "-Toolbar-" << ndx; tbName.put('\0');
	toolbars[ndx] = aToolbar = new XmToolBar(tbName.str(), (s == XmSleft || s == XmSright) ? XmSvertical : XmShorizontal);
	add(aToolbar);
	checkLayout();
	return(aToolbar);
}

bool XmWindow::removeToolbar(XmToolBar* tb)
{
	int ndx = -1;

	for(int i = 0; i < 4; i++)
	{	if(toolbars[i] == tb)
		{	ndx = i;
			break;
		}
	}
	if(ndx == -1)
		return(FALSE);
	toolbars[ndx] = NULL;
	checkLayout();
	remove((char* )tb->getName());
	return(TRUE);
}

XmToolBar* XmWindow::toolbar(xmStyle s)
{
	return(toolbars[orNdx(s)]);
}

XmStatusBar* XmWindow::createStatusBar()
{
	ostrstream stbName;

	if(statusbar)
		return(NULL);

	stbName << wname << "-Statusbar"; stbName.put('\0');
	statusbar = new XmStatusBar(stbName.str());
	add(statusbar);
	checkLayout();
	return(statusbar);
}

#ifndef __GNUG__

#define tbWinStyle (XmSdlgModeless | XmSbordered | XmSmoveable | XmScloseable | XmStitled | XmSsysMenu)

XmToolBox::XmToolBox(char* n, XmObject* par, xmStyle s, int r) : XmDialogWindow(n, par, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

XmToolBox::XmToolBox(char* n, XmObject* par, XmManager* m, xmStyle s, int r) : XmDialogWindow(n, par, m, tbWinStyle), XmToolBar(n, s)
{
	setRows(r);
	XmDialogWindow::add(this);
}

#endif

#ifdef TCL

template<class T> class mincoll
{
	T* contents;
	int curSz, maxSz;
	
	int grow()	{ T* oldcont = contents; contents = new T[maxSz += 10]; for(int i = 0; i < curSz; i++) contents[i] = oldcont[i]; delete oldcont; return(curSz); }
	int stretch(int start)	{ if(curSz == maxSz) grow(); for(int i = curSz; i > start; i--) contents[i] = contents[i - 1]; return(curSz); }
	int shrink(int start)	{ curSz--; for(int i = start; i < curSz; i++) contents[i] = contents[i + 1]; return(curSz); }
public:
	mincoll()	{ contents = new T[maxSz = 10]; curSz = 0; }
	~mincoll()	{ delete contents; }

	T at(int ndx)	{ return(ndx < curSz ? contents[ndx] : (T) 0L); }
	void atPut(int ndx, T el)	{ if(ndx < curSz) contents[ndx] = el; }
	int indexOf(T el)		{ int ndx = -1; for(int i = 0; i < curSz; i++) if(at(i) == el) { ndx = i; break; } return(i); }
	int size()			{ return(curSz); }

	int add(T el)	{ if(curSz == maxSz) grow(); contents[curSz++] = el; return(curSz - 1); }
	int insert(T el, int ndx)	{ stretch(ndx); contents[ndx] = el; return(ndx); }
	int remove(int ndx)		{ if(ndx >= 0 && ndx < curSz) shrink(ndx); return(ndx); }
	int remove(T el)		{ return(remove(indexOf(el))); }

	void zap()	{ for(int i = 0; i < curSz; i++) delete contents[i]; curSz = 0; }
};

#ifndef NO_TCP
#include "../../misc/ipc.h"

IPCStrm* tcpStrm = NULL;

void tclInitIO()
{
	char* dest;
	int port;

	if((dest = App->getCmdArg(1)) && (port = atoi(App->getCmdArg(2))))
	{	tcpStrm = new IPCStrm(port, dest);
		tcpStrm->sync(TRUE);
	}
}
#endif

int tclOutput(char* s)
{
#ifndef NO_TCP
	if(tcpStrm)
		tcpStrm->write(s, strlen(s) + 1);
	else
#endif
		cout << s;
	return(TRUE);	
}

int tclInput(char* inbuf, char sep)
{
	int i = 0;
	char c;

	do
	{
#ifndef NO_TCP
		if(tcpStrm)
		{	char tcpbuf[2];
			tcpStrm->read(tcpbuf, 1);
			c = tcpbuf[0];
		}
		else
#endif
			c = cin.get();
	}
	while((inbuf[i++] = c) != sep);
	inbuf[i - 2] = '\0';	// strip trailing blank also...
	return(TRUE);	
}

#ifdef TCL
#define isTclToplevel() (!(App->getCmdArg(1) && !strncmp(App->getCmdArg(1), "sub", 3)))
#endif

bool tclRootApp = TRUE;

void xmTclInitApp()
{
#ifndef NO_TCP
	tclInitIO();
#endif
	if(!isTclToplevel())
		tclRootApp = FALSE;
}

void xmTclQuitApp()
{
}

/*
 callback protocoll, general form:
events:  0, widget, eventCode, clientData, eventData, NULL
answers: 1, widget, requestSerial, answerData, NULL
*/


void xmTclNextEvent()
{
static char inbuf[4096];

	tclInput(inbuf);

	istrstream in(inbuf);
	int type = 0, code = 0;
	long wid = 0;
	char clData[256], * callData;
	in >> type;

	if(type == 0)		// input event...
	{ 	in >> wid >> code >> clData;
		callData = inbuf + in.tellg();
cerr << "event: " << " " << wid << " " << code << " cld: " << clData << " cad: " << callData << "\n";
		if(!wid)
		{	cerr << "callback: >" << inbuf << "< \ninvalid widget, ignoring...\n";
			return;
		}
		((XmTclController*)wid)->callCallbacks(code, callData, clData);
	}
	else if(type == 1)	// request answer...
	{	long serial;
		in >> wid >> serial;
		callData = inbuf + in.tellg() + 1;	// strip leading blank...
cerr << "answer: " << type << " " << wid << " " << serial << " " << callData << "\n";
		((XmTclController*)wid)->setAnswer(serial, callData);
	}
}

void xmTclMainLoop()
{
	while(1)
		xmTclNextEvent();
}


char* tclWinName(char* n)
{
	if(!n || !*n)
		return(n);
	char* wn = _mkstr(n);
	wn[0] = tolower(wn[0]);
	for(int i = 0; wn[i]; i++)
		if(!isalnum(wn[i]))
			wn[i] = '_';
	return(wn);
}

//int varSerial = 0;

XmTclController::XmTclController(XmObject* anObj, char* n, XmTclController* par)
{
	magic = XM_TCL_MAGIC_CONST;
	name = NULL;
	setName(n);
	ostrstream vn;
	vn << "var" << this /*varSerial++*/; vn.put('\0');
	varName = _mkstr(vn.str());
	curCmd = NULL;
	tclOk = FALSE;
	tclConstrained = TRUE;
	tclAutoRealize = FALSE;
	tclToplevel = FALSE;
	tclRealized = FALSE;
	tclRoot = FALSE;
	callbacks = new mincoll<XmTclCallback*>;
	obj = anObj;
	names = new mincoll<char*>;
	int nameCount = 0;
	parent = NULL;
	master = this;
	setParent(par);
	children = new mincoll<XmTclController*>;
	childrenCount = 0;
	packingOptions = NULL;
	answerSerial = 0;
	answerValue = NULL;
}

XmTclController::~XmTclController()
{
	delete name;
	callbacks->zap();
	names->zap();
	children->zap();
	delete callbacks;
	delete names;
	delete children;
	delete packingOptions;
	delete answerValue;
	magic = 0L;
}

void XmTclController::realize()
{
	XmTclController* c;

	if(tclRealized || (parent && !parent->isVisible()))
		return;
	if(!tclToplevel && !tclAutoRealize)
	{	if(tclConstrained)
		{	cmd() 	<< "pack " << getPath() << " "
				<< (packingOptions ? packingOptions : "") <<"\n";
		}
		else
		{	int nx, ny, nw, nh;
			obj->getFrame(nx, ny, nw, nh);
			cmd() 	<< "place " << getPath()
				<< " -x " << nx << " -y " << ny << "\n";
		}
	}
	execCmd();
	tclRealized = TRUE;	
	for(int i = 0; c = children->at(i); i++)
		c->realize();
}

bool XmTclController::addCbk(char* n, void* d, char* wn)
{
	callbacks->add(new XmTclCallback(n, d, wn));
	return(TRUE);
}

bool XmTclController::removeCbk(char* n, void* d, char* wn)
{
	XmTclCallback* c;
	
	for(int i = 0; c = callbacks->at(i); i++)
	{	if(c->equ(n, d, wn))
		{	callbacks->remove(i);
			return(TRUE);
		}
	}
	return(FALSE);
}

void XmTclController::callCallbacks(int reason, char* clientData, char* callData)
// clientData - data from changed widget
// callData - i.e. name of menu item
{
cerr << name << " callCallbacks " << callbacks->size() << "\n";
	if(!obj)
		return;
	XmTclCallback* cb;
	for(int i = 0; cb = callbacks->at(i); i++)
	{
cerr << cb->reason << " " << (cb->name ? cb->name : "NULL") << " == "
	<< reason << " " << (callData ? callData : "NULL") << " ???\n";
		if(obj->objectType() == Menu)
		{	if(cb->matches(reason, callData))
				menuCallback(this, cb->client_data, cb->name); // clientData);
		}
		else
		{	if(cb->matches(reason))
				genericCallback(this, cb->client_data, clientData);
		}
		if(!isValid())	// object may have been deleted in a callback...!
			break;
	}
}

long curRequestSerial = 0;

char* XmTclController::getValue(char* requestData, int wait)
{
static char reqbuf[500];

	if(!tclRealized)
		return(NULL);

	ostrstream req(reqbuf, 500);
	long sn = curRequestSerial++;

	req 	<< "puts \"1 " << long(this) << " "
		<< sn << " " << requestData << " \\b\"; flush stdout\n";
	req.put('\0');
	tclOutput(req.str());

	for(int i = 0; i < wait; i++)
	{	xmTclNextEvent();
		if(answerSerial == sn)
			return(answerValue);
	}
	return(NULL);
}

char* XmTclController::getVarValue(int wait)
{
	if(!varName)
		return(NULL);
	ostrstream vn;
	vn << "$" << varName;
	vn.put('\0');
	return(getValue(vn.str(), wait));
}

int XmTclController::addName(char* n)
{
	return(names->add(tclWinName(n)));
}

int XmTclController::insertName(char* n, int ndx)
{
	return(names->insert(tclWinName(n), ndx));
}

int XmTclController::removeName(char* n)
{
	int ndx = nameIndex(n);
	if(ndx != -1)
	{	delete names->at(ndx);
		return(names->remove(ndx));
	}
	return(-1);
}

int XmTclController::nameIndex(char* n)
{
	char* c, * s = tclWinName(n);
	
	for(int i = 0; c = names->at(i); i++)
	{	if(!strcmp(s, c))
		{	delete s;
			return(i);
		}
	}
	delete s;
	return(-1);
}

char tclRPB[500];

char* XmTclController::namePath(char* n)
{
	getPath();
	strcat(tclRPB, ".");
	return(strcat(tclRPB, n));
}
	
char* XmTclController::getPath(bool append)
{
	if(!append)
		tclRPB[0] = '\0';
	if(parent)
		parent->getPath(TRUE);
	else if(!tclRootApp)
	{	strcat(tclRPB, ".");
		strcat(tclRPB, App->getName());
	}
	else if(tclRoot)
		return(strcat(tclRPB, append ? "" : "."));
	strcat(tclRPB, ".");
	return(strcat(tclRPB, name));
}

void XmTclController::setName(char* n)
{
	delete name;
	name = tclWinName(n ? n : "unknown");
}

void XmTclController::addChild(XmTclController* c)
{
	children->add(c);
}

void XmTclController::removeChild(XmTclController* c)
{
	children->remove(c);
}

XmTclController* XmTclController::getChild(char* n)
{
	XmTclController* c;
	char* wn = tclWinName(n);
	
	for(int i = 0; c = children->at(i); i++)
	{	if(	(c->name && !strcmp(wn, c->name)) ||
			/*c->nameIndex(wn) ||*/
			(c = c->getChild(n)))
		{	delete wn;
			return(c);
		}
	}
	delete wn;
	return(0);
}

void XmTclController::execChildrenCmds()
{
	XmTclController* c;
	
	for(int i = 0; c = children->at(i); i++)
		c->execCmd();
}

char argstmtbuf[2000];

char* XmTclController::processArgs(ArgList lst, int n)
{
	if(n)
	{	ostrstream out(argstmtbuf, 2000);
		for(int i = 0; i < n; i++)
		{	if(!strcmp(lst[i].name, "text"))
			{	((XmControl*)obj)->setText((char*)lst[i].value);
				continue;
			}
		}
		out.put('\0');
		return(out.str());
	}
	return("");
}

char* XmTclController::resizeCmd(int, int)
{
	return("");
}

char* XmTclController::callbackString(char* clientData, char* callData, int reason)
{
static char cbkbuf[500];
	ostrstream cbk(cbkbuf, 500);
	cbk << "{ puts \"0 " << long(this) << " " << reason << " "
	    << (clientData ? clientData : "none") << " "
	    << (callData ? callData : "none")
            << " \\b\"; flush stdout }"; cbk.put('\0');
	return(cbk.str());
}


inline bool useTextCoordsFor(XmControl* ctr)
{
	return((ctr->isA(StaticText) || ctr->isA(PushButton) || ctr->isA(CheckBox) ||
		ctr->isA(RadioButton) || ctr->isAn(Edit) || ctr->isA(ListBox)) ? TRUE : FALSE);
}

#define tclFontWidth 7
#define tclFontHeight 16

int x2tclX(XmObject* o, int n)
{
	if(o->objectType() == Control)
	{	if(useTextCoordsFor((XmControl*)o))
			return(n / tclFontWidth);
	}
	return(n);
}

int x2tclY(XmObject* o, int n)
{
	if(o->objectType() == Control)
	{	if(useTextCoordsFor((XmControl*)o))
			return(n / tclFontHeight);
	}
	return(n);
}

int tcl2xX(XmObject*, int n) { return(n); }
int tcl2xY(XmObject*, int n) { return(n); }

#endif
