#ifndef WidgetLayout_included
#define WidgetLayout_included

#include <qobject.h>
#include <qwidget.h>
#include <qlist.h>
#include <qdict.h>
#include <qrect.h>
#include <qcursor.h>
#include <qpopmenu.h>
#include <qstrlist.h>

#include "WidgetFrame.h"

#include "widget/DlgWidget.h"

class DlgParser;
class DialogWnd;
class BoxLayout;
class LayoutItem;

struct Selected
{
    Selected( QWidget *w )
	{ widget = w; };

    QWidget *widget;
    QPoint   newPos;

    QRect		itsResizeSquares[8];
    // squares to grab and drag in the widget's coordinates for the
    // selected widget
};

class SelectList : public QList<Selected>
{
protected:
    int compareItems( GCI i1, GCI i2 )
	{ return (((const Selected*)i1)->widget != 
		  ((const Selected*)i2)->widget ); };
};

class WidgetLayout : public QObject
{
    Q_OBJECT

    friend class DialogPropDialog;
    // The property dialog needs access to the layout
    
public: // methods
    
    WidgetLayout
    (
	DialogWnd*	parent = NULL,
	const char*	name = NULL
    );


    virtual ~WidgetLayout();
    

    void Start( void );
    // Start the widget layout

    DlgWidget* AddWidget
    ( 	
	DlgWidget::Type	type
    );
    
    void SetDragging
    (
	QWidget*	widget
    );

    void SetNoMove
    (
	QWidget*	widget
    );
    // don't allow user to move top left corner

    QWidget* GetSelectedWidget( void );

    DlgWidget *GetSelectedDlgWidget();

    void GetWidgetNames
    (
	QStrList&	nameList
    ) const;
    // Get a list of the names of the widgets

    void GetWidgetTypeNames
    (
	DlgWidget::Type type,
	QStrList&	nameList
    ) const;
    // Get a list of the names of the given type of widget

    const DlgWidget* GetWidget
    (	
	const QString&	name
    ) const;
    // Given a name of a widget, return it or NULL if none match

    void SetDlgEditWnd
    (
	WidgetFrame*	editWnd
    );
    // set the window used to hold the widgets

    WidgetFrame *GetDlgEditWnd() const;
    // retrieve the window used to hold the widgets

    void SaveContents
    (
	QTextStream&
    );
    // save contents, including the widgets

    void ResetContents( void );
    // reset the contents of the layout

    void RestoreContents
    (
	DlgParser *
    );
    // restore contents, creating the widgets

    void GenerateDataHeader
    (
	QTextStream&	stream,
	const QString&	dataName,
	const QString&	dataBaseName,
	const QString&	dataBaseHeader
    );
    // generate the dialog data header file

    void GenerateDataSource
    (
	QTextStream&	stream,
	const QString&	dataHeader,
	const QString&	dataName,
	const QString&	dataBaseName,
	bool		modalWindow,
	WFlags		windowFlags
    );
    // generate the dialog data source file

    void RaiseDlgWidget
    (
	const DlgWidget*	dlgWidget
    );
    // raise the dlg widget in the z ordering

    void LowerDlgWidget
    (
	const DlgWidget*	dlgWidget
    );
    // lower the dlg widget in the z ordering
    
    void EnableLayout();
    // Enable interaction

    void DisableLayout();
    // Disable interaction

    void AutoSelect( bool = TRUE );
    // Automatically select all new widgets

    bool CopySelected( QTextStream & );
    bool CutSelected( QTextStream & );

    //
    // Geometry Layout
    //

    enum LayoutType
    {
	None,
	Box
    };
    // the type of layout used with this window

    LayoutType GetLayoutType( void ) const;
    
    void SetLayoutType
    (
	LayoutType	type
    );
    // Get/Set methods for the type of layout used

    enum ScreenFlags
    {
	SF_None			= 0x0000,
	SF_WidgetSelected	= 0x0001,
	SF_Align		= 0x0002,
	SF_Layout		= 0x0004,
	SF_BoxLayout		= 0x0008
    };
    // Flags used to set the availability of operations through the menus
    // and toolbars

    QSize GetWindowSize( void ) const;
    void SetWindowSize( const QSize&	theSize );
    // Get/Set methods on the window's size

    const QSize& GetWindowMinSize( void ) const;
    void SetWindowMinSize( const QSize&	theSize );
    // Get/Set methods on the window's min size

    const QSize& GetWindowMaxSize( void ) const;
    void SetWindowMaxSize( const QSize&	theSize );
    // Get/Set methods on the window's max size

    const QPoint& GetWindowPos( void ) const;
    void SetWindowPos( const QPoint&	thePoint );
    // Get/Set methods on the window's initial position, -1,-1 for no initial
    // position
    

signals:

    void Changed();

    void WidgetWndChanged();

    void WidgetGeometryChanged( int, int, int, int );
    // emit when a widget's geometry has changed with its x and y coordinate
    // and width and height

    void WidgetSelected( const QString&, const QString& );
    // emit when a widget is selected with the widget's name and 
    //mapped variable
    
    void WidgetDeselected();
    // emit when all widgets where deselected (Mouse button 2)

    void UpdateScreen( int );
    // Tell the dialog window to update the screen
    
public slots: // methods

    // Process events on the widgets

    void ProcessMousePress
    (
	QWidget* 	widget,
	QMouseEvent*	event
    );
    
    void ProcessMouseDoubleClick( QWidget*	widget,
				  QMouseEvent*	event );
    
    void ProcessMouseMove
    (	
	QWidget*	widget,
	QMouseEvent*	event
    );
    
    void ProcessMouseRelease();
    
    void ProcessPaint
    (	
	QWidget*	widget
    );
    
    void ProcessResize
    (
	QWidget*	widget,
	QResizeEvent*	event
    );

    void RaiseSelectedWidget();

    void LowerSelectedWidget();

    void DeleteSelectedWidget();
    
    void SelectedWidgetProperties();
    void OptionsGrid();

    void ContentsRestored();
    // All Widgets are restored, disconnect from Parser

    void AlignTop();
    // align selected widgets to top of first widget

    void AlignBottom();
    // align selected widgets to bottom of first widget

    void AlignLeft();
    // align selected widgets to left of first widget

    void AlignRight();
    // align selected widgets to right of first widget

    void SizeHorizontal();
    // give selected widgets the same width as the first widget

    void SizeVertical();
    // give selected widgets the same height as the first widget

    void SizeBothHV();
    // give selected widgets the same width and height as the first widget

    void SizeToFit();
    // Resizes the widget to its content's size

    void MoveUp();
    // Move the selected widgets up one grid spacing

    void MoveDown();
    // Move the selected widgets down one grid spacing

    void MoveLeft();
    // Move the selected widgets left one grid spacing

    void MoveRight();
    // Move the selected widgets right one grid spacing

    void ResizeShorter();
    // Resize the selected widgets one grid spacing shorter

    void ResizeWider();
    // Resize the selected widgets one grid spacing widget

    void ResizeNarrower();
    // Resize the selected widgets one grid spacing narrower

    void ResizeTaller();
    // Resize the selected widgets one grid spacing taller

    void InsertBoxLayout();
    // Insert a box layout into the currently selected layout

    void InsertGridLayout();
    // Insert a grid layout into the currently selected layout

    void InsertBoxSpacing();
    // Insert spacing into the currently selected box layout

    void InsertBoxStretch();
    // Insert stretch into the currently selected box layout

    void LayoutMousePress
    (
	QWidget*	widget,
	QMouseEvent*	event
    );
    // The user has pressed the mouse on a layout item

    void LayoutPaint
    (
	QWidget*	widget
    );
    // A layout item has been painted

private slots: // methods

    void RestoreNewWidget
    ( 
	QString &widgetName 
    );
    // Restore a new Widget

    void RestoreLayoutKeyValue
    ( 
	QString &key, 
	QString &value 
    );        
    // Restore a single key/value pair for the Layout 

    void RestoreWidgetKeyValue
    ( 
	QString &key, 
	QString &value 
    );        
    // Restore a single key/value pair for a Widget

    void RestoreEndWidget();        
    // Constructing a new widget is finished

    void RestoreNewLayoutItem( QString&	itemName );
    // Restore a new layout item

    void RestoreLayoutItemKeyValue( QString&	key,
				    QString&	value );
    // Restore a single key/value pair for a layout item

    void RestoreEndLayoutItem();
    // Constructing a new layout item is finished

private: // methods

    QRect CalculateGeometryRect
    (
	QWidget*	widget,
	QPoint		pos
    );
    // calculate the new geometry rect for the widget


    DlgWidget* GetDlgWidget
    (
	const QWidget*	widget 
    );
    // given a widget, return its DlgWidget
    
    void SetSelected
    (
	QWidget*	widget,
	bool 		selected
    );

    void ClearSelection();

    void AddToSelection( QWidget * );

    void RemoveFromSelection( QWidget * );

    bool IsSelected( QWidget*	widget );

    void UpdateCursor
    (
	QWidget*	widget
    );

    void MoveWidget( const QPoint&	newPos );
    // move the widget be the new position relative to the top left corner

    void ResizeWidget( QWidget*		widget,
		       const QRect&	newRect );
    // Resize the widget to the specified rect

    void WidgetGeometryChange( QWidget* widget );
    // process a change in widget geometry

    void ConnectToWidget( QWidget* widget );
    // connect the necessary events from the widget to the slots for 
    // manipulating the widget

    void LowerLayout( void );
    // Lower all the widgets that represent layout items
        
    LayoutItem* FindSelectedLayoutItem( void ) const;
    // Return the selected layout item or NULL if none

    BoxLayout* FindLayoutForSelectedItem( void ) const;
    // Return the layout the selected layout item is on or NULL if none

    void UpdateWidgetLayoutState
    (
	const DlgWidget*	dlgWidget
    );
    // Update the state of the widget

    void AddScreenFlags( int flags );
    // Add the given flags to the set of flags and update the screen

    void RemoveScreenFlags( int flags );
    // Remove the given flags from the set of flags and update the screen

    void GenerateWidgetSource
    (
	DlgWidget*		widget,
	QTextStream&		stream
    );
    // generate the source for one widget

    void GenerateWidgetExtraSource( DlgWidget* 		widget,
				    QTextStream&	stream );
    // Generate extra source for the widget after all have been constructed
    // and initialized.

    void GenerateLayoutSource( QTextStream&	stream );
    // generate the source for the layout of the window


private: // attributes

    enum ResizeSquare
    {
	topLeft 	= 0,
	left 		= 1,
	bottomLeft	= 2,
	bottom		= 3,
	bottomRight	= 4,
	right		= 5,
	topRight	= 6,
	top		= 7,
	none		= 8
    };

    enum RestoreAttr
    {
	APPLICATION,
	LAYOUT,
	WIDGET
    };

    RestoreAttr attrType;
    // The type of the actual read attribute
  
    DialogWnd *itsParent;
    // The application
  
    WidgetFrame*		itsDlgEditWnd;
    // window on which the widgets are created and manipulated
    
    QPopupMenu* itsWidgetPopup;
    // popup menu for widgets

    QPopupMenu*	itsWidgetFramePopup;
    // popup menu for the widget frame
    
    SelectList itsSelectedWidget;
    // List of actual selected Widgets

    QRect		itsResizeSquares[8];
    // squares to grab and drag in the widget's coordinates for the
    // selected widget

    QPoint		itsWidgetClick;
    bool		isDragging;
    ResizeSquare	itsDragSquare;

    const QCursor*	itsCursors[9];

    QDict< DlgWidget >	itsDlgWidgets;
    // dictionary of dlgwidgets indexed by the name of the widget

    QDict< QWidget >	itsNoMoveWidgets;
    // widgets whose top left corner doesn't move

    QDict< QWidget >	itsFixedSizeWidgets;
    // widgets with a fixed size only

    QList< DlgWidget >	itsZOrdering;
    // z ordering of the widgets on the screen.  index 0 is the bottommost
    // widget

    DlgParser *itsParser;
    // The Parser which will be used to read the dilaog file

    DlgWidget *widgetInConstruction;
    // The actual Widget which is just build

    LayoutItem* itsLayoutItemInConstruction;
    // The layout item being constructed

    bool autoSelect;
    // Automatically select all new widgets

    static int layoutNumber;
    // Numbering all the Layouts to allow automatic name generation

    LayoutType	itsLayoutType;
    // the type of layout used 

    BoxLayout*	itsBoxLayout;
    // the box layout used, only defined if the layout type is BoxLayout

    QWidget* itsSelectedLayoutWidget;
    // the widget representing the currently selected layout item

    int itsScreenFlags;
    // the flags used to determine which screen elements are enabled

    QSize itsWindowMinSize;
    QSize itsWindowMaxSize;
    // Min and Max sizes of the window
    
    QPoint itsWindowPos;
    // The initial position of the window, -1,-1 for no initial pos
};


inline void WidgetLayout::Start( void )
{
    emit UpdateScreen( itsScreenFlags );
}


inline const DlgWidget* WidgetLayout::GetWidget
(
    const QString&	name
) const
{
    return itsDlgWidgets[ name ];
}


inline WidgetFrame *WidgetLayout::GetDlgEditWnd() const
{
    return itsDlgEditWnd;
}

inline void WidgetLayout::SetNoMove
(
    QWidget*	widget
)
{
    itsNoMoveWidgets.insert( widget->name(), widget );
}


inline void WidgetLayout::SetDlgEditWnd
(
    WidgetFrame*	editWnd
)
{
    itsDlgEditWnd = editWnd;
}


inline void WidgetLayout::AutoSelect( bool aS )
{
    autoSelect = aS;
}


inline WidgetLayout::LayoutType WidgetLayout::GetLayoutType( void ) const
{
    return itsLayoutType;
}


inline void WidgetLayout::AddScreenFlags( int flags )
{
    itsScreenFlags |= flags;
    emit UpdateScreen( itsScreenFlags );
}


inline void WidgetLayout::RemoveScreenFlags( int flags )
{
    itsScreenFlags &= ~flags;
    emit UpdateScreen( itsScreenFlags );
}


inline QSize WidgetLayout::GetWindowSize( void ) const
{
    return itsDlgEditWnd->size();
}


inline void WidgetLayout::SetWindowSize( const QSize&	theSize )
{
    itsDlgEditWnd->setFixedSize( theSize );
    emit WidgetWndChanged();
}


inline const QSize& WidgetLayout::GetWindowMinSize( void ) const
{
    return itsWindowMinSize;
}


inline void WidgetLayout::SetWindowMinSize( const QSize&	theSize )
{
    itsWindowMinSize = theSize;
}


inline const QSize& WidgetLayout::GetWindowMaxSize( void ) const
{
    return itsWindowMaxSize;
}


inline void WidgetLayout::SetWindowMaxSize( const QSize&	theSize )
{
    itsWindowMaxSize = theSize;
}


inline const QPoint& WidgetLayout::GetWindowPos( void ) const
{
    return itsWindowPos;
}


inline void WidgetLayout::SetWindowPos( const QPoint&	thePoint )
{
    itsWindowPos = thePoint;
}


#endif // WidgetLayout_included



