#include <math.h>
#include <stdlib.h>

#include <qpen.h>
#include <qpainter.h>
#include <qbitmap.h>
#include <qimage.h>
#include <qtstream.h>

#include "pixmap/PixmapWnd.h"
#include "pixmap/PixmapDraw.h"

void EmbedData( const QByteArray &input, QFile *output );

inline int Abs( int i )
{
  return i < 0 ? -i : i;
}

PixmapDraw::PixmapDraw( QWidget *p, const char *name )
  : QFrame( p, name ),
    itsHorScr( new QScrollBar( QScrollBar::Horizontal, this )),
    itsVerScr( new QScrollBar( QScrollBar::Vertical, this ))
{
  itsHorScr->hide();
  itsVerScr->hide();

  itsTimer = NULL;

  actCellSize = 8;

  previewWindow = NULL;

  regionActive = FALSE;

  itsPixmap = new QPixmap;
  itsClipboard = new QPixmap;

  leftColor = black;
  rightColor = backgroundColor();
  backGround = white;

  lastPoint = QPoint( -1, -1 );

  connect( itsHorScr, SIGNAL(valueChanged(int)), 
           this, SLOT(HorScrBarChanged(int)));
  connect( itsVerScr, SIGNAL(valueChanged(int)), 
           this, SLOT(VerScrBarChanged(int)));

  mode = DrawPoint;

  setMouseTracking( TRUE );

  horOff = verOff = 0;
}

PixmapDraw::~PixmapDraw()
{
}

void PixmapDraw::SetSize( const QSize &size )
{
  numRows = size.height();
  numCols = size.width();

  actCellSize = CalcCellSize();
 
  QPixmap temp( *itsPixmap );
  itsPixmap->resize( size );
  itsPixmap->fill( backGround );
  
  if( ! temp.isNull() )
    bitBlt( itsPixmap, 0, 0, &temp );

  repaint();

  if( previewWindow )
    previewWindow->Resize();

  emit ZoomChanged( actCellSize );
}

QSize PixmapDraw::GetSize()
{
  return itsPixmap->size();
}

void PixmapDraw::SetBackgroundColor( const QColor &col )
{
  // This deletes all previous work...
  // itsPixmap->fill( backGround );
  int b;
  QImage image = itsPixmap->convertToImage();

  if( image.depth() < 16 )
  {
    QRgb *colorTable = image.colorTable();

    for( b = 0; b < image.numColors(); b++ )
    {
      if( colorTable[b] == backGround.rgb() )
        break;
    }

    if( b == image.numColors() )
    {
      warning( "Changing background color failed.\n" );

      return;
    }

    image.setColor( b, col.rgb() );
    itsPixmap->convertFromImage( image );  

    repaint();
    if( previewWindow )
      previewWindow->repaint();

    backGround = col;
  } else
  {
    warning( "Please use fill to change background color in 16 bit images" );
  }
}

void PixmapDraw::SetupPreview( Previewer *previewer )
{
  previewWindow = previewer;
  previewWindow->SetPixmap( itsPixmap );
}

void PixmapDraw::Rotate( float a )
{
  QWMatrix matrix;

  QPixmap pixmap;
  QSize origSize;
  QPoint origCenter;
  QPoint offset;
  QPixmap newPixmap;

  if( regionActive )
  {
    origSize = clipboardRect.size();
    origCenter = clipboardRect.center();
    offset = clipboardRect.topLeft();
    pixmap.resize( origSize );
    bitBlt( &pixmap, QPoint( 0, 0 ), itsPixmap, clipboardRect );
  } else
  {
    origSize = itsPixmap->size();
    origCenter = itsPixmap->rect().center();
    offset = QPoint( 0, 0 );
    pixmap = *itsPixmap;
  }

  matrix.translate( origCenter.x(), origCenter.y() );
  matrix.rotate( a );

  pixmap = pixmap.xForm( matrix );
  QRect rect( QPoint( pixmap.width() / 2 - origSize.width() / 2,
                      pixmap.height() / 2 - origSize.height() / 2 ),
              origSize );
 
  bitBlt( itsPixmap, offset, &pixmap, rect );

  if( previewWindow )
    previewWindow->repaint();

}

void PixmapDraw::Flip( FlipDirection dir )
{
  QWMatrix matrix;

  QPixmap pixmap;

  if( regionActive )
  {
    pixmap.resize( clipboardRect.size() );
    bitBlt( &pixmap, QPoint( 0, 0 ), itsPixmap, clipboardRect );
  } else
  {
    pixmap = *itsPixmap;
  }

  if( dir == Vertical )
    matrix.setMatrix( 1, 0, 0, -1, 0, pixmap.height() );
  else
    matrix.setMatrix( -1, 0, 0, 1, pixmap.width(), 0 );

  pixmap = pixmap.xForm( matrix );

  if( regionActive )
    bitBlt( itsPixmap, clipboardRect.topLeft(), &pixmap );
  else
    *itsPixmap = pixmap;

  matrix.reset();
  matrix.scale( actCellSize, actCellSize );

  repaint( FALSE );

  if( previewWindow )
    previewWindow->repaint();

}

bool PixmapDraw::LoadFile( const QString &fileName )
{
  bool ok = itsPixmap->load( fileName );

  if( ok )
  {
    SetSize( itsPixmap->size() );
    repaint();
    if( previewWindow )
    {
      previewWindow->Resize();
      previewWindow->repaint();
    }
  }
  return ok;
}

bool PixmapDraw::WriteFile( const QString &fileName, const QString &format )
{
  return itsPixmap->save( fileName, format );
}

void PixmapDraw::GeneratePixmap( QString &varName, QString &varLengthName, 
                                 QFile *source )
{
  // A little hack to simplyfy things here:
  // Generate temporary name, save pixmap to this filename,
  // use ordinary file access methods to get pixmaps data. This avoids 
  // nasty conversions to QImage, and scanning through scanlines

  // Code taken from qembed (C) by Troll Tech As. who grants all rights
  // to everyone

  QString outName( tmpnam( 0 ) );

  if( outName.isEmpty() )
  {
    warning( "Couldn't create a temporary file name for this pixmap" );
    return;
  }

  itsPixmap->save( outName, "BMP" );

  QFile f( outName );
	if ( !f.open(IO_ReadOnly) ) {
    warning( "Cannot reopen temporarly saved pixmap %s", (const char *) outName );
    return;
	}
	QByteArray a( f.size() + 1 );
  if( ! a.data() )
    fatal( "No memory for %d bytes", f.size() );

	if ( f.readBlock(a.data(), f.size()) != (int)f.size() ) {
    warning( "Cannot read file %s", (const char *) outName );
    f.close();
    return;
	}

  QTextStream stream( source );
  stream << "const unsigned int " << varLengthName << " = " <<
    f.size() << ";\n";
  stream << "const unsigned char " << varName << "[] = {";
  
  EmbedData( a, source );

  f.close();
  
  // remove temp file 
  remove( outName ); 

  stream << "\n};\n\n";
}

void PixmapDraw::drawContents( QPainter *painter )
{
  painter = painter;

  QSize rSize;

  if( itsPixmap )
  {
    // Performance hack
    if( actCellSize == 1 )
    {
      bitBlt( this, 0, 0, itsPixmap, horOff, verOff );
    } else
    {
      QImage image = itsPixmap->convertToImage();
      int depth = image.depth();

      rSize = QSize( actCellSize, actCellSize );
      QRect rect( QPoint( 0, 0 ), rSize );
      int x, y;

      // Calculate the size which is seen
      int width = image.width();
      width = width < contRectWidth ? width : contRectWidth;
      int imageHeight = image.height();
      int height = image.height();
      height = height < contRectHeight ? height : contRectHeight;

      // Draw the pixmap
      int offX, offY;
      for( y = verOff, offY = 1; 
           y < verOff + height; 
           y++, offY += actCellSize )
      {
        // Under somewhat obscure circumstances we reach the limit and 
        // dump core. Reproduction: Pixmap 232x232, scroll 3/4 down, zoom out  
        if( y  >= imageHeight )
          continue;
        for( x = horOff, offX = 1; 
             x < horOff + width; 
             x++, offX += actCellSize )
        {
          rect.moveTopLeft( QPoint( offX, offY ) );
          uint *line = (uint *) image.scanLine( y );
          if( depth < 16 )
          {
            painter->fillRect( rect, 
                               QBrush( image.color(*(image.scanLine(y) + x)))
                               );
          } else
          {
            painter->fillRect( rect, 
                               QBrush( ( *( line + x ) ) )
                               );
          }

          if( useGrid )
            painter->drawPoint( rect.topLeft() );
        }
      }
    }
    // Surround it with a black line
    rSize.setWidth( itsPixmap->width() * actCellSize + 2 );
    rSize.setHeight( itsPixmap->height() * actCellSize + 2 );
    painter->drawRect( QRect( QPoint( 0, 0 ), rSize ));
  }
}

void PixmapDraw::resizeEvent ( QResizeEvent *rE )
{
  actCellSize = CalcCellSize();

  // Positioning the scrollbars
  itsVerScr->setGeometry( size().width() - SCRBAR_DIMENSION, 0,
                          SCRBAR_DIMENSION, size().height() - SCRBAR_DIMENSION );

  itsHorScr->setGeometry( 0, size().height() - SCRBAR_DIMENSION, 
                          size().width() - SCRBAR_DIMENSION, SCRBAR_DIMENSION );

  QFrame::resizeEvent( rE );

  CalcContRect();  
}

void PixmapDraw::mousePressEvent( QMouseEvent *mE )
{
  QPainter painter;
  painter.begin( itsPixmap );

  int colNum = CoordTrans( mE->pos().x() ) + horOff;
  int rowNum = CoordTrans( mE->pos().y() ) + verOff;

  emit MouseTracking( colNum, rowNum, 0, 0 );

  QColor c = black;
  if( mE->button() == LeftButton )
    c = leftColor;
  else if( mE->button() == RightButton )
    c = rightColor;

  if( regionActive )
    painter.setClipRect( clipboardRect );
  
  switch( mode )
  {
  case DrawPoint:
    painter.fillRect( colNum, rowNum, pixelWidth, pixelWidth, c );
    emit Changed();
    break;
  case DrawLine:
  case DrawRect:
  case DrawCircle:
  case DrawRegion:
    firstPoint = QPoint( (colNum-horOff) * actCellSize + actCellSize / 2,
                         (rowNum-verOff) * actCellSize + actCellSize / 2 );
    break;
  case DrawFill:
    Fill( colNum, rowNum, c );
    emit Changed();
    break;
  case DrawPaste:
    if( regionActive )
    {
      QRect r = clipboardRect;
      r.moveTopLeft( QPoint( colNum, rowNum ) );
      r = r.intersect( clipboardRect );
      int cN = colNum;
      int rN = rowNum;

      if( colNum <= r.left() )
      {
         cN = r.left();
      }
      if( rowNum <= r.top() )
      {
        rN = r.top();
      }

      r.moveTopLeft( r.topLeft() - QPoint( colNum, rowNum ) );
      if( ! r.isEmpty() )
        bitBlt( itsPixmap, QPoint( cN, rN ), itsClipboard, r );
    } else
    { 
      bitBlt( itsPixmap, colNum, rowNum, itsClipboard );
    }
    break;
  default:
    break;
  }
  painter.end();
  repaint( FALSE );
}

void PixmapDraw::mouseMoveEvent( QMouseEvent *mE )
{
  QPoint start( CoordTrans( firstPoint.x() ) + horOff, 
                CoordTrans( firstPoint.y() ) + verOff );
  QPoint end( CoordTrans( mE->pos().x() ) + horOff,
              CoordTrans( mE->pos().y() ) + verOff );

  int maxX = itsPixmap->width() * actCellSize;
  int maxY = itsPixmap->height() * actCellSize;

  if( !( mE->state() & LeftButton || 
         mE->state() & RightButton ||
         mode == DrawPaste ))
  {
    emit MouseTracking( end.x(), end.y(), 0, 0 );
    return;
  }

  emit MouseTracking( start.x(), start.y(), 
                      Abs( start.x() - end.x() ), 
                      Abs( start.y() - end.y() ) );
  
  QPainter painter;
  painter.begin( this );
  painter.setClipRect( 0, 0, maxX, maxY );
  QPen pen;
  pen.setColor( black );
  pen.setStyle( (PenStyle) lineStyle );
  pen.setWidth( pixelWidth * actCellSize );
  painter.setPen( pen );

  painter.setRasterOp( NotXorROP );
      
  if( regionActive )
  {
    QRect rect = clipboardRect;
    rect.moveTopLeft( rect.topLeft() * actCellSize );
    rect.setSize( rect.size() * actCellSize );

    painter.setClipRect( rect );
  }

  switch( mode )
  {
  case DrawPoint:
    break;
  case DrawLine:
    {
      if( lastPoint.x() != -1 )
        painter.drawLine( firstPoint, lastPoint );
      lastPoint = mE->pos();
      painter.drawLine( firstPoint, lastPoint );
    }
  break;
  case DrawRect:
    {
      if( lastPoint.x() != -1 )
        painter.drawRect( QRect( firstPoint, lastPoint ) );
      lastPoint = mE->pos();
      if( mE->state() & ControlButton )
        lastPoint.setX( firstPoint.x() - firstPoint.y() + lastPoint.y() );
      painter.drawRect( QRect( firstPoint, lastPoint ) );
    }
  break;
  case DrawCircle:
    {
      if( lastPoint.x() != -1 )
        painter.drawArc( QRect( firstPoint, lastPoint ), 
                         0, 16*360 );
      lastPoint = mE->pos();
      if( mE->state() & ControlButton )
        lastPoint.setX( firstPoint.x() - firstPoint.y() + lastPoint.y() );
      painter.drawArc( QRect( firstPoint, lastPoint ), 
                       0, 16*360 );
    }
  break;
  case DrawFill:
    break;
  case DrawRegion:
    {
      // Reset painter
      pen.setStyle( (PenStyle) SolidLine );
      pen.setWidth( 1 );
      painter.setPen( pen );
      
      if( lastPoint.x() != -1 )
        painter.drawRect( QRect( firstPoint, lastPoint ) );
      lastPoint = mE->pos();
      if( mE->state() & ControlButton )
        lastPoint.setX( firstPoint.x() - firstPoint.y() + lastPoint.y() );
      painter.drawRect( QRect( firstPoint, lastPoint ) );
    }
  break;
  case DrawPaste:
    {
      // Reset painter
      pen.setStyle( (PenStyle) SolidLine );
      pen.setWidth( 1 );
      painter.setPen( pen );

      QRect rect = clipboardRect;
      rect.setSize( rect.size() * actCellSize );
      
      if( lastPoint.x() != -1 )
      {
        rect.moveTopLeft( lastPoint );
        painter.drawRect( rect );
      }
      lastPoint = mE->pos();
      rect.moveTopLeft( lastPoint );
      painter.drawRect( rect );
    }
  break;
  default:
    break;
  }
  painter.end();
}
  
void PixmapDraw::mouseReleaseEvent( QMouseEvent *mE )
{
  QPainter painter;
  painter.begin( itsPixmap );

  QPoint start( CoordTrans( firstPoint.x() ) + horOff, 
                CoordTrans( firstPoint.y() ) + verOff );
  QPoint end( CoordTrans( mE->pos().x() ) + horOff,
              CoordTrans( mE->pos().y() ) + verOff );
  if( mE->state() & ControlButton )
    end.setX( start.x() - start.y() + end.y() );

  QPen pen;
  if( mE->button() == LeftButton )
    pen.setColor( leftColor );
  else if( mE->button() == RightButton )
    pen.setColor( rightColor );
  pen.setStyle( (PenStyle) lineStyle );
  pen.setWidth( pixelWidth );
  painter.setPen( pen );

  if( regionActive )
    painter.setClipRect( clipboardRect );
  
  switch( mode )
  {
  case DrawPoint:
    break;
  case DrawLine:
    painter.drawLine( start, end );
    emit Changed();
    break;
  case DrawRect:
    painter.drawRect( QRect( start, end ) );
    emit Changed();
    break;
  case DrawCircle:
    painter.drawArc( QRect( start, end ), 0, 16*360 );
    emit Changed();
    break;
  case DrawFill:
    break;
  case DrawRegion:
    if( ! itsClipboard )
      itsClipboard = new QPixmap;
    itsClipboard->resize( QRect( start, end ).size() );
    clipboardRect = QRect( start, end );
    if( ! itsTimer )
    {
      itsTimer = new QTimer( this );
      connect( itsTimer, SIGNAL(timeout()), this, SLOT(TimerEvent()));
    }
    itsTimer->start( 500 );
    selectionVisible = FALSE;
    emit CPAvailable();
    break;
  default:
    break;
  }
  painter.end();
  lastPoint = QPoint( -1, -1 );

  repaint( FALSE );

  if( previewWindow )
    previewWindow->repaint();
}

void PixmapDraw::enterEvent( QEvent * )
{
  if( mode == DrawPaste )
    repaint( FALSE );
}

void PixmapDraw::leaveEvent( QEvent * )
{
  int maxX = itsPixmap->width() * actCellSize;
  int maxY = itsPixmap->height() * actCellSize;

  QPainter painter;
  painter.begin( this );
  painter.setClipRect( 0, 0, maxX, maxY );
  QPen pen;
  pen.setColor( black );
  painter.setPen( pen );

  painter.setRasterOp( NotXorROP );
      

  if( mode == DrawPaste )
  {
    QRect rect = clipboardRect;
    rect.setSize( rect.size() * actCellSize );
      
    if( lastPoint.x() != -1 )
    {
      rect.moveTopLeft( lastPoint );
      painter.drawRect( rect );
    }
    lastPoint.setX( -1 );
  }
  painter.end();
}    

void PixmapDraw::Fill( int sX, int sY, const QColor &nC )
{
  QImage image = itsPixmap->convertToImage();

  uint nR, oR;

  if( image.depth() < 16 )
  {
    QRgb nRgb = nC.rgb();
  
    // Check the existance of nC in images color table
    QRgb *colorTable = image.colorTable();
    
    int b;
    for( b = 0; b < image.numColors(); b++ )
    {
      if( colorTable[b] == nRgb )
        break;
    }
    
    if( b == image.numColors() )
    {
      image.setNumColors( b + 1 );
      image.setColor( b, nRgb );
    }

    // Colortable index
    nR = b;

    // Get the old color
    colorTable = image.colorTable();
    QRgb oRgb = image.color( *( image.scanLine( sY ) + sX ));
    for( b = 0; b < image.numColors(); b++ )
    {
      if( colorTable[b] == oRgb )
        break;
    }
  
    oR = b;

    // Security check
    if( oR == nR )
      return;
  } else
  {
    nR = nC.rgb();
    oR = *((uint *) image.scanLine( sY ) + sX );
  }
  
  // Start the filling
  FillLine( image, sX, sY, oR, nR );

  itsPixmap->convertFromImage( image );  

  repaint();
  if( previewWindow )
    previewWindow->repaint();
}  

/*
 * FillLine : Fill the row startY, starting at row startX, changing oC to nC
 *  Start at (startX/startY) and go right until there's another Color than oC
 *  change the color and check if there's a pixel with color oC in the line
 *  above or below.
 */
void PixmapDraw::FillLine( QImage &image, int startX, int startY, 
                           unsigned int oC, unsigned int nC )
{
  int x = startX;

  // The destinction if we have a more than 16 bit image or merely an 8 bit
  // one is to be done overall in this function, so these blocks are just
  // copied
  if( image.depth() < 16 )
  {
    while( image.scanLine( startY )[x] == oC )
    {
      if( ! regionActive || 
          ( regionActive &&  clipboardRect.contains( QPoint( x, startY )) ))
      {
        image.scanLine( startY )[x] = nC;
    
        if( startY > 0 )
          if( image.scanLine( startY - 1 )[x] == oC )
            FillLine( image, x, startY - 1, oC, nC );
        if( startY < image.height() - 1 )
          if( image.scanLine( startY + 1 )[x] == oC )
            FillLine( image, x, startY + 1, oC, nC );
      }
      ++x;
      if( x == image.width() )
        break;
    }

    x = startX - 1;
    while( image.scanLine( startY )[x] == oC )
    {
      if( ! regionActive || 
          ( regionActive &&  clipboardRect.contains( QPoint( x, startY )) ))
      {
        image.scanLine( startY )[x] = nC;
    
        if( startY > 0 )
          if( image.scanLine( startY - 1 )[x] == oC )
            FillLine( image, x, startY - 1, oC, nC );
        if( startY < image.height() - 1 )
          if( image.scanLine( startY + 1 )[x] == oC )
            FillLine( image, x, startY + 1, oC, nC );
      }
      --x;
      if( x == -1 )
        break;
    }
  } else
  {
    while( ((uint *)image.scanLine( startY ))[x] == oC )
    {
      if( ! regionActive || 
          ( regionActive &&  clipboardRect.contains( QPoint( x, startY )) ))
      {
        ((uint *)image.scanLine( startY ))[x] = nC;
    
        if( startY > 0 )
          if( ((uint *)image.scanLine( startY - 1 ))[x] == oC )
            FillLine( image, x, startY - 1, oC, nC );
        if( startY < image.height() - 1 )
          if( ((uint *)image.scanLine( startY + 1 ))[x] == oC )
            FillLine( image, x, startY + 1, oC, nC );
      }
      ++x;
      if( x == image.width() )
        break;
    }

    x = startX - 1;
    while( ((uint *)image.scanLine( startY ))[x] == oC )
    {
      if( ! regionActive || 
          ( regionActive &&  clipboardRect.contains( QPoint( x, startY )) ))
      {
        ((uint*)image.scanLine( startY ))[x] = nC;
    
        if( startY > 0 )
          if( ((uint *)image.scanLine( startY - 1 ))[x] == oC )
            FillLine( image, x, startY - 1, oC, nC );
        if( startY < image.height() - 1 )
          if( ((uint *)image.scanLine( startY + 1 ))[x] == oC )
            FillLine( image, x, startY + 1, oC, nC );
      }
      --x;
      if( x == -1 )
        break;
    }
  }
}

void PixmapDraw::HorScrBarChanged( int val )
{
  horOff = val;
  repaint();
}

void PixmapDraw::VerScrBarChanged( int val )
{
  verOff = val;
  repaint();
}

int PixmapDraw::CalcCellSize()
{
  int actR = numRows;
  int actC = numCols;

  if( actR && actC )
  {
    int spcNeededR = actR * actCellSize;
    int spcNeededC = actC * actCellSize;

    // cell size against table size
    int h = size().height() - SCRBAR_DIMENSION - 4;
    if( spcNeededR < h )
    {
      itsVerScr->hide();
      itsVerScr->setRange( 0, 0 );
    } else
    {
      itsVerScr->show();
      itsVerScr->setRange( 0, (spcNeededR - h) / actCellSize );
    }

    int w = size().width() - SCRBAR_DIMENSION - 4;
    if( spcNeededC < w )
    {
      itsHorScr->hide();
      itsHorScr->setRange( 0, 0 );
    } else
    {
      itsHorScr->show();
      itsHorScr->setRange( 0, (spcNeededC - w) / actCellSize );
    }
  } 

  return actCellSize;
}
  
void PixmapDraw::ZoomIn()
{
  if( actCellSize == 1 )
    actCellSize = 2;

  if( actCellSize < 20 )
    actCellSize = int( actCellSize * 1.25 + 0.5);

  if( actCellSize > 20 )
    actCellSize = 20;

  CalcCellSize();
  CalcContRect();  

  emit ZoomChanged( actCellSize );

  repaint();
}

void PixmapDraw::ZoomOut()
{
  if( actCellSize > 1  )
    actCellSize = int( actCellSize * 0.75 );
  
  if( actCellSize < 1 )
    actCellSize = 1;

  CalcCellSize();
  CalcContRect();

  emit ZoomChanged( actCellSize );

  repaint();
}

void PixmapDraw::UseGrid( bool useIt )
{
  useGrid = useIt;

  repaint();
}



bool PixmapDraw::Copy()
{
  if( ! itsClipboard || itsClipboard->isNull() )
    return FALSE;

  bitBlt( itsClipboard, QPoint( 0, 0 ), itsPixmap, clipboardRect );

  return TRUE;
}

bool PixmapDraw::Cut()
{
  if( ! itsClipboard || itsClipboard->isNull() )
    return FALSE;

  bitBlt( itsClipboard, QPoint( 0, 0 ), itsPixmap, clipboardRect );


  QPainter painter;
  painter.begin( itsPixmap );
  painter.fillRect( clipboardRect, backGround );
  painter.end();

  repaint();

  if( previewWindow )
    previewWindow->repaint();
  
  return TRUE;
}

bool PixmapDraw::Paste()
{
  if( ! itsClipboard || itsClipboard->isNull() )
    return FALSE;

  mode = DrawPaste;
  setCursor( crossCursor );

  return TRUE;
}

void PixmapDraw::CalcContRect()
{  
  QRect contRect = contentsRect();
  contRectWidth  = (contRect.width() - SCRBAR_DIMENSION) / actCellSize;
  contRectHeight = (contRect.height() - SCRBAR_DIMENSION) / actCellSize;
}

void PixmapDraw::TimerEvent()
{
  QPainter painter;
  painter.begin( this );
  QPen pen;
  pen.setColor( black );
  painter.setPen( pen );

  painter.setRasterOp( NotXorROP );

  QRect rect = clipboardRect;
  rect.moveTopLeft( rect.topLeft() * actCellSize );
  rect.setSize( rect.size() * actCellSize );
      
  painter.drawRect( rect );
  selectionVisible = ! selectionVisible;

  painter.end();
}    
  
void PixmapDraw::Deselect()
{
  if( itsTimer )
    itsTimer->stop();

  if( selectionVisible )
  { 
    QPainter painter;
    painter.begin( this );
    QPen pen;
    pen.setColor( black );
    painter.setPen( pen );

    painter.setRasterOp( NotXorROP );

    if( mode == DrawPaste )
    {
      QRect rect = clipboardRect;
      rect.setSize( rect.size() * actCellSize );
      
      painter.drawRect( rect );
    }

    painter.end();
  }

  regionActive = FALSE;
}

void PixmapDraw::ClipRegion()
{
  regionActive = ! regionActive;
}  

// Code taken from qembed (C) by Troll Tech As. who grants all rights
// to everyone
void EmbedData( const QByteArray &input, QFile *output )
{
  static char hexdigits[] = "0123456789abcdef";
  QString s( 100 );
  int nbytes = input.size();
  char *p = s.data();
  for ( int i=0; i<nbytes; i++ ) {
    if ( (i%14) == 0 ) {
	    strcpy( p, "\n    " );
	    output->writeBlock( s.data(), s.length() );
	    p = s.data();
    }
    int v = (int)((uchar)input[i]);
    *p++ = '0';
    *p++ = 'x';
    *p++ = hexdigits[(v >> 4) & 15];
    *p++ = hexdigits[v & 15];
    if ( i < nbytes-1 )
	    *p++ = ',';
  }
  if ( p > s.data() ) {
    *p = '\0';
    output->writeBlock( s.data(), s.length() );
  }
}
