/* ****************************** -*-c++-*- *******************************/
/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

#ifndef TYPES_H
#define TYPES_H

#include <am_inc.h>

#include AM_IO__H

#ifndef NULL
#define NULL (void*)0
#endif

#ifdef NEED_BOOL
// Define a Boolean type because some c++ do not support the bool type, yet.
// Hopefully, this will someday disappear.
#define true 1
#define false 0
typedef int bool;
#endif

// Am_ID_Tag is used to name classes derived from Am_Wrapper.
typedef unsigned short Am_ID_Tag;

// Call Am_Error to generate a software generated error.  Hopefully, someday
// this will be replaced by exceptions.
extern void Am_Error (const char* error_string);
// Am_Error prints out "** Program Aborted"
extern void Am_Error ();

//This version can be used with a print-out string like:
//  Am_ERROR(object << " is not valid")
#define Am_ERROR(error_string)  			\
{ cerr << "** Amulet_Error: " << error_string << endl;  \
  Am_Error();						\
}


// A helper function that generates Am_ID_Tag's for classes derived from
// Am_Wrapper.  Providing a base tag will OR that base onto the returned id.
// Providing a base with an already defined subnumber will return that number.
extern Am_ID_Tag Am_Get_Unique_ID_Tag (const char* type_name,
                                       Am_ID_Tag base = 0);
extern Am_ID_Tag Am_Set_ID_Class (const char* type_name, Am_ID_Tag base);


// These helper functions strip apart the various portions of an ID tag.
// The first returns the base number of the id.
inline Am_ID_Tag Am_Type_Base (Am_ID_Tag in_id)
{
  return in_id & 0x0FFF;
}

// The second returns the class portion of the id.
inline Am_ID_Tag Am_Type_Class (Am_ID_Tag in_id)
{
  return in_id & 0xF000;
}

// The Am_Registered_Type class.  Classes derived from this type have runtime
// type ID's, and pointers to them can be registered in the name registry.

class Am_Registered_Type {
public:
  virtual Am_ID_Tag ID() const = 0;
// returns unique tag for each derived type.
};

// The Am_Wrapper type.  Classes derived from this type can be stored in an
// object slot directly.
class Am_Wrapper : public Am_Registered_Type {
 public:
  virtual void Note_Reference () = 0; // Note that wrapper is being stored
  virtual unsigned Ref_Count () = 0; // Return the reference count.
  virtual Am_Wrapper* Make_Unique () = 0; // Return a unique copy of the data.
  virtual void Release () = 0;    // Indicates data is no longer being used.
  virtual bool operator== (Am_Wrapper& test_value) = 0; // Equality test.
  virtual Am_ID_Tag ID () const = 0; // Returns unique tag
                                     // for each derived type.
  virtual void Print_Name (ostream& out); // to print out the name of
                                          // the wrapper.
};

#if defined(_MSC_VER) || __MWERKS__
typedef unsigned char* Am_Ptr;
#else
typedef void* Am_Ptr;
#endif

// A procedure type to use as a placeholder
typedef void Am_Generic_Procedure ();

/////////////////////////////////////////////////////////////////////////////
// Wrappers for methods
/////////////////////////////////////////////////////////////////////////////

// A procedure type to use as a placeholder
typedef void Am_Generic_Procedure ();

class Am_Method_Wrapper : public Am_Registered_Type {
 public:
  // constructor registers method's name in name registry.
  Am_Method_Wrapper(Am_ID_Tag* id_ptr, Am_Generic_Procedure *p,
                    const char * name = 0);
  // prints the name or the procedure pointer address
  static void Print(ostream& os, Am_Method_Wrapper* wrapper);
  Am_ID_Tag ID() const {return *ID_Ptr;}
  Am_Generic_Procedure *Call;
protected:
  Am_ID_Tag* ID_Ptr;
};  


// Am_Value_Type is an enumeration of all the distinct types that can be
// stored in a Am_Value object.
typedef Am_ID_Tag Am_Value_Type;

// Basic types.
const Am_Value_Type Am_NONE    =  0;
const Am_Value_Type Am_UNINIT  =  1;
const Am_Value_Type Am_INT     =  2;
const Am_Value_Type Am_LONG    =  3;
const Am_Value_Type Am_BOOL    =  4;
const Am_Value_Type Am_FLOAT   =  5;
const Am_Value_Type Am_DOUBLE  =  6;
const Am_Value_Type Am_CHAR    =  7;
const Am_Value_Type Am_STRING  =  8;
const Am_Value_Type Am_VOIDPTR =  9;
const Am_Value_Type Am_PROC    = 10;

// Basic supertypes.
const Am_Value_Type Am_WRAPPER = 0x1000;
const Am_Value_Type Am_METHOD  = 0x2000;

extern void Am_Print_Type (ostream& os, Am_Value_Type type);


/////////  TOP LEVEL MACRO FOR DEFINING NEW TYPES OF METHODS
/// args must be a parenthesized list of the parameter names and types.

#define Am_Define_Method_Type_Impl(Type_name)                \
  Am_ID_Tag Type_name::Type_name##_ID =                      \
      Am_Get_Unique_ID_Tag (#Type_name, Am_METHOD);          \
ostream& operator<< (ostream& os, const Type_name& method) { \
  Am_Method_Wrapper::Print(os, method.from_wrapper);         \
  return os;                                                 \
}

//This is assigned into the Call slot when there is no procedure pointer.
//If debugging, and get a null method, it will print out an error
//message.  If not debugging, and de-reference the 0, will get a
//"Illegal instruction" error at run time.  The
//Am_Null_Method_Error_Function is defined with ... so it can take any params.
typedef void Am_Any_Procedure(...);
#ifdef DEBUG
extern Am_Any_Procedure* Am_Null_Method_Error_Function;
#else
#define Am_Null_Method_Error_Function 0
#endif

#define Am_Define_Method_Type(Type_name, Return_type, Args)                  \
/* now, a typedef for the procedure type */                                  \
typedef Return_type Type_name##_Type Args;                                   \
class Type_name {                                                            \
 public:                                                                     \
  /* Variables: */                                                           \
  /*   the wrapper I was created from  */                                    \
  Am_Method_Wrapper* from_wrapper;                                           \
  /*   a pointer to the procedure of the appropriate type */                 \
  Type_name##_Type *Call;                                                    \
  /* allocate an ID for this type       called Type_name_ID*/                \
  static Am_ID_Tag Type_name##_ID;                                           \
  bool Valid () const {                                                      \
    return Call != (Type_name##_Type*)Am_Null_Method_Error_Function;         \
  }                                                                          \
  /* method to see if a procedure is my type (and valid) */                  \
  static int Test (Am_Method_Wrapper* wrapper) {                             \
     return wrapper && wrapper->ID() == Type_name::Type_name##_ID &&         \
            wrapper->Call !=                                                 \
               (Am_Generic_Procedure*)Am_Null_Method_Error_Function; }       \
  static int Test (const Am_Value& value) {                                  \
     return value.value.method_value &&                                      \
     value.type == Type_name::Type_name##_ID &&                              \
     value.value.method_value->Call !=                                       \
        (Am_Generic_Procedure*)Am_Null_Method_Error_Function; }              \
  static Am_Value_Type Type_ID ()                                            \
  { return Type_name##_ID; }                                                 \
  /* empty constructor */                                                    \
  Type_name () {                                                             \
    Call = (Type_name##_Type*)Am_Null_Method_Error_Function;                 \
    from_wrapper = 0; }                                                      \
  /* constructor of this class from a wrapper */                             \
  Type_name (Am_Method_Wrapper* wrapper) {                                   \
    from_wrapper = wrapper;                                                  \
    if (from_wrapper)                                                        \
      Call = (Type_name##_Type*)from_wrapper->Call;                          \
    else                                                                     \
      Call = (Type_name##_Type*)Am_Null_Method_Error_Function;               \
  }                                                                          \
  /* constructor of this class from a wrapper, with the                      \
     procedure pointer passed for type-checking purposes */                  \
  Type_name (Am_Method_Wrapper* wrapper,                                     \
             Type_name##_Type*) {			                     \
    from_wrapper = wrapper;                                                  \
    if (from_wrapper)                                                        \
      Call = (Type_name##_Type*)from_wrapper->Call;                          \
    else                                                                     \
      Call = (Type_name##_Type*)Am_Null_Method_Error_Function;               \
  }                                                                          \
  Type_name (const Am_Value& value) {                                        \
    from_wrapper = value;                                                    \
    if (from_wrapper) {                                                      \
      if (value.type != Type_name::Type_name##_ID)                           \
        Am_Error("** Tried to assign a " #Type_name " with a non "           \
                 #Type_name " method wrapper.");                             \
      Call = (Type_name##_Type*)from_wrapper->Call;                          \
    }                                                                        \
    else                                                                     \
      Call = (Type_name##_Type*)Am_Null_Method_Error_Function;               \
  }                                                                          \
  Type_name& operator= (Am_Method_Wrapper* wrapper) {                        \
    from_wrapper = wrapper;                                                  \
    if (from_wrapper)                                                        \
      Call = (Type_name##_Type*)from_wrapper->Call;                          \
    else                                                                     \
      Call = (Type_name##_Type*)Am_Null_Method_Error_Function;               \
    return *this;                                                            \
  }                                                                          \
  Type_name& operator= (const Am_Value& value) {                             \
    from_wrapper = value;                                                    \
    if (from_wrapper) {                                                      \
      if (value.type != Type_name::Type_name##_ID)                           \
        Am_Error("** Tried to assign a " #Type_name " with a non "           \
                 #Type_name " method wrapper.");                             \
      Call = (Type_name##_Type*)from_wrapper->Call;                          \
    }                                                                        \
    else                                                                     \
      Call = (Type_name##_Type*)Am_Null_Method_Error_Function;               \
    return *this;                                                            \
  }                                                                          \
  /* convert me into a Am_Method_Wrapper so I can be stored into a slot */   \
  operator Am_Method_Wrapper* () const {return from_wrapper;}                \
};                                                                           \
extern ostream& operator<< (ostream& os, const Type_name& method);

/////////  MACRO FOR DEFINING NEW METHODS OF A PRE_DEFINED TYPE 
/// return_type and args must be the same as used for defining Type_name

#define Am_Define_Method(Type_name, Return_type, Method, Args)              \
/* declare the procedure so can make a pointer to it */                     \
static Return_type Method##_proc Args;                                      \
/* allocate a wrapper object holding the procedure */                       \
Am_Method_Wrapper Method##_inst  =                                          \
      Am_Method_Wrapper(&Type_name::Type_name##_ID,                         \
            (Am_Generic_Procedure*)&Method##_proc,                          \
            #Method);                                                       \
/* Create a Type_name class object for the procedure, and call it Method    \
   (passing the wrapper object is sufficient for construction, but we       \
   also pass the procedure pointer to type-check its signature against      \
   the signature expected of Type_name methods). */                         \
Type_name Method (&Method##_inst, Method##_proc);	                    \
/* now the procedure header, so this can be followed by the code */         \
Return_type Method##_proc Args          
/* code goes here { ... }  */


class Am_String; // Forward reference (see below)

// This class is a union of all the distinct types that can be stored as a
// single entity.  It is used as object slots and value list items.  Certain
// Get and Set functions use the type to make it possible to set/retrieve
// the value without having to know its type explicitly.
class Am_Value {
 public:
  Am_Value_Type type;
  union {  // Storage for the value.
    Am_Wrapper*           wrapper_value;
    Am_Ptr                 voidptr_value;
    long                  long_value;
    bool                  bool_value;
    float                 float_value;
    char                  char_value;
    Am_Method_Wrapper*    method_value;
    Am_Generic_Procedure* proc_value; //like void* for procedures
  } value;

  // Casting operators to mitigate some syntax.
  operator Am_Wrapper* () const;
  operator Am_Ptr () const;
  operator int () const;
  operator long () const;
#ifndef NEED_BOOL
  operator bool () const;
#endif
  operator float () const;
  operator double () const;
  operator char () const;
  operator Am_Generic_Procedure* () const;
  operator Am_Method_Wrapper* () const;

  operator== (Am_Wrapper* test_value) const;
  operator== (Am_Ptr test_value) const;
  operator== (int test_value) const;
  operator== (long test_value) const;
#ifndef NEED_BOOL
  operator== (bool test_value) const;
#endif
  operator== (float test_value) const;
  operator== (double test_value) const;
  operator== (char test_value) const;
  operator== (const char* test_value) const;
  operator== (const Am_String& test_value) const;
  operator== (Am_Generic_Procedure* test_value) const;
  operator== (Am_Method_Wrapper* test_value) const;
  operator== (const Am_Value& test_value) const;

  operator!= (Am_Wrapper* test_value) const;
  operator!= (Am_Ptr test_value) const;
  operator!= (int test_value) const;
  operator!= (long test_value) const;
#ifndef NEED_BOOL
  operator!= (bool test_value) const;
#endif
  operator!= (float test_value) const;
  operator!= (double test_value) const;
  operator!= (char test_value) const;
  operator!= (const char* test_value) const;
  operator!= (const Am_String& test_value) const;
  operator!= (Am_Generic_Procedure* test_value) const;
  operator!= (Am_Method_Wrapper* test_value) const;
  operator!= (const Am_Value& test_value) const;

  // Returns true if slot exists and is not = 0
  bool Valid () const;

  // Resets and clears contents.
  void Set_Empty ();

  // Clears contents and makes value uninitialized.
  void Set_Uninitialized ();

  // Creation operations provided to aid initialization.
  Am_Value ()
  {
    type = Am_NONE;
  }
  Am_Value (Am_Wrapper* initial);
  Am_Value (Am_Ptr initial)
  {
    type = Am_VOIDPTR;
    value.voidptr_value = initial;
  }
  Am_Value (int initial)
  {
    type = Am_INT;
    value.long_value = initial;
  }
  Am_Value (long initial)
  {
    type = Am_LONG;
    value.long_value = initial;
  }
#if !defined(NEED_BOOL)
  Am_Value (bool initial)
  {
    type = Am_BOOL;
    value.bool_value = initial;
  }
#endif
  Am_Value (float initial)
  {
    type = Am_FLOAT;
    value.float_value = initial;
  }
  Am_Value (double initial);
  Am_Value (char initial)
  {
    type = Am_CHAR;
    value.char_value = initial;
  }
  Am_Value (const char* initial);
  Am_Value (const Am_String& initial);
  Am_Value (Am_Generic_Procedure* initial)
  {
    type = Am_PROC;
    value.proc_value = initial;
  }
  Am_Value (Am_Method_Wrapper* initial);
  Am_Value (const Am_Value& initial);

  ~Am_Value ();

  // Assignment operators have been provided to ease some syntax juggling.
  Am_Value& operator= (Am_Wrapper* in_value);
  Am_Value& operator= (Am_Ptr in_value);
  Am_Value& operator= (int in_value);
  Am_Value& operator= (long in_value);
#ifndef NEED_BOOL
  Am_Value& operator= (bool in_value);
#endif
  Am_Value& operator= (float in_value);
  Am_Value& operator= (double in_value);
  Am_Value& operator= (char in_value);
  Am_Value& operator= (const char* in_value);
  Am_Value& operator= (const Am_String& in_value);
  Am_Value& operator= (Am_Generic_Procedure* in_value);
  Am_Value& operator= (Am_Method_Wrapper* in_value);
  Am_Value& operator= (const Am_Value& in_value);
};

extern Am_Value Am_No_Value;

extern ostream& operator<< (ostream& os, const Am_Value& value);


#define Am_WRAPPER_DATA_DECL(type_name)                  \
   public:                                               \
    void Note_Reference ()                               \
    { ++refs; }                                          \
    unsigned Ref_Count ()                                \
    { return refs; }                                     \
    Am_Wrapper* Make_Unique ();                          \
    void Release ()                                      \
    { if (!--refs) delete this; }                        \
    bool operator== (Am_Wrapper& test_value);            \
    Am_ID_Tag ID () const                                \
    { return id; }                                       \
    bool Is_Unique ()                                    \
    { return refs == 1; }                                \
    bool Is_Zero ()                                      \
    { return refs == 0; }                                \
    static type_name##_Data* Narrow (Am_Wrapper* value); \
    static Am_ID_Tag type_name##_Data_ID ()              \
    { return id; }                                       \
   private:                                              \
    unsigned refs;                                       \
    static Am_ID_Tag id;                           


#define Am_WRAPPER_DATA_IMPL_NO_ID(type_name, create_args)     \
type_name##_Data* type_name##_Data::Narrow (Am_Wrapper* value) \
{                                                              \
  if (value && (value->ID () == id))                           \
    return (type_name##_Data*)value;                           \
  else                                                         \
    return NULL;                                               \
}                                                              \
bool type_name##_Data::operator== (Am_Wrapper& test_value)     \
{                                                              \
  if (id == test_value.ID ())                                  \
    return (&test_value == this) ||                            \
           (this->operator== ((type_name##_Data&)test_value)); \
  else                                                         \
    return false;                                              \
}                                                              \
Am_Wrapper* type_name##_Data::Make_Unique ()                   \
{                                                              \
  if (refs == 1)                                               \
    return this;                                               \
  else {                                                       \
    Release ();                                                \
    return new type_name##_Data create_args;                   \
  }                                                            \
}

#define Am_WRAPPER_DATA_IMPL(type_name, create_args) \
Am_WRAPPER_DATA_IMPL_NO_ID (type_name, create_args)  \
Am_ID_Tag type_name##_Data::id = Am_Get_Unique_ID_Tag (#type_name, Am_WRAPPER);

#define Am_WRAPPER_DATA_IMPL_ID(type_name, create_args, in_id) \
Am_WRAPPER_DATA_IMPL_NO_ID (type_name, create_args)            \
Am_ID_Tag type_name##_Data::id = Am_Get_Unique_ID_Tag (#type_name, in_id);

#define Am_WRAPPER_DECL(type_name)                    \
 public:                                              \
  type_name (const type_name&);                       \
  type_name (const Am_Value&);                        \
  type_name (type_name##_Data* in_data)               \
  { data = in_data; }                                 \
  ~type_name ();                                      \
  type_name& operator= (const type_name&);            \
  type_name& operator= (const Am_Value&);             \
  type_name& operator= (type_name##_Data* in_data);   \
  operator Am_Wrapper* () const;                      \
  bool Valid () const;                                \
  static type_name Narrow (Am_Wrapper*);              \
  static bool Test (const Am_Wrapper*);               \
  static bool Test (const Am_Value& in_value);        \
  static Am_Value_Type Type_ID ();                    \
 protected:                                           \
  type_name##_Data* data;

#define Am_WRAPPER_IMPL(type_name)                                  \
type_name::type_name (const type_name& prev)                        \
{                                                                   \
  data = prev.data;                                                 \
  if (data)                                                         \
    data->Note_Reference ();                                        \
}                                                                   \
type_name::type_name (const Am_Value& in_value)                     \
{                                                                   \
  data = (type_name##_Data*)in_value.value.wrapper_value;           \
  if (data) {                                                       \
    if (type_name##_Data::type_name##_Data_ID () != in_value.type)  \
      Am_Error ("** Tried to set a " #type_name " with a non "      \
                #type_name " wrapper.");                            \
    data->Note_Reference ();                                        \
  }                                                                 \
}                                                                   \
type_name::~type_name ()                                            \
{                                                                   \
  if (data) {                                                       \
    if (data->Is_Zero ())                                           \
      Am_Error ("** Tried to delete a " #type_name " twice.");                                                  \
    data->Release ();                                               \
  }                                                                 \
  data = NULL;                                                      \
}                                                                   \
type_name& type_name::operator= (const type_name& prev)             \
{                                                                   \
  if (data)                                                         \
    data->Release ();                                               \
  data = prev.data;                                                 \
  if (data)                                                         \
    data->Note_Reference ();                                        \
  return *this;                                                     \
}                                                                   \
type_name& type_name::operator= (const Am_Value& in_value)          \
{                                                                   \
  if (data)                                                         \
    data->Release ();                                               \
  data = (type_name##_Data*)in_value.value.wrapper_value;           \
  if (data) {                                                       \
    if (in_value.type != type_name##_Data::type_name##_Data_ID ())  \
      Am_Error ("** Tried to set a " #type_name " with a non "      \
                #type_name " wrapper.");                            \
    data->Note_Reference ();                                        \
  }                                                                 \
  return *this;                                                     \
}                                                                   \
type_name& type_name::operator= (type_name##_Data* in_data)         \
{                                                                   \
  if (data) data->Release ();                                       \
  data = in_data;                                                   \
  return *this;                                                     \
}                                                                   \
type_name::operator Am_Wrapper* () const                            \
{                                                                   \
  if (data)                                                         \
    data->Note_Reference ();                                        \
  return data;                                                      \
}                                                                   \
bool type_name::Valid () const                                      \
{                                                                   \
  return data != NULL;                                              \
}                                                                   \
type_name type_name::Narrow (Am_Wrapper* in_data)                   \
{                                                                   \
  if (in_data) {                                                    \
    if (type_name##_Data::type_name##_Data_ID () == in_data->ID ()) \
      return (type_name##_Data*)in_data;                            \
    else                                                            \
      Am_Error ("** Tried to set a " #type_name " with a non "      \
                #type_name " wrapper.");                            \
  }                                                                 \
  return (type_name##_Data*)NULL;                                   \
}                                                                   \
bool type_name::Test (const Am_Wrapper* in_data)                    \
{                                                                   \
  return (in_data &&                                                \
          (in_data->ID () ==                                        \
           type_name##_Data::type_name##_Data_ID ()));              \
}								    \
bool type_name::Test (const Am_Value& in_value)                     \
{                                                                   \
  return (in_value.value.wrapper_value &&                           \
          (in_value.type ==                                         \
           type_name##_Data::type_name##_Data_ID ()));              \
}								    \
Am_Value_Type type_name::Type_ID ()                                 \
{								    \
  return type_name##_Data::type_name##_Data_ID ();                  \
}
  
// A simple string class Used to store retrieve strings from Am_Values and
// similar places.  Uses wrapper reference counting for automatic deallocation.
class Am_String_Data;

class Am_String {
  Am_WRAPPER_DECL (Am_String)
 public:
  Am_String ()
  { data = NULL; }
  Am_String (const char* string, bool copy = true);

  Am_String& operator= (const char* string);

  operator const char* () const;
  operator char* ();

  operator== (const Am_String& test_string) const;
  operator== (const char* test_string) const;
#if defined(_WINDOWS)
  operator== (char* test_string) const
    { return operator== ((const char*)test_string); }
#endif
};


/* for iterators */
enum Am_Insert_Position { Am_BEFORE, Am_AFTER };
enum Am_Add_Position { Am_HEAD, Am_TAIL };

// The NULL string.
extern Am_String Am_No_String;

extern ostream& operator<< (ostream& os, const Am_String& string);

#endif
