/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

#include <am_inc.h>

#include STANDARD_SLOTS__H
#include VALUE_LIST__H
#include WIDGETS_ADVANCED__H //for Am_Compute_MG_Feedback_Object

#include WIDGETS__H
#include DEBUGGER__H
#include MISC__H
#include REGISTRY__H
#include GESTURE__H

#include UNDO_DIALOG__H

#define CLASSIFIER_FILENAME "lib/select.cl"

Am_Slot_Key R0 = Am_Register_Slot_Name ("R0");
Am_Slot_Key NEW_OBJECT_PROTOTYPE = Am_Register_Slot_Name ("NEW_OBJECT_PROTOTYPE");

Am_Object window;
Am_Object create_inter;
Am_Object create_poly_inter;
Am_Object freehand_inter;
Am_Object rfeedback;
Am_Object lfeedback;
Am_Object undo_handler;
Am_Object my_selection;
Am_Object last_created_obj;
Am_Object created_objs;
Am_Object scroller;
Am_Object my_undo_dialog;
Am_Object menu_bar;
Am_Object arrow_bm;
Am_Object circle_bm;
Am_Object rect_bm;
Am_Object line_bm;
Am_Object poly_bm;
Am_Object freehand_bm;
Am_Object amulet_icon;
Am_Object tool_panel;
Am_Object color_panel;
Am_Object cut_command;

Am_Object Freehand_Line;


Am_Object create_new_obj_internal (Am_Object prototype, Am_Inter_Location data)
{
  static int new_object_cnt = 1;
  Am_Object ref_obj;
  int a, b, c, d;
  bool create_line;
  data.Get_Location(create_line, ref_obj, a, b, c, d);
  if (ref_obj != created_objs) {
    Am_Translate_Coordinates(ref_obj, a, b, created_objs, a, b);
    if (create_line)
      Am_Translate_Coordinates(ref_obj, c, d, created_objs, c, d);
  }
  
  Am_Style color;
  const char* colorstr;
  Am_Object color_rect;
  color_rect = color_panel.Get(Am_VALUE);
  if (color_rect.Valid()) {
    color = color_rect.Get(Am_FILL_STYLE);
    colorstr = Am_Get_Name_Of_Item(color);
    colorstr = colorstr ? colorstr : "(unknown color)";
  }
  else {
    switch (new_object_cnt) {
    case 1: color = Am_Black; colorstr = "Black"; break;
    case 2: color = Am_Green; colorstr = "Green";break;
    case 3: color = Am_Blue; colorstr = "Blue";break;
    case 4: color = Am_Yellow; colorstr = "Yellow";break;
    case 5: color = Am_Purple; colorstr = "Purple";break;
    case 6: color = Am_Cyan; colorstr = "Cyan";break;
    case 7: color = Am_Orange; colorstr = "Orange";break;
    case 8: color = Am_Red; colorstr = "Red";break;
    case 9: color = Am_White; colorstr = "White";break;
    case 10: color = Am_Motif_Orange; colorstr = "Motif_Orange";break;
    case 11: color = Am_Motif_Light_Orange; colorstr ="Motif_Light_Orange";break;
    case 12: color = Am_Motif_Gray; colorstr = "Motif_Gray";break;
    case 13: color = Am_Motif_Light_Gray; colorstr = "Motif_Light_Gray";break;
    case 14: color = Am_Motif_Blue; colorstr = "Motif_Blue";break;
    case 15: color = Am_Motif_Light_Blue; colorstr = "Motif_Light_Blue";break;
    case 16: color = Am_Motif_Green; colorstr = "Motif_Green";break;
    case 17: color = Am_Motif_Light_Green; colorstr = "Motif_Light_Green";break;
    }
  }
  new_object_cnt = (new_object_cnt % 17) + 1;

  Am_Object new_obj;
  if (create_line)
    new_obj = prototype.Create ()
      .Set (Am_LINE_STYLE, color)
      .Set (Am_X1, a)
      .Set (Am_Y1, b)
      .Set (Am_X2, c)
      .Set (Am_Y2, d)
      ;
  else if (prototype.Valid())
    new_obj = prototype.Create ()
      .Set (Am_FILL_STYLE, color)
      .Set (Am_LEFT, a)
      .Set (Am_TOP, b)
      .Set (Am_WIDTH, c)
      .Set (Am_HEIGHT, d)
      ;
  else new_obj = Am_No_Object;

  if (new_obj.Valid()) {
    cout << "\n--++-- Created new object " << new_obj << " at ("
	 << a << "," << b << "," << c << "," << d << ") color="
	 << colorstr << endl << flush;
    created_objs.Add_Part (new_obj);
    last_created_obj = new_obj;
  }
  return new_obj;
}

// Am_Create_New_Object_Proc for new object command
Am_Define_Method(Am_Create_New_Object_Method, Am_Object, create_new_object,
		 (Am_Object /* inter */, Am_Inter_Location data,
		  Am_Object old_object)) {
  Am_Object new_obj;
  
  if (old_object.Valid()) {
    new_obj = old_object.Copy();
    cout << "\n--++-- Created new object " << new_obj << " as a copy of "
	 << old_object << endl << flush;
    if (new_obj.Valid()) {
      created_objs.Add_Part (new_obj);
      last_created_obj = new_obj;
    }
  }
  else {
    Am_Object proto;
    Am_Object v;
    v = tool_panel.Get(Am_VALUE);
    if (v == circle_bm) proto = Am_Arc;
    else if (v == rect_bm) proto = Am_Rectangle;
    else if (v == line_bm) proto = Am_Line;
    new_obj = create_new_obj_internal (proto, data);
  }
  
  cout << "returning " << new_obj << endl << flush;
  return new_obj;
}

Am_Define_Object_Formula (compute_feedback_obj) {
  Am_Object which_feedback;
  if ((bool)self.GV(Am_AS_LINE)) {
    cout << "&-&&- Recompute formula; using LINE feedback\n";
    which_feedback = lfeedback;
  }
  else {
    cout << "&-&&- Recompute formula; using RECT feedback\n";
    which_feedback = rfeedback;
  }
  return which_feedback;
}

Am_Define_No_Self_Formula (bool, line_tool)
{
  Am_Value v;
  tool_panel.GVM(Am_VALUE, v);
  if (v.type == Am_OBJECT && v == line_bm) return true;
  else return false;
}

Am_Define_No_Self_Formula (bool, selection_tool)
{
  Am_Value v;
  tool_panel.GVM(Am_VALUE, v);
  if (v.type == Am_OBJECT && v == arrow_bm) return true;
  else return false;
}

Am_Define_No_Self_Formula (bool, somethings_selected)
{
  if (my_selection.Valid()) {
    Am_Value v;
    v = my_selection.GV(Am_VALUE);
    return !v.Valid ();
  } else return false;
}

Am_Define_No_Self_Formula (bool, rubber_bandable_tool_is_selected)
{
  //Am_Object v;
  // v = tool_panel.GV(Am_VALUE);
  Am_Object v(tool_panel.GV(Am_VALUE));
  return (v == circle_bm || v == rect_bm || v == line_bm);
}

Am_Define_No_Self_Formula (bool, polygon_tool_is_selected)
{
  return tool_panel.GV(Am_VALUE) == poly_bm;
}

Am_Define_No_Self_Formula (bool, freehand_tool_is_selected)
{
  return tool_panel.GV(Am_VALUE) == freehand_bm;
}


Am_Object partial_poly; // polygon under construction

Am_Style Thick_Black_Line(0.0f, 0.0f, 0.0f, 10);

Am_Define_Method(Am_Mouse_Event_Method, void, polygon_start,
		 (Am_Object inter, int x, int y,
		  Am_Object event_window, Am_Input_Char ic))
{
  if (event_window != created_objs)
    Am_Translate_Coordinates (event_window, x, y, created_objs, x, y);

  // start the polygon
  partial_poly = Am_Polygon.Create ()
    .Set (Am_POINT_LIST, Am_Point_List())
    .Set (Am_FILL_STYLE, Am_No_Style)
    .Set (Am_LINE_STYLE, ic.shift ? Thick_Black_Line : Am_Black);
    ;
  created_objs.Add_Part (partial_poly);

  // enable vertex-selection interactor
//   poly_vertex_inter.Set (Am_ACTIVE, true);

  // show feedback
  Am_Object feedback(inter.Get(Am_FEEDBACK_OBJECT));
  if (feedback.Valid ()) {
    feedback.Set(Am_VISIBLE, true);
    Am_To_Top(feedback);
    // feedback position will be set by polygon_interim_do
  }
}

Am_Define_Method(Am_Mouse_Event_Method, void, polygon_interim_do,
		 (Am_Object inter, int x, int y,
		  Am_Object event_window, Am_Input_Char ic))
{
  if (event_window != created_objs)
    Am_Translate_Coordinates (event_window, x, y, created_objs, x, y);

  // move endpoint of feedback line
  Am_Object feedback;
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  if (feedback.Valid ()) {
    bool feedback_vis = feedback.Get(Am_VISIBLE);
    if (feedback_vis == false) {
      feedback.Set(Am_VISIBLE, true);
    }
    feedback.Set(Am_X2, x);
    feedback.Set(Am_Y2, y);
  }

  static Am_Input_Char vertex_event("any_left_down");
  if (ic == vertex_event) {
    // user has left-clicked!
    if (!Am_Point_In_All_Owners (partial_poly, x, y, created_objs)) {
      // clicked outside of drawing window -- throw away this point and
      // stop
      Am_Stop_Interactor (inter, Am_No_Object, Am_Input_Char(), Am_No_Object, 0, 0);
    }
    else {
      Am_Point_List pl = partial_poly.Get (Am_POINT_LIST);
      if (pl.Empty()) {
	// the click that started it all -- first point of the polygon
	partial_poly.Set (Am_POINT_LIST, pl.Add((float)x, (float)y));
	feedback.Set(Am_X1, x);
	feedback.Set(Am_Y1, y);
      }
      else {
	int first_x;
	int first_y;
	pl.Start(); pl.Get (first_x, first_y);
	
	int delta_x = x - first_x;
	int delta_y = y - first_y;
	
	if (delta_x < 5 && delta_x > -5 && delta_y < 5 && delta_y > -5) {
	  // clicked on (er, near) the initial point again -- close
	  // the polygon and stop
	  pl.Add ((float)first_x, (float)first_y, Am_TAIL, false);
	  Am_Stop_Interactor (inter, Am_No_Object, Am_Input_Char(), Am_No_Object, 0, 0);
	}
	else {
	  // add new point to polygon, reset feedback origin to it, and
	  // keep running
	  pl.Add ((float)x, (float)y, Am_TAIL, false);
	  feedback.Set(Am_X1, x);
	  feedback.Set(Am_Y1, y);
	}
	partial_poly.Note_Changed (Am_POINT_LIST);
      }
    }
  }

}

Am_Define_Method(Am_Mouse_Event_Method, void, polygon_abort,
		 (Am_Object inter, int x, int y,
		  Am_Object event_window, Am_Input_Char /* ic */))
{
  if (event_window != created_objs)
    Am_Translate_Coordinates (event_window, x, y, created_objs, x, y);

  // hide feedback
  Am_Object feedback;
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  if (feedback.Valid ()) {
    feedback.Set(Am_VISIBLE, false);
  }

  // disable vertex-selection interactor
//   poly_vertex_inter.Set (Am_ACTIVE, false);

  // destroy polygon under construction
  partial_poly.Destroy ();
}

Am_Define_Method(Am_Mouse_Event_Method, void, polygon_do,
		 (Am_Object inter, int /*x*/, int /*y*/,
		  Am_Object /*event_window*/, Am_Input_Char /* ic */))
{
  // hide feedback
  Am_Object feedback;
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  if (feedback.Valid ()) {
    feedback.Set(Am_VISIBLE, false);
  }

  // disable vertex-selection interactor
//   poly_vertex_inter.Set (Am_ACTIVE, false);

  Am_Object new_object =  partial_poly;
  Am_Style color = color_panel.Get_Object(Am_VALUE).Get(Am_FILL_STYLE);
  new_object.Set (Am_FILL_STYLE, color);
  // new_object has already been added to created_objs

  // take care of undo/redo
  inter.Set (Am_VALUE, new_object);
  inter.Set(Am_OBJECT_MODIFIED, new_object);
  extern void Am_Copy_Values_To_Command(Am_Object from_object);
  Am_Copy_Values_To_Command(inter);
}


void Undo_Redo_Selective(Am_Slot_Key allowed_slot, Am_Slot_Key method_slot,
			 const char* prompt_str) {
  Am_Object last_command;
  bool use_new = (method_slot == Am_SELECTIVE_REPEAT_ON_NEW_METHOD);
    Am_Value_List l;
    l = undo_handler.Get(Am_COMMAND);
    Am_String s;
    Am_Object cmd;
    Am_Selective_Allowed_Method allowed_method;
    Am_Selective_New_Allowed_Method new_allowed_method;
    Am_Value current_selection, obj_modified;
    if (use_new) {
      new_allowed_method = undo_handler.Get(allowed_slot);
      //get the selection
      my_selection.Get(Am_VALUE, current_selection);
    }
    else allowed_method = undo_handler.Get(allowed_slot);

    bool allowed;
    int cnt;
    for (l.Start(), cnt = 0; !l.Last(); l.Next(), cnt++) {
      cmd = l.Get();
      s = cmd.Get(Am_LABEL);
      cmd.Get(Am_OBJECT_MODIFIED, obj_modified);
      cout << cnt << " " << s << " on " << obj_modified
	   << " (cmd = " << cmd << ") ";
      if (use_new)
	allowed = new_allowed_method.Call(cmd, current_selection);
      else allowed = allowed_method.Call(cmd);
      if (allowed) cout << "OK\n";
      else cout << "NOT OK\n";
    }
    cout << "--Type index of command to " << prompt_str << " (or -1 to exit): "
	 << flush;
    int which;
    cin >> which;
    if (which < 0) return;
    for (l.Start(), cnt = 0; cnt < which; l.Next(), cnt++) ;
    cmd = l.Get();
    if (use_new) {
      Am_Handler_Selective_Repeat_New_Method method;
      method = undo_handler.Get(method_slot);
      cout << prompt_str << " on cmd " << cmd << " method = " << method
	   << " new obj = " << current_selection << endl << flush;
      method.Call(undo_handler, cmd, current_selection);
    }
    else {
      Am_Handler_Selective_Undo_Method method;
      method = undo_handler.Get(method_slot);
      cout << prompt_str << " on cmd " << cmd << " method = " << method
	   << endl << flush;
      method.Call(undo_handler, cmd);
    }
}

Am_Define_Method(Am_Object_Method, void, change_setting, (Am_Object cmd)) {
  Am_Object inter = cmd.Get_Owner();
  Am_Input_Char c = Am_Input_Char::Narrow(inter.Get(Am_START_CHAR));

  cout << "---- got " << c << endl;
  switch (c.As_Char()) {
  case 'n': { //no value
    my_selection.Set(Am_VALUE, Am_Value_List());
    Am_Value v;
    my_selection.Get(Am_VALUE, v);
    cout << "setting Am_VALUE of " << my_selection << " to " << v
	 << endl << flush;
    break;
  }
  case 'l': { // set to one rectangle object
    if (last_created_obj.Valid()) {
      my_selection.Set(Am_VALUE, Am_Value_List()
		       .Add(last_created_obj));
      Am_Value v;
      my_selection.Get(Am_VALUE, v);
      cout << "setting Am_VALUE of " << my_selection << " to " << v
	   << " so only " << last_created_obj << endl << flush;
    }
    break;
  }
  case 'e': { // everything selected
    Am_Value_List l;
    l = created_objs.Get(Am_GRAPHICAL_PARTS);
    my_selection.Set(Am_VALUE, l);
    Am_Value v;
    my_selection.Get(Am_VALUE, v);
    cout << "setting Am_VALUE of " << my_selection << " to " << v
	 << endl << flush;
    break;
  }
  case 'i': {
    static bool tracing = false;
    if (tracing) Am_Set_Inter_Trace(Am_INTER_TRACE_NONE);
    else Am_Set_Inter_Trace(Am_INTER_TRACE_ALL);
    tracing = !tracing;
    break;
  }
  case 'a': { // toggle active
    bool active = !(bool)my_selection.Get(Am_ACTIVE);
    cout << "setting Am_ACTIVE to " << active << endl << flush;
    my_selection.Set(Am_ACTIVE, active);
    break;
  }
  case 's': { // toggle queuing of selections
    Am_Object cmd = my_selection.Get_Part(Am_COMMAND);
    int queued = cmd.Get(Am_IMPLEMENTATION_PARENT);
    if (queued == Am_NOT_USUALLY_UNDONE) {
      cout << "Changing " << cmd << " so will be queued\n" << flush;
      cmd.Set(Am_IMPLEMENTATION_PARENT, 0);
    }
    else {
      cout << cmd << " parent now " << queued
	   << " changing so will be NOT be queued\n" << flush;
      cmd.Set(Am_IMPLEMENTATION_PARENT, Am_NOT_USUALLY_UNDONE);
    }
    break;
  }
  case 'm': {
    static bool using_left = true;
    if (using_left) {
      using_left = false;
      cout << "switching to middle button\n" << flush;
      my_selection.Set(Am_START_WHEN, "ANY_MIDDLE_DOWN");
    }
    else {
      using_left = true;
      cout << "switching to left button\n" << flush;
      my_selection.Set(Am_START_WHEN, "ANY_LEFT_DOWN");
    }
    break;
  }
	
  case 'L': {
    static int line_styles_cnt = 1;
    switch (line_styles_cnt) {
    case 1: scroller.Set(Am_LINE_STYLE, Am_Red); break;
    case 2: scroller.Set(Am_LINE_STYLE, Am_Line_8); break;
    case 3: scroller.Set(Am_LINE_STYLE, Am_No_Style); break;
    case 4: scroller.Set(Am_LINE_STYLE, Am_Black); break;
    }
    line_styles_cnt = (line_styles_cnt % 4) + 1;
    cout << "Changing line style of scroller" << endl << flush;
    break;
  }
    
    
  case 'c': {
    static int handles_color_cnt = 1;
    Am_Style color;
    char* colorstr;
    switch (handles_color_cnt) {
    case 1: color = Am_Red; colorstr = "Red"; break;
    case 2: color = Am_Green; colorstr = "Green";break;
    case 3: color = Am_Blue; colorstr = "Blue";break;
    case 4: color = Am_Black; colorstr = "Black";break;
    }
    handles_color_cnt = (handles_color_cnt % 4) + 1;
    cout << "Changing handles color to " << colorstr << endl << flush;
    my_selection.Set(Am_FILL_STYLE, color);
    break;
  }

  case 'u':
    {
      Am_Object last_command;
      cout << " checking undo...";
      cout << " undo handler = " << undo_handler << endl << flush;
      last_command = undo_handler.Get(Am_UNDO_ALLOWED);
      if (last_command.Valid()) {
	Am_String s;
        s = last_command.Get(Am_LABEL);
	cout << " undoing cmd " << last_command << " = " << s << endl << flush;
	Am_Object_Method undoit;
	undoit = undo_handler.Get(Am_PERFORM_UNDO);
	undoit.Call (undo_handler);
      }
      else cout << " nothing to undo\n" << flush;
    break;
    }
  case 'r':
    {
      Am_Object last_command;
      cout << " checking redo (undo the undo) ...";
      cout << " undo handler = " << undo_handler << endl << flush;
      last_command = undo_handler.Get(Am_REDO_ALLOWED);
      if (last_command.Valid()) {
	Am_String s;
        s = last_command.Get(Am_LABEL);
	cout << " re-doing cmd " << last_command << " = " << s << endl
	     << flush;
	Am_Object_Method redoit;
	redoit = undo_handler.Get(Am_PERFORM_REDO);
	redoit.Call (undo_handler);
      }
      else cout << " nothing to redo\n" << flush;
    break;
    }
  case 'x': {
    Undo_Redo_Selective(Am_SELECTIVE_UNDO_ALLOWED, Am_SELECTIVE_UNDO_METHOD,
			"Selective Undo");
    break;
  }
  case 'R': {
    Undo_Redo_Selective(Am_SELECTIVE_REPEAT_SAME_ALLOWED,
			Am_SELECTIVE_REPEAT_SAME_METHOD,
			"Selective Repeat");
    break;
  }
  case 'N': {
    Undo_Redo_Selective(Am_SELECTIVE_REPEAT_NEW_ALLOWED,
			Am_SELECTIVE_REPEAT_ON_NEW_METHOD,
			"Selective Repeat on New");
    break;
  }
  case 'd': {
    my_undo_dialog.Set(Am_VISIBLE, true);
    Am_To_Top(my_undo_dialog);
    cout << "Displayed undo dialog box\n" << flush;
    break;
  }
  case 'f': {
    Am_Object inter;
    Am_Value feedback;
    inter = my_selection.Get_Part(Am_MOVE_INTERACTOR);
    inter.Get(Am_FEEDBACK_OBJECT, feedback);
    if (feedback.Valid()) {
      cout << "Turning off feedback\n" << flush;
      inter.Set(Am_FEEDBACK_OBJECT, NULL);
    }
    else {
      cout << "Turning on feedback\n" << flush;
      inter.Set(Am_FEEDBACK_OBJECT, Am_Compute_MG_Feedback_Object);
    }
    break;
  }
  case 'q':
    Am_Exit_Main_Event_Loop ();
    break;
  default:
    cout << "** Illegal, want:\n"
	 << "   n = no selection\n"
	 << "   l = last object selected\n"
	 << "   e = everything selected\n"
	 << "   c = change color of handles\n"
	 << "   a = toggle active\n"
	 << "   m = switch to middle mouse button, and back\n"
	 << "   f = toggle whether move uses feedback object or not\n"
	 << "   s = toggle queuing of selections for undo\n"
         << "   i = inter debug\n"
	 << "   L = line style of scrolling group\n"
	 << "   Undoing:\n"
	 << "      u=undo, r=redo\n"
	 << "      x=selective undo, R=selective repeat same, N=repeat new\n"
         << "      d = display undo dialog box\n"
         << "   q = quit\n"
         << endl << flush;
    break;
  } // end switch
}
//used:  acdeilmnqrsux LNRS
	 
Am_Define_Method(Am_Object_Method, void, my_do, (Am_Object cmd)) {
  Am_Value value;
  cmd.Get(Am_VALUE, value);
  cout << "\n+-+-+- Command " << cmd  << " value = "
       << value << endl << flush;
}



////// Freehand drawing


Am_Define_Method(Am_Object_Method, void, freehand_create,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  int left = inter.Get (Am_MIN_X);
  int top = inter.Get (Am_MIN_Y);
  int width = (int)inter.Get (Am_MAX_X) - left;
  int height = (int)inter.Get (Am_MAX_Y) - top;
  Am_Inter_Location data(false, inter.Get_Owner(), left, top, width, height); 

  Am_Style color = color_panel.Get_Object(Am_VALUE).Get(Am_FILL_STYLE);
  Am_Object new_object = Freehand_Line.Create ()
    .Set (Am_POINT_LIST, inter.Get (Am_POINT_LIST))
    .Set (Am_LINE_STYLE, color)
    ;
  created_objs .Add_Part (new_object);

  // Set appropriate slots in implementation parent (to be called next).
  // This parent command is actually the New_Points_Interactor's 
  // implementation command.
  Am_Object parent_cmd;
  parent_cmd = cmd.Get(Am_IMPLEMENTATION_PARENT);
  parent_cmd
    .Set(Am_IMPLEMENTATION_PARENT, 0)
    .Set(Am_SAVED_OLD_OWNER, create_inter)
    .Set(Am_OBJECT_MODIFIED, new_object)
    .Set(Am_START_OBJECT, new_object.Copy())
    .Set(Am_VALUE, new_object)
    .Set(Am_AS_LINE, true)
    .Set(Am_INTERIM_VALUE, data)
    .Set(Am_HAS_BEEN_UNDONE, false)
    ;
}



////// Gesture stuff 

// the Do Method for an unrecognized gesture
Am_Define_Method(Am_Object_Method, void, gesture_unrecognized,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  cout << "** UNRECOGNIZED" << endl;
  cout << "(nap = " << (double)inter.Get (Am_NONAMBIGUITY_PROB) << ", dist="
    << (double)inter.Get (Am_DIST_TO_MEAN) << ")" << endl;
}

// the Do method for a LINE gesture
Am_Define_Method(Am_Object_Method, void, gesture_linelike,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  Am_Object prototype;
  prototype = cmd.Get (NEW_OBJECT_PROTOTYPE);
  int x1 = inter.Get (Am_START_X);
  int y1 = inter.Get (Am_START_Y);
  int x2 = inter.Get (Am_END_X);
  int y2 = inter.Get (Am_END_Y);
  Am_Inter_Location data(true, inter.Get_Owner(), x1, y1, x2, y2);

  Am_Object new_object;
  new_object = create_new_obj_internal (prototype, data);

  // Set appropriate slots in implementation parent (to be called next).
  // This parent command is actually the New_Points_Interactor's 
  // implementation command.
  Am_Object parent_cmd;
  parent_cmd = cmd.Get(Am_IMPLEMENTATION_PARENT);
  parent_cmd
    .Set(Am_IMPLEMENTATION_PARENT, 0)
    .Set(Am_SAVED_OLD_OWNER, create_inter)
    .Set(Am_OBJECT_MODIFIED, new_object)
    .Set(Am_START_OBJECT, new_object.Copy())
    .Set(Am_VALUE, new_object)
    .Set(Am_AS_LINE, true)
    .Set(Am_INTERIM_VALUE, data)
    .Set(Am_HAS_BEEN_UNDONE, false)
    ;
}

// the Do method for a CIRCLE or RECTANGLE gesture
Am_Define_Method(Am_Object_Method, void, gesture_boxlike,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  Am_Object prototype;
  prototype = cmd.Get (NEW_OBJECT_PROTOTYPE);
  int left = inter.Get (Am_MIN_X);
  int top = inter.Get (Am_MIN_Y);
  int width = (int)inter.Get (Am_MAX_X) - left;
  int height = (int)inter.Get (Am_MAX_Y) - top;
  Am_Inter_Location data(false, inter.Get_Owner(), left, top, width, height); 

  Am_Object new_object;
  new_object = create_new_obj_internal (prototype, data);

  // Set appropriate slots in implementation parent (to be called next).
  // This parent command is actually the New_Points_Interactor's 
  // implementation command.
  Am_Object parent_cmd;
  parent_cmd = cmd.Get(Am_IMPLEMENTATION_PARENT);
  parent_cmd
    .Set(Am_IMPLEMENTATION_PARENT, 0)
    .Set(Am_SAVED_OLD_OWNER, create_inter)
    .Set(Am_OBJECT_MODIFIED, new_object)
    .Set(Am_START_OBJECT, new_object.Copy())
    .Set(Am_VALUE, new_object)
    .Set(Am_AS_LINE, false)
    .Set(Am_INTERIM_VALUE, data)
    .Set(Am_HAS_BEEN_UNDONE, false)
    ;
}

// the Do method for a CUT gesture
Am_Define_Method(Am_Object_Method, void, gesture_cut,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  int g_minx = inter.Get (Am_MIN_X);
  int g_miny = inter.Get (Am_MIN_Y);
  int g_maxx = inter.Get (Am_MAX_X);
  int g_maxy = inter.Get (Am_MAX_Y);

  // find all the objects whose bounding boxes intersect the gesture's bounding
  // box.
  Am_Value_List objs;
  objs = created_objs.Get (Am_GRAPHICAL_PARTS);
  Am_Value_List selected_objs;

  for (objs.Start(); !objs.Last(); objs.Next()) {
    Am_Object obj;
    obj = objs.Get();
    int o_minx = obj.Get (Am_LEFT);
    int o_miny = obj.Get (Am_TOP);
    int o_maxx = o_minx + (int)obj.Get (Am_WIDTH);
    int o_maxy = o_miny + (int)obj.Get (Am_HEIGHT);
    
    if (g_maxx > o_minx && o_maxx > g_minx &&
	g_maxy > o_miny && o_maxy > g_miny)
      selected_objs.Add (obj);
  }

  if (selected_objs.Empty()) {
    // nothing to do!  don't execute Cut command, don't register for undo
    cmd.Set (Am_IMPLEMENTATION_PARENT, true);
  }
  else {
    cmd.Set (Am_IMPLEMENTATION_PARENT, cut_command);
    my_selection.Set (Am_VALUE, selected_objs);
  }
}


// obsolete: the Do method for a DOT gesture, when DOT meant "toggle selection"
// (now it means "undo")
Am_Define_Method(Am_Object_Method, void, gesture_dot,
		 (Am_Object cmd))
{
  Am_Object inter;
  inter = cmd.Get (Am_SAVED_OLD_OWNER);
  int x = inter.Get (Am_START_X);
  int y = inter.Get (Am_START_Y);
  Am_Object obj;

  obj = Am_Point_In_Part (created_objs, x, y, created_objs);
  if (obj.Valid()) {
    Am_Value_List list;
    list = my_selection.Get (Am_VALUE);

    // toggle selectedness of object
    list.Start();
    if(list.Member(obj))
      list.Delete();
    else
      list.Add(obj);

    my_selection.Set (Am_VALUE, list);

    Am_Object parent_command;
    parent_command = cmd.Get(Am_IMPLEMENTATION_PARENT);
    if (parent_command.Valid()) {
      Am_Value old_value;
      parent_command.Get(Am_VALUE, old_value);
      parent_command.Set(Am_OLD_VALUE, old_value);
      parent_command.Set(Am_VALUE, list);
      parent_command.Set(Am_OBJECT_MODIFIED, obj);
    }
  }
}


Am_Define_Formula(int, scroll_height) {
  return (int)self.GV_Owner().GV(Am_HEIGHT) - 10 - (int)self.GV(Am_TOP);
}
Am_Define_Formula(int, scroll_width) {
  return (int)self.GV_Owner().GV(Am_WIDTH) - 10 - (int)self.GV(Am_LEFT);
}

#if defined(_WINDOWS) || defined(_MACINTOSH)

#define AMULET_BITMAP "lib/amside.gif"
#define ARROW_BITMAP "lib/arrow.gif"
#else
#define AMULET_BITMAP "lib/amuletside.xbm"
#define ARROW_BITMAP "lib/arrow.xbm"			 
#endif

Am_Image_Array icon, arrow;

void load_bitmaps() {
  char* pathname = Am_Merge_Pathname(AMULET_BITMAP);
  icon = Am_Image_Array(pathname);
  delete pathname;
  amulet_icon = Am_Bitmap.Create("Amulet_Icon")
    .Set (Am_IMAGE, icon)
    .Set (Am_LINE_STYLE, Am_Black)
    ;									   
  circle_bm = Am_Arc.Create("Rect not bitmap")
    .Set (Am_LEFT, 0)
	.Set (Am_TOP, 0)
	.Set (Am_WIDTH, 16)
	.Set (Am_HEIGHT, 16)
	.Set (Am_LINE_STYLE, Am_Black)
	.Set (Am_FILL_STYLE, Am_Blue)
	;

  pathname = Am_Merge_Pathname(ARROW_BITMAP);
  arrow = Am_Image_Array(pathname);
  delete pathname;
  if (!arrow.Valid()) Am_Error ("Arrow bitmap image not found");
  arrow_bm = Am_Bitmap.Create("Arrow bitmap")
    .Set (Am_IMAGE, arrow)
    .Set (Am_LINE_STYLE, Am_Black)
    ;

  line_bm = Am_Line.Create("Line, not bitmap")
	.Set (Am_LEFT, 0)
	.Set (Am_TOP, 0)   
	.Set (Am_WIDTH, 16)
	.Set (Am_HEIGHT, 8)
	;

  rect_bm = Am_Rectangle.Create("Rect not bitmap")
    .Set (Am_LEFT, 0)
	.Set (Am_TOP, 0)
	.Set (Am_WIDTH, 12)
	.Set (Am_HEIGHT, 8)
	.Set (Am_LINE_STYLE, Am_Black)
	.Set (Am_FILL_STYLE, Am_Red)
	;						   

  poly_bm = Am_Polygon.Create("Polygon not bitmap")
    .Set (Am_POINT_LIST, Am_Point_List()
	  // an irregular closed polygon
	  .Add(10.0f,0.0f).Add(0.0f,30.0f).Add(30.0f,40.0f).Add(30.0f,20.0f).Add(15.0f,20.0f).Add(10.0f,0.0f))
	.Set (Am_WIDTH, 16)
	.Set (Am_HEIGHT, 16)
	.Set (Am_LEFT, 0)
	.Set (Am_TOP, 0)
	.Set (Am_LINE_STYLE, Am_Black)
	.Set (Am_FILL_STYLE, Am_Green)
	;

  // (I sketched a curly-que freehand, then copied the points below.) 
  static int ar[] = { 133,127,  122,159,  121,195,  126,211,  
		      140,229,  156,239,  190,243,  208,220,
		      200,182,  196,179,  194,179,  168,186,
		      144,201,  135,217,  132,235,  134,255,
		      143,267,  195,293,  219,305,  249,306 };
  freehand_bm = Am_Polygon.Create("Freehand not bitmap")
    .Set (Am_POINT_LIST, Am_Point_List(ar, sizeof(ar)/sizeof(int)))
	.Set (Am_WIDTH, 16)
	.Set (Am_HEIGHT, 16)
	.Set (Am_LEFT, 0)
	.Set (Am_TOP, 0)
	.Set (Am_FILL_STYLE, Am_No_Style)
	.Set (Am_LINE_STYLE, Am_Black)
	;
}

Am_Define_Method(Am_Object_Method, void, make_no_selection,
		 (Am_Object /*cmd*/)) {
  my_selection.Set(Am_VALUE, NULL);
}


Am_Define_Method(Am_Get_Object_Property_Value_Method, void, get_obj_color,
	 (Am_Object /* command_obj */, Am_Object obj, Am_Value &old_value)) {
  if (obj.Is_Instance_Of(Am_Line)) obj.Get(Am_LINE_STYLE, old_value);
  else obj.Get(Am_FILL_STYLE, old_value);
}
Am_Define_Method(Am_Set_Object_Property_Value_Method, void, set_obj_color,
	 (Am_Object /* command_obj */, Am_Object obj, Am_Value new_value)) {
  if (obj.Is_Instance_Of(Am_Line) || obj.Is_Instance_Of(Freehand_Line)) 
    obj.Set(Am_LINE_STYLE, new_value);
  else obj.Set(Am_FILL_STYLE, new_value);
}

void main ()
{
  Am_Object cmd;
  
  Am_Initialize ();

  load_bitmaps();
  
  undo_handler = Am_Multiple_Undo_Object.Create("my_multi_undo");
  
  window = Am_Window.Create ("window")
    .Set (Am_LEFT, 20)
    .Set (Am_TOP,  45)
    .Set (Am_WIDTH, 500)
    .Set (Am_TITLE, "Amulet Test Selection Widget")
    .Set (Am_HEIGHT, 500)
    .Set (Am_UNDO_HANDLER, undo_handler)
    .Set (Am_FILL_STYLE, Am_Amulet_Purple)
    //always exit when destroy the main window
    .Set (Am_DESTROY_WINDOW_METHOD, Am_Window_Destroy_And_Exit_Method)
    ;




  scroller = Am_Scrolling_Group.Create("scroller")
    .Set (Am_LEFT, 55)
    .Set (Am_TOP, 40)
    .Set (Am_INNER_WIDTH, 1000)  
    .Set (Am_INNER_HEIGHT, 1000)
    .Set (Am_INNER_FILL_STYLE, Am_White)
    .Set (Am_WIDTH, scroll_width)
    .Set (Am_HEIGHT, scroll_height)
    ;
  created_objs = Am_Group.Create("created_objs")
    .Set (Am_LEFT, 0)
    .Set (Am_TOP, 0)
    .Set (Am_WIDTH, 1000)  
    .Set (Am_HEIGHT, 1000)
    ;

  window.Add_Part(scroller);
  scroller.Add_Part(created_objs);
  
  tool_panel = Am_Button_Panel.Create ("tool panel")
    .Set (Am_LEFT, 10)
    .Set (Am_TOP, 40)
    .Set (Am_FIXED_HEIGHT, true)
    .Set (Am_FINAL_FEEDBACK_WANTED, true)
    .Set (Am_V_SPACING, -4)
    .Set (Am_ITEMS, Am_Value_List()
	  .Add(arrow_bm)
	  .Add(line_bm)
	  .Add(rect_bm)
	  .Add(circle_bm)
	  .Add(poly_bm)
	  .Add(freehand_bm)
	  )
    .Set(Am_VALUE, arrow_bm)
    ;
  tool_panel.Get_Part(Am_COMMAND)
    .Set (Am_DO_METHOD, make_no_selection)
    .Set (Am_IMPLEMENTATION_PARENT, true) //don't queue for UNDO
    ;
  
  window.Add_Part(tool_panel);

  Am_Object color_proto = Am_Rectangle.Create("color proto")
    .Set(Am_WIDTH, 16)
    .Set(Am_HEIGHT, 16)
    .Set(Am_LINE_STYLE, Am_No_Style)
    .Set(Am_FILL_STYLE, Am_Black)
    ;

  rfeedback = Am_Rectangle.Create ("rfeedback")
    .Set (Am_FILL_STYLE, 0)
    .Set (Am_LINE_STYLE, Am_Dotted_Line)
    .Set (Am_VISIBLE, 0) 
    ;
  lfeedback = Am_Line.Create ("lfeedback")
    .Set (Am_LINE_STYLE, // thick dotted line 
	   Am_Style (0.0f, 0.0f, 0.0f, 6, Am_CAP_BUTT, Am_JOIN_MITER,
   		     Am_LINE_ON_OFF_DASH))
    .Set (Am_VISIBLE, 0)
    ;
  created_objs.Add_Part(lfeedback)
    .Add_Part(rfeedback)
    ;
  
  create_inter = Am_New_Points_Interactor.Create("create_objects")
//    .Set(Am_START_WHEN, "ANY_RIGHT_DOWN")
    .Set(Am_AS_LINE, line_tool)
    .Set(Am_FEEDBACK_OBJECT, compute_feedback_obj)
    .Set(Am_CREATE_NEW_OBJECT_METHOD, create_new_object)
    .Set(Am_ACTIVE, rubber_bandable_tool_is_selected)
    ;
  created_objs.Add_Part (create_inter);

  create_poly_inter = Am_New_Points_Interactor.Create("create_polygons")
    .Set(Am_START_WHEN, "ANY_SINGLE_LEFT_DOWN")
    .Set(Am_STOP_WHEN, "ANY_DOUBLE_LEFT_DOWN")
    .Set(Am_AS_LINE, true)
    .Set(Am_FEEDBACK_OBJECT, lfeedback)
    .Set(Am_START_DO_METHOD, polygon_start)
    .Set(Am_INTERIM_DO_METHOD, polygon_interim_do)
    .Set(Am_DO_METHOD, polygon_do)
    .Set(Am_ABORT_DO_METHOD, polygon_abort)
    .Set(Am_CREATE_NEW_OBJECT_METHOD, create_new_object)
    .Set(Am_ACTIVE, polygon_tool_is_selected);
  created_objs
    .Add_Part (create_poly_inter);


  Am_Object change_settings_inter =
    Am_One_Shot_Interactor.Create("change_settings")
    .Set(Am_START_WHEN, "ANY_KEYBOARD")
    .Set(Am_PRIORITY, 0.9)  // so it runs after the menu accelerators
    ;
  change_settings_inter.Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, change_setting)
    .Set(Am_IMPLEMENTATION_PARENT, true) //not undo-able
    .Set_Name("change_settings_command")
    ;
  window.Add_Part (change_settings_inter);
  
  my_selection = Am_Selection_Widget.Create("my_selection")
    .Set(Am_OPERATES_ON, created_objs)
    .Set(Am_ACTIVE, selection_tool)
    ;
  my_selection.Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, my_do);
  my_selection.Get_Part(Am_MOVE_GROW_COMMAND)
    .Set(Am_DO_METHOD, my_do);
  
  scroller.Add_Part(my_selection);

  Am_Object black_color_item;
  color_panel = Am_Button_Panel.Create ("color_panel")
    .Set (Am_LEFT, 10)
    .Set (Am_TOP, 250)
    .Set (Am_V_SPACING, -4)
    .Set (Am_FINAL_FEEDBACK_WANTED, true)
    .Set (Am_ITEMS, Am_Value_List()
    	  .Add(black_color_item =
	       color_proto.Create().Set(Am_FILL_STYLE, Am_Black))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_White))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_Red))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_Orange))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_Yellow))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_Green))
	  .Add(color_proto.Create().Set(Am_FILL_STYLE, Am_Blue))
	  )
    .Add_Part(Am_COMMAND,
	      Am_Graphics_Set_Property_Command.Create("Change_Color")
	      .Set(Am_LABEL, "Change color")
	      .Set(Am_GET_OBJECT_VALUE_METHOD, get_obj_color)
	      .Set(Am_SET_OBJECT_VALUE_METHOD, set_obj_color)
	      .Set(Am_SLOT_FOR_VALUE, (int)Am_FILL_STYLE)
	      .Set(Am_SELECTION_WIDGET, my_selection)
	      )
    ;
  color_panel.Set(Am_VALUE, black_color_item);

  window.Add_Part(color_panel);

  Am_Initialize_Undo_Dialog_Box();
  my_undo_dialog = Am_Undo_Dialog_Box.Create("My_Undo_Dialog")
    .Set(Am_LEFT, 550)
    .Set(Am_TOP, 200)
    .Set(Am_UNDO_HANDLER_TO_DISPLAY, undo_handler)
    .Set(Am_SELECTION_WIDGET, my_selection)
    .Set(Am_SCROLLING_GROUP, scroller)
    .Set(Am_VISIBLE, false)
    ;
  Am_Screen.Add_Part(my_undo_dialog);
  
  menu_bar = Am_Menu_Bar.Create("menu_bar")
      .Set(Am_ITEMS, Am_Value_List ()
	 .Add (Am_Command.Create("Amulet_Command")
	       .Set(Am_IMPLEMENTATION_PARENT, true) //not undoable
	       .Set(Am_DO_METHOD, my_do)
	       .Set(Am_LABEL, amulet_icon)
	       .Set(Am_ITEMS, Am_Value_List ()
                     .Add ("About Testselectionwidget...")
                     .Add ("About Amulet...")
		    )
	       .Set(Am_ACTIVE, false)
	       )
	 .Add (Am_Command.Create("File_Command")
	       .Set(Am_IMPLEMENTATION_PARENT, true) //not undoable
	       .Set(Am_LABEL, "File")
	       .Set(Am_DO_METHOD, my_do)
	       .Set(Am_ITEMS, Am_Value_List ()
                     .Add (Am_Command.Create("Open...")
			   .Set(Am_LABEL, "Open...")
			   .Set(Am_ACTIVE, false))
                     .Add (Am_Command.Create("Save As...")
			   .Set(Am_LABEL, "Save As...")
			   .Set(Am_ACTIVE, false))
                     .Add (Am_Command.Create("Save_Command")
                           .Set(Am_LABEL, "Save")
			   .Set(Am_ACTIVE, false)
			   )
                     .Add (Am_Quit_No_Ask_Command.Create())
		     )
	       )
	   .Add (Am_Command.Create("Edit_Command")
		 .Set(Am_LABEL, "Edit")
		 .Set(Am_DO_METHOD, my_do)
		 .Set(Am_IMPLEMENTATION_PARENT, true) //not undoable
		 .Set(Am_ITEMS, Am_Value_List ()
		      .Add (Am_Undo_Command.Create())
		      .Add (Am_Redo_Command.Create())
		      .Add (Am_Show_Undo_Dialog_Box_Command.Create()
			    .Set(Am_UNDO_DIALOG_BOX, my_undo_dialog))
		      .Add (Am_Menu_Line_Command.Create())
		      .Add (cut_command = Am_Graphics_Cut_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_Copy_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_Paste_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_Clear_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_Clear_All_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Menu_Line_Command.Create())
 		      .Add (Am_Graphics_Duplicate_Command.Create()
 			    .Set(Am_SELECTION_WIDGET, my_selection)
 			    )
		      .Add (Am_Selection_Widget_Select_All_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      )
		 )
	   .Add (Am_Command.Create("Arrange_Command")
		 .Set(Am_LABEL, "Arrange")
		 .Set(Am_DO_METHOD, my_do)
		 .Set(Am_IMPLEMENTATION_PARENT, true) //not undoable
		 .Set(Am_ITEMS, Am_Value_List ()
		      .Add (Am_Graphics_To_Top_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_To_Bottom_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Menu_Line_Command.Create())
		      .Add (Am_Graphics_Group_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      .Add (Am_Graphics_Ungroup_Command.Create()
			    .Set(Am_SELECTION_WIDGET, my_selection)
			    )
		      )
		 )	   
	   )
  ;
  window.Add_Part(menu_bar);

  Am_Object gesture_feedback = Am_Polygon.Create("gesture feedback")
     .Set (Am_FILL_STYLE, Am_No_Style);
  scroller.Add_Part (gesture_feedback);
  

  // Freehand drawing tool
  Freehand_Line = Am_Polygon.Create ("Freehand")
    .Set (Am_FILL_STYLE, Am_No_Style);

  Am_Object new_points_impl_cmd = 
    Am_New_Points_Interactor.Get_Part(Am_IMPLEMENTATION_COMMAND);
  Am_Object freehand_inter = Am_Gesture_Interactor.Create("gesture")
    .Set (Am_START_WHEN, "any_left_down")
    .Set (Am_FEEDBACK_OBJECT, gesture_feedback)
    .Set (Am_ACTIVE, freehand_tool_is_selected)
    ;
  freehand_inter.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, freehand_create)
    .Set (Am_IMPLEMENTATION_PARENT, new_points_impl_cmd);
  created_objs. Add_Part(freehand_inter);

  char* pathname = Am_Merge_Pathname(CLASSIFIER_FILENAME);
  Am_Gesture_Classifier gc(pathname);
  delete pathname;
  if (!gc.Valid())
    Am_Error ("gesture classifier not found");
  Am_Object gesture_reader = Am_Gesture_Interactor.Create("gesture");
  gesture_reader
    .Set (Am_START_WHEN, "any_right_down")
    .Set (Am_FEEDBACK_OBJECT, gesture_feedback)
    .Set (Am_CLASSIFIER, gc)
    //.Set (Am_MIN_NONAMBIGUITY_PROB, 0.95)
    .Set (Am_MAX_DIST_TO_MEAN, 200)
    .Set (Am_ITEMS, Am_Value_List ()
	  .Add (Am_Command.Create()
		.Set (Am_LABEL, ":LINE")
		.Set (NEW_OBJECT_PROTOTYPE, Am_Line)
		.Set (Am_DO_METHOD, gesture_linelike)
		.Set (Am_IMPLEMENTATION_PARENT, new_points_impl_cmd))
	  .Add (Am_Command.Create()
		.Set (Am_LABEL, ":CIRCLE")
		.Set (NEW_OBJECT_PROTOTYPE, Am_Arc)
		.Set (Am_DO_METHOD, gesture_boxlike)
		.Set (Am_IMPLEMENTATION_PARENT, new_points_impl_cmd))
	  .Add (Am_Command.Create()
		.Set (Am_LABEL, ":RECTANGLE")
		.Set (NEW_OBJECT_PROTOTYPE, Am_Rectangle)
		.Set (Am_DO_METHOD, gesture_boxlike)
		.Set (Am_IMPLEMENTATION_PARENT, new_points_impl_cmd))
	  .Add (Am_Command.Create ()
		.Set (Am_LABEL, ":CUT")
		.Set (Am_DO_METHOD, gesture_cut))
	  .Add (Am_Undo_Command.Create()
		.Set (Am_LABEL, ":DOT"))
	  )
    ;
  gesture_reader.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, gesture_unrecognized)
    .Set (Am_IMPLEMENTATION_PARENT, true)  // inhibit undo
    ;
  
  created_objs.Add_Part (gesture_reader);

  Am_Screen.Add_Part(window);

  cout << "Ready\n" << flush;
  Am_Main_Event_Loop ();

  Am_Cleanup ();
  
}
