/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-2000 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <typeinfo>
#include <string>
#include <algorithm>

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/dialog.hh>
#include <toad/pushbutton.hh>
#include <toad/listbox.hh>
#include <toad/combobox.hh>
#include <toad/textfield.hh>
#include <toad/checkbox.hh>
#include <toad/filedialog.hh>
#include <toad/lba/StaticCString.hh>
#include <toad/lba/STLVectorCString.hh>

#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>

//!TFileDialog
//. A dialog for file selections.
//. <P>
//. (Will be modified soon.)

class TLBA_FileType: public TLBAdapter
{
		vector<TFileDialog::TFileType> &vec;
	public:
		TLBA_FileType(vector<TFileDialog::TFileType> &v):vec(v){}
		virtual ~TLBA_FileType(){}
		unsigned ItemHeight();
		unsigned ItemCount();
		void PrintItem(int x,int y,unsigned item,TPen &);
};

static TLBAdapter* NewListBoxAdapter(vector<TFileDialog::TFileType> &v)
{
	return new TLBA_FileType(v);
}

// constructor
//---------------------------------------------------------------------------
TFileDialog::TFileDialog(TWindow *p,const string &t)
	:TDialog(p,t)
{
	result = IDABORT;
	filter = "";

	SetSize(442,315);

	TTextField *tf;
	tf = new TTextField(this, "filename", &filename);
		tf->SetDirect(true);
		tf->SetShape(8,24,329,21);
		OLD_CONNECT(this, btnOk,  tf, tf->sigActivate);

	lb_dir=new TListBox(this,"dirbox", &directory);
		OLD_CONNECT(this,dirSelect, lb_dir,lb_dir->sigSelect);
		lb_dir->SetShape(8,68,161,197);

	lb_file=new TListBox(this,"filebox", &file);
		OLD_CONNECT(this,fileSelect, lb_file,lb_file->sigSelect);
		OLD_CONNECT(this,btnOk, lb_file,lb_file->sigDoubleClick);
		lb_file->SetShape(176,68,161,197);
	
	cb=new TComboBox(this, "filetype", NewListBoxAdapter(filetype));
		cb->SetSelection(0);
		cb->SetShape(8,288,329,21);
		OLD_CONNECT(this, actFileType, cb,cb->sigSelect);

	TPushButton *btn;
	btn = new TPushButton(this, "OK", IDOK);
		OLD_CONNECT(this,btnOk, btn,btn->sigActivate);
		btn->SetShape(348,8,85,21);
	btn = new TPushButton(this, "Abort", IDABORT);
		OLD_CONNECT(this,btnAbort, btn,btn->sigActivate);
		btn->SetShape(348,32,85,21);
	btn = new TPushButton(this, "Reload", 102);
		btn->SetShape(348,68,85,21);
		btn->SetEnabled(false);

	bShowHiddenFiles=false;
	TCheckBox *chkb;
	chkb = new TCheckBox(this, "show hidden files");
		OLD_CONNECT(this,hidden, chkb,chkb->sigValueChanged);
		chkb->SetShape(348,104,85,29);
		chkb->SetValue(bShowHiddenFiles);

	if (filetype.size()>0)
		filter = filetype[0].ext;
	
	char buffer[4097];
	getcwd(buffer, 4096);
	filename = buffer;
	filename += "/"+filter;
	
	LoadDir();
}

void TFileDialog::create()
{
	if (filetype.empty()) {
		cb->bExplicitCreate = true;
		SetSize(Width(), Height()-28-16);
#if 0
		cb->bFocusTraversal = false;
		cb->bNoFocus = true;
#endif
	}
}

TFileDialog::~TFileDialog()
{
	ClrDir();
}

// paint
//---------------------------------------------------------------------------
void TFileDialog::paint()
{
	TPen pen(this);
	pen.DrawString(8+8,24-16, "Selection");
	pen.DrawString(8+8,68-16,"Directories");
	pen.DrawString(176+8,68-16,"Files");
	if (!filetype.empty())
		pen.DrawString(8+8,288-16,"Filetype");
}

unsigned TFileDialog::GetResult() const
{
	return result;
}

const string& TFileDialog::GetFilename() const
{
	return filename;
}

// filename selected in listbox
//---------------------------------------------------------------------------
// file was selected
void TFileDialog::fileSelect(TListBox* lb)
{
	unsigned p = filename.rfind("/");
	filename = p!=string::npos ? filename.substr(0,p) : "";
	filename+="/"+file[lb->GetFirstSelectedItem()]; 
	TDataManipulator::Assimilate(&filename);
}

// directory selected in listbox
//---------------------------------------------------------------------------
void TFileDialog::dirSelect(TListBox* lb)
{
	string where = directory[lb->GetFirstSelectedItem()];
	if (where==".")
		return;
	
	unsigned p = filename.rfind("/");
	if (p!=string::npos) {
		filename=filename.substr(0,p);
	} else {
		filename="";
	}
	if (where=="..") {
		p = filename.rfind("/");
		filename = p!=string::npos ? filename.substr(0,p) : "";
	} else {
		filename+="/"+where;
	}

	filename+="/"+filter;

	LoadDir();

	lb_file->SetTopItemPosition(0);
	lb_file->AdapterChanged();

	lb_dir->SetTopItemPosition(0);
	lb_dir->AdapterChanged();
	TDataManipulator::Assimilate(&filename);
}

// file type selected in combobox
//---------------------------------------------------------------------------
void TFileDialog::actFileType(TComboBox *cb)
{
	// copy new file extension to 'filter'
	//-------------------------------------
	filter = filetype[cb->GetFirstSelectedItem()].ext;
	
	// update textfield
	//------------------
	unsigned wild_pos = min( filename.find("*"), filename.find("?") );

	unsigned slash_pos = filename.rfind("/", wild_pos);
	if (slash_pos==string::npos) {
		filename="/"+filename;
		slash_pos = 0;
	}
	
	string path = filename.substr(0,slash_pos+1);
	filename = filename.substr(slash_pos+1);	
	
	unsigned n = filename.rfind(".");
	if (n!=string::npos) {
		if (filter.size()>=1)
			filename = filename.substr(0,n) + filter.substr(1);
	} else {
		filename = filter;
	}

	filename = path + filename;
	
	TDataManipulator::Assimilate(&filename);	// update 
	
	// update file and directory window
	//----------------------------------
	LoadDir();

	lb_file->SetTopItemPosition(0);
	lb_file->AdapterChanged();
	lb_dir->SetTopItemPosition(0);
	lb_dir->AdapterChanged();
}


void *TFileDialog::GetXtra() const
{
	unsigned p=cb->GetFirstSelectedItem();
	if (p==(unsigned)-1) {
		#ifdef DEBUG
		printf("void *TFileDialog::GetXtra(): no selection, returning NULL\n");
		#endif
		return NULL;
	}
	return filetype[p].xtra;
}

/*---------------------------------------------------------------------------*
 | hidden                                                                    |
 *---------------------------------------------------------------------------*/
void TFileDialog::hidden(TCheckBox* cb)
{
	bShowHiddenFiles = cb->Value();
	LoadDir();

	lb_file->SetTopItemPosition(0);
	lb_file->AdapterChanged();

	lb_dir->SetTopItemPosition(0);
	lb_dir->AdapterChanged();
} 
 
/*---------------------------------------------------------------------------*
 | destroy                                                                   |
 *---------------------------------------------------------------------------*/
void TFileDialog::destroy()
{
	lb_file = lb_dir = NULL;
	ENTRYEXIT("TFileDialog::destroy()");
	ClrDir();
	chdir(currentdir.c_str());
}

/*---------------------------------------------------------------------------*
 | btnOk                                                                     |
 *---------------------------------------------------------------------------*/
void TFileDialog::btnOk()
{
	filter = filename;
	if (filter.find("*")!=(unsigned)-1 || filter.find("?")!=(unsigned)-1)	{
		LoadDir();
		lb_file->SetTopItemPosition(0);
		lb_file->AdapterChanged();
		lb_dir->SetTopItemPosition(0);
		lb_dir->AdapterChanged();
	}	else {
		if (filter.size()==0)
			result = IDABORT;
		else
			result = IDOK;
		EndDialog(this);
	}
}

/*---------------------------------------------------------------------------*
 | btnAbort                                                                  |
 *---------------------------------------------------------------------------*/
void TFileDialog::btnAbort()
{
	result = IDABORT;
	EndDialog(this);
}

 
/*---------------------------------------------------------------------------*
 | LoadDir                                                                   |
 *---------------------------------------------------------------------------*/
struct TComp
{
	bool operator()(const char* a, const char* b)	const{
		return strcasecmp(a,b)<0;
	}
	bool operator()(const string& a, const string& b)	const{
		return strcasecmp(a.c_str(), b.c_str())<0;
	}
};
static TComp comp;

void TFileDialog::LoadDir()
{
	// clear file and directory buffer
	//---------------------------------
	ClrDir();
	
	// read current directory
	//------------------------
	unsigned p = filename.rfind("/");
	string cwd = p!=string::npos ? filename.substr(0,p) : "";
	cwd+="/";

	dirent *de;
	

//cout << "loading directory: \"" << cwd << "\"\n";

	DIR *dd = opendir(cwd.c_str());
	if (dd==NULL)
		return;

	while( (de=readdir(dd))!=NULL )	{
		string filename = cwd + de->d_name;
//cout << "   \"" << name << "\"\n";
		struct stat st;
		stat(filename.c_str(), &st);
		bool bShow = true;

		// check if hidden file
		//----------------------
		if (!bShowHiddenFiles && de->d_name[0]=='.') {
			bShow = false;
			if (de->d_name[1]==0) {
				bShow=true;
			} else {
				if (de->d_name[1]=='.' && de->d_name[2]==0)
					bShow=true;
			}
		}
		if (bShow) {
			if (S_ISDIR(st.st_mode)) {
				directory.push_back(de->d_name);
			} else {
				if (pFilterCompare(de->d_name))
					file.push_back(de->d_name);
			}
		}
	}
	closedir(dd);
	
	// sort file and directory entrys
	//--------------------------------
	sort(file.begin(),file.end(), comp);
	sort(directory.begin(),directory.end(),comp);

	// directory without read permission
	//----------------------------------
	if (directory.size()==0) {
		directory.push_back("..");
	}
}	

// compare filename 'str' with 'filter'
//--------------------------------------
bool TFileDialog::pFilterCompare(const char *str)
{
	const char *flt = filter.c_str();
	bool wild = false;
	int fp=0,sp=0, f,s;

	moonchild:
	while(flt[fp]=='*')	{
		fp++;
		wild = true;
	}
	if (flt[fp]==0)
		return true;

	while(true)	{
		f=fp; s=sp;
		while(flt[f]==str[s] || flt[f]=='?') {
			if (flt[f]==0 || str[s]==0)
				return (flt[f]==str[s]);
			s++;
			f++;
		}
		if (flt[f]=='*') {
			fp=f;
			sp=s;
			goto moonchild;
		}
		if (!wild)
			return false;
		sp++;
		if (str[sp]==0)
			return false;
	}
	return false;
}

void TFileDialog::ClrDir()
{
	file.erase(file.begin(), file.end());
	directory.erase(directory.begin(), directory.end());
}

void TFileDialog::SetFileType(unsigned n) {
	cb->SetSelection(n);
}

void TFileDialog::SetFilename(const string& s)
{
	filename = s;
	TDataManipulator::Assimilate(&filename);
	TFTList::iterator p=filetype.begin(), e=filetype.end();
	unsigned i = 0;
	while(p!=e) {
		int n = s.size()-(*p).ext.size()+1;
		if ( n>=0 && (*p).ext.size()>0 &&	s.substr(n) == (*p).ext.substr(1) ) {
			SetFileType(i);
			return;
		}
		p++;
		i++;
	}
	SetFileType(0);
}

void TFileDialog::AddFileType(const string &name,const string &ext,void* xtra)
{
	filetype.push_back(TFileType(name,ext,xtra));
	const string &s = filename;
	unsigned i = 0;
	TFTList::iterator p=filetype.begin(), e=filetype.end();
	while(p!=e) {
		int n = s.size()-(*p).ext.size()+1;
		if ( n>=0 && (*p).ext.size()>0 &&	s.substr(n) == (*p).ext.substr(1) ) 
		{
			SetFileType(i);
			return;
		}
		p++;
		i++;
	}
	SetFileType(0);
}


// ListBoxAdapter for TFileType vector
//---------------------------------------------------------------------------

void TLBA_FileType::PrintItem(int x,int y,unsigned item,TPen &pen)
{
	if (item>=vec.size()) return;
	string s = vec[item].name;
	if (vec[item].ext.size()>0)	{
		s+=" (";
		s+=vec[item].ext;
		s+=")";
	}
	pen.DrawString(x,y,s);
}

unsigned TLBA_FileType::ItemHeight()
{
	return TOADBase::DefaultFont().Height();
}

unsigned TLBA_FileType::ItemCount()
{
	return vec.size();
}
