/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */
#include <cstdio>
#include <cstring>

#include <toad/toad.hh>
#include <toad/bitmap.hh>
#include <toad/bitmapfilter.hh>
#include <toad/io/urlstream.hh>
#include <toad/io/binstream.hh>
#include <toad/filter_bmp.hh>

TFilterBMP::EResult TFilterBMP::Load(istream &is)
{
	TInBinStream file(&is);

	TBMPFileInfo bfi;
	ulong dummy;

	// load file header 
	//------------------
	if (!file.CompareString("BM",2)) {
		SetError("'%s' is not a bitmap file");
		return WRONG;
	}
		
	file.ReadDWord();
	file.ReadDWord();
	bfi.bitmap_offset = file.ReadDWord();

	// bitmap info || bitmap core info
	bfi.color_offset = file.TellRead();

	dummy = file.ReadDWord();
	bfi.color_offset += dummy;		// add size of bitmap info
	bfi.color_count  = (bfi.bitmap_offset - bfi.color_offset) / 4;
	if (dummy!=40) {
		SetError("Can't load OS/2 1.2 Bitmap Files");
		return ERROR;
	}

	bfi.width						= file.ReadDWord();
	bfi.height 					=	file.ReadDWord();
	bfi.planes					= file.ReadWord();
	bfi.bits_per_pixel	=	file.ReadWord();
	bfi.compression			= file.ReadDWord();
	bfi.image_size			= file.ReadDWord();
	if (bfi.planes!=1 || bfi.compression!=0) {
		SetError("Can't load compressed bitmap files or files with more than "
						 "one plane.");
		return ERROR;
	}

	if (( bfi.bits_per_pixel!=1 && bfi.bits_per_pixel!=4
		 && bfi.bits_per_pixel!=8 && bfi.bits_per_pixel!=24))
	{
		SetError("Bits per pixel resolution must be 1,4,8 or 24");
		return ERROR;
	}

	// load color palette 
	//--------------------
	TRGB f;
	if (bfi.bits_per_pixel!=24)	{
		CreateBuffer(bfi.width,bfi.height,TBITMAP_INDEXED);
		file.SeekRead(bfi.color_offset);
		for(unsigned i=0; i<bfi.color_count; i++)	{
			f.b = file.ReadByte();
			f.g = file.ReadByte();
			f.r = file.ReadByte();
			file.ReadByte();
			SetIndexColor(i,f);
		}
	}	else {
		CreateBuffer(bfi.width, bfi.height, TBITMAP_TRUECOLOR);
	}
	
	// load bitmap data 
	//------------------
	file.SeekRead(bfi.bitmap_offset);

	int i;
	unsigned n,c,x;
	int y;
	unsigned char b;

	switch(bfi.bits_per_pixel) {
		case 1:
			for(y=bfi.height-1; y>=0; y--) {
				c = 0;
				for(x=0; x<bfi.width; x+=8)	{
					b = file.ReadByte();
					c++;
					for(i=0; i<8; i++) {
						n=b&1;
						b>>=1;
						SetIndexPixel(x+7-i,y, n);
					}
				}
				c=4-(c%4); 
				if (c<4) 
					for(i=c; i>0; i--) 
						file.ReadByte();
			}
			break;

		case 4:
			for(y=bfi.height-1; y>=0; y--) {
				c = 0;
				for(x=0; x<bfi.width; x+=2)	{
					c++;
					b = file.ReadByte();
					n=b & 15;
					SetIndexPixel(x+1,y,n);
					n=(b>>4) & 15;
					SetIndexPixel(x,y,n);
				}
				c=4-(c%4); 
				if (c<4) 
					for(i=c; i>0; i--) 
						file.ReadByte();
			}
			break;
		case 8:
			for(y=bfi.height-1; y>=0; y--) {
				c = 0;
				for(x=0; x<bfi.width; x++) {
					c++;
					SetIndexPixel(x,y,file.ReadByte());
				}
				c=4-(c%4); 
				if (c<4) 
					for(i=c; i>0; i--) 
						file.ReadByte();
			}
			break;
		case 24:
			unsigned char c;
			for(y=bfi.height-1; y>=0; y--) {
				c=0;
				for(x=0; x<bfi.width; x++) {
					f.b = file.ReadByte();
					f.g = file.ReadByte();
					f.r = file.ReadByte();
					c+=3;
					SetColorPixel(x,y, f);
				}
				c=4-(c%4); 
				if (c<4) 
					for(i=c; i>0; i--) 
						file.ReadByte();
			}
			break;
	}
	return OK;
}

bool TFilterBMP::Save(ostream &os)
{
	TOutBinStream file(&os);

	file.WriteString("BM");												// type						(0)
	file.WriteDWord(14+40+w*h*3UL);								// size						(2)
	file.WriteDWord(0);														// reserved				(6)
	file.WriteDWord(14+40);												// bitmap offset	(10)

	// bitmap info header
	//--------------------
	file.WriteDWord(40);													// size						(14)
	file.WriteDWord(w);
	file.WriteDWord(h);
	file.WriteWord(1);														// planes
	file.WriteWord(24);														// bit per pixel
	file.WriteDWord(0);														// compression
	file.WriteDWord(w*h*3UL);											// size if image
	file.WriteDWord(0);
	file.WriteDWord(0);
	file.WriteDWord(0);
	file.WriteDWord(0);

	short c;
	TRGB f;
	for(int y=h-1; y>=0; y--)	{
		c=0;
		for(int x=0; x<w; x++) {
			if (IsIndex())
				GetIndexColor(GetIndexPixel(x,y), &f);
			else
				GetColorPixel(x,y, &f);
			file.WriteByte(f.b);
			file.WriteByte(f.g);
			file.WriteByte(f.r);
			c+=3;
		}
		c=4-(c%4);
		if (c<4)
			for(int x=c; x>0; x--)
				file.WriteByte(0);
	}
	return true;
}
