/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-2000 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

// #define EXPERIMENTAL

/*
 * - improve paint:
 *   - horizontal clip
 *   - vertical clip   (ok)
 * - add regexp to `Find'
 * - select with mouse (scrolling is missing, reduce flicker)
 * - select with shift+cursor keys
 * - something MVC like for multiple views
 * - `GotoLine' resulted in a difference of one line and messed up the
 *   screen; pity is I don't know how to reproduce it
 */

#ifdef EXPERIMENTAL
	#include <X11/Xlib.h>
	#include <X11/Xutil.h>
	#define _TOAD_PRIVATE
#endif

#include <toad/toad.hh>
#include <toad/simpletimer.hh>
#include <toad/scrolledarea.hh>
#include <toad/textarea.hh>

// timer for blinking caret
//---------------------------------------------------------------------------
class TTextArea::TBlink:
	public TSimpleTimer
{
	public:
	TBlink() {
		current = NULL;
	}
	TTextArea *current;
	bool visible;						// true: draw text cursor
	bool blink;							// blink state (it would be better to sync the timer on
													// every keyDown & mouseLDown event!)
	void tick();
};

static TTextArea::TBlink blink;

void TTextArea::TBlink::tick()
{
	if (!current)				// sanity check, not needed
		return;
	blink = !blink;
	if (visible!=blink) {
		visible = blink;
		current->_invalidate_line(current->_cy);
		current->PaintNow();
	}
}

CONSTRUCTOR(TTextArea)
{
	_Init();
}

//. initialisation for all constructors
void TTextArea::_Init()
{
	SetMouseMoveMessages(TMMM_LBUTTON);

	_enabled = true;
	bTabKey = true;

	font = new TFont(TFont::TYPEWRITER, TFont::PLAIN, 12);
	if (font)
		SetItemSize(font->TextWidth("X"), font->Height());

	_Reset();
	bDoubleBuffer = true;
}

TTextArea::~TTextArea()
{
	if (blink.current==this) {
		blink.StopTimer();
		blink.current=NULL;
	}
	delete font;
}

void TTextArea::SetEnabled(bool b)
{
	if (b==_enabled)
		return;
	_enabled = b;

	if (_enabled) {
		SetBackground(TColor::WHITE);
	}	else {
		SetBackground(TColor::DIALOG);
	}
	Invalidate();
}

void TTextArea::_Reset()
{
	area_x = area_y = 0;
	_cx = _cy = _sx = 0;
	_tabwidth = 2;
	_leftborder = 0;
	_rightborder = 60;
	_autoident = true;
		
	_selection_left = 0;
	_selection_right = 0;
	_selection_visible = false;
	_wordstar_control = false;
	
	_line_modified = false;
	_modified = false;
	_bline = 0;

	// calculate text width & height on screen
	_tw = _th = 1;
	unsigned bol, eol;
	bol = 0;
	do {
		eol = _data.find('\n', bol);
		if (eol==string::npos)
			eol = _data.size();
		unsigned w = eol-bol;
		if (w!=0) {
			string s = _data.substr(bol, eol-bol);
			w = _line2x(s.size(), s);
		}
		if (w>_tw)
			_tw = w;
		_th++;
		bol=eol+1;
	} while(bol<=_data.size());
	
	eol = _data.find('\n');
	_line = _data.substr(0, eol);
	
	SetAreaSize(_tw, _th);
	pArrangeSB();
}

void TTextArea::putData(const string &data)
{
	_data = data;

	_Reset();

	Invalidate();
	sigStatus();
}

const string& TTextArea::getData()
{
	_store_line();
	return _data;
}

//. Search starting from the current cursor position for a regular 
//. expression and place cursor at the result.
//. <P>
//. The current implementation doesn't use regular expressions.
//---------------------------------------------------------------------------
void TTextArea::Find(const string &expr)
{
	_store_line();
	_cursor_right();
#if 0
cout << "_bline+_cx:" << _bline+_cx << endl;
cout << "_data.size:" << _data.size() << endl;
#endif
	unsigned pos = _data.find(expr, _bline+_cx);
	if (pos!=string::npos) {
		_go2bufferpos(pos);
		sigStatus();
	}
}

//. Go to a specific line in a range from 1 to n.
unsigned TTextArea::GotoLine(unsigned line)
{
	if (line>0)
		line--;
	_bline = _y2buffer(&line);

	// get new line
	//--------------
	_invalidate_line(_cy);
	_cy = line;
	_invalidate_line(_cy);
	_fetch_line();
	_cx = _xy2buffer(0,line) - _bline;
	_calculate_sliders();
	sigStatus();
	return line+1;
}

//. Place cursor at a specific buffer position.
//---------------------------------------------------------------------------
void TTextArea::_go2bufferpos(unsigned pos)
{
	_store_line();
	unsigned bol, eol;
	bol = 0;
	_cy = _cx = 0;

	while(true) {
		eol = _data.find('\n', bol);
		if (pos<eol)
			break;
		if (eol==string::npos)
			eol = _data.size();
		bol=eol+1;
		_cy++;
	}
	_bline = bol;
	_fetch_line();
	_cx = pos-bol;
	_calculate_sliders();
}


void TTextArea::keyDown(TKey key, char* str, unsigned modifier)
{
	if (_cx==npos) {
		if (key!=TK_UP && key!=TK_DOWN)
			_cx = _x2line(_sx, _line);
		if (key==TK_LEFT && _cx==_line.size()) {
//			cout << "no left" << endl;
			_invalidate_line(_cy);
			return;
		}
	}

	if (_wordstar_control) {
		_wordstar_control = false;
		switch(key) {
			case 'b':
			case 'B':
				_selection_left = _bline+_cx;
				_selection_visible = true;
				Invalidate();
				break;
			case 'k':
			case 'K':
				_selection_right = _bline+_cx;
				_selection_visible = true;
				Invalidate();
				break;
			case 'h':
			case 'H':
				_selection_visible = !_selection_visible;
				Invalidate();
				break;
			case 'y':	// erase selection
			case 'Y':
				_selection_erase();
				sigStatus();
				break;
			case 'c': // copy selection
			case 'C':
				_selection_copy();
				sigStatus();
				break;
			case 'm': // move selection
			case 'M':
				_selection_move();
				break;
		}
		return;
	}

	if (modifier & MK_CONTROL)
	{
		switch(key) {
			case 'y':
			case 'Y':
				_delete_line();
				break;
			case 'k':
			case 'K':
				_wordstar_control=true;
				break;
		}
	} else {
		switch(key)	{
			case TK_RIGHT:
				_cursor_right();
				break;
			case TK_LEFT:
				_cursor_left();
				break;
			case TK_UP:
//				cout << "before cursor up: _cx=" << _cx << " _sx=" << _sx << endl;
				_cursor_up();
//				cout << "after cursor up: _cx=" << _cx << " _sx=" << _sx << endl;
				break;
			case TK_DOWN:
//				cout << "before cursor down: _cx=" << _cx << " _sx=" << _sx << endl;
				_cursor_down();
//				cout << "after cursor down: _cx=" << _cx << " _sx=" << _sx << endl;
				break;
			case TK_HOME:
				_cursor_home();
				break;
			case TK_END:
				_cursor_end();
				break;
			case TK_RETURN:
				_return();
				break;
			case TK_DELETE:
				_delete();
				break;
			case TK_BACKSPACE:
				_backspace();
				break;
			case TK_TAB:
				_insert('\t');
				break;
			case TK_PAGEUP:
				_page_up();
				break;
			case TK_PAGEDOWN:
				_page_down();
				break;
			default:
				if (key>=32 && key<=255)
					_insert(key);
		}
//		cout << "line:" << _cy << " total height:" << _th << " area_y:" << area_y << " area_h:" << area_h << " visi_h:" << visi_h << endl;
	}
	
	sigStatus();
}

//. calculate the buffer postion of line `y'
//------------------------------------------
unsigned TTextArea::_y2buffer(unsigned *y) const
{
	if (*y==_cy)
		return _bline;

	unsigned bline = 0;
	unsigned i;
	for(i=0; i<*y; i++) {
		unsigned p = _data.find('\n', bline);
		if (p==string::npos)
			break;
		p++;
		bline=p;
	}
	*y=i;
//cout << "_y2buffer: " << i << endl;
	return bline;
}

//. calculate the buffer position of position (x,y)
//--------------------------------------------------
unsigned TTextArea::_xy2buffer(unsigned x, unsigned y) const
{
	unsigned bol;
	string line;
	
	if (y!=_cy) {
		bol = _y2buffer(&y);
		unsigned eol = _data.find('\n', bol);
		if (eol!=string::npos)
			eol-=bol;
		line = _data.substr(bol, eol);
	} else {
		bol = _bline;
		line = _line;
	}
	return bol + _x2line(x, line);
}

//. screen position to position in string `line'
unsigned TTextArea::_x2line(unsigned x, const string &line) const
{
	unsigned cx = 0, i;
	for(i=0; i<line.size(); i++) {
		if (line[i]=='\t') {
			unsigned n = (_tabwidth-cx-1) % _tabwidth;
			if (n>0)
				cx+=n;
		}
		cx++;
		if (cx>x)
			break;
	}
	return i;
}

//. position in `line' to screen position
unsigned TTextArea::_line2x(unsigned x, const string &line) const
{
	unsigned i, cx=0;
	for(i=0; i<x; i++) {
		if (line[i]=='\t') {
			unsigned n = (_tabwidth-cx-1) % _tabwidth;
			if (n>0)
				cx+=n; 
		}
		cx++;
	}
	return cx;
}

void TTextArea::mouseLDown(int x, int iy, unsigned m)
{
	if (_cx==npos)
		_cx = _x2line(_sx, _line);

	if (m & MK_DOUBLE) {
		_selection_left = _bline;
		_selection_right = _data.find('\n', _bline);
		_selection_right++;	// assumes string::npos==-1
		_selection_visible = true;
		Invalidate();
		return;
	}

	unsigned y = iy;

	SetFocus();
	_store_line();

	// calculate text position
	x=(x/item_w)+area_x;
	y=(y/item_h)+area_y;

	_bline = _y2buffer(&y);

	// get new line
	//--------------
	_invalidate_line(_cy);
	_cy = y;
	_invalidate_line(_cy);
	_fetch_line();

	_cx = _xy2buffer(x,y) - _bline;

	blink.visible=true;

	_calculate_sliders();
	
	sigStatus();
}

void TTextArea::mouseMove(int x, int y, unsigned)
{
	if (x<0 || y<0)
		return;

	// calculate text position
	x=(x/item_w)+area_x;
	y=(y/item_h)+area_y;

	unsigned uy = y;
	unsigned line = _y2buffer(&uy);
	unsigned ux = _xy2buffer(x,y) - line;
	
	unsigned p1=_bline+_cx;
	unsigned p2=line+ux;
	if (p1<p2) {
		_selection_left = p1;
		_selection_right = p2;
	} else {
		_selection_left = p2;
		_selection_right = p1;
	}
	
	_selection_visible = true;
	Invalidate(false);
}

void TTextArea::mouseLUp(int,int,unsigned)
{
}

void TTextArea::mouseMDown(int, int, unsigned)
{
	bool ai = _autoident;
	_autoident = false;

	_selection_left = 0;
	_selection_right = 0;
	_selection_visible = false;
	_wordstar_control = false;
	string data = GetSelection();
	for(unsigned i=0; i<data.size(); i++) {
		if (data[i]=='\n') {
			_return();
		} else {
			_insert(data[i]);
		}
	}
	_autoident = ai;
	sigStatus();
}

void TTextArea::focus()
{
	_invalidate_line(_cy);
	if (IsFocus()) {
		blink.current=this;
		blink.visible=true;
		blink.blink=true;
		blink.StartTimer(0,500000);
	} else {
		blink.StopTimer();
		blink.current=NULL;
	}
}

//. Insert a single char at the current cursor position
void TTextArea::_insert(unsigned char c)
{
	c = filter(c);
	if (c>0) {
		_line.insert(_cx, 1, c);

		if (_cx+_bline <= _selection_left)
			_selection_left++;
		if (_cx+_bline <= _selection_right)
			_selection_right++;

		// update area size
		//------------------
		unsigned w = _line2x(_line.size(), _line);
		if (w>_tw) {
			_tw=w;
			SetAreaSize(_tw, _th);
			_calculate_sliders();
		}
		_line_modified = true;
		_cursor_right();
	}
}

//. insert a string at the current cursor position
//---------------------------------------------------------------------------
void TTextArea::_insert(const string &str)
{
	unsigned n = str.size();
	if (n==0) return;
	_line.insert(_cx, str);
	if (_cx+_bline <= _selection_left)
		_selection_left+=n;
	if (_cx+_bline <= _selection_right)
		_selection_right+=n;

	// update area size
	//------------------
	// width is missing!

	unsigned p = 0;
	unsigned lines_inserted=0;
	while(true) {
		p=str.find('\n', p);
		if (p==string::npos)
			break;
		p++;
		_th++;
		lines_inserted++;
	}
	if (lines_inserted) {
		SetAreaSize(_tw, _th);
		_calculate_sliders();
		Invalidate();		// SCROLL
	} else {
		_invalidate_line(_cy);
	}

	_line_modified = true;
	_store_line();
	_fetch_line();
}

//. Break the current line before cursor position into two lines.
//---------------------------------------------------------------------------
void TTextArea::_return()
{
	_line.insert(_cx, "\n");

	if (_cx+_bline <= _selection_left)
		_selection_left++;
	if (_cx+_bline < _selection_right)
		_selection_right++;

	_line_modified=true;
	_store_line();			// isn't this done by `cursor_down'?

	string ident;
	if (_autoident) {
		unsigned n = _line.find_first_not_of(" \t");
		if (n!=string::npos && n!=0) {
			ident = _line.substr(0,n);
    }
	}

	_fetch_line();			// isn't this done by `cursor_down'?
	_cursor_down();
	_th++;
	SetAreaSize(_tw, _th);

	_cursor_home();
	_insert(ident);
	_cx=ident.size();

	// scroll down
	unsigned y = (_cy - area_y)*item_h;
	TRectangle rect(0,y,Width(),Height()-y);
	ScrollRectangle(rect, 0,item_h);

	_calculate_sliders();
}

void TTextArea::_delete()
{
	if (_line.size()>_cx)	{
		_line.erase(_cx,1);
		_line_modified = true;
		_invalidate_line(_cy);
	}	else {
		// join current line with the next line
		_store_line();
		
		unsigned p = _data.find('\n', _bline);
		if (p==string::npos)
			return;
		_data.erase(p,1);

		// scroll up
		unsigned y = (_cy - area_y)*item_h;
		TRectangle rect(0,y,Width(),Height()-y);
		ScrollRectangle(rect, 0,-item_h);
		_invalidate_line(_cy);
		
		_fetch_line();
		_th--;
		if (_line.size() > _tw)
			_tw = _line.size();
		SetAreaSize(_tw, _th);
int dummy = area_y;					// workaround for the current TScrolledArea
		_calculate_sliders();
if (dummy!=area_y) Invalidate();
	}

	if (_cx+_bline < _selection_left)
		_selection_left--;
	if (_cx+_bline < _selection_right)
		_selection_right--;
}

void TTextArea::_backspace()
{
	if (_bline>0 || _cx>0) {
		_cursor_left();
		_delete();
	}
}

void TTextArea::_delete_line()
{
	if (_bline < _selection_left) {
		if (_bline+_line.size() < _selection_left) {
			_selection_left-=_line.size();
		} else {
			_selection_left = _bline;
		}
	}
	if (_bline < _selection_right) {
		if (_bline+_line.size() < _selection_right) {
			_selection_right-=_line.size();
		} else {
			_selection_right = _bline;
		}
	}

	_cx = 0;
	_line.erase();
	_line_modified = true;
	_delete();
// Invalidate();
}

void TTextArea::_cursor_left()
{
	if (_cx>0) {
		_cx--;
		_invalidate_line(_cy);
	} else {
		if (_cy>0) {
			_cursor_up();
			_cursor_end();
		}
	}
	_calculate_sliders();
	blink.visible = true;
}

void TTextArea::_cursor_right()
{
	if (static_cast<unsigned>(_cx)<_line.size()) {
		_cx++;
		_invalidate_line(_cy);
	} else {
		unsigned cy = _cy;
		_cursor_down();
		if (cy!=_cy)
			_cursor_home();
	}
	_calculate_sliders();
	blink.visible = true;
}

void TTextArea::_cursor_down()
{
	if (_cx!=npos) {
		_sx = _line2x(_cx, _line);
		_cx = npos;
	}
	_store_line();

	// go to the next line
	unsigned p = _data.find('\n', _bline);
	if (p==string::npos)
		return;
	p++;
	_bline=p;

	_fetch_line();
	_invalidate_line(_cy);
	_cy++;
	_invalidate_line(_cy);
	_calculate_sliders();
	blink.visible = true;
}

void TTextArea::_cursor_up()
{
	if (_cy > 0) {
		if (_cx!=npos) {
			_sx = _line2x(_cx, _line);
			_cx = npos;
		}
		_store_line();
		
		// go to the previous line
		if (_bline>=2)
			_bline = _data.rfind('\n', _bline-2)+1;
		else
			_bline = 0;
		if (_bline==string::npos)
			_bline = 0;

		_fetch_line();
		_invalidate_line(_cy);
		_cy--;
		_invalidate_line(_cy);
		_calculate_sliders();
		blink.visible = true;
	}
}

void TTextArea::_cursor_home()
{
	_cx = 0;
	_invalidate_line(_cy);
	_calculate_sliders();
}

void TTextArea::_cursor_end()
{
	_cx = _line.size();
	_invalidate_line(_cy);
	_calculate_sliders();
}

void TTextArea::_page_down()
{
	_store_line();

	int old_area_y = area_y;
	unsigned old_cy = _cy;

	for(int i=0; i<visi_h; i++) {
		unsigned p = _data.find('\n', _bline);
		if (p==string::npos)
			break;
		p++;
		_cy++;
		area_y++;
		_bline=p;
	}
	_fetch_line();
	if (area_y!=old_area_y) {
		_calculate_sliders();
		Invalidate();
	} else if (_cy!=old_cy) {
		_invalidate_line(old_cy);
		_invalidate_line(_cy);
	}
}

void TTextArea::_page_up()
{
	_store_line();

	int old_area_y = area_y;
	unsigned old_cy = _cy;
	
	for(int i=0; i<visi_h; i++) {	
		// go to the previous line
		if (_bline>=2)
			_bline = _data.rfind('\n', _bline-2)+1;
		else
			_bline = 0;
		if (_bline==string::npos)
			_bline = 0;
		if (_cy>0)
			_cy--;
		if (area_y>0)
			area_y--;
	}
	_fetch_line();
	if (area_y!=old_area_y) {
		_calculate_sliders();
		Invalidate();
	} else if (_cy!=old_cy) {
		_invalidate_line(old_cy);
		_invalidate_line(_cy);
	}
}

void TTextArea::_selection_copy()
{
	if (_selection_left<_selection_right) {
		string str = _data.substr(_selection_left, 
															_selection_right-_selection_left);
		_insert(str);
		_selection_left=_bline+_cx;
		_selection_right=_selection_left+str.size();
		Invalidate();
	}
}

void TTextArea::_selection_move()
{
	if (_selection_left<_selection_right) {
		string str = _data.substr(_selection_left, 
															_selection_right-_selection_left);
		_selection_erase();
		_insert(str);
		_selection_left=_bline+_cx;
		_selection_right=_selection_left+str.size();
		Invalidate();
	}
}

void TTextArea::_selection_erase()
{
	if (_selection_left<_selection_right) {
		_store_line();
		unsigned size = _selection_right-_selection_left;
		
		unsigned lines = 0;
		unsigned breaks_till_selection=0;
		unsigned p=_selection_left;
		for(;;) {
			if (p<_bline)
				breaks_till_selection++;
			p=_data.find('\n', p);
			if (p==string::npos || p>_selection_right)
				break;
			p++;
			lines++;
		}
//		cout << "removing " << lines << " lines" << endl;

		_th-=lines;
#if 0		
		cout << "_bline: " << _bline << endl;
		cout << "_cx   : " << _cx << endl;
		cout << "_cy   : " << _cy << endl;
#endif

		if (_selection_left < _bline+_cx && _bline+_cx < _selection_right) {
			// cursor is inside the selection
//			cout << "cursor is inside the selection" << endl;
//			cout << "  breaks till selection: " << breaks_till_selection << endl;
			
			// line breaks between _selection_left & _bline
			_cy -= breaks_till_selection;
			
			// start of the first line containing the selection
			_bline = _data.rfind('\n', _selection_left);
			if (_bline==string::npos)
				_bline=0;
			else
				_bline++;
			
			_cx = _selection_left-_bline;
//			cout << "new _cx: " << _cx << endl;
			
		} else {
			// cursor is outside the selection
//			cout << "cursor is outside the selection" << endl;
			if (_bline < _selection_right && _selection_right < _bline+_cx) {
				if (_selection_left<_bline) {
					_cx -= _selection_right - _bline;
//					cout << "  decrement _cx by " << _selection_right - _bline << endl;
				} else {
					_cx -= _selection_right - _selection_left;
//					cout << "  decrement _cx by " << _selection_right - _selection_left << endl;
				}
			}

			if (_selection_left < _bline) {
				if (_bline < _selection_right) {
//cout << "_cx [1] = " << _cx << endl;
					_bline = _data.rfind('\n', _selection_left);
					if (_bline==string::npos)
						_bline=0;
					else
						_bline++;
					_cx += _selection_left - _bline;
//cout << "_cx [2] = " << _cx << endl;
				} else {
					_bline -= size;
				}
				_cy -= lines;
			}
		}
#if 0		
		cout << "done adjustments:" << endl;
		cout << "_bline: " << _bline << endl;
		cout << "_cx   : " << _cx << endl;
		cout << "_cy   : " << _cy << endl;
#endif
		_data.erase(_selection_left, size);

		_selection_left = _selection_right = 0;

		_fetch_line();
		_modified=true;
		SetAreaSize(_tw, _th);
		_calculate_sliders();
		Invalidate();
		sigStatus();
	}
}

//. copy current line from linebuffer to textbuffer
void TTextArea::_store_line()
{
	// line wasn't modified => dont't waste time
	if (!_line_modified)
		return;
		
	_line_modified = false;
	_modified = true;
	
	unsigned eol = _data.find('\n', _bline);
	if (eol==string::npos)
		eol=_data.size();
	_data.replace(_bline, eol-_bline, _line);
}

//. copy current line from textbuffer to linebuffer
void TTextArea::_fetch_line()
{
	unsigned eol = _data.find('\n', _bline);
	if (eol!=string::npos)
		eol-=_bline;
	_line = _data.substr(_bline, eol);

	// #warning "ignoring tab stops"
	// shouldn't do this check anyway, better to check and adjust cursor position
	// during: insert, cursor left & right, delete, backspace, ...
	if (_cx!=npos && _cx>_line.size())
		_cx = _line.size();
}

void TTextArea::paint()
{
	TPen pen(this);
	ClipPen(pen);
	
	// get clip area (in text coordinates)
	//-------------------------------------
	TRectangle clipbox;
	pen.GetClipBox(&clipbox);		// should be called after `ClipPen'!
//cout << "textarea: paint y=" << clipbox.y << " h=" << clipbox.h << endl;
	unsigned x1 = clipbox.x / item_w;
	unsigned y1 = clipbox.y / item_h;
	unsigned x2 = (clipbox.x+clipbox.w-1) / item_w;
	unsigned y2 = (clipbox.y+clipbox.h-1) / item_h;
//cout << "lines " << y1 << " to " << y2 << endl;
#if 0
cout << "clipbox:" << clipbox.x << ", " 
									 << clipbox.y << ", "
									 << clipbox.w << ", "
									 << clipbox.h << endl;
cout << "item size: " << item_w << ", " << item_h << endl;
cout << "clipbox:" << x1 << ", " 
									 << y1 << ", "
									 << x2 << ", "
									 << y2 << endl;
#endif	

	// prepare pen
	//------------------------------------
	if (font)
		pen.SetFont(font);
	pen.SetOrigin(-area_x*item_w, -area_y*item_h);

	unsigned bol, eol;
	int sy = 0;
	bol = 0;
	unsigned cl = 0;

	// calculate difference between the line in the line buffer
	// and the line in the data buffer
	//-------------------------------------------------------
	eol = _data.find('\n', _bline);
	if (eol!=string::npos)
		eol-=_bline;
	int difference = _line.size() - _data.substr(_bline, eol).size();

	// find first visible line (starting from current line)
	// that's actually a bad algorithm, the previous visible
	// is a better date to start with
	//-------------------------------------------------------
	bol = _bline;
	cl  = _cy;
	sy  = _cy * item_h;

	while (cl>area_y) {
		// go one line backwards
		if (bol>0)
			bol--;
		if (bol>0 && _data[bol-1]!='\n') {
			bol--;
			bol = _data.rfind('\n', bol);
			if (bol==string::npos)
				bol=0;
			else
				bol++;
		}
		cl--;
		sy-=item_h;
	}

	// paint visible lines
	//---------------------
	string line;
	string uline;
	int sx;
	while(true) {
		eol = _data.find('\n', bol);

//		if (cl>=area_y) {
		if (y1<=cl-area_y && cl-area_y<=y2) {
// cout << "PAINTING LINE " << cl << endl;
			uline = line = cl==_cy ? _line : _data.substr(bol, eol-bol);

			// add tabs
			//----------
			int sx = _cx;
			for(unsigned i=0; i<line.size(); i++) {
				if (line[i]=='\t') {
					unsigned n = (_tabwidth-i-1) % _tabwidth;
					if (n>0) {
						line.replace(i, 1, n+1, ' ');
						if (sx>i)
							sx+=n;
						i+=n;
					}
				}
			}
			if (_cx==-1)
				sx=_sx;

			// draw line
			// (use `substr' for horizontal optimization)
			//--------------------------------------------
			bool cursor_in_selection = false;
			if (_selection_left<_selection_right && _selection_visible) {
			
				// draw line with selection
				//--------------------------
				unsigned b,e;
				unsigned ml, mr;
				bool inside_selection = false;
				ml = 0;
				mr = uline.size();
				b = bol;
				e = eol;
				if (bol>_bline)
					b+=difference;
				if (bol>=_bline)
					e+=difference;
				
				if (b<=_selection_left && _selection_left<=e) {
					ml = _selection_left-b;
					inside_selection = true;
				}
				if (b<=_selection_right && _selection_right<=e) {
					mr = _selection_right-b;
					inside_selection=true;
				}
				if (_selection_left<=b && e<=_selection_right)
					inside_selection=true;
				if (inside_selection && !line.empty()) {
#if 0
					cout<< "line:" << cl
							<< "  ml:" << ml
							<< "  mr:" << mr
							<< endl;
#endif
#if 1
					// adjust ml && mr to the tabstops
					//---------------------------------
					{
						for(unsigned i=0; i<uline.size(); i++) {
							if (uline[i]=='\t') {
								unsigned n = (_tabwidth-i-1) % _tabwidth;
								if (n>0) {
									uline.replace(i, 1, n+1, ' ');
									if (ml>i)
										ml+=n;
									if (mr>i)
										mr+=n;
									i+=n;
								}
							}
						}
					}
#endif
#if 0
					cout<< "       "
							<< "  ml:" << ml
							<< "  mr:" << mr
							<< endl;
#endif					
					if(ml>0)
						pen.FillString(0, sy, line.substr(0,ml));
					pen.SetColor(255,255,255);
					pen.SetBackColor(0,0,0);
					pen.FillString(ml*item_w,sy, line.substr(ml, mr-ml));
					pen.SetColor(0,0,0);
					pen.SetBackColor(255,255,255);
					if (mr<line.size())
						pen.FillString(mr*item_w,sy, line.substr(mr));
					
					if (ml<=_cx && _cx<mr)
						cursor_in_selection = true;
				} else {
					pen.FillString(0,sy, line);
				}
			} else {
			
				// draw simple line without selection
				//------------------------------------
				pen.FillString(0,sy, line);
			}
			
			// draw cursor
			if (blink.visible && cl==_cy && IsFocus()) {
				if (cursor_in_selection)
					pen.SetColor(255,255,255);
				sx*=item_w;
				pen.DrawLine(sx	 ,sy,								 sx,	 sy+pen.Height()-1);
				pen.DrawLine(sx-2,sy,	               sx+2, sy );
				pen.DrawLine(sx-2,sy+pen.Height()-1, sx+2, sy+pen.Height()-1);
				if (cursor_in_selection)
					pen.SetColor(0,0,0);
			}
		}
		cl++;
		if (eol==string::npos || cl>area_y+visi_h)
			break;
		bol=eol+1;
		sy+=item_h;
	}
}

void TTextArea::_invalidate_line(unsigned l)
{
	if (l<area_y)
		return;
	unsigned pos = (l-area_y)*item_h;
	if (pos>Height())
		return;
// cout << "invalidating screen line: " << pos << endl;
	Invalidate(0,pos,Width(),item_h);
}

// the code around `ScrollWindow' is the same as in TScrolledArea! :(
void TTextArea::_calculate_sliders()
{
	SetAreaSize(_tw, _th);
	
	unsigned cx = _cx==npos ? _sx : _line2x(_cx, _line);

	// scroll to cursor (it's currently ignoring the tab stops!)
	if (cx>area_x+visi_w) {
		area_x = cx-visi_w;
		Invalidate();
	}
	if (cx<area_x) {
		area_x = cx;
		Invalidate();
	}

	if (_cy>=area_y+visi_h) {
		int ny = _cy-visi_h+1;
		int dy = ny - area_y;
		area_y = ny;
		if (abs(dy)<visi_h)
			ScrollWindow(0,-dy*item_h);
		else
			Invalidate();
//cout << "scroll1" << endl;
	}

	if (_cy<area_y) {
		int ny = _cy;
		int dy = ny - area_y;
		area_y = ny;
		if (abs(dy)<visi_h)
			ScrollWindow(0,-dy*item_h);
		else
			Invalidate();
//cout << "scroll2" << endl;
	}
		
	pArrangeSB();
}

unsigned char TTextArea::filter(unsigned char c)
{
	return c;
}
