/***********************************************************************/
/*                                                                     */
/* Project      : GRAV                                                 */
/* Current Rel. : 3.5                                                  */
/* Creator      : Michael Knigge                                       */
/* Creation Date: 01/21/95                                             */
/* Description  : Functions needed for reading and sorting directories */
/*                                                                     */
/* Functions    : grav_check_filename()                                */
/*                grav_read_directory()                                */
/*                grav_show_directory()                                */
/*                grav_change_directory()                              */
/*                grav_sort_directory()                                */
/*                                                                     */
/***********************************************************************/

#define SUBPROG

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <ncurses.h>
#include "defines.h"
#include "standard.h"
#include "gbm.h"
#include "directory.h"
#include "error.h"
#include "misc.h"
#include "screen.h"
#include "info.h"
#include "selector.h"


int grav_check_filename(char *filename)
{
   struct stat buffer;
   
   /*
    * if stat() returns -1, an error occured and we
    * suppose that the file/directory doen't exist
    */

   if ((stat(filename, &buffer)) == -1)
      return(IS_NOT_FOUND);

   /*
    * check if the given argument is a file or a directory
    */

   if ((buffer.st_mode & 0x7000) == 0x4000)
       return(IS_DIRECTORY);
   else
       return(IS_FILE);
}

int grav_read_directory(int use_color, char *path, int *files, DIRTAB dirtab[])
{
   DIR           *dir;
   struct dirent *dir_entry;
   char           current_dir[256];
   int            local;
   int            line;
   int            ft;
   int            flag;
   int            type;
   int            percent;
   int            index;
   int            prog_bar;
   int            bar;
   int            error_id;
   char           indicator[4] = "-\\|/"; 

   GBM            gbm;
   GBMFT          gbmft;
   GBMRGB         gbmrgb[0x100];   


  /*
   * first draw a little nice window ;-)
   */
  line = ( LINES / 2 ) + 2;
   
  if (use_color == TRUE)
  {
     attron(A_BOLD);
     grav_draw_box(4,45,line,6,YELLOW_ON_RED);
     attron(COLOR_PAIR(YELLOW_ON_RED));
     mvaddstr(line, 17, "Reading directory...");
  }
  else
  {
     grav_draw_box(4,45,line,6,COLOR_REVERSE);
     attron(A_REVERSE);
     mvaddstr(line,17, "Reading directory...");
  }
  
     
   /*
    * get current directory, open the directory given in "path" 
    * and return a pointer
    */
   
   getcwd(current_dir, 255);
   dir          = opendir(path);
   local        = 0;
   index        = 0;
   percent      = 0;
   
   /*
    * call readdir() until all entries in directory were returned
    */
   
   while((dir_entry=readdir(dir)) != NULL)
   {
      mvaddch(line+1, 25, indicator[index]);
      refresh();
      index++;
      if (index > 3)
         index = 0;

      /*
       * add always if entry is a directory
       */
      if ((type=grav_check_filename(dir_entry->d_name)) == IS_DIRECTORY)
         flag = TRUE;
      else
         flag = FALSE;
      
      /*
       * never add "." directory
       */
       
       if (strcmp(dir_entry->d_name, ".") == 0)
          flag = FALSE;

      /*
       * don't add ".." when in root-directory
       */
       
      if ( strcmp(current_dir, "/")  == 0    && 
           strcmp(dir_entry->d_name, "..") == 0 )
         flag = FALSE;
      
      /*
       * check if the current file is a graphic that can be 
       * displayed. If not, don't add the entry to the structure.
       * if we got more than MAX_DIRTAB entries, we just return TRUE.
       */
          
      if (type == IS_FILE)      
      {
         if (grav_guess_filetype(dir_entry->d_name, &ft) != FALSE)
         {
            flag = TRUE;
            if (local == MAX_DIRTAB)
               return(TRUE);
         }
      }

      if (flag == TRUE)
      {
         strcpy(dirtab[local].filename, dir_entry->d_name);
         dirtab[local].filetype = type;
         dirtab[local].tagged   = FALSE;
         dirtab[local].gfx_type = ft;

         dirtab[local].width    = 0;
         dirtab[local].height   = 0;
         dirtab[local].bpp      = 0;

         local++;
         *files = local;
      }
   }
   closedir(dir);

   /*
    * a new text in our window. we are going to get the resolution
    * and bpp of every image in the current directory. Also switch
    * the   nodelay   to TRUE because we want getch() _NOT_ to wait
    * for a key pressed....
    */
   mvaddstr(line,17, "Getting fileinfo... ");
   attron(A_ALTCHARSET);
   nodelay(stdscr,TRUE);

   for(index=0; index<43; index++)
     mvaddch(line+1, 6+index, ACS_BOARD);
   
   /*
    * and now loop over every file and get the fileinformations
    * (interruptable by any key)
    */
   for(index=0; index<local; index++)
   {
      if ((getch()) != ERR)
         index = local;
                     
      percent  = (index+1) * 100 / local;
      prog_bar = 43 * percent / 100;
      
      for (bar=0; bar <prog_bar; bar++)
         mvaddch(line+1, 6+bar, ACS_BLOCK);

      refresh();
      
      if (dirtab[index].filetype == IS_FILE)
      {
         error_id = grav_get_file_info(index, dirtab, &gbm, &gbmft, gbmrgb);
         if (error_id == GRAV_NO_ERROR)
         {
            dirtab[index].width  = gbm.w;
            dirtab[index].height = gbm.h;
            dirtab[index].bpp    = gbm.bpp;
         }
         else
         {
            dirtab[index].width  = -1;
            dirtab[index].height = -1;
            dirtab[index].bpp    = -1;
         }
      }

   }

   /*
    * now switch the   nodelay  to FALSE - so getch() will wait for a
    * key again....
    */
   nodelay(stdscr,FALSE);
   attroff(A_ALTCHARSET);

   if (use_color == TRUE)
      attroff(A_BOLD);
   else
      attroff(A_REVERSE);
  
   /*
    * and at last, sort the directory
    */
   qsort (&(dirtab[1]),
          local - 1,
          sizeof(DIRTAB),
          (void *)grav_sort_directory);
                                     
   return(TRUE);
}

int grav_show_directory(int use_color, int page, int max_files, DIRTAB dirtab[])
{
   int     local;
   int     count;
   int     line;
   int     step;
   char    current_dir[61];

   
   /*
    * first we have to draw the file-window
    */
    
   if (use_color)
   {
      grav_draw_box(LINES-9,47,8,5,BLACK_ON_CYAN);
      attron(COLOR_PAIR(BLACK_ON_CYAN));
   }
   else
   {
      grav_draw_box(LINES-9,47,8,5,COLOR_REVERSE);
      attron(A_REVERSE);   
   }

   step  = LINES - 11;
   line  = 1;
   local = page * step;
   
   for (count=local;count < local + step; count++)
   {

      /*
       * if we are going to display more entries than allowed,
       * just jump out!
       */
       
      if (count >= max_files)
         break;
         
      /*
       * if the current filen in dirtab is a directory,
       * we will put this filename in "<" and ">". otherwise
       * we will use the unmodified filename
       */

      if (dirtab[count].tagged == FALSE)
      {
         if(use_color)
            attron(COLOR_PAIR(BLACK_ON_CYAN));
         else
            attron(A_REVERSE);
            
         grav_display_entry(line, POS_LEFT, count, dirtab);
      }
      else
      {
         attron(A_BOLD);
         
         if (use_color)
            attron(COLOR_PAIR(YELLOW_ON_CYAN));
         else
            attron(A_REVERSE);
            
         grav_display_entry(line, POS_LEFT, count, dirtab);
         attroff(A_BOLD);
      }

      line++;
   }
   
   if (use_color)
      attron(COLOR_PAIR(WHITE_ON_BLUE));
   else
      attroff(A_REVERSE);

   getcwd(current_dir,60);
   strncat(current_dir,BLANK60,60-strlen(current_dir));
   mvaddstr(5, 18, current_dir);                     

   refresh();

   return(TRUE);                     
}

int grav_change_directory(int *error, int index, DIRTAB dirtab[])
{
   *error = GRAV_NO_ERROR;
   
   if (chdir(dirtab[index].filename) == -1)
   {
      *error = GRAV_ERROR_CHDIR;
      return(FALSE);
   }
   
   return(TRUE);
}

int grav_sort_directory(void *ptr1, void *ptr2)
{
   DIRTAB *tab1;
   DIRTAB *tab2;
   
   tab1 = (DIRTAB *)ptr1;
   tab2 = (DIRTAB *)ptr2;


   if (tab1->filetype == IS_DIRECTORY &&
       tab2->filetype == IS_DIRECTORY)
      return(strcmp(tab1->filename,tab2->filename));

   if (tab1->filetype == IS_FILE &&
       tab2->filetype == IS_FILE)
      return(strcmp(tab1->filename,tab2->filename));

   if (tab1->filetype == IS_FILE &&
       tab2->filetype == IS_DIRECTORY)
      return(1);
      
   if (tab1->filetype == IS_DIRECTORY &&
       tab2->filetype == IS_FILE)
      return(-1);      

   /*
    * we can never reach this point....
    */
   return(0);
}
