/***********************************************************************/
/*                                                                     */
/* Project      : GRAV                                                 */
/* Current Rel. : 3.5                                                  */
/* Creator      : Michael Knigge                                       */
/* Creation Date: 04/20/95                                             */
/* Description  : read file (Photo-CD)                                 */
/*                                                                     */
/* Functions    : grav_read_pcd_file()                                 */
/*                                                                     */
/***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <malloc.h>
#include <ncurses.h>
#include "standard.h"
#include "gbm.h"
#include "defines.h"
#include "error.h"
#include "readpcd.h"
#include "view.h"

char *grav_read_pcd_file(int *error_id, int flag, int index, DIRTAB dirtab[], GBM *gbm, GBMFT *gbmft, GBMRGB gbmrgb[])
{
   int  r,g,b,gray;
   int  line;
   int  column;
   int  pos;
   int  bitmap_width;
   long bitmap_size;

   unsigned int  rotate;
   unsigned char gray1[768];
   unsigned char gray2[768];
   unsigned char red[384];
   unsigned char blue[384];

   FILE *pcd_file;
   char *header;
   char *bitmap;
   char *work_bitmap;

   static char   *ft = "PhotoCD Image\0";

   static int gray_LUT[256] = {
       0,   1,   2,   4,   5,   7,   8,   9,  11,  12,  14,  15,  16,  18,  19,  21,
      22,  24,  25,  26,  28,  29,  31,  32,  33,  35,  36,  38,  39,  41,  42,  43,
      45,  46,  48,  49,  50,  52,  53,  55,  56,  57,  59,  60,  62,  63,  65,  66,
      67,  69,  70,  72,  73,  74,  76,  77,  79,  80,  82,  83,  84,  86,  87,  89,
      90,  91,  93,  94,  96,  97,  99, 100, 101, 103, 104, 106, 107, 108, 110, 111,
     113, 114, 115, 117, 118, 120, 121, 123, 124, 125, 127, 128, 130, 131, 132, 134,
     135, 137, 138, 140, 141, 142, 144, 145, 147, 148, 149, 151, 152, 154, 155, 156,
     158, 159, 161, 162, 164, 165, 166, 168, 169, 171, 172, 173, 175, 176, 178, 179,
     181, 182, 183, 185, 186, 188, 189, 190, 192, 193, 195, 196, 198, 199, 200, 202,
     203, 205, 206, 207, 209, 210, 212, 213, 214, 216, 217, 219, 220, 222, 223, 224,
     226, 227, 229, 230, 231, 233, 234, 236, 237, 239, 240, 241, 243, 244, 246, 247,
     248, 250, 251, 253, 254, 256, 257, 258, 260, 261, 263, 264, 265, 267, 268, 270,
     271, 272, 274, 275, 277, 278, 280, 281, 282, 284, 285, 287, 288, 289, 291, 292,
     294, 295, 297, 298, 299, 301, 302, 304, 305, 306, 308, 309, 311, 312, 313, 315,
     316, 318, 319, 321, 322, 323, 325, 326, 328, 329, 330, 332, 333, 335, 336, 338,
     339, 340, 342, 343, 345, 346, 347, 349, 350, 352, 353, 355, 356, 357, 359, 360  
   };

   static int blue_LUT[256] = {
     -318, -316, -314, -312, -310, -308, -306, -304, -302, -300, -298, -296, -294, -292, -290, -288,
     -286, -284, -282, -280, -278, -276, -274, -272, -270, -268, -266, -264, -262, -260, -258, -256,
     -254, -252, -250, -248, -246, -244, -242, -240, -238, -236, -234, -232, -230, -228, -226, -224,
     -222, -220, -218, -216, -214, -212, -210, -208, -206, -204, -202, -200, -198, -196, -194, -192,
     -190, -188, -186, -184, -182, -180, -178, -176, -174, -172, -170, -168, -166, -164, -162, -160,
     -158, -156, -154, -152, -150, -148, -146, -144, -142, -140, -138, -136, -134, -132, -130, -128,
     -126, -124, -122, -120, -118, -116, -114, -112, -110, -108, -106, -104, -102, -100,  -98,  -96,
      -94,  -92,  -90,  -88,  -86,  -84,  -82,  -80,  -78,  -76,  -74,  -72,  -70,  -68,  -66,  -64,
      -62,  -60,  -58,  -56,  -54,  -52,  -50,  -48,  -46,  -44,  -42,  -40,  -38,  -36,  -34,  -32,
      -30,  -28,  -26,  -24,  -22,  -20,  -18,  -16,  -14,  -12,  -10,   -8,   -6,   -4,   -2,    0,
        2,    4,    6,    8,   10,   12,   14,   16,   18,   20,   22,   24,   26,   28,   30,   32,
       34,   36,   38,   40,   42,   44,   46,   48,   50,   52,   54,   56,   58,   60,   62,   64,
       66,   68,   70,   72,   74,   76,   78,   80,   82,   84,   86,   88,   90,   92,   94,   96,
       98,  100,  102,  104,  106,  108,  110,  112,  114,  116,  118,  120,  122,  124,  126,  128,
      130,  132,  134,  136,  138,  140,  142,  144,  146,  148,  150,  152,  154,  156,  158,  160,
      162,  164,  166,  168,  170,  172,  174,  176,  178,  180,  182,  184,  186,  188,  190,  192,
   };

   static int red_LUT[256] = {
     -274, -272, -270, -268, -266, -264, -262, -260, -258, -256, -254, -252, -250, -248, -246, -244,
     -242, -240, -238, -236, -234, -232, -230, -228, -226, -224, -222, -220, -218, -216, -214, -212,
     -210, -208, -206, -204, -202, -200, -198, -196, -194, -192, -190, -188, -186, -184, -182, -180,
     -178, -176, -174, -172, -170, -168, -166, -164, -162, -160, -158, -156, -154, -152, -150, -148,
     -146, -144, -142, -140, -138, -136, -134, -132, -130, -128, -126, -124, -122, -120, -118, -116,
     -114, -112, -110, -108, -106, -104, -102, -100,  -98,  -96,  -94,  -92,  -90,  -88,  -86,  -84,
      -82,  -80,  -78,  -76,  -74,  -72,  -70,  -68,  -66,  -64,  -62,  -60,  -58,  -56,  -54,  -52,
      -50,  -48,  -46,  -44,  -42,  -40,  -38,  -36,  -34,  -32,  -30,  -28,  -26,  -24,  -22,  -20,
      -18,  -16,  -14,  -12,  -10,   -8,   -6,   -4,   -2,    0,    2,    4,    6,    8,   10,   12,
       14,   16,   18,   20,   22,   24,   26,   28,   30,   32,   34,   36,   38,   40,   42,   44,
       46,   48,   50,   52,   54,   56,   58,   60,   62,   64,   66,   68,   70,   72,   74,   76,
       78,   80,   82,   84,   86,   88,   90,   92,   94,   96,   98,  100,  102,  104,  106,  108,
      110,  112,  114,  116,  118,  120,  122,  124,  126,  128,  130,  132,  134,  136,  138,  140,
      142,  144,  146,  148,  150,  152,  154,  156,  158,  160,  162,  164,  166,  168,  170,  172,
      174,  176,  178,  180,  182,  184,  186,  188,  190,  192,  194,  196,  198,  200,  202,  204,
      206,  208,  210,  212,  214,  216,  218,  220,  222,  224,  226,  228,  230,  232,  234,  236,
   };


   /*
    * this code is "stolen" from pcdview, written by Gerd Knorr.
    * I hope that I get a reference of the photo-cd format soon, so
    * I can add more support for photo-cd's....
    */

   pcd_file = fopen(dirtab[index].filename, "r");

   if (pcd_file == NULL)
   {
      *error_id = GRAV_ERROR_OPEN;
      return(NULL);
   }

   /*
    * Now we will verify, if this file is a valid PCD-Image. Therefore
    * we will allocate memory for the PCD-Header and read it.
    */
    if ((header = malloc(3*0x800)) == NULL)
    {
       fclose(pcd_file);
       *error_id = GRAV_ERROR_ALLOC;
       return(NULL);
    }      
    
    fread(header,3*0x800, 1,pcd_file);

    if (strncmp(header,"PCD_OPA",7) != 0   &&
        strncmp(header+0x800, "PCD", 3))
    {
       *error_id = GRAV_ERROR_HEADER;
       fclose(pcd_file);
       free(header);
       return(NULL);
    }
    
   /*
    * next, we will set the filepointer to the beginning of
    * the 768x512 image....
    */
   fseek(pcd_file, 196608, SEEK_SET);

   /*
    * calculate the bitmap-size. the size must be a multiple of 4.
    * this exactly match the format used by OS/2 and WINDOWS
    * bitmaps...
    */

   pos              = 0;
   gbm->h           = 512;
   gbm->w           = 768;
   gbm->bpp         = 24;
   gbmft->long_name = ft;
   bitmap_size      = gbm->h * (((gbm->w * gbm->bpp + 31) / 32) * 4);
   rotate           = header[0x0e02] & 0x03;

   /*
    * We don't need the header any longer....
    */
    free(header);

   if (flag == READ_HEADER_ONLY)
   {
      *error_id     = GRAV_NO_ERROR;
      fclose(pcd_file);
      return(NULL);
   }

   /*
    * allocate memory for the bitmap...
    */
    
   if ((bitmap = malloc((int) bitmap_size)) == NULL)
   {
      fclose(pcd_file);
      *error_id = GRAV_ERROR_ALLOC;
      return(NULL);
   }  
   
   /*
    * initialize the bitmap
    */
    
   memset(bitmap, 0, bitmap_size);

   /*
    * and last but not least, read the graphic into memory (stored
    * as a bitmap)
    */
    
   bitmap_width = grav_get_bitmap_width(gbm);
   work_bitmap  = bitmap + bitmap_size;

   for(line=0; line<gbm->h; line+=2)
   {
      fread(gray1,gbm->w,  1,pcd_file);
      fread(gray2,gbm->w,  1,pcd_file);
      fread(blue, gbm->w/2,1,pcd_file);
      fread(red, gbm->w/2, 1,pcd_file);
      
      if (feof(pcd_file))
      {
         fclose(pcd_file);
         *error_id = GRAV_ERROR_READ; 
         return(NULL);
      }

      work_bitmap = work_bitmap - bitmap_width;
      pos         = 0;
               
      for(column=0; column<gbm->w; column++)
      {
         /*
          * calculate colors....
          */
         gray = gray_LUT[gray1[column]];
         b    = gray + blue_LUT[blue[column >> 1]];
         r    = gray + red_LUT [red [column >> 1]];
         g    = (10 * gray - b - 3 * r) / 6;
         
         /*
          * store them into the bitmap...
          */
         work_bitmap[pos++] = (b & ~0xff) ? ((b < 0) ? 0 : 255 ) : b;
         work_bitmap[pos++] = (g & ~0xff) ? ((g < 0) ? 0 : 255 ) : g;
         work_bitmap[pos++] = (r & ~0xff) ? ((r < 0) ? 0 : 255 ) : r;
      }
      
      pos         = 0;
      work_bitmap = work_bitmap - bitmap_width;
      
      for(column=0; column<gbm->w; column++)
      {      
         /*
          * calculate colors....
          */
         gray = gray_LUT[gray2[column]];
         b    = gray + blue_LUT[blue[column >> 1]];
         r    = gray + red_LUT [red [column >> 1]];
         g    = (10 * gray - b - 3 * r) / 6;
         
         /*
          * store them into the bitmap...
          */
         work_bitmap[pos++] = (b & ~0xff) ? ((b < 0) ? 0 : 255 ) : b;
         work_bitmap[pos++] = (g & ~0xff) ? ((g < 0) ? 0 : 255 ) : g;
         work_bitmap[pos++] = (r & ~0xff) ? ((r < 0) ? 0 : 255 ) : r;
      }
   }

   /*
    * we don't need gbm_file any longer, so close it
    */
    
   fclose(pcd_file);

   *error_id = GRAV_NO_ERROR;
   return(bitmap);
} 



