// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _MainWindow_H_
#define _MainWindow_H_

#include <qapp.h>
#include <qwidget.h>
#include <qmenubar.h>
#include <qpopmenu.h>
#include <qlabel.h>
#include <qlcdnum.h>
#include <qpainter.h>
#include <qpushbt.h>
#include <qbitmap.h>
#include "xcd.h"
#include "CompactDisc.h"
#include "Cdrom.h"

//: A modified QLabel class.
//. This is  a modified QLabel class that passes all mousePressEvents and
//. keyPressEvents to its parent.  The position of the mouse event is adjusted
//. to be relative to the parents location.
class MyLabel : public QLabel {
protected:
    //: Pass all QMouseEvents to the parent widget.
    //. The x and y coordinates are adjusted so that they are offsets from the
    //. parent widgets upper left corner instead of the childs.
    virtual void mousePressEvent(QMouseEvent* e);

    //: Pass all QKeyEvents to the parent widget.
    virtual void keyPressEvent(QKeyEvent* e) { qApp->sendEvent(parentWidget(),e); }
public:
    //: Create a MyLabel.
    MyLabel(const char* text, QWidget* parent) : QLabel(text, parent) {}
};

//: A modified QLCDNumber class.
//. This is a modified QLCDNumber class that passes all mousePressEvents and
//. keyPressEvents to its parent.  The position of the mouse event is adjusted
//. to be relative to the parents location.
class MyLCDNumber : public QLCDNumber {
protected:
    //: Pass all QMouseEvents to the parent widget.
    //. The x and y coordinates are adjusted so that they are offsets from the
    //. parent widgets upper left corner instead of the childs.
    virtual void mousePressEvent(QMouseEvent* e);

    //: Pass all QKeyEvents to the parent widget.
    virtual void keyPressEvent(QKeyEvent* e) { qApp->sendEvent(parentWidget(),e); }
public:
    //: Create a MyLCDNumber.
    MyLCDNumber(int size, QWidget* parent);
};

//: The main interface to XCd.
//. This is the only window that is guaranteed to be onscreen at all times.  It
//. allows full manipulation of the CDDrive, and provides access to the other
//. windows.
class MainWindow : public QWidget {
    Q_OBJECT
	QBitmap             _icon;
    QPopupMenu*         _mainMenu;    // the main popup menu.
    QPopupMenu*         _trackMenu;   // list of tracks on the disc.
	MyLabel*            _stepLabel;
    MyLCDNumber*        _stepValue;   // step value display.
	MyLabel*            _timeLabel;
    MyLCDNumber*        _timeValue;   // time value display.
	MyLabel*            _trackLabel;
    MyLCDNumber*        _trackValue;  // track number display.
	MyLabel*            _volumeLabel;
    MyLCDNumber*        _volumeValue; // volume value display.
    MyLabel*            _trackName;   // track title display.
	QPushButton*        _shuffle;
	QPushButton*        _repeat;
    QString             _step;        // current step value.
    QString             _track;       // current track number.
    QString             _time;        // current time value.
    QString             _volume;      // current volume value.
    QString             _discTitle;   // disc title.
    QString             _discArtist;  // disc artist.
    QString             _trackTitle;  // current tracks title.
    int                 _trackNumber; // current track number.
    CompactDisc*        _cd;
	CdromStatus         _status;

    void redrawIcon();
private slots:
    // Process a selection from the Commands menu.
    void procCommandsMenu(int id);

    // Process a selection from the Time menu.
    void procTimeMenu(int id);

    // Process a selection from the Icon menu.
    void procIconMenu(int id);

    // Process a selection from the Windows menu.
    void procMainMenu(int id);

    // Process a selection from the Track menu.
    void procTrackMenu(int id);

    void procShuffle( bool b );
    void procRepeat( bool b );
protected:
    //: Use the mouse to popup the main menu and the track list.
    virtual void mousePressEvent(QMouseEvent* e);
	virtual void timerEvent( QTimerEvent* e );
public:
    //: Create a MainWindow just like any other QWidget.
    MainWindow(QWidget* parent=0, const char* name=0);
public slots:
    //: Update the Step display.
    void changeStep(int step);

    //: Update the Track display.
    void changeTrack(int track, const char* name);

    //: Update the Time display.
    void changeTime(int minute, int second);

    //: Update the Volume display.
    void changeVolume(int volume);

    //: Configure the MainWindow for no disc in the CDROM drive.
    void noDisc();

    //: Configure the MainWindow for the new disc.
    void newDisc(CompactDisc* cd);

    //: Change the icon mode pref.
    void changeIconMode();

    //: Change the discs title.
    void changeDiscTitle(const char* title);

    //: Change the discs artist.
    void changeDiscArtist(const char* artist);

    //: Change the tracks title.
    void changeTrackTitle(int track, const char* title);

    //: Toggle the time display mode.
    void changeTimeMode();

    void nextIconMode();
    void nextTimeMode();
    void changeShuffle();
    void changeRepeat();
    void changeProgram( int length, int* tracks );
	void changeStatus( CdromStatus status );
signals:
    //: Emitted when the user presses play.
    void play();

    //: Emitted when the user presses pause.
    void pause();

    //: Emitted when the user presses stop.
    void stop();

    //: Emitted when the user presses eject.
    void eject();

    //: Emitted when the user presses nextTrack.
    void nextTrack();

    //: Emitted when the user presses previousTrack.
    void previousTrack();

    //: Emitted when the user presses incrVolume.
    void incrVolume();

    //: Emitted when the user presses decrVolume.
    void decrVolume();

    //: Emitted when the user selects a track.
    void playTrack(int track);
};

#endif
