// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _Program_H_
#define _Program_H_

#include <qlist.h>
#include <qobject.h>
#include "CompactDiscID.h"

//: Maintain an ordered list of tracks to be played.
//. The Program class contains an ordered list of tracks on an audio disc.  Each
//. track in the program is considered to be a step.  The program step can be
//. reset to the beginning of the program, incremented, and decremented.  Each
//. operation that changes the program step returns the new current track
//. number.  This is the abstract base class for all types of Programs.
class Program : public QObject {
    Q_OBJECT
	
    // Dont let the compiler create these...
    Program(const Program& prog);
    Program& operator=(const Program& prog);
protected:
    QList<int>   _orderedTracks;
    QList<int>   _tracks;
    unsigned int _step;
public:
    Program( const CompactDiscID& id );
    Program( const int length, const int* tracks );

    ~Program() { } // Needed to satisfy g++ ;).

    //: Reset the program step, and return the new current track.
    int reset() { _step = 0; emit updateStep(_step+1); return *_tracks.at(_step); }

    //: Increment the program step, and return the new current track.
    int next() { _step++; if (_step >= _tracks.count()) _step = 0; emit updateStep(_step+1); return *_tracks.at(_step); }

    //: Decrement the program step, and return the new current track.
    int previous() { if (_step == 0) _step = _tracks.count() - 1; else _step--; emit updateStep(_step+1); return *_tracks.at(_step); }

    //: Check to see if we are at the end of the Program.
    bool isEnd() const { return _tracks.count() == (_step+1); }

    //: Check to see if we are at the start of the Program.
    bool isStart() const { return _tracks.count() == (_step+1); }

    //: Re-shuffle the tracks.
    void shuffle();

    //: Unshuffle the tracks.
    void unshuffle();

    //: Return the list of programmed tracks in order.
    QList<int>& ordered() { return _orderedTracks; }

signals:
    //: Emitted when the program step changes.
    void updateStep(int step);
};

#endif
