// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _CDDrive_H_
#define _CDDrive_H_

#include <qobject.h>
#include "xcd.h"
#include "CompactDisc.h"
#include "CdromBase.h"

//: Provide an interface to the Cdrom class.
//. This class provides an interface to the Cdrom class for manipulating audio
//. comapct discs.  All the normal operations (play/pause/stop/eject/etc...)
//. are supported.  However cue and review are not currently supported because
//. of problems with some drives.
class CDDrive : public QObject {
    Q_OBJECT                             // need this to send/receive signals.
    int                 _timerID;        // timer for polling the drive status.
    int                 _track;          // currently playing track.
	int                 _relTime;
    int                 _desiredTrack;   // the track that the user _wants_ to be playing.
    CdromStatus         _status;         // playing/paused/no status/error.
    int                 _volume;         // output volume of cdrom to sound card.
    CompactDisc*        _cd;             // info for the current disc in the drive.
    enum DiscStatus {
        NoDisc,
        NewDisc,
        OldDisc
    } _discStatus;                       // the presence of a disc in the drive.

    //: Calculate the current play time taking into account the time mode.
    void doTime( int absTime, int relTime );

    //: Process a change in the currently playing track.
    void doUpdateTrack();

    //: Convert from frames to minutes and seconds, and emit the time.
    void doUpdateTime( int frame );

    //: Process the detection of a new disc.
    bool doNewDisc();

    //: Process the detection of the abscence of a disc.
    void doNoDisc();

    //: Process a new status reading from the CDDrive.
    void changeStatus( CdromStatus status, int absTime, int relTime );

protected:
    //: Poll the cdrom for its current status, play time, and track.
    void timerEvent(QTimerEvent* e);

public:
    //. Initialize the CDDrive object, and start the polling timer.  There
	//. should be only one instance of this object for the entire application.
    CDDrive();

public slots:
    //: Go to the previous track.
    void previousTrack();

    //: Stop any active play operation.
    void stop();

    //: Begin playing the current desired track.
    void play();

    //: Pause the current play operation.
    void pause();

    //: Go to the next track.
    void nextTrack();

    //: Eject the drive tray.
    void eject();

    //: Increase the volume by one unit.
    void incrVolume();

    //: Decrease the volume by one unit.
    void decrVolume();

    //: Begin playing a specific track.
    void playTrack(int track);

    //: Process a change in the program step.
    void changeStep(int step);

	//: Toggle random play mode on/off.
    void changeShuffle();

    //: Change the poll time pref.
    void changePollTime();

	//: Change the tracks in the UserProgram.
	void changeProgram(int length, int* tracks);

	//: Re-read the disc info from the new database file.
	void changeDbasePath();
signals:
    //: Emitted when the program step changes.
    void updateStep(int step);

    //: Emitted when the track number changes.
    void updateTrack(int track, const char* name);

    //: Emitted when the time changes.
    void updateTime(int minute, int second);

    //: Emitted when the volume changes.
	//. The range of the volume is from 0 to 63.
    void updateVolume(int volume);

	//: Emitted when the audio status of the CDROM changes.
	//. Possible values for the status are: Playing, Paused, Stopped,
	//. Completed, and Error.
	void updateStatus( CdromStatus status );

    //: Emitted when the current play operation completes.
    void done();

    //: Emitted when the absence of a disc is detected.
    void noDisc();

    //: Emitted when a new disc is detected.
	//. The CompactDisc object of the new disc is given.
    void newDisc(CompactDisc* cd);
};

#endif
