// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _CompactDisc_H_
#define _CompactDisc_H_

#include <qstring.h>
#include <qobject.h>
#include "Program.h"
#include "CompactDiscID.h"

//: Store information about an audio compact disc.
//. Maintain the title, artist, and track names for an audio compact disc.
//. Each CompactDisc has an associated ComapctDiscID that uniquely identifies
//. it within the Database.
class CompactDisc : public QObject {
    Q_OBJECT
    CompactDiscID       _id;     // unique id for this disc.
    QString             _title;  // disc title.
    QString             _artist; // artist name.
    QString*            _tracks; // track titles.
	Program*            _program;

    // Don't let the compiler create these...
    CompactDisc(CompactDisc& cd);
    CompactDisc& operator=(CompactDisc& cd);
public:
    //: Create a CompactDisc.
    CompactDisc( const CompactDiscID& id, QString& title, QString& artist, QString* tracks, int progLength, int* progTracks );

    //: Delete a CompactDisc.
    ~CompactDisc();

    Program* program() { return _program; }

    //: Return a reference to our CompactDiscID.
    const CompactDiscID& id() const { return _id; }
    
    //: Return the disc's title.
    const char* title() const { return (const char*)_title; }

    //: Return the disc's artist.
    const char* artist() const { return (const char*)_artist; }

    //: Return the name of track number <VAR>track</VAR> on the disc.
    const char* track(int track) const { return (const char*)_tracks[track]; }

    void changeProgram( int length, int* tracks );

    //: Is our CompactDiscID equal to <VAR>id</VAR>?
    int isEqual( const CompactDiscID &cid ) const { return _id == cid; }
    int isEqual( const CompactDisc &cd ) const { return _id == cd.id(); }
public slots:
    //: Set the disc's title to <VAR>title</VAR>.
    void setTitle(const char* title);

    //: Set the disc's artist to <VAR>artist</VAR>.
    void setArtist(const char* artist);

    //: Set the name of track number <VAR>track</VAR> to <VAR>name</VAR>.
    void setTrack(int track, const char* name);

    //: Update the CompactDisc's entry in the Database.
    void saveDiscInfo();
signals:
    //: Emitted when the disc's title changes.
    void updateTitle(const char* title);

    //: Emitted when the disc's artist changes.
    void updateArtist(const char* artist);

    //: Emitted when the name of a track on the disc changes.
    void updateTrack(int track, const char* name);
};

#endif
