// XCd - An X11 CD Player
// Copyright (C) 1996  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _CompactDiscID_H_
#define _CompactDiscID_H_

#include "xcd.h"

//: Store some info that uniquely identifies a CompactDisc.
//. Store some information about an audio compact disc that can be derived
//. directly from the physical disc itself.  The following attributes are used
//. to identify the disc: number of tracks on the disc, absolute starting frame
//. of each track (including leadout track).
class CompactDiscID {
    int         _numTracks;  // number of tracks on the disc.
    int         _firstTrack; // physical number of the first track.
    int*        _trackStart; // starting frame of each track.

    // Don't let the compiler generate these...
    CompactDiscID& operator=(CompactDiscID& id);
public:
    //: Create a CompactDiscID from the disc's data.
    CompactDiscID(int num, int first, int start[]);

    //: Create a CompactDiscID from another CompactDiscID.
    CompactDiscID(const CompactDiscID& id);

    //: Destroy a CompactDiscID.
    ~CompactDiscID();

    //: Return the number of tracks on the disc.
    int numTracks() const { return _numTracks; }

    //: Return the first track number on the disc.
    int firstTrack() const { return _firstTrack; }

    //: Return the absolute starting frame of track number <VAR>track</VAR>.
    int trackStart(int track) const { return _trackStart[track]; }

    //: Return the length of track number <VAR>track</VAR> in frames.
    int trackLength(int track) const;

    //: Return the minutes part of track number <VAR>track</VAR>'s length.
    int trackMinute(int track) const { return trackLength(track)/XcdSecs/XcdFrames; }

    //: Return the seconds part of track number <VAR>track</VAR>'s length.
    int trackSecond(int track) const { return trackLength(track)/XcdFrames%XcdSecs; }

    //: Determine if two CompactDiscIDs are equal.
    bool operator==(const CompactDiscID& id) const;
};

#endif
