/* $Header: /wwg/motif/xltcdplay/RCS/cdrom.cpp,v 1.4 1997/02/02 04:34:57 wwg Rel $
 * Warren W. Gay VE3WWG		Sat Jan 18 00:54:21 1997
 *
 * CD-ROM Device Functions:
 *
 * 	X LessTif CD Play :
 * 
 * 	Copyright (C) 1997  Warren W. Gay VE3WWG
 * 
 * This  program is free software; you can redistribute it and/or modify it
 * under the  terms  of  the GNU General Public License as published by the
 * Free Software Foundation version 2 of the License.
 * 
 * This  program  is  distributed  in  the hope that it will be useful, but
 * WITHOUT   ANY   WARRANTY;   without   even  the   implied   warranty  of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details (see enclosed file COPYING).
 * 
 * You  should have received a copy of the GNU General Public License along
 * with this  program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * Send correspondance to:
 * 
 * 	Warren W. Gay VE3WWG
 * 	5536 Montevideo Road #17
 *	Mississauga, Ontario L5N 2P4
 * 
 * Email:
 * 	wwg@ica.net			(current ISP of the month :-) )
 * 	bx249@freenet.toronto.on.ca	(backup)
 * 
 * $Log: cdrom.cpp,v $
 * Revision 1.4  1997/02/02 04:34:57  wwg
 * Changed the way info[].lengthf gets computed for data tracks.
 * Originally I used a strange computation that was used by
 * workbone, but this does not add up to total track time--
 * so for now I'm just going to assume that it is correct here
 * and that it was an error in the workbone project.
 *
 * Revision 1.3  1997/01/30 04:22:26  wwg
 * I think we've done all the damage we need now.. hopefully
 * now finally releasing a Beta release.
 *
 * Revision 1.2  1997/01/26 05:04:50  wwg
 * Many fixes and enhancements: now includes all features
 * but the progress indicators.
 *
 * Revision 1.1  1997/01/20 04:38:22  wwg
 * Initial revision
 *
 */
#include <unistd.h>
#include "cdrom.h"

static char *rcsid[] = { "@(#)cdrom.cpp $Revision: 1.4 $", _cdrom_h_ };

extern "C" int ioctl(int fd,int cmd,void *argptr);

static char ErrorText[300];					// Raised Error Text

// Constructor:

CDROM::CDROM() {
	fd = -1;
	devicePath = NULL;
	cEntries = 0;
}

// Destructor:

CDROM::~CDROM() {
	if ( fd != -1 )
		close();
}

// Returns TRUE if device is open:

int
CDROM::isOpen() {
	return fd >= 0 ? 1 : 0;
}

// Open CDROM device on pathname: The side effect of this attempt, even if
// it fails is that the devicePath member remains pointing to a device name
// pathname. This is utilized when CDROM::eject is called, if the device is
// not already open (yes, its a hack).

void
CDROM::open(const char *pathname)
throw(char *) {
	unsigned x;
	enum { NoReset, AfterReset, AfterStart } e = NoReset;

	if ( fd >= 0 )
		throw "Internal Error: CDROM already open!";

	/*
	 * Open the device: This normally succeeds immediately
	 */
	if ( (fd = ::open(devicePath = (char *) pathname,0)) == -1 ) {
		sprintf(ErrorText,"%s: opening CDROM %s",sys_errlist[errno],devicePath);
		throw ErrorText;
	}

	/*
	 * Read the Table of Contents Header: on my obstinate CD-ROM drive, this gives
	 * me a lot of trouble. I think my NEC Multispin 6V is one of those nasty
	 * drives that needs to be diddled with at initialization to work properly.
	 * With enough fooling around, it eventually does succeed (I've found that a software
	 * eject followed by a manual inject, and a pause often does the job).
	 *
	 * Anyway, its because of these stubborn drives that you see this odd looking
	 * initialization sequence. If you have a normal drive, you may want to #if 0
	 * some of this crap out.
	 */
retry:	if ( ioctl(fd,CDROMREADTOCHDR,&toc_hdr) == -1 ) {

		if ( e == NoReset ) {
			// Try a reset, after the first failure
			ioctl(fd,CDROMRESET,0);
			e = AfterReset;
			goto retry;

		} else if ( e == AfterReset ) {
			ioctl(fd,CDROMSTART,0);		// Ignore errors here
			e = AfterStart;
			goto retry;

		} else	{
			// We give up!
			sprintf(ErrorText,"%s: Reading TOC header from %s",
				sys_errlist[errno],devicePath);
			goto errxit;
		}
	}

	cEntries = toc_hdr.cdth_trk1;

	/*
	 * Read in the tracks of info:
	 */
	for ( x=0; x <= cEntries; ++x ) {
		info[x].cdtoc.cdte_track = x == cEntries ? CDROM_LEADOUT : x + 1;
		info[x].cdtoc.cdte_format = CDROM_MSF;

		if ( ioctl(fd,CDROMREADTOCENTRY,&info[x].cdtoc) == -1 ) {
			sprintf(ErrorText,"%s: reading TOC entry # %u %sfor %s",
				sys_errlist[errno],
				(unsigned) x,
				x != cEntries ? "" : "(leadout) ",
				devicePath);
			goto errxit;
		}

		info[x].bSuppress = info[x].bDataTrack = info[x].cdtoc.cdte_ctrl & CDROM_DATA_TRACK ? 1 : 0;
		info[x].startf = info[x].cdtoc.cdte_addr.msf.minute * 60U * 75U
			+ info[x].cdtoc.cdte_addr.msf.second * 75U
			+ info[x].cdtoc.cdte_addr.msf.frame;

		if ( x > 0 )
			info[x-1].lengthf = info[x].startf - info[x-1].startf;

		if ( x == cEntries )
			info[x].lengthf = 0;
	}

	cdend.minute = (info[cEntries].startf - 1) / (60U * 75U);
	cdend.second = ((info[cEntries].startf - 1) - cdend.minute * (60U * 75U)) / 75;
	cdend.frame  = (info[cEntries].startf - 1) % 75;

	_getState();				// Read in subchannel info
	return;

	/*
	 * The I give up exit.
	 */
errxit:	::close(fd);
	fd = -1;
	throw ErrorText;
}

// This method updates the subchannel status structure,
// or throws an error if the drive just went offline:

unsigned
CDROM::_getState()
throw(char *) {
	/*
	 * Read subchannel info:
	 */
	subch.cdsc_format = CDROM_MSF;		// Use Minute-Second-Frame format

	if ( ioctl(fd,CDROMSUBCHNL,&subch) == -1 ) {
		sprintf(ErrorText,"%s: reading Subchannel from %s",
			sys_errlist[errno],
			devicePath);
		throw ErrorText;
	}
	return subch.cdsc_audiostatus;
}

// This method requests the CD-ROM drive to STOP:

unsigned
CDROM::_stop()
throw(char *) {

	switch ( _getState() ) {		// Check current state
	case CDROM_AUDIO_COMPLETED :	
	case CDROM_AUDIO_INVALID :
	case CDROM_AUDIO_ERROR :
	case CDROM_AUDIO_NO_STATUS :
		break;				// No action necessary

	case CDROM_AUDIO_PLAY :
	case CDROM_AUDIO_PAUSED :
	default :
		stop();				// Stop the drive
	}

	return _getState();			// Return state info
}	

// Close the CD-ROM drive file descriptor, and this object:

void
CDROM::close()
throw(char *) {

	if ( fd < 0 )
		throw "Internal Error: CDROM not open!";
	::close(fd);
	fd = -1;
	cEntries = 0;
}

// Eject the CD from the drive:  This method will use the open file
// descriptor, if a prior open was successful. Otherwise, it will do
// a short open (no TOC read) using the last pathname tried, and cause
// a CD eject.

void
CDROM::eject()
throw(char *) {

	if ( fd < 0 ) {			// Was this object successfully opened ?
		int tmp_fd;		// Nope: Try a short open skipping TOC read

		if ( devicePath != NULL && (tmp_fd = ::open(devicePath,0)) >= 0 ) {
			ioctl(fd,CDROMSTOP,0);		// Ignore errors
			ioctl(tmp_fd,CDROMEJECT,0);	// Best effort basis here..
			::close(tmp_fd);		// Done with this fd
			return;				// Exit with tail between legs..
		}

		// Well, we tried:
		throw "Internal Error: Device is not open!";
	}
		
	_stop();			// Stop the drive if it requires it

	/*
	 * Eject the CD now
	 */
	if ( ioctl(fd,CDROMEJECT,0) == -1 ) {
		sprintf(ErrorText,"%s: ioctl(%d,EJECT) on %s",sys_errlist[errno],fd,devicePath);
		throw ErrorText;
	}
}

// Play the CD: This method requires that this object is open.
// The state is checked, and if paused, play is resumed. Otherwise,
// STOP is issued, and then play is issued for the selected track:

void
CDROM::play(unsigned track)
throw(char *) {
	struct cdrom_ti ti;

	switch ( _getState() ) {
	case CDROM_AUDIO_PAUSED :			// Currently PAUSEd ?
		if ( ioctl(fd,CDROMRESUME,0) == -1 ) {	// Yes, RESUME..
			sprintf(ErrorText,"%s: resuming play on %s",sys_errlist[errno],devicePath);
			throw ErrorText;		// Whoops!
		}
		break;

	case CDROM_AUDIO_PLAY :				// Currently PLAYing?
		stop();					// Yes, better STOP first..
							// ..and then..
	default :
		ioctl(fd,CDROMSTART,0);			// Start the motor drive (ignore errs)
		ti.cdti_ind0 = 1;
		ti.cdti_ind1 = 1;
		ti.cdti_trk0 = track;			// Start here
		ti.cdti_trk1 = toc_hdr.cdth_trk1;	// End at last track
	
		if ( ioctl(fd,CDROMPLAYTRKIND,&ti) == -1 ) {
			if ( errno != EINVAL )
				sprintf(ErrorText,"%s: starting play of track # %u on %s",
					sys_errlist[errno],(unsigned) track,devicePath);
			else	sprintf(ErrorText,
					"EINVAL CDROMPLAYTRKIND: ind0=%u, ind1=%u, trk0=%u, trk1=%u;",
					(unsigned)ti.cdti_ind0,
					(unsigned)ti.cdti_ind1,
					(unsigned)ti.cdti_trk0,
					(unsigned)ti.cdti_trk1);
			throw ErrorText;		// Report error
		}
	}
}

// This method moves the current playing selection forward/back +/- n seconds:

void
CDROM::play(int seconds) throw(char *) {
	cdrom_msf msf;
	unsigned curmsf;

	switch ( _getState() ) {
	case CDROM_AUDIO_PAUSED :			// Currently PAUSEd ?
		if ( ioctl(fd,CDROMRESUME,0) == -1 ) {	// Yes, RESUME..
			sprintf(ErrorText,"%s: resuming play on %s",sys_errlist[errno],devicePath);
			throw ErrorText;		// Whoops!
		}

	case CDROM_AUDIO_PLAY :				// Currently PLAYing?
		// Start with current time..
		curmsf = ((subch.cdsc_absaddr.msf.minute * 60U) + subch.cdsc_absaddr.msf.second) * 75U
			+ subch.cdsc_absaddr.msf.frame;

		if ( seconds >= 0 )
			curmsf += (unsigned) seconds * 75U;
		else	curmsf -= ((unsigned) -seconds) * 75U;

		msf.cdmsf_min0 = curmsf / (60U * 75U);
		msf.cdmsf_sec0 = (curmsf - ((unsigned)msf.cdmsf_min0 * (60U * 75U))) / 75U;
		msf.cdmsf_frame0 = curmsf % 75U;

		// Now plug in end time:
		msf.cdmsf_min1 = cdend.minute;
		msf.cdmsf_sec1 = cdend.second;
		msf.cdmsf_frame1 = cdend.frame;

		// Now tell the device what we want:

		if ( ioctl(fd,CDROMPLAYMSF,&msf) == -1 ) {
			if ( errno != EINVAL )
				sprintf(ErrorText,"%s: %s play of track # %u on %s",
					sys_errlist[errno],
					seconds >= 0 ? "FF" : "REW",
					(unsigned)subch.cdsc_trk,
					devicePath);
			else	sprintf(ErrorText,
					"EINVAL CDROMPLAYMSF: min0=%u,sec0=%u,frm0=%u;min1=%u,sec1=%u,frm1=%u;",
					msf.cdmsf_min0,
					msf.cdmsf_sec0,
					msf.cdmsf_frame0,
					msf.cdmsf_min1,
					msf.cdmsf_sec1,
					msf.cdmsf_frame1);
			throw ErrorText;		// Report error
		}
	}
}

// PAUSE the CDROM drive: This object must be open first.

void
CDROM::pause()
throw(char *) {

	if ( fd < 0 )
		throw "PAUSE function requires open CDROM device";

	switch ( _getState() ) {
	case CDROM_AUDIO_INVALID :			// status may be unsupported
	case CDROM_AUDIO_PLAY :
		if ( ioctl(fd,CDROMPAUSE,0) == -1 ) {
			sprintf(ErrorText,"%s: pausing CDROM %s",sys_errlist[errno],devicePath);
			throw ErrorText;
		}
	}
}

// STOP the CDROM drive: This object must be open prior to call

void
CDROM::stop()
throw(char *) {

	if ( fd < 0 )
		throw "STOP function requires open CDROM device";

	switch ( _getState() ) {
	case CDROM_AUDIO_COMPLETED :
		break;
	default :
		if ( ioctl(fd,CDROMSTOP,0) == -1 ) {
			sprintf(ErrorText,"%s: stopping CDROM %s",
				sys_errlist[errno],
				devicePath);
			throw ErrorText;
		}
	}
}

// Get CD Track and Minute and Second (position): This will throw an
// error if the CD has been manually removed.

char *
CDROM::getPos()
throw(char *) {

	switch ( _getState() ) {			// This call can throw

	case CDROM_AUDIO_PLAY :
		sprintf(pos,"%02u %02u:%02u",
			(unsigned)subch.cdsc_trk,
			(unsigned)subch.cdsc_reladdr.msf.minute,
			(unsigned)subch.cdsc_reladdr.msf.second);
		break;

	case CDROM_AUDIO_PAUSED :
		sprintf(pos,"%02u PAUSE",(unsigned)subch.cdsc_trk);
		break;
	
	case CDROM_AUDIO_COMPLETED :
		sprintf(pos,"%02u -END-",(unsigned)subch.cdsc_trk);
		break;		

	case CDROM_AUDIO_ERROR :
		sprintf(pos,"%02u ERROR",(unsigned)subch.cdsc_trk);
		break;

	case CDROM_AUDIO_INVALID :
	case CDROM_AUDIO_NO_STATUS :
	default :
		sprintf(pos,"%02u --:--",(unsigned)subch.cdsc_trk);
	}

	return pos;					// Return edited display
}

// Return TRUE if the CDROM device is playing:

int
CDROM::isPlaying() throw(char *) {
	if ( fd > 0 && _getState() == CDROM_AUDIO_PLAY )
		return 1;				// True
	return 0;					// Nope
}

// Get current Track Number:

unsigned
CDROM::getTrack() throw(char *) {
	_getState();
	return (unsigned) subch.cdsc_trk;
}

// Get the number of Tracks on this Audio CD: (from TOC)

unsigned
CDROM::getTracks() throw(char *) {
	_getState();
	return (unsigned) toc_hdr.cdth_trk1;
}

// Private method: update vol settings member:

void
CDROM::_getVol() throw(char *) {
	_getState();
	if ( ioctl(fd,CDROMVOLREAD,&vol) == -1 ) {
		sprintf(ErrorText,"%s: reading CD volume levels from %s",
			sys_errlist[errno],devicePath);
		throw ErrorText;
	}
}

// Private method: set vol settings based upon vol member:

void
CDROM::_setVol() throw(char *) {
	_getState();
	if ( ioctl(fd,CDROMVOLCTRL,&vol) == -1 ) {
		sprintf(ErrorText,"%s: writing CD volume levels to %s",
			sys_errlist[errno],devicePath);
		throw ErrorText;
	}
}

// Get left volume level (0 - 255):

unsigned
CDROM::getLeftVol() throw(char *) {
	_getVol();
	return (unsigned) vol.RIGHTCH;
}

// Get right volume level (0 - 255):

unsigned
CDROM::getRightVol() throw(char *) {
	_getVol();
	return (unsigned) vol.LEFTCH;
}

// Set left volume level (0 - 255):

void
CDROM::setLeftVol(unsigned level) throw(char *) {
	_getVol();					// Get current settings
	vol.RIGHTCH = (unsigned char) (level & 0xFF);	// Update channel
	_setVol();					// Set new settings
}

// Set left volume level (0 - 255):

void
CDROM::setRightVol(unsigned level) throw(char *) {
	_getVol();
	vol.LEFTCH = (unsigned char) (level & 0xFF);
	_setVol();
}

// Perform a hard CDROM drive reset:

void
CDROM::reset() throw(char *) {

	if ( fd >= 0 )
		reset(NULL);			// This object is open, use its fd & devicePath

	else if ( devicePath != NULL )
		reset(devicePath);		// This object is not open, but has a path

	else	reset("/dev/cdrom");		// Not open & no path, assume one..
}

// Perform a CDROM reset when this object is not open: do temp open and give RESET cmd:

void
CDROM::reset(const char *pathname) throw(char *) {
	int tmp_fd = -1;

	if ( pathname == NULL ) {		// Pathname given?
		pathname = devicePath;		// No.. assume this object is open
		tmp_fd = fd;			// .. and use the open fd too..

	} else if ( (tmp_fd = ::open(pathname,0)) == -1 ) { 	// Else do temporary open..
		sprintf(ErrorText,"%s: device open of CDROM %s",sys_errlist[errno],pathname);
		throw ErrorText;
	}

	if ( ioctl(tmp_fd,CDROMRESET,0) == -1 ) {		// Now do RESET
		sprintf(ErrorText,"%s: reseting CDROM device %s",sys_errlist[errno],pathname);
		if ( tmp_fd != fd )
			::close(tmp_fd);			// Close if we opened it..
		throw ErrorText;
	}

	if ( tmp_fd != fd )
		::close(tmp_fd);				// Close if we opened it here..
}

int
CDROM::isDataTrack(unsigned track) throw(char *) {

	if ( fd < 0 )
		throw "Internal Error: CDROM device is not open.";
	if ( track > cEntries || track < 1 )
		throw "Track # out of range.";
	return info[track-1].bDataTrack ? 1 : 0;
}

void
CDROM::getProgress(unsigned *pTrk,unsigned *pCD)
throw(char *) {
	unsigned track_frames;					// Track frames length
	unsigned total_frames;					// CD total in frames
	unsigned x;

	_getState();						// Update info	

	// First, compute Track position:

	if ( subch.cdsc_trk < 1 ) {
		*pTrk = *pCD = 0;				// No progress
		return;
	}
	
	track_frames = subch.cdsc_reladdr.msf.minute * 60U * 75U
		+ subch.cdsc_reladdr.msf.second * 75U
		+ subch.cdsc_reladdr.msf.frame;

	if ( info[subch.cdsc_trk-1].lengthf > 0 )
		*pTrk = (track_frames * 100U) / info[subch.cdsc_trk-1].lengthf;
	else	*pTrk = 0;				// Safety valve

	// Now compute CD position:

	if ( subch.cdsc_trk > 1 )
		// Add in times for prior tracks to current time in frames:
		for ( x=0; x<(unsigned)subch.cdsc_trk-1; ++x )
			track_frames += info[x].lengthf;

	total_frames = cdend.minute * 60U * 75U + cdend.second * 75U + cdend.frame;

	if ( total_frames > 0 )
		*pCD = (track_frames * 100U) / total_frames;
	else	*pCD = 0;				// Safety valve
}

void
CDROM::getTrackMSF(unsigned track,unsigned *pMin,unsigned *pSec,unsigned *pFrame)
throw(char *) {

	_getState();

	if ( track < 1 ) {
		*pMin = *pSec = *pFrame = 0U;
		return;
	}

	if ( track - 1 > cEntries )
		throw "Internal error: Track number out of range in CDROM::getTrackMSF()";
	*pMin = info[track-1].lengthf / (60U * 75U);
	*pSec = ( (info[track-1].lengthf - *pMin * 60U * 75U) ) / 75U;
	*pFrame = info[track-1].lengthf % 75U;
}

void
CDROM::getCDMSF(unsigned *pMin,unsigned *pSec,unsigned *pFrame)
throw(char *) {

	if ( fd < 0 )
		throw "Internal Error: cdrom device is not open in CDROM::getCDMSF()";

	*pMin = cdend.minute;
	*pSec = cdend.second;
	*pFrame = cdend.frame;
}

void
CDROM::setTrack(unsigned trackNo) throw(char *) {
	struct cdrom_ti ti;

	if ( fd < 0 )
		throw "Internal Error: cdrom device is not open in CDROM::setTrack()";
	
	ti.cdti_ind0 = 1;
	ti.cdti_ind1 = 1;
	ti.cdti_trk0 = trackNo;			// Start here
	ti.cdti_trk1 = trackNo;			// For simplicity here

	ioctl(fd,CDROMPLAYTRKIND,&ti);		// We just want to register the track # (ignore errors)
	stop();					// This will report errors
}

// Return the number of tracks that are data tracks:

unsigned
CDROM::getDataTracks() throw(char *) {
	unsigned trk = 1;
	unsigned data_tracks_count = 0;

	for ( ; trk <= getTracks(); ++trk )
		if ( isDataTrack(trk) )
			++data_tracks_count;
	return data_tracks_count;
}

// $Source: /wwg/motif/xltcdplay/RCS/cdrom.cpp,v $
