/*
 * Simple little CD player.
 *
 * Copyright is owned by:	William E. Roadcap (roadcapw@cfw.com)
 *				Waynesboro, Va 22980 
 *				USA
 *
 * All rights are reserved under to GNU Public License Agreement.
 *
 * You may freely copy and distribute this software in source and/or 
 * binary form for non-commercial use.
 *
 */

#include <stdio.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

/*
 * Humdrum....
 */
#define TRUE 1
#define FALSE 0
#define NOT !
#define NORMAL  0
#define REPEAT_TRACK 1
#define REPEAT_DISK  2


typedef char flag;


/*
 * IOCTL structures (see /usr/include/linux/cdrom.h)
 */
struct cdrom_tochdr tochdr;  /* Table of contents header */
struct cdrom_tocentry song;  /* Table of contents entries */
struct cdrom_ti ti;          /* Track(s) to play / Track indexing */
struct cdrom_subchnl sc;     /* CD-ROM Drive status */
struct cdrom_volctrl vol;    /* Volume control */

/*
 * IOCTL structures (see /usr/include/asm/termios.h & termios man page)
 */
struct termios old_tc; /* to remember setting on entry */
struct termios new_tc;

int drive = 0;		/* CD drive file descriptor */

flag stopped = TRUE;	/* internal status flag */
flag paused  = FALSE;	/* internal status flag */

char timer_mode;		/* keeps track of the timer mode */
u_char track, minute, second;	/* timer thingies */
u_int  cd_seconds;		/* length of entire CD in seconds */
u_int  seconds_remaining;	/* what's left to play */

u_char first, last;		/* first and last tracks of CD */

char play_mode = REPEAT_DISK;	/* play mode for machine */
char *play_mode_str;		/* play mode for humans */

char status;		/* drive status for machines */
char *status_str;	/* drive status for humans */

u_char prev_track;	/* to keep track of track changes */
u_char prev_index = 0;	/* to keep track of timer changes */

int volume = 75;	/* initial volume level */

u_int tracks[50];

main () {
    char cmd;
    flag cont_prog = TRUE; /* Loop escape */

    int stall_counter = 0;/* For stopped CD Kludge */


    set_raw_term();	/* stty -opost -icanon min 0 time 1 */

    init_new_disk();	/* open the device and initialize everything */
    adj_volume(0);	/* set initial volume level */
    cycle_play_modes();	/* initialize normal play */
    display_commands();	/* give some hints          */


    while (cont_prog) {
        cmd = getchar();	/* Whatcha gonna do willy? */

        if (stopped == FALSE) { /* is the disk playing? */
            get_cd_status();	/* Update status info   */

            if (second != prev_index)  {	/* display status once per second */
                display_status();
                prev_index = second;
                stall_counter = 0; /* KLUDGE reset */
            }
            else if (paused == FALSE)
                stall_counter++; /* KLUDGE count */

            if (stall_counter >= 30)
                seconds_remaining = 0; /* KLUDGE force stop */
    
            /*
             * Handle play modes only if there's a track change and 
             * the keyboard is idle, or when the disk is finished.
             */
            if (prev_track != track || seconds_remaining == 0) {
                switch (play_mode) {
                case REPEAT_TRACK: track = prev_track;
                                   cmd = 'r';
                                   break;
                case REPEAT_DISK:  if (prev_track == last) {
                                       track = first;
                                       cmd = 'r';
                                   }
                                   break;
                /* As of kernel version 1.3.35, my Panasonic CD-ROM won't
                 * register CDROM_AUDIO_COMPLETE when the disk is done. 
                 *
                 * Instead, CDROM_AUDIO_PLAY is returned with spurious
                 * track numbers.
                 * Kludge: Stop the CD when there is no time remaining.
                 */
                case NORMAL:  cmd = (seconds_remaining == 0) ? -5 : cmd ;
                              break;
                }
            }

            switch (cmd) {
            case '>': play_next_track();
                      break;
            case '<': play_prev_track();
                      break;
            case 'r': play_at_track(track); 
                      break;
            case '.': pause(paused);
                      break;
            case -5 : stop();
                      break;
            case 's': stop();
                      track = 0; /* prevents locking into repeat mode */
                      break; 
            }
            prev_track = track;
        }

        switch(cmd) {
        case 'p': if (paused)
                      pause(paused);
                  else 
                      init_new_disk();
                  break;
        case 'm': cycle_play_modes();
                  break;
        case 't': toggle_timer_mode();
                  break;
        case '-': adj_volume(-25);	/* 1 tenth */
                  break;
        case '+': adj_volume(25);	/* ditto */
                  break;
        case 'e': eject();
                  break;
        case 'q': printf("\r\n");
                  cont_prog = FALSE;
                  break;
        }
    }

    tcsetattr(0, TCSANOW, &old_tc);
}



play_at_track (index)
    u_char index;
{

    if (drive < 1 || index < first || index > last) /* can't play this! */
        return(1);

    ti.cdti_trk0 = index;			/* set begin and */
    ti.cdti_trk1 = last;			/* end tracks    */
    ti.cdti_ind0 = ti.cdti_ind1 = 0;	/* set indexes   */

    ioctl(drive,CDROMSTOP);			/* must stop before  */
    ioctl(drive,CDROMPLAYTRKIND, &ti);	/* switching tracks  */

    display_status();
}

play_next_track () {
    if(stopped) return(0);
    play_at_track(track == last ? track = first : ++track);
}

play_prev_track () {
    if(stopped) return(0);
    play_at_track(track == first ? track = last : --track);
}

/*
 * What's going on with the CD.  Is it playing, paused, ejected?
 * What song are we playing, etc, etc...
 */
get_cd_status() {
    stopped = TRUE;	/* reset flag until proven otherwise */
    paused  = FALSE;	/* ditto */

    /*
     * An ejected tray or non-audio CD is not reported clearly by
     * ioctl calls. If drive file descriptor is not valid then assume
     * there is no disk. Reset counters & timers to zero and split.
     */
    if (drive < 1) {
      status_str = "No Disk";
      track = first = last = 0;
      minute = second = cd_seconds = seconds_remaining = 0;
      return(0);
    }

    sc.cdsc_format = CDROM_MSF;		/* in minute:second format */
    ioctl(drive,CDROMSUBCHNL,&sc);	/* get status */

    status  = sc.cdsc_audiostatus;  /* playing, paused, etc. */

    if (status == CDROM_AUDIO_NO_STATUS) /* must be stopped */
        track = minute = second = 0;
    else {
        track   = sc.cdsc_trk;
        minute  = sc.cdsc_absaddr.msf.minute;
        second  = sc.cdsc_absaddr.msf.second;
    }

    seconds_remaining = cd_seconds - second - minute * 60; /* seconds */

    /*
    * Convert status integer to a descriptive string.
    */
    switch (status) {
    case CDROM_AUDIO_PLAY:      status_str = "Playing";
				stopped = FALSE;
                                break;
    case CDROM_AUDIO_PAUSED:    status_str = "Paused";
				paused = TRUE;
                                stopped = FALSE;
                                break;
    case CDROM_AUDIO_ERROR:     status_str = "Error";
                                break;
    case CDROM_AUDIO_COMPLETED: status_str = "Completed";
                                break;
    case CDROM_AUDIO_INVALID:   status_str = "Invalid";
                                break;
    case CDROM_AUDIO_NO_STATUS: status_str = "Stopped";	/* must be stopped */
    }
}

/*
 * Called at start of program and when 'p'lay after an eject.
 * Opening the device causes the tray to retract on Panasonics and
 * maybe others.
 */
init_new_disk () {
    int i = 0;
    int junk = 0;

    drive = open("/dev/cdrom", O_RDONLY);

    read_toc();		/* Get first & last tracks and CD length */
    get_cd_status();	/* What's up with this CD? */

    /*
     * If not already playing then start the first song!
     */
    if (status == CDROM_AUDIO_PLAY) {
        /* supposedly playing, but let's check for a stalled player anyway */
        junk = second;
        for(i=1;i<=10;i++) getchar(); /* wait one 1 second */
        get_cd_status(); /* check the time */

        if(junk == second) { /* must be stalled! */
            play_at_track(first);
            prev_track = first;
        }
        else
            prev_track = track;
    }
    else {
        play_at_track(first);
        prev_track = first;	/* used to detect a track change */
    }
}

read_toc() {
    /*
     * Get table of contents header
     */
    ioctl(drive,CDROMREADTOCHDR, &tochdr);
    first = tochdr.cdth_trk0;	/* first track */
    last  = tochdr.cdth_trk1;	/* last track */

    /*
     * How long is this CD?
     * Get time stamp of end of last song.
     * Convert to totals seconds.
     */
    song.cdte_track  = CDROM_LEADOUT;		/* track to get */
    song.cdte_format = CDROM_MSF;		/* in minute:second:frame */
    ioctl(drive, CDROMREADTOCENTRY, &song);	/* retreive it */
    cd_seconds = song.cdte_addr.msf.second +
                 song.cdte_addr.msf.minute * 60;		/* convert */
}

/*
* Stop the drive motor, refresh CDROM_AUDIO_NO_STATUS,
* eject the disk and close the device.
*
* I close the device here so the CD tray will be retracted if
* the device is reopened.
*/
eject() {
    stop();		/* stop the drive */
    ioctl(drive,CDROMEJECT);
    get_cd_status();	/* player won't eject without this!?!? */
    close(drive);
    drive = 0;		/* indicate the tray is open */
    display_status();   /* refresh */
}

stop() {
    ioctl(drive,CDROMSTOP);
    display_status();
}

pause(current_state) 
    char current_state;
{
    ioctl(drive, current_state ? CDROMRESUME : CDROMPAUSE);
    display_status();
}

display_commands() {
    printf("\r\n\n");
    printf("PLAYCD v%s          ",VERSION);
    printf("Copyright 1995 -- William Roadcap (roadcapw@cfw.com)\n");
    printf("\r\n");
    printf("p  play/resume      >  next track          +  volume up\r\n");
    printf(".  pause/resume     <  previous track      -  volume down\r\n");
    printf("s  stop             r  restart track\r\n");
    printf("                                           m  play mode\r\n");
    printf("e  eject            q  quit                t  timer mode\r\n");
    printf("\r\n");
}

display_status() {
    u_char remain_minutes, remain_seconds;

    get_cd_status();

    if (timer_mode) {	/* Remaining */
        remain_minutes    = seconds_remaining / 60; /* convert to minutes */
        remain_seconds    = seconds_remaining % 60; /* and seconds. */
    
        printf("\r%-10s  Track %2u/%-2u   %02u:%02u remains   %s    Volume %2d", \
        status_str, track, last, remain_minutes, remain_seconds, play_mode_str, volume/25);
    }
    else {		/* Elapsed */
        printf("\r%-10s  Track %2u/%-2u   %02u:%02u elapsed   %s    Volume %2d", \
        status_str, track, last, minute, second, play_mode_str, volume/25);
    }
}

/*
 * Volume is scaled from 1-255 to 1-10.
 */
adj_volume(adjustment)
    int adjustment;
{
    if(drive <= 0) return;

    if (ioctl(drive, CDROMVOLREAD, &vol) == 0)
        volume = vol.channel0;

    volume += adjustment;	/* apply adjustment */
    volume = volume < 0 ? 0 : volume > 255 ? 255 : volume; /* limits */
    vol.channel0 = volume;		/* left */
    vol.channel1 = volume;		/* right */
    ioctl(drive, CDROMVOLCTRL, &vol);	/* set it */

    display_status();
}

/*
 * Remember current terminal state and set to input & output.
 */
set_raw_term() {
    tcgetattr(0, &old_tc);
    new_tc = old_tc;
    new_tc.c_lflag &= ~(ECHO | ICANON);		/* raw output */
    new_tc.c_cc[VMIN] = 0;			/* don't wait for keypresses */
    new_tc.c_cc[VTIME] = 1;
    tcsetattr(0, TCSANOW, &new_tc);
}

reset_term() {
    tcsetattr(0, TCSANOW, &old_tc);
}			

toggle_timer_mode () {
    timer_mode = timer_mode ? 0 : 1; /* toggle */
    display_status();
}

cycle_play_modes() {
    switch (play_mode) {
    case NORMAL:        play_mode = REPEAT_TRACK;
			play_mode_str = "Repeat Track";
                        break;
    case REPEAT_TRACK:  play_mode = REPEAT_DISK;
			play_mode_str = "Repeat Disk ";
                        break;
    case REPEAT_DISK:   play_mode = NORMAL;
			play_mode_str = "Normal Play ";
                        break;
    }

    display_status();
}
