/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.jdbc;

import java.io.Closeable;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.SQLNonTransientConnectionException;
import java.sql.SQLTimeoutException;
import java.sql.SQLTransientConnectionException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.iceberg.BaseMetastoreCatalog;
import org.apache.iceberg.CatalogUtil;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.SupportsNamespaces;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.exceptions.NotFoundException;
import org.apache.iceberg.hadoop.Configurable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.jdbc.JdbcClientPool;
import org.apache.iceberg.jdbc.JdbcTableOperations;
import org.apache.iceberg.jdbc.JdbcUtil;
import org.apache.iceberg.jdbc.UncheckedInterruptedException;
import org.apache.iceberg.jdbc.UncheckedSQLException;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.util.LocationUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcCatalog
extends BaseMetastoreCatalog
implements Configurable<Configuration>,
SupportsNamespaces,
Closeable {
    public static final String PROPERTY_PREFIX = "jdbc.";
    private static final String NAMESPACE_EXISTS_PROPERTY = "exists";
    private static final Logger LOG = LoggerFactory.getLogger(JdbcCatalog.class);
    private static final Joiner SLASH = Joiner.on((String)"/");
    private FileIO io;
    private String catalogName = "jdbc";
    private String warehouseLocation;
    private Object conf;
    private JdbcClientPool connections;
    private Map<String, String> catalogProperties;

    public void initialize(String name, Map<String, String> properties) {
        Preconditions.checkNotNull(properties, (Object)"Invalid catalog properties: null");
        String uri = properties.get("uri");
        Preconditions.checkNotNull((Object)uri, (Object)"JDBC connection URI is required");
        String inputWarehouseLocation = properties.get("warehouse");
        Preconditions.checkArgument((inputWarehouseLocation != null && inputWarehouseLocation.length() > 0 ? 1 : 0) != 0, (Object)"Cannot initialize JDBCCatalog because warehousePath must not be null or empty");
        this.warehouseLocation = LocationUtil.stripTrailingSlash(inputWarehouseLocation);
        this.catalogProperties = properties;
        if (name != null) {
            this.catalogName = name;
        }
        String fileIOImpl = properties.getOrDefault("io-impl", "org.apache.iceberg.hadoop.HadoopFileIO");
        this.io = CatalogUtil.loadFileIO(fileIOImpl, properties, this.conf);
        try {
            LOG.debug("Connecting to JDBC database {}", (Object)properties.get("uri"));
            this.connections = new JdbcClientPool(uri, properties);
            this.initializeCatalogTables();
        }
        catch (SQLTimeoutException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Query timed out", new Object[0]);
        }
        catch (SQLNonTransientConnectionException | SQLTransientConnectionException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Connection failed", new Object[0]);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog", new Object[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to initialize", new Object[0]);
        }
    }

    private void initializeCatalogTables() throws InterruptedException, SQLException {
        LOG.trace("Creating database tables (if missing) to store iceberg catalog");
        this.connections.run(conn -> {
            DatabaseMetaData dbMeta = conn.getMetaData();
            ResultSet tableExists = dbMeta.getTables(null, null, "iceberg_tables", null);
            if (tableExists.next()) {
                return true;
            }
            LOG.debug("Creating table {} to store iceberg catalog", (Object)"iceberg_tables");
            return conn.prepareStatement("CREATE TABLE iceberg_tables(catalog_name VARCHAR(255) NOT NULL,table_namespace VARCHAR(255) NOT NULL,table_name VARCHAR(255) NOT NULL,metadata_location VARCHAR(5500),previous_metadata_location VARCHAR(5500),PRIMARY KEY (catalog_name, table_namespace, table_name))").execute();
        });
        this.connections.run(conn -> {
            DatabaseMetaData dbMeta = conn.getMetaData();
            ResultSet tableExists = dbMeta.getTables(null, null, "iceberg_namespace_properties", null);
            if (tableExists.next()) {
                return true;
            }
            LOG.debug("Creating table {} to store iceberg catalog namespace properties", (Object)"iceberg_namespace_properties");
            return conn.prepareStatement("CREATE TABLE iceberg_namespace_properties(catalog_name VARCHAR(255) NOT NULL,namespace VARCHAR(255) NOT NULL,property_key VARCHAR(5500),property_value VARCHAR(5500),PRIMARY KEY (catalog_name, namespace, property_key))").execute();
        });
    }

    @Override
    protected TableOperations newTableOps(TableIdentifier tableIdentifier) {
        return new JdbcTableOperations(this.connections, this.io, this.catalogName, tableIdentifier, this.catalogProperties);
    }

    @Override
    protected String defaultWarehouseLocation(TableIdentifier table) {
        return SLASH.join((Object)this.defaultNamespaceLocation(table.namespace()), (Object)table.name(), new Object[0]);
    }

    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        int deletedRecords;
        TableOperations ops = this.newTableOps(identifier);
        TableMetadata lastMetadata = null;
        if (purge) {
            try {
                lastMetadata = ops.current();
            }
            catch (NotFoundException e) {
                LOG.warn("Failed to load table metadata for table: {}, continuing drop without purge", (Object)identifier, (Object)e);
            }
        }
        if ((deletedRecords = this.execute("DELETE FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? ", this.catalogName, JdbcUtil.namespaceToString(identifier.namespace()), identifier.name())) == 0) {
            LOG.info("Skipping drop, table does not exist: {}", (Object)identifier);
            return false;
        }
        if (purge && lastMetadata != null) {
            CatalogUtil.dropTableData(ops.io(), lastMetadata);
        }
        LOG.info("Dropped table: {}", (Object)identifier);
        return true;
    }

    public List<TableIdentifier> listTables(Namespace namespace) {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        return this.fetch(row -> JdbcUtil.stringToTableIdentifier(row.getString("table_namespace"), row.getString("table_name")), "SELECT * FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ?", this.catalogName, JdbcUtil.namespaceToString(namespace));
    }

    public void renameTable(TableIdentifier from, TableIdentifier to) {
        int updatedRecords = this.execute((SQLException err) -> {
            if (err instanceof SQLIntegrityConstraintViolationException || err.getMessage() != null && err.getMessage().contains("constraint failed")) {
                throw new AlreadyExistsException("Table already exists: %s", new Object[]{to});
            }
        }, "UPDATE iceberg_tables SET table_namespace = ? , table_name = ?  WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? ", JdbcUtil.namespaceToString(to.namespace()), to.name(), this.catalogName, JdbcUtil.namespaceToString(from.namespace()), from.name());
        if (updatedRecords == 1) {
            LOG.info("Renamed table from {}, to {}", (Object)from, (Object)to);
        } else {
            if (updatedRecords == 0) {
                throw new NoSuchTableException("Table does not exist: %s", new Object[]{from});
            }
            LOG.warn("Rename operation affected {} rows: the catalog table's primary key assumption has been violated", (Object)updatedRecords);
        }
    }

    public String name() {
        return this.catalogName;
    }

    @Override
    public void setConf(Configuration conf) {
        this.conf = conf;
    }

    public void createNamespace(Namespace namespace, Map<String, String> metadata) {
        if (this.namespaceExists(namespace)) {
            throw new AlreadyExistsException("Namespace already exists: %s", new Object[]{namespace});
        }
        ImmutableMap createMetadata = metadata == null || metadata.isEmpty() ? ImmutableMap.of((Object)NAMESPACE_EXISTS_PROPERTY, (Object)"true") : ImmutableMap.builder().putAll(metadata).put((Object)NAMESPACE_EXISTS_PROPERTY, (Object)"true").buildOrThrow();
        this.insertProperties(namespace, (Map<String, String>)createMetadata);
    }

    public List<Namespace> listNamespaces() {
        List<Object> namespaces = Lists.newArrayList();
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("table_namespace")), "SELECT DISTINCT table_namespace FROM iceberg_tables WHERE catalog_name = ?", this.catalogName));
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("namespace")), "SELECT DISTINCT namespace FROM iceberg_namespace_properties WHERE catalog_name = ?", this.catalogName));
        namespaces = namespaces.stream().filter(n -> n.levels().length >= 1).map(n -> Namespace.of((String[])((String[])Arrays.stream(n.levels()).limit(1L).toArray(String[]::new)))).distinct().collect(Collectors.toList());
        return namespaces;
    }

    public List<Namespace> listNamespaces(Namespace namespace) throws NoSuchNamespaceException {
        if (namespace.isEmpty()) {
            return this.listNamespaces();
        }
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        List<Object> namespaces = Lists.newArrayList();
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("table_namespace")), "SELECT DISTINCT table_namespace FROM iceberg_tables WHERE catalog_name = ? AND table_namespace LIKE ?", this.catalogName, JdbcUtil.namespaceToString(namespace) + "%"));
        namespaces.addAll(this.fetch(row -> JdbcUtil.stringToNamespace(row.getString("namespace")), "SELECT DISTINCT namespace FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace LIKE ?", this.catalogName, JdbcUtil.namespaceToString(namespace) + "%"));
        int subNamespaceLevelLength = namespace.levels().length + 1;
        namespaces = namespaces.stream().filter(n -> !n.equals((Object)namespace)).filter(n -> n.levels().length >= subNamespaceLevelLength).map(n -> Namespace.of((String[])((String[])Arrays.stream(n.levels()).limit(subNamespaceLevelLength).toArray(String[]::new)))).distinct().collect(Collectors.toList());
        return namespaces;
    }

    public Map<String, String> loadNamespaceMetadata(Namespace namespace) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        HashMap properties = Maps.newHashMap();
        properties.putAll(this.fetchProperties(namespace));
        if (!properties.containsKey("location")) {
            properties.put("location", this.defaultNamespaceLocation(namespace));
        }
        properties.remove(NAMESPACE_EXISTS_PROPERTY);
        return ImmutableMap.copyOf((Map)properties);
    }

    private String defaultNamespaceLocation(Namespace namespace) {
        if (namespace.isEmpty()) {
            return this.warehouseLocation;
        }
        return SLASH.join((Object)this.warehouseLocation, (Object)SLASH.join((Object[])namespace.levels()), new Object[0]);
    }

    public boolean dropNamespace(Namespace namespace) throws NamespaceNotEmptyException {
        if (!this.namespaceExists(namespace)) {
            return false;
        }
        List<TableIdentifier> tableIdentifiers = this.listTables(namespace);
        if (tableIdentifiers != null && !tableIdentifiers.isEmpty()) {
            throw new NamespaceNotEmptyException("Namespace %s is not empty. %s tables exist.", new Object[]{namespace, tableIdentifiers.size()});
        }
        int deletedRows = this.execute("DELETE FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace = ?", this.catalogName, JdbcUtil.namespaceToString(namespace));
        return deletedRows > 0;
    }

    public boolean setProperties(Namespace namespace, Map<String, String> properties) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        Preconditions.checkNotNull(properties, (Object)"Invalid properties to set: null");
        if (properties.isEmpty()) {
            return false;
        }
        Preconditions.checkArgument((!properties.containsKey(NAMESPACE_EXISTS_PROPERTY) ? 1 : 0) != 0, (String)"Cannot set reserved property: %s", (Object)NAMESPACE_EXISTS_PROPERTY);
        Map<String, String> startingProperties = this.fetchProperties(namespace);
        HashMap inserts = Maps.newHashMap();
        HashMap updates = Maps.newHashMap();
        for (String key : properties.keySet()) {
            String value = properties.get(key);
            if (startingProperties.containsKey(key)) {
                updates.put(key, value);
                continue;
            }
            inserts.put(key, value);
        }
        boolean hadInserts = false;
        if (!inserts.isEmpty()) {
            hadInserts = this.insertProperties(namespace, inserts);
        }
        boolean hadUpdates = false;
        if (!updates.isEmpty()) {
            hadUpdates = this.updateProperties(namespace, updates);
        }
        return hadInserts || hadUpdates;
    }

    public boolean removeProperties(Namespace namespace, Set<String> properties) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        Preconditions.checkNotNull(properties, (Object)"Invalid properties to remove: null");
        if (properties.isEmpty()) {
            return false;
        }
        return this.deleteProperties(namespace, properties);
    }

    @Override
    public void close() {
        this.connections.close();
    }

    public boolean namespaceExists(Namespace namespace) {
        return JdbcUtil.namespaceExists(this.catalogName, this.connections, namespace);
    }

    private int execute(String sql, String ... args) {
        return this.execute((SQLException err) -> {}, sql, args);
    }

    private int execute(Consumer<SQLException> sqlErrorHandler, String sql, String ... args) {
        try {
            return this.connections.run(conn -> {
                PreparedStatement preparedStatement = conn.prepareStatement(sql);
                Throwable throwable = null;
                try {
                    for (int pos = 0; pos < args.length; ++pos) {
                        preparedStatement.setString(pos + 1, args[pos]);
                    }
                    Integer n = preparedStatement.executeUpdate();
                    return n;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (preparedStatement != null) {
                        JdbcCatalog.$closeResource(throwable, preparedStatement);
                    }
                }
            });
        }
        catch (SQLException e) {
            sqlErrorHandler.accept(e);
            throw new UncheckedSQLException(e, "Failed to execute: %s", sql);
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e, "Interrupted in SQL command", new Object[0]);
        }
    }

    private <R> List<R> fetch(RowProducer<R> toRow, String sql, String ... args) {
        try {
            return this.connections.run(conn -> {
                ArrayList result = Lists.newArrayList();
                PreparedStatement preparedStatement = conn.prepareStatement(sql);
                Throwable throwable = null;
                try {
                    for (int pos = 0; pos < args.length; ++pos) {
                        preparedStatement.setString(pos + 1, args[pos]);
                    }
                    ResultSet rs = preparedStatement.executeQuery();
                    Throwable throwable2 = null;
                    try {
                        while (rs.next()) {
                            result.add(toRow.apply(rs));
                        }
                    }
                    catch (Throwable throwable3) {
                        throwable2 = throwable3;
                        throw throwable3;
                    }
                    finally {
                        if (rs != null) {
                            JdbcCatalog.$closeResource(throwable2, rs);
                        }
                    }
                }
                catch (Throwable throwable4) {
                    throwable = throwable4;
                    throw throwable4;
                }
                finally {
                    if (preparedStatement != null) {
                        JdbcCatalog.$closeResource(throwable, preparedStatement);
                    }
                }
                return result;
            });
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to execute query: %s", sql);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in SQL query", new Object[0]);
        }
    }

    private Map<String, String> fetchProperties(Namespace namespace) {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", new Object[]{namespace});
        }
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        List<Map.Entry> entries = this.fetch(row -> new AbstractMap.SimpleImmutableEntry<String, String>(row.getString("property_key"), row.getString("property_value")), "SELECT *  FROM iceberg_namespace_properties WHERE catalog_name = ? AND namespace = ? ", this.catalogName, namespaceName);
        return ImmutableMap.builder().putAll(entries).buildOrThrow();
    }

    private boolean insertProperties(Namespace namespace, Map<String, String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        String[] args = (String[])properties.entrySet().stream().flatMap(entry -> Stream.of(this.catalogName, namespaceName, (String)entry.getKey(), (String)entry.getValue())).toArray(String[]::new);
        int insertedRecords = this.execute(JdbcUtil.insertPropertiesStatement(properties.size()), args);
        if (insertedRecords == properties.size()) {
            return true;
        }
        throw new IllegalStateException(String.format("Failed to insert: %d of %d succeeded", insertedRecords, properties.size()));
    }

    private boolean updateProperties(Namespace namespace, Map<String, String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        Stream caseArgs = properties.entrySet().stream().flatMap(entry -> Stream.of((String)entry.getKey(), (String)entry.getValue()));
        Stream<String> whereArgs = Stream.concat(Stream.of(this.catalogName, namespaceName), properties.keySet().stream());
        String[] args = (String[])Stream.concat(caseArgs, whereArgs).toArray(String[]::new);
        int updatedRecords = this.execute(JdbcUtil.updatePropertiesStatement(properties.size()), args);
        if (updatedRecords == properties.size()) {
            return true;
        }
        throw new IllegalStateException(String.format("Failed to update: %d of %d succeeded", updatedRecords, properties.size()));
    }

    private boolean deleteProperties(Namespace namespace, Set<String> properties) {
        String namespaceName = JdbcUtil.namespaceToString(namespace);
        String[] args = (String[])Stream.concat(Stream.of(this.catalogName, namespaceName), properties.stream()).toArray(String[]::new);
        return this.execute(JdbcUtil.deletePropertiesStatement(properties), args) > 0;
    }

    private static /* synthetic */ /* end resource */ void $closeResource(Throwable x0, AutoCloseable x1) {
        if (x0 != null) {
            try {
                x1.close();
            }
            catch (Throwable throwable) {
                x0.addSuppressed(throwable);
            }
        } else {
            x1.close();
        }
    }

    @FunctionalInterface
    static interface RowProducer<R> {
        public R apply(ResultSet var1) throws SQLException;
    }
}

