/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.impl;

import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.google.common.annotations.VisibleForTesting;
import java.util.Locale;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.s3a.RemoteFileChangedException;
import org.apache.hadoop.fs.s3a.impl.LogExactlyOnce;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public abstract class ChangeDetectionPolicy {
    private static final Logger LOG = LoggerFactory.getLogger(ChangeDetectionPolicy.class);
    @VisibleForTesting
    public static final String CHANGE_DETECTED = "change detected";
    private final Mode mode;
    private final boolean requireVersion;
    private final LogExactlyOnce logNoVersionSupport = new LogExactlyOnce(LOG);

    protected ChangeDetectionPolicy(Mode mode, boolean requireVersion) {
        this.mode = mode;
        this.requireVersion = requireVersion;
    }

    public Mode getMode() {
        return this.mode;
    }

    public abstract Source getSource();

    public boolean isRequireVersion() {
        return this.requireVersion;
    }

    public LogExactlyOnce getLogNoVersionSupport() {
        return this.logNoVersionSupport;
    }

    public static ChangeDetectionPolicy getPolicy(Configuration configuration) {
        Mode mode = Mode.fromConfiguration(configuration);
        Source source = Source.fromConfiguration(configuration);
        boolean requireVersion = configuration.getBoolean("fs.s3a.change.detection.version.required", true);
        return ChangeDetectionPolicy.createPolicy(mode, source, requireVersion);
    }

    @VisibleForTesting
    public static ChangeDetectionPolicy createPolicy(Mode mode, Source source, boolean requireVersion) {
        switch (source) {
            case ETag: {
                return new ETagChangeDetectionPolicy(mode, requireVersion);
            }
            case VersionId: {
                return new VersionIdChangeDetectionPolicy(mode, requireVersion);
            }
        }
        return new NoChangeDetection();
    }

    public abstract String getRevisionId(ObjectMetadata var1, String var2);

    public abstract void applyRevisionConstraint(GetObjectRequest var1, String var2);

    public ImmutablePair<Boolean, RemoteFileChangedException> onChangeDetected(String revisionId, String newRevisionId, String uri, long position, String operation, long timesAlreadyDetected) {
        switch (this.mode) {
            case None: {
                return new ImmutablePair((Object)false, null);
            }
            case Warn: {
                if (timesAlreadyDetected == 0L) {
                    LOG.warn(String.format("%s change detected on %s %s at %d. Expected %s got %s", new Object[]{this.getSource(), operation, uri, position, revisionId, newRevisionId}));
                    return new ImmutablePair((Object)true, null);
                }
                return new ImmutablePair((Object)false, null);
            }
        }
        return new ImmutablePair((Object)true, (Object)new RemoteFileChangedException(uri, operation, String.format("%s change detected while reading at position %s. Expected %s got %s", new Object[]{this.getSource(), position, revisionId, newRevisionId})));
    }

    static class NoChangeDetection
    extends ChangeDetectionPolicy {
        NoChangeDetection() {
            super(Mode.None, false);
        }

        @Override
        public Source getSource() {
            return Source.None;
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            return null;
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
        }

        public String toString() {
            return "NoChangeDetection";
        }
    }

    static class VersionIdChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        VersionIdChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            String versionId = objectMetadata.getVersionId();
            if (versionId == null) {
                this.getLogNoVersionSupport().warn("fs.s3a.change.detection.mode set to " + (Object)((Object)Source.VersionId) + " but no versionId available while reading {}. Ensure your bucket has object versioning enabled. You may see inconsistent reads.", uri);
            }
            return versionId;
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
            LOG.debug("Restricting request to version {}", (Object)revisionId);
            request.withVersionId(revisionId);
        }

        @Override
        public Source getSource() {
            return Source.VersionId;
        }

        public String toString() {
            return "VersionIdChangeDetectionPolicy mode=" + (Object)((Object)this.getMode());
        }
    }

    static class ETagChangeDetectionPolicy
    extends ChangeDetectionPolicy {
        ETagChangeDetectionPolicy(Mode mode, boolean requireVersion) {
            super(mode, requireVersion);
        }

        @Override
        public String getRevisionId(ObjectMetadata objectMetadata, String uri) {
            return objectMetadata.getETag();
        }

        @Override
        public void applyRevisionConstraint(GetObjectRequest request, String revisionId) {
            LOG.debug("Restricting request to etag {}", (Object)revisionId);
            request.withMatchingETagConstraint(revisionId);
        }

        @Override
        public Source getSource() {
            return Source.ETag;
        }

        public String toString() {
            return "ETagChangeDetectionPolicy mode=" + (Object)((Object)this.getMode());
        }
    }

    public static enum Mode {
        Client("client"),
        Server("server"),
        Warn("warn"),
        None("none");

        private final String mode;

        private Mode(String mode) {
            this.mode = mode;
        }

        private static Mode fromString(String trimmed) {
            for (Mode value : Mode.values()) {
                if (!value.mode.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.mode value: \"{}\"", (Object)trimmed);
            return Mode.fromString("server");
        }

        static Mode fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.mode", "server").trim().toLowerCase(Locale.ENGLISH);
            return Mode.fromString(trimmed);
        }
    }

    public static enum Source {
        ETag("etag"),
        VersionId("versionid"),
        None("none");

        private final String source;

        private Source(String source) {
            this.source = source;
        }

        private static Source fromString(String trimmed) {
            for (Source value : Source.values()) {
                if (!value.source.equals(trimmed)) continue;
                return value;
            }
            LOG.warn("Unrecognized fs.s3a.change.detection.source value: \"{}\"", (Object)trimmed);
            return Source.fromString("etag");
        }

        static Source fromConfiguration(Configuration configuration) {
            String trimmed = configuration.get("fs.s3a.change.detection.source", "etag").trim().toLowerCase(Locale.ENGLISH);
            return Source.fromString(trimmed);
        }
    }
}

