/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.contract;

import java.io.Closeable;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.UUID;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathCapabilities;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.io.IOUtils;
import org.junit.Assert;
import org.junit.internal.AssumptionViolatedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContractTestUtils
extends Assert {
    private static final Logger LOG = LoggerFactory.getLogger(ContractTestUtils.class);
    public static final String IO_CHUNK_BUFFER_SIZE = "io.chunk.buffer.size";
    public static final int DEFAULT_IO_CHUNK_BUFFER_SIZE = 128;
    public static final String IO_CHUNK_MODULUS_SIZE = "io.chunk.modulus.size";
    public static final int DEFAULT_IO_CHUNK_MODULUS_SIZE = 128;

    public static void assertPropertyEquals(Properties props, String key, String expected) {
        String val = props.getProperty(key);
        if (expected == null) {
            ContractTestUtils.assertNull((String)("Non null property " + key + " = " + val), (Object)val);
        } else {
            ContractTestUtils.assertEquals((String)("property " + key + " = " + val), (Object)expected, (Object)val);
        }
    }

    public static void writeAndRead(FileSystem fs, Path path, byte[] src, int len, int blocksize, boolean overwrite, boolean delete) throws IOException {
        fs.mkdirs(path.getParent());
        ContractTestUtils.writeDataset(fs, path, src, len, blocksize, overwrite);
        byte[] dest = ContractTestUtils.readDataset(fs, path, len);
        ContractTestUtils.compareByteArrays(src, dest, len);
        if (delete) {
            ContractTestUtils.rejectRootOperation(path);
            boolean deleted = fs.delete(path, false);
            ContractTestUtils.assertTrue((String)"Deleted", (boolean)deleted);
            ContractTestUtils.assertPathDoesNotExist(fs, "Cleanup failed", path);
        }
    }

    public static void writeDataset(FileSystem fs, Path path, byte[] src, int len, int buffersize, boolean overwrite) throws IOException {
        ContractTestUtils.writeDataset(fs, path, src, len, buffersize, overwrite, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void writeDataset(FileSystem fs, Path path, byte[] src, int len, int buffersize, boolean overwrite, boolean useBuilder) throws IOException {
        ContractTestUtils.assertTrue((String)("Not enough data in source array to write " + len + " bytes"), (src.length >= len ? 1 : 0) != 0);
        try (FSDataOutputStream out = useBuilder ? fs.createFile(path).overwrite(overwrite).replication((short)1).bufferSize(buffersize).blockSize((long)buffersize).build() : fs.create(path, overwrite, fs.getConf().getInt("io.file.buffer.size", 4096), (short)1, (long)buffersize);){
            out.write(src, 0, len);
        }
        ContractTestUtils.assertFileHasLength(fs, path, len);
    }

    public static byte[] readDataset(FileSystem fs, Path path, int len) throws IOException {
        byte[] dest = new byte[len];
        int offset = 0;
        try (FSDataInputStream in = fs.open(path);){
            int nbytes;
            for (int nread = 0; nread < len; nread += nbytes) {
                nbytes = in.read(dest, offset + nread, len - nread);
                if (nbytes >= 0) continue;
                throw new EOFException("End of file reached before reading fully.");
            }
        }
        return dest;
    }

    public static void verifyFileContents(FileSystem fs, Path path, byte[] original) throws IOException {
        ContractTestUtils.assertIsFile(fs, path);
        FileStatus stat = fs.getFileStatus(path);
        String statText = stat.toString();
        ContractTestUtils.assertEquals((String)("wrong length " + statText), (long)original.length, (long)stat.getLen());
        byte[] bytes = ContractTestUtils.readDataset(fs, path, original.length);
        ContractTestUtils.compareByteArrays(original, bytes, original.length);
    }

    public static void verifyRead(FSDataInputStream stm, byte[] fileContents, int seekOff, int toRead) throws IOException {
        byte[] out = new byte[toRead];
        stm.seek((long)seekOff);
        stm.readFully(out);
        byte[] expected = Arrays.copyOfRange(fileContents, seekOff, seekOff + toRead);
        ContractTestUtils.compareByteArrays(expected, out, toRead);
    }

    public static void compareByteArrays(byte[] original, byte[] received, int len) {
        ContractTestUtils.assertEquals((String)"Number of bytes read != number written", (long)len, (long)received.length);
        int errors = 0;
        int firstErrorByte = -1;
        for (int i = 0; i < len; ++i) {
            if (original[i] == received[i]) continue;
            if (errors == 0) {
                firstErrorByte = i;
            }
            ++errors;
        }
        if (errors > 0) {
            String message = String.format(" %d errors in file of length %d", errors, len);
            LOG.warn(message);
            int overlap = 10;
            for (int i = Math.max(0, firstErrorByte - 10); i < Math.min(firstErrorByte + 10, len); ++i) {
                byte actual = received[i];
                byte expected = original[i];
                String letter = ContractTestUtils.toChar(actual);
                String line = String.format("[%04d] %2x %s%n", i, actual, letter);
                if (expected != actual) {
                    line = String.format("[%04d] %2x %s -expected %2x %s%n", i, actual, letter, expected, ContractTestUtils.toChar(expected));
                }
                LOG.warn(line);
            }
            ContractTestUtils.fail((String)message);
        }
    }

    public static String toChar(byte b) {
        if (b >= 32) {
            return Character.toString((char)b);
        }
        return String.format("%02x", b);
    }

    public static String toChar(byte[] buffer) {
        StringBuilder builder = new StringBuilder(buffer.length);
        for (byte b : buffer) {
            builder.append(ContractTestUtils.toChar(b));
        }
        return builder.toString();
    }

    public static byte[] toAsciiByteArray(String s) {
        char[] chars = s.toCharArray();
        int len = chars.length;
        byte[] buffer = new byte[len];
        for (int i = 0; i < len; ++i) {
            buffer[i] = (byte)(chars[i] & 0xFF);
        }
        return buffer;
    }

    public static void cleanup(String action, FileSystem fileSystem, String cleanupPath) {
        if (fileSystem == null) {
            return;
        }
        Path path = new Path(cleanupPath).makeQualified(fileSystem.getUri(), fileSystem.getWorkingDirectory());
        ContractTestUtils.cleanup(action, fileSystem, path);
    }

    public static void cleanup(String action, FileSystem fileSystem, Path path) {
        ContractTestUtils.noteAction(action);
        try {
            ContractTestUtils.rm(fileSystem, path, true, false);
        }
        catch (Exception e) {
            LOG.error("Error deleting in " + action + " - " + path + ": " + e, (Throwable)e);
        }
    }

    public static boolean rm(FileSystem fileSystem, Path path, boolean recursive, boolean allowRootDelete) throws IOException {
        if (fileSystem != null) {
            ContractTestUtils.rejectRootOperation(path, allowRootDelete);
            if (fileSystem.exists(path)) {
                return fileSystem.delete(path, recursive);
            }
        }
        return false;
    }

    public static void rename(FileSystem fileSystem, Path src, Path dst) throws IOException {
        ContractTestUtils.rejectRootOperation(src, false);
        ContractTestUtils.assertTrue((boolean)fileSystem.rename(src, dst));
        ContractTestUtils.assertPathDoesNotExist(fileSystem, "renamed", src);
    }

    public static void rejectRootOperation(Path path, boolean allowRootOperation) throws IOException {
        if (path.isRoot() && !allowRootOperation) {
            throw new IOException("Root directory operation rejected: " + path);
        }
    }

    public static void rejectRootOperation(Path path) throws IOException {
        ContractTestUtils.rejectRootOperation(path, false);
    }

    public static FileStatus[] deleteChildren(FileSystem fileSystem, Path path, boolean recursive) throws IOException {
        FileStatus[] children;
        for (FileStatus entry : children = ContractTestUtils.listChildren(fileSystem, path)) {
            fileSystem.delete(entry.getPath(), recursive);
        }
        return children;
    }

    public static FileStatus[] listChildren(FileSystem fileSystem, Path path) throws IOException {
        FileStatus[] entries = fileSystem.listStatus(path);
        if (entries.length == 1 && path.equals((Object)entries[0].getPath())) {
            return new FileStatus[0];
        }
        return entries;
    }

    public static void noteAction(String action) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("==============  " + action + " =============");
        }
    }

    public static void downgrade(String message, Throwable failure) {
        LOG.warn("Downgrading test " + message, failure);
        AssumptionViolatedException ave = new AssumptionViolatedException((Object)failure, null);
        throw ave;
    }

    public static void unsupported(String message) {
        ContractTestUtils.skip(message);
    }

    public static void skip(String message) {
        LOG.info("Skipping: {}", (Object)message);
        throw new AssumptionViolatedException(message);
    }

    public static void fail(String text, Throwable thrown) {
        throw new AssertionError(text, thrown);
    }

    public static void assertFileHasLength(FileSystem fs, Path path, int expected) throws IOException {
        FileStatus status = fs.getFileStatus(path);
        ContractTestUtils.assertEquals((String)("Wrong file length of file " + path + " status: " + status), (long)expected, (long)status.getLen());
    }

    public static void assertIsDirectory(FileSystem fs, Path path) throws IOException {
        FileStatus fileStatus = fs.getFileStatus(path);
        ContractTestUtils.assertIsDirectory(fileStatus);
    }

    public static void assertIsDirectory(FileStatus fileStatus) {
        ContractTestUtils.assertTrue((String)("Should be a directory -but isn't: " + fileStatus), (boolean)fileStatus.isDirectory());
    }

    public static void assertErasureCoded(FileSystem fs, Path path) throws IOException {
        FileStatus fileStatus = fs.getFileStatus(path);
        ContractTestUtils.assertTrue((String)(path + " must be erasure coded!"), (boolean)fileStatus.isErasureCoded());
    }

    public static void assertNotErasureCoded(FileSystem fs, Path path) throws IOException {
        FileStatus fileStatus = fs.getFileStatus(path);
        ContractTestUtils.assertFalse((String)(path + " should not be erasure coded!"), (boolean)fileStatus.isErasureCoded());
    }

    public static byte[] writeTextFile(FileSystem fs, Path path, String text, boolean overwrite) throws IOException {
        byte[] bytes = new byte[]{};
        if (text != null) {
            bytes = ContractTestUtils.toAsciiByteArray(text);
        }
        ContractTestUtils.createFile(fs, path, overwrite, bytes);
        return bytes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void createFile(FileSystem fs, Path path, boolean overwrite, byte[] data) throws IOException {
        FSDataOutputStream stream = fs.create(path, overwrite);
        try {
            if (data != null && data.length > 0) {
                stream.write(data);
            }
            stream.close();
        }
        finally {
            IOUtils.closeStream((Closeable)stream);
        }
    }

    public static void appendFile(FileSystem fs, Path path, byte[] data) throws IOException {
        try (FSDataOutputStream stream = fs.appendFile(path).build();){
            if (data != null && data.length > 0) {
                stream.write(data);
            }
        }
    }

    public static void touch(FileSystem fs, Path path) throws IOException {
        ContractTestUtils.createFile(fs, path, true, null);
    }

    public static void assertDeleted(FileSystem fs, Path file, boolean recursive) throws IOException {
        ContractTestUtils.assertDeleted(fs, file, recursive, false);
    }

    public static void assertDeleted(FileSystem fs, Path file, boolean recursive, boolean allowRootOperations) throws IOException {
        ContractTestUtils.rejectRootOperation(file, allowRootOperations);
        ContractTestUtils.assertPathExists(fs, "about to be deleted file", file);
        boolean deleted = fs.delete(file, recursive);
        String dir = ContractTestUtils.ls(fs, file.getParent());
        ContractTestUtils.assertTrue((String)("Delete failed on " + file + ": " + dir), (boolean)deleted);
        ContractTestUtils.assertPathDoesNotExist(fs, "Deleted file", file);
    }

    public static void assertRenameOutcome(FileSystem fs, Path source, Path dest, boolean expectedResult) throws IOException {
        boolean result = fs.rename(source, dest);
        if (expectedResult != result) {
            ContractTestUtils.fail((String)String.format("Expected rename(%s, %s) to return %b, but result was %b", source, dest, expectedResult, result));
        }
    }

    public static String readBytesToString(FileSystem fs, Path path, int length) throws IOException {
        try (FSDataInputStream in = fs.open(path);){
            byte[] buf = new byte[length];
            in.readFully(0L, buf);
            String string = ContractTestUtils.toChar(buf);
            return string;
        }
    }

    public static String readUTF8(FileSystem fs, Path path, int length) throws IOException {
        if (length < 0) {
            FileStatus status = fs.getFileStatus(path);
            length = (int)status.getLen();
        }
        try (FSDataInputStream in = fs.open(path);){
            byte[] buf = new byte[length];
            in.readFully(0L, buf);
            String string = new String(buf, "UTF-8");
            return string;
        }
    }

    public static String fileStatsToString(FileStatus[] stats, String separator) {
        StringBuilder buf = new StringBuilder(stats.length * 128);
        for (int i = 0; i < stats.length; ++i) {
            buf.append(String.format("[%02d] %s", i, stats[i])).append(separator);
        }
        return buf.toString();
    }

    public static String ls(FileSystem fileSystem, Path path) throws IOException {
        FileStatus[] stats;
        if (path == null) {
            return "/";
        }
        String pathtext = "ls " + path;
        try {
            stats = fileSystem.listStatus(path);
        }
        catch (FileNotFoundException e) {
            return pathtext + " -file not found";
        }
        catch (IOException e) {
            return pathtext + " -failed: " + e;
        }
        return ContractTestUtils.dumpStats(pathtext, stats);
    }

    public static String dumpStats(String pathname, FileStatus[] stats) {
        return pathname + ' ' + ContractTestUtils.fileStatsToString(stats, System.lineSeparator());
    }

    public static void assertIsFile(FileSystem fileSystem, Path filename) throws IOException {
        ContractTestUtils.assertPathExists(fileSystem, "Expected file", filename);
        FileStatus status = fileSystem.getFileStatus(filename);
        ContractTestUtils.assertIsFile(filename, status);
    }

    public static void assertIsFile(FileContext fileContext, Path filename) throws IOException {
        ContractTestUtils.assertPathExists(fileContext, "Expected file", filename);
        FileStatus status = fileContext.getFileStatus(filename);
        ContractTestUtils.assertIsFile(filename, status);
    }

    public static void assertIsFile(Path filename, FileStatus status) {
        String fileInfo = filename + "  " + status;
        ContractTestUtils.assertFalse((String)("File claims to be a directory " + fileInfo), (boolean)status.isDirectory());
        ContractTestUtils.assertFalse((String)("File claims to be a symlink " + fileInfo), (boolean)status.isSymlink());
    }

    public static void assertPathsExist(FileSystem fs, String message, Path ... paths) throws IOException {
        for (Path path : paths) {
            ContractTestUtils.assertPathExists(fs, message, path);
        }
    }

    public static void assertPathsDoNotExist(FileSystem fs, String message, Path ... paths) throws IOException {
        for (Path path : paths) {
            ContractTestUtils.assertPathDoesNotExist(fs, message, path);
        }
    }

    public static byte[] dataset(int len, int base, int modulo) {
        byte[] dataset = new byte[len];
        for (int i = 0; i < len; ++i) {
            dataset[i] = (byte)(base + i % modulo);
        }
        return dataset;
    }

    public static void assertPathExists(FileSystem fileSystem, String message, Path path) throws IOException {
        ContractTestUtils.verifyPathExists(fileSystem, message, path);
    }

    public static FileStatus verifyPathExists(FileSystem fileSystem, String message, Path path) throws IOException {
        try {
            return fileSystem.getFileStatus(path);
        }
        catch (FileNotFoundException e) {
            LOG.error("{}: not found {}; parent listing is:\n{}", new Object[]{message, path, ContractTestUtils.ls(fileSystem, path.getParent())});
            throw (IOException)new FileNotFoundException(message + ": not found " + path + " in " + path.getParent()).initCause(e);
        }
    }

    public static void assertPathExists(FileContext fileContext, String message, Path path) throws IOException {
        if (!fileContext.util().exists(path)) {
            throw new FileNotFoundException(message + ": not found " + path + " in " + path.getParent());
        }
    }

    public static void assertPathDoesNotExist(FileSystem fileSystem, String message, Path path) throws IOException {
        try {
            FileStatus status = fileSystem.getFileStatus(path);
            ContractTestUtils.fail((String)(message + ": unexpectedly found " + path + " as  " + status));
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    public static void assertPathDoesNotExist(FileContext fileContext, String message, Path path) throws IOException {
        try {
            FileStatus status = fileContext.getFileStatus(path);
            ContractTestUtils.fail((String)(message + ": unexpectedly found " + path + " as  " + status));
        }
        catch (FileNotFoundException fileNotFoundException) {
            // empty catch block
        }
    }

    public static void assertListStatusFinds(FileSystem fs, Path dir, Path subdir) throws IOException {
        FileStatus[] stats = fs.listStatus(dir);
        boolean found = false;
        StringBuilder builder = new StringBuilder();
        for (FileStatus stat : stats) {
            builder.append(stat.toString()).append(System.lineSeparator());
            if (!stat.getPath().equals((Object)subdir)) continue;
            found = true;
        }
        ContractTestUtils.assertTrue((String)("Path " + subdir + " not found in directory " + dir + ":" + builder), (boolean)found);
    }

    public static void assertMkdirs(FileSystem fs, Path dir) throws IOException {
        ContractTestUtils.assertTrue((String)("mkdirs(" + dir + ") returned false"), (boolean)fs.mkdirs(dir));
    }

    public static boolean isOSX() {
        return System.getProperty("os.name").contains("OS X");
    }

    public static void validateFileContent(byte[] concat, byte[][] bytes) {
        int idx = 0;
        boolean mismatch = false;
        byte[][] byArray = bytes;
        int n = byArray.length;
        for (int i = 0; i < n; ++i) {
            byte[] bb;
            for (byte b : bb = byArray[i]) {
                if (b == concat[idx++]) continue;
                mismatch = true;
                break;
            }
            if (mismatch) break;
        }
        ContractTestUtils.assertFalse((String)("File content of file is not as expected at offset " + idx), (boolean)mismatch);
    }

    public static void verifyReceivedData(FileSystem fs, Path path, long expectedSize, int bufferLen, int modulus) throws IOException {
        byte[] testBuffer = new byte[bufferLen];
        long totalBytesRead = 0L;
        int nextExpectedNumber = 0;
        NanoTimer timer = new NanoTimer();
        try (FSDataInputStream inputStream = fs.open(path);){
            int bytesRead;
            while ((bytesRead = inputStream.read(testBuffer)) >= 0) {
                totalBytesRead += (long)bytesRead;
                for (int i = 0; i < bytesRead; ++i) {
                    if (testBuffer[i] != nextExpectedNumber) {
                        throw new IOException("Read number " + testBuffer[i] + " but expected " + nextExpectedNumber);
                    }
                    if (++nextExpectedNumber != modulus) continue;
                    nextExpectedNumber = 0;
                }
            }
            if (totalBytesRead != expectedSize) {
                throw new IOException("Expected to read " + expectedSize + " bytes but only received " + totalBytesRead);
            }
        }
        timer.end("Time to read %d bytes", expectedSize);
        ContractTestUtils.bandwidth(timer, expectedSize);
    }

    public static long generateTestFile(FileSystem fs, Path path, long size, int bufferLen, int modulus) throws IOException {
        byte[] testBuffer = new byte[bufferLen];
        for (int i = 0; i < testBuffer.length; ++i) {
            testBuffer[i] = (byte)(i % modulus);
        }
        long bytesWritten = 0L;
        try (FSDataOutputStream outputStream = fs.create(path, false);){
            while (bytesWritten < size) {
                long diff = size - bytesWritten;
                if (diff < (long)testBuffer.length) {
                    outputStream.write(testBuffer, 0, (int)diff);
                    bytesWritten += diff;
                    continue;
                }
                outputStream.write(testBuffer);
                bytesWritten += (long)testBuffer.length;
            }
            long l = bytesWritten;
            return l;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void createAndVerifyFile(FileSystem fs, Path parent, long fileSize) throws IOException {
        int testBufferSize = fs.getConf().getInt(IO_CHUNK_BUFFER_SIZE, 128);
        int modulus = fs.getConf().getInt(IO_CHUNK_MODULUS_SIZE, 128);
        String objectName = UUID.randomUUID().toString();
        Path objectPath = new Path(parent, objectName);
        NanoTimer timer = new NanoTimer();
        ContractTestUtils.assertEquals((long)fileSize, (long)ContractTestUtils.generateTestFile(fs, objectPath, fileSize, testBufferSize, modulus));
        ContractTestUtils.assertPathExists(fs, "not created successful", objectPath);
        timer.end("Time to write %d bytes", fileSize);
        ContractTestUtils.bandwidth(timer, fileSize);
        try {
            ContractTestUtils.verifyReceivedData(fs, objectPath, fileSize, testBufferSize, modulus);
        }
        finally {
            fs.delete(objectPath, false);
        }
    }

    public static String toHuman(long nanos) {
        return String.format(Locale.ENGLISH, "%,d", nanos);
    }

    public static void bandwidth(NanoTimer timer, long bytes) {
        LOG.info("Bandwidth = {}  MB/S", (Object)timer.bandwidthDescription(bytes));
    }

    public static double bandwidthMBs(long bytes, long durationNS) {
        return (double)bytes / 1048576.0 * 1.0E9 / (double)durationNS;
    }

    public static TreeScanResults createSubdirs(FileSystem fs, Path current, int depth, int width, int files, int filesize) throws IOException {
        return ContractTestUtils.createSubdirs(fs, current, depth, width, files, filesize, "dir-", "file-", "0");
    }

    public static TreeScanResults createSubdirs(FileSystem fs, Path current, int depth, int width, int files, int filesize, String dirPrefix, String filePrefix, String marker) throws IOException {
        fs.mkdirs(current);
        TreeScanResults results = new TreeScanResults(current);
        if (depth > 0) {
            byte[] data = ContractTestUtils.dataset(filesize, 97, 122);
            for (int i = 0; i < files; ++i) {
                String name = String.format("%s-%s-%04d.txt", filePrefix, marker, i);
                Path path = new Path(current, name);
                ContractTestUtils.createFile(fs, path, true, data);
                results.add(fs, path);
            }
            for (int w = 0; w < width; ++w) {
                String marker2 = String.format("%s-%04d", marker, w);
                Path child = new Path(current, dirPrefix + marker2);
                results.add(ContractTestUtils.createSubdirs(fs, child, depth - 1, width, files, filesize, dirPrefix, filePrefix, marker2));
                results.add(fs, child);
            }
        }
        return results;
    }

    public static boolean collectionsEquivalent(Collection<Path> left, Collection<Path> right) {
        HashSet<Path> leftSet = new HashSet<Path>(left);
        HashSet<Path> rightSet = new HashSet<Path>(right);
        return leftSet.containsAll(right) && rightSet.containsAll(left);
    }

    public static String pathsToString(Collection<Path> paths) {
        StringBuilder builder = new StringBuilder(paths.size() * 100);
        String nl = System.lineSeparator();
        builder.append(nl);
        for (Path path : paths) {
            builder.append("  \"").append(path.toString()).append("\"").append(nl);
        }
        builder.append("]");
        return builder.toString();
    }

    public static boolean collectionsEquivalentNoDuplicates(Collection<Path> left, Collection<Path> right) {
        return ContractTestUtils.collectionsEquivalent(left, right) && !ContractTestUtils.containsDuplicates(left) && !ContractTestUtils.containsDuplicates(right);
    }

    public static boolean containsDuplicates(Collection<Path> paths) {
        return new HashSet<Path>(paths).size() != paths.size();
    }

    public static FileStatus getFileStatusEventually(FileSystem fs, Path path, int timeout) throws IOException, InterruptedException {
        long endTime = System.currentTimeMillis() + (long)timeout;
        FileStatus stat = null;
        do {
            try {
                stat = fs.getFileStatus(path);
            }
            catch (FileNotFoundException e) {
                if (System.currentTimeMillis() > endTime) {
                    ContractTestUtils.assertPathExists(fs, "Path not found after " + timeout + " mS", path);
                    continue;
                }
                Thread.sleep(50L);
            }
        } while (stat == null);
        return stat;
    }

    public static TreeScanResults treeWalk(FileSystem fs, Path path) throws IOException {
        FileStatus[] statuses;
        TreeScanResults dirsAndFiles = new TreeScanResults();
        for (FileStatus status : statuses = fs.listStatus(path)) {
            LOG.info("{}{}", (Object)status.getPath(), (Object)(status.isDirectory() ? "*" : ""));
        }
        for (FileStatus status : statuses) {
            dirsAndFiles.add(status);
            if (!status.isDirectory()) continue;
            dirsAndFiles.add(ContractTestUtils.treeWalk(fs, status.getPath()));
        }
        return dirsAndFiles;
    }

    public static List<LocatedFileStatus> toList(RemoteIterator<LocatedFileStatus> iterator) throws IOException {
        ArrayList<LocatedFileStatus> list = new ArrayList<LocatedFileStatus>();
        while (iterator.hasNext()) {
            list.add((LocatedFileStatus)iterator.next());
        }
        return list;
    }

    public static List<LocatedFileStatus> toListThroughNextCallsAlone(RemoteIterator<LocatedFileStatus> iterator) throws IOException {
        ArrayList<LocatedFileStatus> list = new ArrayList<LocatedFileStatus>();
        try {
            while (true) {
                list.add((LocatedFileStatus)iterator.next());
            }
        }
        catch (NoSuchElementException noSuchElementException) {
            return list;
        }
    }

    public static void assertCapabilities(Object stream, String[] shouldHaveCapabilities, String[] shouldNotHaveCapabilities) {
        ContractTestUtils.assertTrue((String)"Stream should be instanceof StreamCapabilities", (boolean)(stream instanceof StreamCapabilities));
        StreamCapabilities source = (StreamCapabilities)stream;
        if (shouldHaveCapabilities != null) {
            for (String shouldHaveCapability : shouldHaveCapabilities) {
                ContractTestUtils.assertTrue((String)("Should have capability: " + shouldHaveCapability), (boolean)source.hasCapability(shouldHaveCapability));
            }
        }
        if (shouldNotHaveCapabilities != null) {
            for (String shouldNotHaveCapability : shouldNotHaveCapabilities) {
                ContractTestUtils.assertFalse((String)("Should not have capability: " + shouldNotHaveCapability), (boolean)source.hasCapability(shouldNotHaveCapability));
            }
        }
    }

    public static void assertHasPathCapabilities(PathCapabilities source, Path path, String ... capabilities) throws IOException {
        for (String shouldHaveCapability : capabilities) {
            ContractTestUtils.assertTrue((String)("Should have capability: " + shouldHaveCapability + " under " + path), (boolean)source.hasPathCapability(path, shouldHaveCapability));
        }
    }

    public static void assertLacksPathCapabilities(PathCapabilities source, Path path, String ... capabilities) throws IOException {
        for (String shouldHaveCapability : capabilities) {
            ContractTestUtils.assertFalse((String)("Path  must not support capability: " + shouldHaveCapability + " under " + path), (boolean)source.hasPathCapability(path, shouldHaveCapability));
        }
    }

    public static int read(InputStream in) {
        try {
            return in.read();
        }
        catch (IOException ex) {
            throw new AssertionError((Object)ex);
        }
    }

    public static long readStream(InputStream in) {
        long count = 0L;
        while (ContractTestUtils.read(in) >= 0) {
            ++count;
        }
        return count;
    }

    public static final class NanoTimer {
        private long startTime = this.now();
        private long endTime;

        public void reset() {
            this.endTime = 0L;
            this.startTime = this.now();
        }

        public long end() {
            this.endTime = this.now();
            return this.duration();
        }

        public long end(String format, Object ... args) {
            long d = this.end();
            LOG.info("Duration of {}: {} nS", (Object)String.format(format, args), (Object)ContractTestUtils.toHuman(d));
            return d;
        }

        public long now() {
            return System.nanoTime();
        }

        public long duration() {
            return this.endTime - this.startTime;
        }

        public long elapsedTime() {
            return this.now() - this.startTime;
        }

        public long elapsedTimeMs() {
            return this.elapsedTime() / 1000000L;
        }

        public double bandwidth(long bytes) {
            return ContractTestUtils.bandwidthMBs(bytes, this.duration());
        }

        public double bandwidthBytes(long bytes) {
            double duration = this.duration();
            return duration > 0.0 ? (double)bytes / duration : 0.0;
        }

        public long nanosPerOperation(long operations) {
            return this.duration() / operations;
        }

        public String bandwidthDescription(long bytes) {
            return String.format("%,.6f", this.bandwidth(bytes));
        }

        public long getStartTime() {
            return this.startTime;
        }

        public long getEndTime() {
            return this.endTime;
        }
    }

    public static final class TreeScanResults {
        private Path basePath;
        private final List<Path> files = new ArrayList<Path>();
        private final List<Path> directories = new ArrayList<Path>();
        private final List<Path> other = new ArrayList<Path>();

        public TreeScanResults() {
        }

        public TreeScanResults(Path basePath) {
            this.basePath = basePath;
        }

        public TreeScanResults(RemoteIterator<LocatedFileStatus> results) throws IOException {
            while (results.hasNext()) {
                this.add((FileStatus)results.next());
            }
        }

        public TreeScanResults(FileStatus[] stats) {
            Assert.assertNotNull((String)"Null file status array", (Object)stats);
            for (FileStatus stat : stats) {
                this.add(stat);
            }
        }

        public <F extends FileStatus> TreeScanResults(Iterable<F> stats) {
            for (FileStatus stat : stats) {
                this.add(stat);
            }
        }

        public TreeScanResults add(TreeScanResults that) {
            this.files.addAll(that.files);
            this.directories.addAll(that.directories);
            this.other.addAll(that.other);
            return this;
        }

        public void add(FileStatus status) {
            if (status.isFile()) {
                this.files.add(status.getPath());
            } else if (status.isDirectory()) {
                this.directories.add(status.getPath());
            } else {
                this.other.add(status.getPath());
            }
        }

        public void add(FileSystem fs, Path path) throws IOException {
            this.add(fs.getFileStatus(path));
        }

        public String toString() {
            return String.format("%d director%s and %d file%s", this.getDirCount(), this.getDirCount() == 1L ? "y" : "ies", this.getFileCount(), this.getFileCount() == 1L ? "" : "s");
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TreeScanResults that = (TreeScanResults)o;
            return this.getFileCount() == that.getFileCount() && this.getDirCount() == that.getDirCount();
        }

        public int hashCode() {
            return super.hashCode();
        }

        public void assertSizeEquals(String text, long f, long d, long o) {
            String self = this.toString();
            Assert.assertEquals((String)(text + ": file count in " + self), (long)f, (long)this.getFileCount());
            Assert.assertEquals((String)(text + ": directory count in " + self), (long)d, (long)this.getDirCount());
            Assert.assertEquals((String)(text + ": 'other' count in " + self), (long)o, (long)this.getOtherCount());
        }

        public void assertEquivalent(TreeScanResults that) {
            this.assertFieldsEquivalent("files", that, this.files, that.files);
            this.assertFieldsEquivalent("directories", that, this.directories, that.directories);
            this.assertFieldsEquivalent("other", that, this.other, that.other);
        }

        public void assertFieldsEquivalent(String fieldname, TreeScanResults that, List<Path> ours, List<Path> theirs) {
            String ourList = ContractTestUtils.pathsToString(ours);
            String theirList = ContractTestUtils.pathsToString(theirs);
            Assert.assertFalse((String)("Duplicate  " + fieldname + " in " + this + ": " + ourList), (boolean)ContractTestUtils.containsDuplicates(ours));
            Assert.assertFalse((String)("Duplicate  " + fieldname + " in other " + that + ": " + theirList), (boolean)ContractTestUtils.containsDuplicates(theirs));
            Assert.assertTrue((String)(fieldname + " mismatch: between " + ourList + " and " + theirList), (boolean)ContractTestUtils.collectionsEquivalent(ours, theirs));
        }

        public List<Path> getFiles() {
            return this.files;
        }

        public List<Path> getDirectories() {
            return this.directories;
        }

        public List<Path> getOther() {
            return this.other;
        }

        public Path getBasePath() {
            return this.basePath;
        }

        public long getFileCount() {
            return this.files.size();
        }

        public long getDirCount() {
            return this.directories.size();
        }

        public long getOtherCount() {
            return this.other.size();
        }

        public long totalCount() {
            return this.getFileCount() + this.getDirCount() + this.getOtherCount();
        }
    }
}

