/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.microsoft.azure.storage.AccessCondition;
import com.microsoft.azure.storage.OperationContext;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobRequestOptions;
import com.microsoft.azure.storage.blob.BlockEntry;
import com.microsoft.azure.storage.blob.BlockListingFilter;
import com.microsoft.azure.storage.blob.BlockSearchMode;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.fs.StreamCapabilities;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.fs.azure.AzureException;
import org.apache.hadoop.fs.azure.NativeAzureFileSystemHelper;
import org.apache.hadoop.fs.azure.SelfRenewingLease;
import org.apache.hadoop.fs.azure.StorageInterface;
import org.apache.hadoop.io.ElasticByteBufferPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockBlobAppendStream
extends OutputStream
implements Syncable,
StreamCapabilities {
    private final String key;
    private boolean blobExist;
    private SelfRenewingLease lease = null;
    private final boolean compactionEnabled;
    private static final int DEFAULT_ACTIVATE_COMPACTION_BLOCK_COUNT = 32000;
    private int activateCompactionBlockCount = 32000;
    private final AtomicInteger maxBlockSize;
    private ByteBuffer outBuffer;
    private final AtomicLong committedBlobLength = new AtomicLong(0L);
    private volatile long blobLength = 0L;
    private static final int CLOSE_UPLOAD_DELAY = 10;
    private static final int THREADPOOL_KEEP_ALIVE = 30;
    private final StorageInterface.CloudBlockBlobWrapper blob;
    private final OperationContext opContext;
    private final ConcurrentLinkedQueue<UploadCommand> activeBlockCommands = new ConcurrentLinkedQueue();
    private volatile boolean closed = false;
    private final AtomicReference<IOException> firstError = new AtomicReference();
    private boolean firstErrorThrown = false;
    private final Semaphore uploadingSemaphore = new Semaphore(4, true);
    private final ElasticByteBufferPool poolReadyByteBuffers = new ElasticByteBufferPool();
    private final List<BlockEntry> blockEntries = new ArrayList<BlockEntry>(1024);
    private static final int DEFAULT_CAPACITY_BLOCK_ENTRIES = 1024;
    private final ConcurrentLinkedDeque<BlockEntry> uncommittedBlockEntries = new ConcurrentLinkedDeque();
    private static final int UNSET_BLOCKS_COUNT = -1;
    private long nextBlockCount = -1L;
    private String blockIdPrefix = null;
    private static final int MAX_NUMBER_THREADS_IN_THREAD_POOL = 4;
    private static final int MAX_BLOCK_UPLOAD_RETRIES = 3;
    private static final int BLOCK_UPLOAD_RETRY_INTERVAL = 1000;
    private static final Logger LOG = LoggerFactory.getLogger(BlockBlobAppendStream.class);
    private static final int MAX_BLOCK_COUNT = 100000;
    private ThreadPoolExecutor ioThreadPool;
    private final AccessCondition accessCondition = new AccessCondition();
    private final AtomicInteger threadSequenceNumber;
    private static final String THREAD_ID_PREFIX = "append-blockblob";

    public BlockBlobAppendStream(StorageInterface.CloudBlockBlobWrapper blob, String aKey, int bufferSize, boolean compactionEnabled, OperationContext opContext) throws IOException {
        Preconditions.checkArgument((boolean)StringUtils.isNotEmpty((CharSequence)aKey));
        Preconditions.checkArgument((bufferSize >= 0 ? 1 : 0) != 0);
        this.blob = blob;
        this.opContext = opContext;
        this.key = aKey;
        this.maxBlockSize = new AtomicInteger(bufferSize);
        this.threadSequenceNumber = new AtomicInteger(0);
        this.blockIdPrefix = null;
        this.compactionEnabled = compactionEnabled;
        this.blobExist = true;
        this.outBuffer = this.poolReadyByteBuffers.getBuffer(false, this.maxBlockSize.get());
        try {
            this.blockEntries.addAll(blob.downloadBlockList(BlockListingFilter.COMMITTED, new BlobRequestOptions(), opContext));
            this.blobLength = blob.getProperties().getLength();
            this.committedBlobLength.set(this.blobLength);
            this.lease = new SelfRenewingLease(blob, true);
            this.accessCondition.setLeaseID(this.lease.getLeaseID());
        }
        catch (StorageException ex) {
            if (ex.getErrorCode().equals("BlobNotFound")) {
                this.blobExist = false;
            }
            if (ex.getErrorCode().equals("LeaseAlreadyPresent")) {
                throw new AzureException("Unable to set Append lease on the Blob: " + (Object)((Object)ex), ex);
            }
            LOG.debug("Encountered storage exception. StorageException : {} ErrorCode : {}", (Object)ex, (Object)ex.getErrorCode());
            throw new AzureException(ex);
        }
        this.setBlocksCountAndBlockIdPrefix(this.blockEntries);
        this.ioThreadPool = new ThreadPoolExecutor(4, 4, 30L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), new UploaderThreadFactory());
    }

    @VisibleForTesting
    synchronized void setMaxBlockSize(int size) {
        this.maxBlockSize.set(size);
        this.outBuffer = ByteBuffer.allocate(this.maxBlockSize.get());
    }

    @VisibleForTesting
    void setCompactionBlockCount(int activationCount) {
        this.activateCompactionBlockCount = activationCount;
    }

    @VisibleForTesting
    List<BlockEntry> getBlockList() throws StorageException, IOException {
        return this.blob.downloadBlockList(BlockListingFilter.COMMITTED, new BlobRequestOptions(), this.opContext);
    }

    @Override
    public void write(int byteVal) throws IOException {
        this.write(new byte[]{(byte)(byteVal & 0xFF)});
    }

    @Override
    public synchronized void write(byte[] data, int offset, int length) throws IOException {
        Preconditions.checkArgument((data != null ? 1 : 0) != 0, (Object)"null data");
        if (offset < 0 || length < 0 || length > data.length - offset) {
            throw new IndexOutOfBoundsException();
        }
        if (this.closed) {
            throw new IOException("Stream is closed!");
        }
        while (this.outBuffer.remaining() < length) {
            int remaining = this.outBuffer.remaining();
            this.outBuffer.put(data, offset, remaining);
            this.addBlockUploadCommand();
            offset += remaining;
            length -= remaining;
        }
        this.outBuffer.put(data, offset, length);
    }

    @Override
    public void flush() throws IOException {
        if (this.closed) {
            return;
        }
        this.addBlockUploadCommand();
        if (this.committedBlobLength.get() < this.blobLength) {
            try {
                this.addFlushCommand().await();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public void hsync() throws IOException {
        if (this.compactionEnabled) {
            this.flush();
        }
    }

    public void hflush() throws IOException {
        if (this.compactionEnabled) {
            this.flush();
        }
    }

    public boolean hasCapability(String capability) {
        if (!this.compactionEnabled) {
            return false;
        }
        switch (capability.toLowerCase(Locale.ENGLISH)) {
            case "hsync": 
            case "hflush": {
                return true;
            }
        }
        return false;
    }

    @Override
    public synchronized void close() throws IOException {
        LOG.debug("close {} ", (Object)this.key);
        if (this.closed) {
            return;
        }
        this.flush();
        this.ioThreadPool.shutdown();
        try {
            if (!this.ioThreadPool.awaitTermination(10L, TimeUnit.MINUTES)) {
                LOG.error("Time out occurred while close() is waiting for IO request to finish in append for blob : {}", (Object)this.key);
                NativeAzureFileSystemHelper.logAllLiveStackTraces();
                throw new AzureException("Timed out waiting for IO requests to finish");
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
        if (this.firstError.get() == null && this.blobExist) {
            try {
                this.lease.free();
            }
            catch (StorageException ex) {
                LOG.debug("Lease free update blob {} encountered Storage Exception: {} Error Code : {}", new Object[]{this.key, ex, ex.getErrorCode()});
                this.maybeSetFirstError(new AzureException(ex));
            }
        }
        this.closed = true;
        if (this.firstError.get() != null && !this.firstErrorThrown) {
            throw this.firstError.get();
        }
    }

    private void setBlocksCountAndBlockIdPrefix(List<BlockEntry> blockEntries) {
        if (this.nextBlockCount == -1L && this.blockIdPrefix == null) {
            Random sequenceGenerator = new Random();
            String blockZeroBlockId = !blockEntries.isEmpty() ? blockEntries.get(0).getId() : "";
            String prefix = UUID.randomUUID().toString() + "-";
            String sampleNewerVersionBlockId = this.generateNewerVersionBlockId(prefix, 0L);
            if (!blockEntries.isEmpty() && blockZeroBlockId.length() < sampleNewerVersionBlockId.length()) {
                this.blockIdPrefix = "";
                this.nextBlockCount = (long)sequenceGenerator.nextInt(Integer.MAX_VALUE) + (long)sequenceGenerator.nextInt(2147383647);
                this.nextBlockCount += (long)blockEntries.size();
            } else {
                this.blockIdPrefix = prefix;
                this.nextBlockCount = blockEntries.size();
            }
        }
    }

    private String generateBlockId() throws IOException {
        if (this.nextBlockCount == -1L || this.blockIdPrefix == null) {
            throw new AzureException("Append Stream in invalid state. nextBlockCount not set correctly");
        }
        return !this.blockIdPrefix.isEmpty() ? this.generateNewerVersionBlockId(this.blockIdPrefix, this.nextBlockCount++) : this.generateOlderVersionBlockId(this.nextBlockCount++);
    }

    private String generateOlderVersionBlockId(long id) {
        byte[] blockIdInBytes = new byte[8];
        for (int m = 0; m < 8; ++m) {
            blockIdInBytes[7 - m] = (byte)(id >> 8 * m & 0xFFL);
        }
        return new String(Base64.encodeBase64((byte[])blockIdInBytes), StandardCharsets.UTF_8);
    }

    private String generateNewerVersionBlockId(String prefix, long id) {
        String blockIdSuffix = String.format("%06d", id);
        byte[] blockIdInBytes = (prefix + blockIdSuffix).getBytes(StandardCharsets.UTF_8);
        return new String(Base64.encodeBase64((byte[])blockIdInBytes), StandardCharsets.UTF_8);
    }

    private void writeBlockRequestInternal(String blockId, ByteBuffer dataPayload, boolean bufferPoolBuffer) {
        int uploadRetryAttempts;
        AzureException lastLocalException = null;
        for (uploadRetryAttempts = 0; uploadRetryAttempts < 3; ++uploadRetryAttempts) {
            try {
                long startTime = System.nanoTime();
                this.blob.uploadBlock(blockId, this.accessCondition, new ByteArrayInputStream(dataPayload.array()), dataPayload.position(), new BlobRequestOptions(), this.opContext);
                LOG.debug("upload block finished for {} ms. block {} ", (Object)TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime), (Object)blockId);
            }
            catch (Exception ioe) {
                LOG.debug("Encountered exception during uploading block for Blob {} Exception : {}", (Object)this.key, (Object)ioe);
                lastLocalException = new AzureException("Encountered Exception while uploading block: " + ioe, ioe);
                try {
                    Thread.sleep(1000 * (uploadRetryAttempts + 1));
                    continue;
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
            break;
        }
        if (bufferPoolBuffer) {
            this.poolReadyByteBuffers.putBuffer(dataPayload);
        }
        if (uploadRetryAttempts == 3) {
            this.maybeSetFirstError(lastLocalException);
        }
    }

    private void maybeSetFirstError(IOException exception) {
        this.firstError.compareAndSet(null, exception);
    }

    private void maybeThrowFirstError() throws IOException {
        if (this.firstError.get() != null) {
            this.firstErrorThrown = true;
            throw this.firstError.get();
        }
    }

    private void writeBlockListRequestInternal() {
        int uploadRetryAttempts;
        AzureException lastLocalException = null;
        for (uploadRetryAttempts = 0; uploadRetryAttempts < 3; ++uploadRetryAttempts) {
            try {
                long startTime = System.nanoTime();
                this.blob.commitBlockList(this.blockEntries, this.accessCondition, new BlobRequestOptions(), this.opContext);
                LOG.debug("Upload block list took {} ms for blob {} ", (Object)TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime), (Object)this.key);
            }
            catch (Exception ioe) {
                LOG.debug("Encountered exception during uploading block for Blob {} Exception : {}", (Object)this.key, (Object)ioe);
                lastLocalException = new AzureException("Encountered Exception while uploading block: " + ioe, ioe);
                try {
                    Thread.sleep(1000 * (uploadRetryAttempts + 1));
                    continue;
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
            break;
        }
        if (uploadRetryAttempts == 3) {
            this.maybeSetFirstError(lastLocalException);
        }
    }

    private synchronized void addBlockUploadCommand() throws IOException {
        this.maybeThrowFirstError();
        if (this.blobExist && this.lease.isFreed()) {
            throw new AzureException(String.format("Attempting to upload a block on blob : %s  that does not have lease on the Blob. Failing upload", this.key));
        }
        int blockSize = this.outBuffer.position();
        if (blockSize > 0) {
            UploadBlockCommand command = new UploadBlockCommand(this.generateBlockId(), this.outBuffer);
            this.activeBlockCommands.add(command);
            this.blobLength += (long)blockSize;
            this.outBuffer = this.poolReadyByteBuffers.getBuffer(false, this.maxBlockSize.get());
            this.ioThreadPool.execute(new WriteRequest(command));
        }
    }

    private synchronized UploadCommand addFlushCommand() throws IOException {
        this.maybeThrowFirstError();
        if (this.blobExist && this.lease.isFreed()) {
            throw new AzureException(String.format("Attempting to upload block list on blob : %s that does not have lease on the Blob. Failing upload", this.key));
        }
        UploadBlockListCommand command = new UploadBlockListCommand();
        this.activeBlockCommands.add(command);
        this.ioThreadPool.execute(new WriteRequest(command));
        return command;
    }

    private class WriteRequest
    implements Runnable {
        private final UploadCommand command;

        WriteRequest(UploadCommand command) {
            this.command = command;
        }

        @Override
        public void run() {
            try {
                this.command.dump();
                long startTime = System.nanoTime();
                this.command.execute();
                this.command.setCompleted();
                LOG.debug("command finished for {} ms", (Object)TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime));
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
            catch (Exception ex) {
                LOG.debug("Encountered exception during execution of command for Blob : {} Exception : {}", (Object)BlockBlobAppendStream.this.key, (Object)ex);
                BlockBlobAppendStream.this.firstError.compareAndSet(null, new AzureException(ex));
            }
        }
    }

    private class UploadBlockListCommand
    extends UploadCommand {
        private BlockEntry lastBlock;

        UploadBlockListCommand() {
            super(BlockBlobAppendStream.this.blobLength);
            this.lastBlock = null;
            if (!BlockBlobAppendStream.this.uncommittedBlockEntries.isEmpty()) {
                this.lastBlock = (BlockEntry)BlockBlobAppendStream.this.uncommittedBlockEntries.getLast();
            }
        }

        @Override
        void awaitAsDependent() throws InterruptedException {
        }

        @Override
        void dump() {
            LOG.debug("commit block list with {} blocks for blob {}", (Object)BlockBlobAppendStream.this.uncommittedBlockEntries.size(), (Object)BlockBlobAppendStream.this.key);
        }

        @Override
        public void execute() throws InterruptedException, IOException {
            UploadCommand activeCommand;
            BlockEntry uncommittedBlock;
            UploadCommand activeCommand2;
            if (BlockBlobAppendStream.this.committedBlobLength.get() >= this.getCommandBlobOffset()) {
                LOG.debug("commit already applied for {}", (Object)BlockBlobAppendStream.this.key);
                return;
            }
            if (this.lastBlock == null) {
                LOG.debug("nothing to commit for {}", (Object)BlockBlobAppendStream.this.key);
                return;
            }
            LOG.debug("active commands: {} for {}", (Object)BlockBlobAppendStream.this.activeBlockCommands.size(), (Object)BlockBlobAppendStream.this.key);
            Iterator iterator = BlockBlobAppendStream.this.activeBlockCommands.iterator();
            while (iterator.hasNext() && (activeCommand2 = (UploadCommand)iterator.next()).getCommandBlobOffset() < this.getCommandBlobOffset()) {
                activeCommand2.dump();
                activeCommand2.awaitAsDependent();
            }
            BlockBlobAppendStream.this.uploadingSemaphore.acquire(4);
            do {
                uncommittedBlock = (BlockEntry)BlockBlobAppendStream.this.uncommittedBlockEntries.poll();
                BlockBlobAppendStream.this.blockEntries.add(uncommittedBlock);
            } while (uncommittedBlock != this.lastBlock);
            if (BlockBlobAppendStream.this.blockEntries.size() > BlockBlobAppendStream.this.activateCompactionBlockCount) {
                LOG.debug("Block compaction: activated with {} blocks for {}", (Object)BlockBlobAppendStream.this.blockEntries.size(), (Object)BlockBlobAppendStream.this.key);
                long startCompaction = System.nanoTime();
                this.blockCompaction();
                LOG.debug("Block compaction finished for {} ms with {} blocks for {}", new Object[]{TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startCompaction), BlockBlobAppendStream.this.blockEntries.size(), BlockBlobAppendStream.this.key});
            }
            BlockBlobAppendStream.this.writeBlockListRequestInternal();
            BlockBlobAppendStream.this.uploadingSemaphore.release(4);
            Iterator it = BlockBlobAppendStream.this.activeBlockCommands.iterator();
            while (it.hasNext() && (activeCommand = (UploadCommand)it.next()).getCommandBlobOffset() <= this.getCommandBlobOffset()) {
                it.remove();
            }
            BlockBlobAppendStream.this.committedBlobLength.set(this.getCommandBlobOffset());
        }

        private void blockCompaction() throws IOException {
            int segmentBegin = 0;
            int segmentEnd = 0;
            long segmentOffsetBegin = 0L;
            long segmentOffsetEnd = 0L;
            int maxSegmentBegin = 0;
            int maxSegmentEnd = 0;
            long maxSegmentOffsetBegin = 0L;
            long maxSegmentOffsetEnd = 0L;
            for (BlockEntry block : BlockBlobAppendStream.this.blockEntries) {
                ++segmentEnd;
                if ((segmentOffsetEnd += block.getSize()) - segmentOffsetBegin <= (long)BlockBlobAppendStream.this.maxBlockSize.get()) continue;
                if (segmentEnd - segmentBegin > 2 && maxSegmentEnd - maxSegmentBegin < segmentEnd - segmentBegin) {
                    maxSegmentBegin = segmentBegin;
                    maxSegmentEnd = segmentEnd;
                    maxSegmentOffsetBegin = segmentOffsetBegin;
                    maxSegmentOffsetEnd = segmentOffsetEnd - block.getSize();
                }
                segmentBegin = segmentEnd - 1;
                segmentOffsetBegin = segmentOffsetEnd - block.getSize();
            }
            if (maxSegmentEnd - maxSegmentBegin > 1) {
                LOG.debug("Block compaction: {} blocks for {}", (Object)(maxSegmentEnd - maxSegmentBegin), (Object)BlockBlobAppendStream.this.key);
                ByteArrayOutputStreamInternal blockOutputStream = new ByteArrayOutputStreamInternal(BlockBlobAppendStream.this.maxBlockSize.get());
                try {
                    long length = maxSegmentOffsetEnd - maxSegmentOffsetBegin;
                    BlockBlobAppendStream.this.blob.downloadRange(maxSegmentOffsetBegin, length, blockOutputStream, new BlobRequestOptions(), BlockBlobAppendStream.this.opContext);
                }
                catch (StorageException ex) {
                    LOG.error("Storage exception encountered during block compaction phase : {} Storage Exception : {} Error Code: {}", new Object[]{BlockBlobAppendStream.this.key, ex, ex.getErrorCode()});
                    throw new AzureException("Encountered Exception while committing append blocks " + (Object)((Object)ex), ex);
                }
                String blockId = BlockBlobAppendStream.this.generateBlockId();
                ByteBuffer byteBuffer = ByteBuffer.wrap(blockOutputStream.getByteArray());
                byteBuffer.position(blockOutputStream.size());
                BlockBlobAppendStream.this.writeBlockRequestInternal(blockId, byteBuffer, false);
                BlockBlobAppendStream.this.blockEntries.subList(maxSegmentBegin + 1, maxSegmentEnd - 1).clear();
                BlockEntry newBlock = (BlockEntry)BlockBlobAppendStream.this.blockEntries.get(maxSegmentBegin);
                newBlock.setId(blockId);
                newBlock.setSearchMode(BlockSearchMode.LATEST);
                newBlock.setSize(maxSegmentOffsetEnd - maxSegmentOffsetBegin);
            }
        }

        private class ByteArrayOutputStreamInternal
        extends ByteArrayOutputStream {
            ByteArrayOutputStreamInternal(int size) {
                super(size);
            }

            byte[] getByteArray() {
                return this.buf;
            }
        }
    }

    private class UploadBlockCommand
    extends UploadCommand {
        private final ByteBuffer payload;
        private final BlockEntry entry;

        UploadBlockCommand(String blockId, ByteBuffer payload) {
            super(BlockBlobAppendStream.this.blobLength);
            BlockEntry blockEntry = new BlockEntry(blockId);
            blockEntry.setSize((long)payload.position());
            blockEntry.setSearchMode(BlockSearchMode.LATEST);
            this.payload = payload;
            this.entry = blockEntry;
            BlockBlobAppendStream.this.uncommittedBlockEntries.add(blockEntry);
        }

        @Override
        void execute() throws InterruptedException {
            BlockBlobAppendStream.this.uploadingSemaphore.acquire(1);
            BlockBlobAppendStream.this.writeBlockRequestInternal(this.entry.getId(), this.payload, true);
            BlockBlobAppendStream.this.uploadingSemaphore.release(1);
        }

        @Override
        void dump() {
            LOG.debug("upload block {} size: {} for blob {}", new Object[]{this.entry.getId(), this.entry.getSize(), BlockBlobAppendStream.this.key});
        }
    }

    class UploaderThreadFactory
    implements ThreadFactory {
        UploaderThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r);
            t.setName(String.format("%s-%d", BlockBlobAppendStream.THREAD_ID_PREFIX, BlockBlobAppendStream.this.threadSequenceNumber.getAndIncrement()));
            return t;
        }
    }

    private abstract class UploadCommand {
        private final long commandBlobOffset;
        private final CountDownLatch completed = new CountDownLatch(1);

        UploadCommand(long offset) {
            this.commandBlobOffset = offset;
        }

        long getCommandBlobOffset() {
            return this.commandBlobOffset;
        }

        void await() throws InterruptedException {
            this.completed.await();
        }

        void awaitAsDependent() throws InterruptedException {
            this.await();
        }

        void setCompleted() {
            this.completed.countDown();
        }

        void execute() throws InterruptedException, IOException {
        }

        void dump() {
        }
    }
}

