/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.eclipse.ebam.emf.interfaces.ICacheTransaction;
import org.eclipse.ebam.emf.utilities.CacheSingleton;
import org.eclipse.ebam.model.entities.Journal;
import org.eclipse.ebam.model.entities.MessageContent;
import org.eclipse.ebam.model.entities.SbiResources;
import org.eclipse.ebam.model.entities.ServiceInstance;
import org.eclipse.persistence.queries.ReadQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class CachingTransaction extends EMF implements ICacheTransaction {
    /**
     * Logger.
     */
    private static Logger logger = LoggerFactory.getLogger(CachingTransaction.class);
	
    
    
    /**
     * Method that executes database query to extract
     * <code>ServiceInstance</code> by label.
     * 
     * @param label
     *            to search
     * @param serviceID
     *            service id
     * @return <code>ServiceInstance</code>
     * @throws Exception
     *             error
     */
    @SuppressWarnings("finally")
    private final ServiceInstance getServiceInstanceByLabel(
	    final EntityManager em, final String label, final Integer serviceID)
	    throws Exception {
	logger.debug("IN");
	long start=new Date().getTime();
	ServiceInstance servInst = null;
	try {
	    Query query = em
		    .createQuery("select e from ServiceInstance "
			    + "e where e.label = :label and e.service.idService = :serviceID");
	    query.setParameter("label", label);
	    query.setParameter("serviceID", serviceID);
	    servInst = (ServiceInstance) query.getSingleResult();
	} catch (NoResultException e) {

	} catch (Exception e) {
	    logger.error(e.getMessage(), e);
	    throw e;
	} finally {
	    logger.debug("OUT");
	    return servInst;
	}
    }

    /**
     * Method that stores Journal, ServiceInstance and relevant data on db.
     * 
     * @param j
     *            <code>Journal</code> to be stored
     * @param serviceInstance
     *            <code>ServiceInstance</code> already present in DB
     * @param messageContents
     *            List of relevant data to be stored
     * @return operation result
     */
	public boolean saveJournal(final Journal j,
			final List<MessageContent> messageContents) {
		logger.debug("IN." + j.getLabel());
		boolean isSaved = false;

		EntityManager em = getEMF().createEntityManager();
		em.setFlushMode(FlushModeType.COMMIT);
		em.getTransaction().begin();

		String siLabel = j.getServiceInstance().getLabel();
		Integer serviceID = j.getServiceInstance().getService().getIdService();
		try {

			ServiceInstance serviceInstance = getServiceInstanceByLabel(em,
					siLabel, serviceID);

			if (serviceInstance == null) {
				if (j.getServiceInstance() != null
						&& j.getServiceInstance().getStartdate() != null
						&& j.getServiceInstance().getLabel() != null
						&& j.getServiceInstance().getService() != null) {
					
					// set end date to 12/12/9999
					SimpleDateFormat sdf = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss");
					Date dtEnd = sdf.parse("12/12/9999 00:00:00");
					j.getServiceInstance().setEnddate(dtEnd);
					em.persist(j.getServiceInstance());
					em.flush();
					serviceInstance=j.getServiceInstance();
				} else {
					logger.error("Service Instance is NOT complete...Message="
							+ j.getXmlData());
				}

				logger.info("Service Instance "
						+ j.getServiceInstance().getLabel() + " saved");
			} else {
				logger.info("There is one Service Instance already on DB");
				
				// Check if the start dates are differents
				if (j.getServiceInstance().getStartdate() != null){
					Date dbDate=serviceInstance.getStartdate();
					Date messageDate=j.getServiceInstance().getStartdate();
					if (dbDate.compareTo(messageDate)<0){
						logger.info("  SAVE the START DATE");
						serviceInstance.setStartdate(j.getServiceInstance().getStartdate());
						j.setServiceInstance(serviceInstance);
						em.merge(serviceInstance);
						em.flush();
					}
				}
				
				if (j.getServiceInstance().getEnddate() != null) {
					logger.info("j.getServiceInstance().getEnddate() != null.  SAVE END DATE");
					serviceInstance.setEnddate(j.getServiceInstance().getEnddate());
					j.setServiceInstance(serviceInstance);
					em.merge(serviceInstance);
					em.flush();
				} else{
					j.setServiceInstance(serviceInstance);
				}
					
			}
			Double serviceInstanceId=serviceInstance.getIdServiceInstance();
			em.persist(j);
			em.flush();

			logger.info("Journal " + j.getJurnalId() + " / " + j.getLabel()
					+ " saved");

			if (messageContents != null) {
				for (int i = 0; i < messageContents.size(); i++) {
					MessageContent mcTmp=(MessageContent)messageContents.get(i);
					mcTmp.setServiceInstanceId(serviceInstanceId);
					em.persist(mcTmp);

					logger.info("Relevant Data Saved:"
							+ ((MessageContent) messageContents.get(i))
									.getName()
							+ " - "
							+ ((MessageContent) messageContents.get(i))
									.getValue());
				}
				logger.info("Relevant data saved");
			}

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
				em.close();
			}
			logger.error("ROLL BACK....", e);
			throw e;
		} finally {
			em.close();
			logger.debug("OUT.isSaved=" + isSaved);
			return isSaved;
		}
	}

	public SbiResources getSbiResourcesByName(final String name) {
		logger.debug("IN:"+name);
		SbiResources sbiResources = null;

		if (CacheSingleton.getInstance().get(name,"SbiResources") != null) {

			sbiResources = (SbiResources) CacheSingleton.getInstance().get(name,"SbiResources");
			logger.info("Take SbiResources from cache...");
		} else {

			EntityManager em = getEMF().createEntityManager();
			try {
				Query query = em
						.createQuery("select r from SbiResources r where r.resourceName = :name ");
				query.setParameter("name", name);
				sbiResources = (SbiResources) query.getSingleResult();
				CacheSingleton.getInstance().put(name, sbiResources,"SbiResources");
				logger.info("Put SbiResources from cache...");
			} catch (NoResultException e) {
			 logger.debug(e.getMessage(), e);
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			} finally {
				em.close();

			}
		}
		logger.debug("OUT");
		return sbiResources;
	}
    
}
