/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.inject.Injector;
import java.util.Iterator;
import java.util.Set;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.incquery.patternlanguage.helper.CorePatternLanguageHelper;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.extensibility.QuerySpecificationRegistry;
import org.eclipse.incquery.snapshot.EIQSnapshot.IncQuerySnapshot;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.incquery.testing.core.XmiModelUtil;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper methods for loading models from files or URIs.
 */
@SuppressWarnings("all")
public class ModelLoadHelper {
  /**
   * Load an instance EMF model from the given file to a new resource set.
   */
  public Resource loadModelFromFile(final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadModelFromUri(_string);
  }
  
  /**
   * Load an instance EMF model from the given platform URI to a new resource set.
   */
  public Resource loadModelFromUri(final String platformUri) {
    Resource _xblockexpression = null;
    {
      final ResourceSetImpl resourceSet = new ResourceSetImpl();
      _xblockexpression = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
    }
    return _xblockexpression;
  }
  
  /**
   * Load an additional resource into the resource set from a given file.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromFile(final ResourceSet resourceSet, final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadAdditionalResourceFromUri(resourceSet, _string);
  }
  
  /**
   * Load an additional resource into the resource set from a given platform URI.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromUri(final ResourceSet resourceSet, final String platformUri) {
    Resource _xblockexpression = null;
    {
      final URI modelURI = XmiModelUtil.resolvePlatformURI(XmiModelUtil.XmiModelUtilRunningOptionEnum.BOTH, platformUri);
      Resource _xifexpression = null;
      boolean _notEquals = (!Objects.equal(modelURI, null));
      if (_notEquals) {
        _xifexpression = resourceSet.getResource(modelURI, true);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Load a pattern model from the given file into a new resource set.
   */
  public PatternModel loadPatternModelFromFile(final IFile file, final Injector injector) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadPatternModelFromUri(_string, injector);
  }
  
  /**
   * Load a pattern model from the given platform URI into a new resource set.
   */
  public PatternModel loadPatternModelFromUri(final String platformUri, final Injector injector, final String... additionalDependencyPlatformUris) {
    PatternModel _xblockexpression = null;
    {
      final ResourceSet resourceSet = XmiModelUtil.prepareXtextResource(injector);
      for (final String additionalUri : additionalDependencyPlatformUris) {
        this.loadAdditionalResourceFromUri(resourceSet, additionalUri);
      }
      final Resource resource = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      PatternModel _xifexpression = null;
      EList<EObject> _contents = resource.getContents();
      int _size = _contents.size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        PatternModel _xifexpression_1 = null;
        EList<EObject> _contents_1 = resource.getContents();
        EObject _get = _contents_1.get(0);
        if ((_get instanceof PatternModel)) {
          EList<EObject> _contents_2 = resource.getContents();
          EObject _get_1 = _contents_2.get(0);
          _xifexpression_1 = ((PatternModel) _get_1);
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Initialize a matcher for the pattern with the given name from the pattern model on the selected EMF root.
   */
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final IncQueryEngine engine, final String patternName) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        EList<Pattern> _patterns = model.getPatterns();
        final Function1<Pattern, Boolean> _function = new Function1<Pattern, Boolean>() {
          public Boolean apply(final Pattern it) {
            String _fullyQualifiedName = CorePatternLanguageHelper.getFullyQualifiedName(it);
            return Boolean.valueOf(Objects.equal(patternName, _fullyQualifiedName));
          }
        };
        final Iterable<Pattern> patterns = IterableExtensions.<Pattern>filter(_patterns, _function);
        int _size = IterableExtensions.size(patterns);
        boolean _equals = (_size == 1);
        Preconditions.checkState(_equals, ("No pattern found with name " + patternName));
        Set<IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>>> _registeredQuerySpecifications = engine.getRegisteredQuerySpecifications();
        final SpecificationBuilder builder = new SpecificationBuilder(_registeredQuerySpecifications);
        Iterator<Pattern> _iterator = patterns.iterator();
        Pattern _next = _iterator.next();
        IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> _orCreateSpecification = builder.getOrCreateSpecification(_next);
        _xblockexpression = engine.getMatcher(_orCreateSpecification);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final Notifier emfRoot, final String patternName) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        final IncQueryEngine engine = IncQueryEngine.on(emfRoot);
        _xblockexpression = this.initializeMatcherFromModel(model, engine, patternName);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Initialize a registered matcher for the pattern FQN on the selected EMF root.
   */
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromRegistry(final Notifier emfRoot, final String patternFQN) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        final IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> querySpecification = QuerySpecificationRegistry.getQuerySpecification(patternFQN);
        IncQueryEngine _on = IncQueryEngine.on(emfRoot);
        _xblockexpression = querySpecification.getMatcher(_on);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given file.
   */
  public IncQuerySnapshot loadExpectedResultsFromFile(final ResourceSet resourceSet, final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadExpectedResultsFromUri(resourceSet, _string);
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given platform URI.
   */
  public IncQuerySnapshot loadExpectedResultsFromUri(final ResourceSet resourceSet, final String platformUri) {
    IncQuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      IncQuerySnapshot _xifexpression = null;
      boolean _notEquals = (!Objects.equal(resource, null));
      if (_notEquals) {
        IncQuerySnapshot _xifexpression_1 = null;
        EList<EObject> _contents = resource.getContents();
        int _size = _contents.size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          IncQuerySnapshot _xifexpression_2 = null;
          EList<EObject> _contents_1 = resource.getContents();
          EObject _get = _contents_1.get(0);
          if ((_get instanceof IncQuerySnapshot)) {
            EList<EObject> _contents_2 = resource.getContents();
            EObject _get_1 = _contents_2.get(0);
            _xifexpression_2 = ((IncQuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Load the recorded match set into a new resource set form the given file.
   */
  public IncQuerySnapshot loadExpectedResultsFromFile(final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadExpectedResultsFromUri(_string);
  }
  
  /**
   * Load the recorded match set into a new resource set form the given platform URI.
   */
  public IncQuerySnapshot loadExpectedResultsFromUri(final String platformUri) {
    IncQuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadModelFromUri(platformUri);
      IncQuerySnapshot _xifexpression = null;
      boolean _notEquals = (!Objects.equal(resource, null));
      if (_notEquals) {
        IncQuerySnapshot _xifexpression_1 = null;
        EList<EObject> _contents = resource.getContents();
        int _size = _contents.size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          IncQuerySnapshot _xifexpression_2 = null;
          EList<EObject> _contents_1 = resource.getContents();
          EObject _get = _contents_1.get(0);
          if ((_get instanceof IncQuerySnapshot)) {
            EList<EObject> _contents_2 = resource.getContents();
            EObject _get_1 = _contents_2.get(0);
            _xifexpression_2 = ((IncQuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given file.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromFile(final ResourceSet resourceSet, final IFile file, final String patternFQN) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    return this.loadExpectedResultsForPatternFromUri(resourceSet, _string, patternFQN);
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given platform URI.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromUri(final ResourceSet resourceSet, final String platformUri, final String patternFQN) {
    Resource _loadAdditionalResourceFromUri = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
    final IncQuerySnapshot snapshot = ((IncQuerySnapshot) _loadAdditionalResourceFromUri);
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    final Function1<MatchSetRecord, Boolean> _function = new Function1<MatchSetRecord, Boolean>() {
      public Boolean apply(final MatchSetRecord it) {
        String _patternQualifiedName = it.getPatternQualifiedName();
        return Boolean.valueOf(_patternQualifiedName.equals(patternFQN));
      }
    };
    final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
    int _size = IterableExtensions.size(matchsetrecord);
    boolean _equals = (_size == 1);
    if (_equals) {
      Iterator<MatchSetRecord> _iterator = matchsetrecord.iterator();
      return _iterator.next();
    }
    return null;
  }
}
