/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.extensibility.QuerySpecificationRegistry;
import org.eclipse.incquery.snapshot.EIQSnapshot.BooleanSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.DateSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.DoubleSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.EIQSnapshotFactory;
import org.eclipse.incquery.snapshot.EIQSnapshot.EMFSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.EnumSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.FloatSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.IncQuerySnapshot;
import org.eclipse.incquery.snapshot.EIQSnapshot.InputSpecification;
import org.eclipse.incquery.snapshot.EIQSnapshot.IntSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.LongSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSubstitutionRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MiscellaneousSubstitution;
import org.eclipse.incquery.snapshot.EIQSnapshot.StringSubstitution;
import org.eclipse.xtext.junit4.util.ParseHelper;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Helper methods for dealing with snapshots and match set records.
 */
@SuppressWarnings("all")
public class SnapshotHelper {
  @Inject
  private ParseHelper parseHelper;
  
  /**
   * Returns the EMF root that was used by the matchers recorded into the given snapshot,
   *  based on the input specification and the model roots.
   */
  public Notifier getEMFRootForSnapshot(final IncQuerySnapshot snapshot) {
    Notifier _xifexpression = null;
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.EOBJECT);
    if (_equals) {
      EObject _xifexpression_1 = null;
      EList<EObject> _modelRoots = snapshot.getModelRoots();
      int _size = _modelRoots.size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        EList<EObject> _modelRoots_1 = snapshot.getModelRoots();
        EObject _get = _modelRoots_1.get(0);
        _xifexpression_1 = _get;
      }
      _xifexpression = _xifexpression_1;
    } else {
      Notifier _xifexpression_2 = null;
      InputSpecification _inputSpecification_1 = snapshot.getInputSpecification();
      boolean _equals_1 = Objects.equal(_inputSpecification_1, InputSpecification.RESOURCE);
      if (_equals_1) {
        Resource _xifexpression_3 = null;
        EList<EObject> _modelRoots_2 = snapshot.getModelRoots();
        int _size_1 = _modelRoots_2.size();
        boolean _greaterThan_1 = (_size_1 > 0);
        if (_greaterThan_1) {
          EList<EObject> _modelRoots_3 = snapshot.getModelRoots();
          EObject _get_1 = _modelRoots_3.get(0);
          Resource _eResource = _get_1.eResource();
          _xifexpression_3 = _eResource;
        }
        _xifexpression_2 = _xifexpression_3;
      } else {
        ResourceSet _xifexpression_4 = null;
        InputSpecification _inputSpecification_2 = snapshot.getInputSpecification();
        boolean _equals_2 = Objects.equal(_inputSpecification_2, InputSpecification.RESOURCE_SET);
        if (_equals_2) {
          Resource _eResource_1 = snapshot.eResource();
          ResourceSet _resourceSet = _eResource_1.getResourceSet();
          _xifexpression_4 = _resourceSet;
        }
        _xifexpression_2 = _xifexpression_4;
      }
      _xifexpression = _xifexpression_2;
    }
    return _xifexpression;
  }
  
  /**
   * Returns the model root that was used by the given matcher.
   */
  public ArrayList<EObject> getModelRootsForMatcher(final IncQueryMatcher matcher) {
    ArrayList<EObject> _xblockexpression = null;
    {
      IncQueryEngine _engine = matcher.getEngine();
      final Notifier root = _engine.getScope();
      ArrayList<EObject> _xifexpression = null;
      if ((root instanceof EObject)) {
        return CollectionLiterals.<EObject>newArrayList(((EObject) root));
      } else {
        ArrayList<EObject> _xifexpression_1 = null;
        if ((root instanceof Resource)) {
          ArrayList<EObject> _arrayList = new ArrayList<EObject>();
          final ArrayList<EObject> roots = _arrayList;
          EList<EObject> _contents = ((Resource) root).getContents();
          roots.addAll(_contents);
          return roots;
        } else {
          ArrayList<EObject> _xifexpression_2 = null;
          if ((root instanceof ResourceSet)) {
            ArrayList<EObject> _arrayList_1 = new ArrayList<EObject>();
            final ArrayList<EObject> roots_1 = _arrayList_1;
            EList<Resource> _resources = ((ResourceSet) root).getResources();
            final Procedure1<Resource> _function = new Procedure1<Resource>() {
              public void apply(final Resource it) {
                EList<EObject> _contents = it.getContents();
                roots_1.addAll(_contents);
              }
            };
            IterableExtensions.<Resource>forEach(_resources, _function);
            return roots_1;
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the input specification for the given matcher.
   */
  public InputSpecification getInputspecificationForMatcher(final IncQueryMatcher matcher) {
    InputSpecification _xblockexpression = null;
    {
      IncQueryEngine _engine = matcher.getEngine();
      final Notifier root = _engine.getScope();
      InputSpecification _xifexpression = null;
      if ((root instanceof EObject)) {
        _xifexpression = InputSpecification.EOBJECT;
      } else {
        InputSpecification _xifexpression_1 = null;
        if ((root instanceof Resource)) {
          _xifexpression_1 = InputSpecification.RESOURCE;
        } else {
          InputSpecification _xifexpression_2 = null;
          if ((root instanceof ResourceSet)) {
            _xifexpression_2 = InputSpecification.RESOURCE_SET;
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Saves the matches of the given matcher (using the partial match) into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public MatchSetRecord saveMatchesToSnapshot(final IncQueryMatcher matcher, final IPatternMatch partialMatch, final IncQuerySnapshot snapshot) {
    final String patternFQN = matcher.getPatternName();
    final MatchSetRecord actualRecord = EIQSnapshotFactory.eINSTANCE.createMatchSetRecord();
    actualRecord.setPatternQualifiedName(patternFQN);
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    _matchSetRecords.add(actualRecord);
    InputSpecification _inputSpecification = snapshot.getInputSpecification();
    boolean _equals = Objects.equal(_inputSpecification, InputSpecification.UNSET);
    if (_equals) {
      EList<EObject> _modelRoots = snapshot.getModelRoots();
      ArrayList<EObject> _modelRootsForMatcher = this.getModelRootsForMatcher(matcher);
      _modelRoots.addAll(_modelRootsForMatcher);
      EList<EObject> _modelRoots_1 = snapshot.getModelRoots();
      _modelRoots_1.remove(snapshot);
      InputSpecification _inputspecificationForMatcher = this.getInputspecificationForMatcher(matcher);
      snapshot.setInputSpecification(_inputspecificationForMatcher);
    }
    MatchRecord _createMatchRecordForMatch = this.createMatchRecordForMatch(partialMatch);
    actualRecord.setFilter(_createMatchRecordForMatch);
    final IMatchProcessor<IPatternMatch> _function = new IMatchProcessor<IPatternMatch>() {
      public void process(final IPatternMatch match) {
        EList<MatchRecord> _matches = actualRecord.getMatches();
        MatchRecord _createMatchRecordForMatch = SnapshotHelper.this.createMatchRecordForMatch(match);
        _matches.add(_createMatchRecordForMatch);
      }
    };
    matcher.forEachMatch(partialMatch, _function);
    return actualRecord;
  }
  
  /**
   * Creates a match record that corresponds to the given match.
   *  Each parameter with a value is saved as a substitution.
   */
  public MatchRecord createMatchRecordForMatch(final IPatternMatch match) {
    final MatchRecord matchRecord = EIQSnapshotFactory.eINSTANCE.createMatchRecord();
    List<String> _parameterNames = match.parameterNames();
    final Procedure1<String> _function = new Procedure1<String>() {
      public void apply(final String param) {
        Object _get = match.get(param);
        boolean _notEquals = (!Objects.equal(_get, null));
        if (_notEquals) {
          EList<MatchSubstitutionRecord> _substitutions = matchRecord.getSubstitutions();
          Object _get_1 = match.get(param);
          MatchSubstitutionRecord _createSubstitution = SnapshotHelper.this.createSubstitution(param, _get_1);
          _substitutions.add(_createSubstitution);
        }
      }
    };
    IterableExtensions.<String>forEach(_parameterNames, _function);
    return matchRecord;
  }
  
  /**
   * Creates a partial match that corresponds to the given match record.
   *  Each substitution is used as a value for the parameter with the same name.
   */
  public IPatternMatch createMatchForMachRecord(final IncQueryMatcher matcher, final MatchRecord matchRecord) {
    final IPatternMatch match = matcher.newEmptyMatch();
    EList<MatchSubstitutionRecord> _substitutions = matchRecord.getSubstitutions();
    final Procedure1<MatchSubstitutionRecord> _function = new Procedure1<MatchSubstitutionRecord>() {
      public void apply(final MatchSubstitutionRecord it) {
        Object target = it.getDerivedValue();
        String _parameterName = it.getParameterName();
        match.set(_parameterName, target);
      }
    };
    IterableExtensions.<MatchSubstitutionRecord>forEach(_substitutions, _function);
    return match;
  }
  
  /**
   * Saves all matches of the given matcher into the given snapshot.
   * If the input specification is not yet filled, it is now filled based on the engine of the matcher.
   */
  public MatchSetRecord saveMatchesToSnapshot(final IncQueryMatcher matcher, final IncQuerySnapshot snapshot) {
    IPatternMatch _newEmptyMatch = matcher.newEmptyMatch();
    MatchSetRecord _saveMatchesToSnapshot = this.saveMatchesToSnapshot(matcher, _newEmptyMatch, snapshot);
    return _saveMatchesToSnapshot;
  }
  
  /**
   * Returns the match set record for the given pattern FQN from the snapshot,
   *  if there is only one such record.
   */
  public MatchSetRecord getMatchSetRecordForPattern(final IncQuerySnapshot snapshot, final String patternFQN) {
    MatchSetRecord _xblockexpression = null;
    {
      EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
      final Function1<MatchSetRecord,Boolean> _function = new Function1<MatchSetRecord,Boolean>() {
        public Boolean apply(final MatchSetRecord it) {
          String _patternQualifiedName = it.getPatternQualifiedName();
          boolean _equals = _patternQualifiedName.equals(patternFQN);
          return Boolean.valueOf(_equals);
        }
      };
      final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
      MatchSetRecord _xifexpression = null;
      int _size = IterableExtensions.size(matchsetrecord);
      boolean _equals = (_size == 1);
      if (_equals) {
        Iterator<MatchSetRecord> _iterator = matchsetrecord.iterator();
        return _iterator.next();
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the match set records for the given pattern FQN from the snapshot.
   */
  public ArrayList<MatchSetRecord> getMatchSetRecordsForPattern(final IncQuerySnapshot snapshot, final String patternFQN) {
    ArrayList<MatchSetRecord> _arrayList = new ArrayList<MatchSetRecord>();
    final ArrayList<MatchSetRecord> matchSetRecords = _arrayList;
    EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
    final Function1<MatchSetRecord,Boolean> _function = new Function1<MatchSetRecord,Boolean>() {
      public Boolean apply(final MatchSetRecord it) {
        String _patternQualifiedName = it.getPatternQualifiedName();
        boolean _equals = _patternQualifiedName.equals(patternFQN);
        return Boolean.valueOf(_equals);
      }
    };
    Iterable<MatchSetRecord> _filter = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
    Iterables.<MatchSetRecord>addAll(matchSetRecords, _filter);
    return matchSetRecords;
  }
  
  /**
   * Creates a substitution for the given parameter name using the given value.
   *  The type of the substitution is decided based on the type of the value.
   */
  public MatchSubstitutionRecord createSubstitution(final String parameterName, final Object value) {
    if ((value instanceof EObject)) {
      final EMFSubstitution sub = EIQSnapshotFactory.eINSTANCE.createEMFSubstitution();
      sub.setValue(((EObject) value));
      sub.setParameterName(parameterName);
      return sub;
    } else {
      if ((value instanceof Integer)) {
        final IntSubstitution sub_1 = EIQSnapshotFactory.eINSTANCE.createIntSubstitution();
        sub_1.setValue((((Integer) value)).intValue());
        sub_1.setParameterName(parameterName);
        return sub_1;
      } else {
        if ((value instanceof Long)) {
          final LongSubstitution sub_2 = EIQSnapshotFactory.eINSTANCE.createLongSubstitution();
          sub_2.setValue((((Long) value)).longValue());
          sub_2.setParameterName(parameterName);
          return sub_2;
        } else {
          if ((value instanceof Double)) {
            final DoubleSubstitution sub_3 = EIQSnapshotFactory.eINSTANCE.createDoubleSubstitution();
            sub_3.setValue((((Double) value)).doubleValue());
            sub_3.setParameterName(parameterName);
            return sub_3;
          } else {
            if ((value instanceof Float)) {
              final FloatSubstitution sub_4 = EIQSnapshotFactory.eINSTANCE.createFloatSubstitution();
              sub_4.setValue((((Float) value)).floatValue());
              sub_4.setParameterName(parameterName);
              return sub_4;
            } else {
              if ((value instanceof Boolean)) {
                final BooleanSubstitution sub_5 = EIQSnapshotFactory.eINSTANCE.createBooleanSubstitution();
                sub_5.setValue((((Boolean) value)).booleanValue());
                sub_5.setParameterName(parameterName);
                return sub_5;
              } else {
                if ((value instanceof String)) {
                  final StringSubstitution sub_6 = EIQSnapshotFactory.eINSTANCE.createStringSubstitution();
                  sub_6.setValue(((String) value));
                  sub_6.setParameterName(parameterName);
                  return sub_6;
                } else {
                  if ((value instanceof Date)) {
                    final DateSubstitution sub_7 = EIQSnapshotFactory.eINSTANCE.createDateSubstitution();
                    sub_7.setValue(((Date) value));
                    sub_7.setParameterName(parameterName);
                    return sub_7;
                  } else {
                    if ((value instanceof EEnumLiteral)) {
                      final EnumSubstitution sub_8 = EIQSnapshotFactory.eINSTANCE.createEnumSubstitution();
                      String _literal = ((EEnumLiteral) value).getLiteral();
                      sub_8.setValueLiteral(_literal);
                      EEnum _eEnum = ((EEnumLiteral) value).getEEnum();
                      sub_8.setEnumType(_eEnum);
                      sub_8.setParameterName(parameterName);
                      return sub_8;
                    } else {
                      final MiscellaneousSubstitution sub_9 = EIQSnapshotFactory.eINSTANCE.createMiscellaneousSubstitution();
                      sub_9.setValue(value);
                      sub_9.setParameterName(parameterName);
                      return sub_9;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  }
  
  /**
   * Registers the matcher factories used by the derived features of snapshot models into the EMF-IncQuery
   * matcher factory registry. This is useful when running tests without extension registry.
   */
  public void prepareSnapshotMatcherFactories() {
    try {
      EObject _parse = this.parseHelper.parse("\n\t\t\tpackage org.eclipse.viatra2.emf.incquery.testing.queries\n\n\t\t\timport \"http://www.eclipse.org/viatra2/emf/incquery/snapshot\"\n\t\t\timport \"http://www.eclipse.org/emf/2002/Ecore\"\n\t\t\t\n\t\t\tpattern RecordRoleValue(\n\t\t\t\tRecord : MatchRecord,\n\t\t\t\tRole\n\t\t\t) = {\n\t\t\t\tMatchSetRecord.filter(MS,Record);\n\t\t\t\tRecordRole::Filter == Role;\n\t\t\t} or {\n\t\t\t\tMatchSetRecord.matches(MS,Record);\n\t\t\t\tRecordRole::Match == Role;\n\t\t\t}\n\t\t\t\n\t\t\tpattern SubstitutionValue(\n\t\t\t\tSubstitution : MatchSubstitutionRecord,\n\t\t\t\tValue\n\t\t\t) = {\n\t\t\t\tMiscellaneousSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tEMFSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tIntSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tLongSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tDoubleSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tFloatSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tBooleanSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tStringSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tDateSubstitution.value(Substitution,Value);\n\t\t\t} or {\n\t\t\t\tEnumSubstitution.valueLiteral(Substitution,Value);\n\t\t\t}\n\t\t");
      final PatternModel patternModel = ((PatternModel) _parse);
      EList<Pattern> _patterns = patternModel.getPatterns();
      final Procedure1<Pattern> _function = new Procedure1<Pattern>() {
        public void apply(final Pattern it) {
          final IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> factory = QuerySpecificationRegistry.getOrCreateQuerySpecification(it);
          QuerySpecificationRegistry.registerQuerySpecification(factory);
        }
      };
      IterableExtensions.<Pattern>forEach(_patterns, _function);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
