/**
 * <copyright>
 * 
 * Copyright (c) 2014-2015 itemis and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     itemis - Initial API and implementation
 * 
 * </copyright>
 */
package org.eclipse.sphinx.jdt.util;

import com.google.common.base.Objects;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.util.regex.Pattern;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.ILog;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.IVMInstall2;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.osgi.service.datalocation.Location;
import org.eclipse.osgi.util.NLS;
import org.eclipse.sphinx.jdt.internal.Activator;
import org.eclipse.sphinx.jdt.internal.messages.Messages;
import org.eclipse.sphinx.platform.util.ExtendedPlatform;
import org.eclipse.sphinx.platform.util.PlatformLogUtil;
import org.eclipse.sphinx.platform.util.StatusUtil;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

@SuppressWarnings("all")
public class JavaExtensions {
  public final static String VERSION_1_8 = "1.8";
  
  public final static String DEFAULT_OUTPUT_FOLDER_NAME = "bin";
  
  public final static Pattern CLASS_NAME_PATTERN = Pattern.compile("([a-z][a-z_0-9]*\\.)*[A-Z_]($[A-Z_]|[\\w_])*");
  
  private final static String PLUGIN_ID_VERSION_SEPARATOR = "_";
  
  public static File getFile(final IClasspathEntry entry) {
    File _xifexpression = null;
    IPath _path = entry.getPath();
    File _file = _path.toFile();
    boolean _exists = _file.exists();
    if (_exists) {
      IPath _path_1 = entry.getPath();
      _xifexpression = _path_1.toFile();
    } else {
      IWorkspace _workspace = ResourcesPlugin.getWorkspace();
      IWorkspaceRoot _root = _workspace.getRoot();
      IPath _location = _root.getLocation();
      IPath _path_2 = entry.getPath();
      IPath _append = _location.append(_path_2);
      _xifexpression = _append.toFile();
    }
    return _xifexpression;
  }
  
  public static IJavaProject getJavaProject(final String projectName) {
    IWorkspace _workspace = ResourcesPlugin.getWorkspace();
    IWorkspaceRoot _root = _workspace.getRoot();
    IProject _project = _root.getProject(projectName);
    return JavaCore.create(_project);
  }
  
  /**
   * Returns the absolute path in the local file system corresponding to given workspace-relative path.
   * 
   * @param workspacePath the workspace-relative path to some resource in the workspace
   * 
   * @return the absolute path in the local file system corresponding to given <code>workspacePath</code>, or null if no path can be determined
   */
  public static IPath getLocation(final IPath workspacePath) {
    IWorkspace _workspace = ResourcesPlugin.getWorkspace();
    IWorkspaceRoot _root = _workspace.getRoot();
    IResource _findMember = _root.findMember(workspacePath);
    IPath _location = null;
    if (_findMember!=null) {
      _location=_findMember.getLocation();
    }
    return _location;
  }
  
  /**
   * Returns the plug-in installation location. It is composed of the Platform's installation location
   * followed by the name of the "plugins" folder inside it.
   * 
   * @return The plug-in installation location.
   */
  public static IPath getPluginInstallLocation() {
    Location _installLocation = Platform.getInstallLocation();
    URL _uRL = _installLocation.getURL();
    String _path = _uRL.getPath();
    Path _path_1 = new Path(_path);
    return _path_1.append("plugins");
  }
  
  /**
   * Returns the common postfix of the classpath root locations of "dev mode" plug-ins. It is equal to the name
   * of the default output folder ("bin") in Java projects.
   * 
   * @return The common classpath root location postfix of "dev" mode plug-ins.
   */
  public static String getDevModePluginClasspathRootLocationPostfix() {
    return JavaExtensions.DEFAULT_OUTPUT_FOLDER_NAME;
  }
  
  public static boolean isInstalledPluginClasspathRootLocationOf(final String pluginId, final IPath classpathLocation) {
    boolean _xblockexpression = false;
    {
      Assert.isNotNull(classpathLocation);
      boolean _and = false;
      IPath _pluginInstallLocation = JavaExtensions.getPluginInstallLocation();
      boolean _isPrefixOf = _pluginInstallLocation.isPrefixOf(classpathLocation);
      if (!_isPrefixOf) {
        _and = false;
      } else {
        String _lastSegment = classpathLocation.lastSegment();
        String[] _split = _lastSegment.split(JavaExtensions.PLUGIN_ID_VERSION_SEPARATOR);
        String _get = _split[0];
        boolean _equals = _get.equals(pluginId);
        _and = _equals;
      }
      _xblockexpression = _and;
    }
    return _xblockexpression;
  }
  
  public static boolean isDevModePluginClasspathLocationOf(final String pluginId, final IPath classpathLocation) {
    boolean _xblockexpression = false;
    {
      Assert.isNotNull(classpathLocation);
      String _string = classpathLocation.toString();
      Path _path = new Path(pluginId);
      String _devModePluginClasspathRootLocationPostfix = JavaExtensions.getDevModePluginClasspathRootLocationPostfix();
      IPath _append = _path.append(_devModePluginClasspathRootLocationPostfix);
      String _string_1 = _append.toString();
      _xblockexpression = _string.contains(_string_1);
    }
    return _xblockexpression;
  }
  
  public static Bundle getBundle(final URL pluginClasspathRootLocationURL) {
    try {
      Object _xblockexpression = null;
      {
        Assert.isNotNull(pluginClasspathRootLocationURL);
        String _path = pluginClasspathRootLocationURL.getPath();
        IPath pluginClasspathRootLocation = new Path(_path);
        IPath _pluginInstallLocation = JavaExtensions.getPluginInstallLocation();
        boolean _isPrefixOf = _pluginInstallLocation.isPrefixOf(pluginClasspathRootLocation);
        if (_isPrefixOf) {
          BundleContext _context = Activator.getContext();
          return _context.getBundle(("reference:" + pluginClasspathRootLocationURL));
        }
        String _lastSegment = pluginClasspathRootLocation.lastSegment();
        String _devModePluginClasspathRootLocationPostfix = JavaExtensions.getDevModePluginClasspathRootLocationPostfix();
        boolean _equals = _lastSegment.equals(_devModePluginClasspathRootLocationPostfix);
        if (_equals) {
          BundleContext _context_1 = Activator.getContext();
          IPath _removeLastSegments = pluginClasspathRootLocation.removeLastSegments(1);
          File _file = _removeLastSegments.toFile();
          URI _uRI = _file.toURI();
          URL _uRL = _uRI.toURL();
          String _plus = ("reference:" + _uRL);
          return _context_1.getBundle(_plus);
        }
        _xblockexpression = null;
      }
      return ((Bundle)_xblockexpression);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static IPath getPluginClasspathRootLocation(final String pluginId) {
    final Bundle bundle = ExtendedPlatform.loadBundle(pluginId);
    boolean _notEquals = (!Objects.equal(bundle, null));
    if (_notEquals) {
      try {
        final URL classpathRootURL = bundle.getResource("/");
        final URL resolvedClasspathRootURL = FileLocator.resolve(classpathRootURL);
        String _path = resolvedClasspathRootURL.getPath();
        Path _path_1 = new Path(_path);
        return _path_1.removeTrailingSeparator();
      } catch (final Throwable _t) {
        if (_t instanceof IOException) {
          final IOException ex = (IOException)_t;
          Activator _default = Activator.getDefault();
          final IStatus status = StatusUtil.createErrorStatus(_default, ex);
          Activator _default_1 = Activator.getDefault();
          ILog _log = _default_1.getLog();
          _log.log(status);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return null;
  }
  
  public static void validateCompilerCompliance(final String compliance) {
    final IVMInstall install = JavaRuntime.getDefaultVMInstall();
    if ((install instanceof IVMInstall2)) {
      final String compilerCompliance = JavaExtensions.getCompilerCompliance(((IVMInstall2)install), compliance);
      boolean _equals = compilerCompliance.equals(compliance);
      boolean _not = (!_equals);
      if (_not) {
        try {
          final float complianceToSet = Float.parseFloat(compliance);
          final float complianceFromVM = Float.parseFloat(compilerCompliance);
          String _bind = NLS.bind(Messages.error_JRECompliance_NotCompatible, compliance, compilerCompliance);
          Assert.isLegal((complianceToSet <= complianceFromVM), _bind);
        } catch (final Throwable _t) {
          if (_t instanceof NumberFormatException) {
            final NumberFormatException ex = (NumberFormatException)_t;
            Activator _default = Activator.getDefault();
            PlatformLogUtil.logAsWarning(_default, ex);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
  }
  
  private static String getCompilerCompliance(final IVMInstall2 vmInstall, final String defaultCompliance) {
    Assert.isNotNull(vmInstall);
    final String version = vmInstall.getJavaVersion();
    boolean _equals = Objects.equal(version, null);
    if (_equals) {
      return defaultCompliance;
    } else {
      boolean _startsWith = version.startsWith(JavaExtensions.VERSION_1_8);
      if (_startsWith) {
        return JavaExtensions.VERSION_1_8;
      } else {
        boolean _startsWith_1 = version.startsWith(JavaCore.VERSION_1_7);
        if (_startsWith_1) {
          return JavaCore.VERSION_1_7;
        } else {
          boolean _startsWith_2 = version.startsWith(JavaCore.VERSION_1_6);
          if (_startsWith_2) {
            return JavaCore.VERSION_1_6;
          } else {
            boolean _startsWith_3 = version.startsWith(JavaCore.VERSION_1_5);
            if (_startsWith_3) {
              return JavaCore.VERSION_1_5;
            } else {
              boolean _startsWith_4 = version.startsWith(JavaCore.VERSION_1_4);
              if (_startsWith_4) {
                return JavaCore.VERSION_1_4;
              } else {
                boolean _startsWith_5 = version.startsWith(JavaCore.VERSION_1_3);
                if (_startsWith_5) {
                  return JavaCore.VERSION_1_3;
                } else {
                  boolean _startsWith_6 = version.startsWith(JavaCore.VERSION_1_2);
                  if (_startsWith_6) {
                    return JavaCore.VERSION_1_3;
                  } else {
                    boolean _startsWith_7 = version.startsWith(JavaCore.VERSION_1_1);
                    if (_startsWith_7) {
                      return JavaCore.VERSION_1_3;
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    return defaultCompliance;
  }
}
