/*$Id: ActiveRuleSet.cpp,v 1.2 2006/08/20 21:03:03 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "ActiveRuleSet.h"
#define OVECCOUNT 30

ActiveRuleSet::ActiveRuleSet(Rules *rules): headers(NULL), url(NULL)
{
	TRACE("constructor %08x", this);
	this->noContent = *rules->getNoContentRuleSet();
	this->rules = rules;
	rulesVersion = rules->getRulesVersion();
}

ActiveRuleSet::~ActiveRuleSet()
{
	TRACE("destructor %08x", this);
	if (url)
		free(url);
	if (headers)
		free(headers);
}

inline bool ActiveRuleSet::match(const RE &re_info, const char *s, int len)
{
	int rc;	
	int ovector[OVECCOUNT];	
	rc = pcre_exec(re_info.re_compiled, re_info.re_extra, s, len, 
		       0, 0, ovector, OVECCOUNT);	
        if (rc < 0)
		return false;
	return true;
}


int 
urlContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;
	
	it = caller->urlMatchedParts.find(rule);
	
	if (it != caller->urlMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		return 0;
	}
	caller->urlMatchedParts[rule].insert(match);

	
	if (caller->urlMatchedParts[rule].size() == rule->url_content.size()){
		caller->urlContentCorrect[rule->actionID].insert(rule);		
	}
	return 0;
}

int 
headersContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;
	
	if (caller->noMatch.find(rule) != caller->noMatch.end() ||
	    (rule->url_content.size() && 
	     caller->urlContentCorrect[rule->actionID].find(rule) ==
	     caller->urlContentCorrect[rule->actionID].end())){
		return 0;
	}


	it = caller->headersMatchedParts.find(rule);

	if (it != caller->headersMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		return 0;
	}
	caller->headersMatchedParts[rule].insert(match);

	
	if (caller->headersMatchedParts[rule].size() == rule->headers_content.size()){
		caller->headersContentCorrect[rule->actionID].insert(rule);		
	}
	return 0;
}

int 
bodyContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;
	
	if (caller->noMatch.find(rule) != caller->noMatch.end() ||
	    (rule->url_content.size() && 
	     caller->urlContentCorrect[rule->actionID].find(rule) ==
	     caller->urlContentCorrect[rule->actionID].end()) ||
	    (rule->headers_content.size() && 
	     caller->headersContentCorrect[rule->actionID].find(rule) ==
	     caller->headersContentCorrect[rule->actionID].end())
	    ){
		TRACE("wrong url or headers %d %d",
		      rule->url_content.size(), rule->headers_content.size());
		if (caller->noMatch.find(rule) != caller->noMatch.end())
			TRACE("in noMatch set");
		return 0;
	}
	
	
	it = caller->bodyMatchedParts.find(rule);
	
	if (it != caller->bodyMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		TRACE("previously matched");
		return 0;
	}
	TRACE("inserting");
	caller->bodyMatchedParts[rule].insert(match);
	
	
	if (caller->bodyMatchedParts[rule].size() == rule->body_content.size()){
		TRACE("ok");
		caller->bodyContentCorrect[rule->actionID].insert(rule);
		list<const Rule *> result;
		/*check if url and headers part of this new rule are ok*/
		caller->checkUrlSet(caller->bodyContentCorrect[rule->actionID], result);
		caller->checkHeadersSet(caller->bodyContentCorrect[rule->actionID], result);
		FK_ASSERT(result.size() == 0);
	}
	return 0;
}

void
ActiveRuleSet::checkUrlSet(set<Rule *> &rules, list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking url");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;
		for(it2 = rule->urlREs.begin(); it2 != rule->urlREs.end(); ++it2){
			TRACE("Matching %s against %s", it2->pattern, url);
			if (!match(*it2, url, urllen)){
				TRACE("not matched");				
				to_erase.push_front(it);
				noMatch.insert(rule);
				break;
			}
			TRACE("matched");
		}
		if (it2 == rule->urlREs.end() && !rule->headersRule() && 
		    !rule->bodyRule()){
			result.push_back(rule);

			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}
	
	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}

}

void
ActiveRuleSet::checkHeadersSet(set<Rule *> &rules, list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking headers");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;

		for(it2 = rule->headersREs.begin(); it2 != rule->headersREs.end(); ++it2){
			TRACE("Matching %s against %s", it2->pattern, headers);
			if (!match(*it2, headers, headerslen)){
				TRACE("not matched");				
				to_erase.push_front(it);
				noMatch.insert(rule);
				break;
			}
			TRACE("matched");
		}
		if (it2 == rule->headersREs.end() && !rule->bodyRule()){
			TRACE("inserting %d", rule->fid);
			result.push_back(rule);
			
			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}
	
	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}
}

void
ActiveRuleSet::checkBodySet(set<Rule *> &rules, const char *body, int bodylen,
			     list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking body");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;
		bool matched = true;
		for(it2 = rule->bodyREs.begin(); it2 != rule->bodyREs.end(); ++it2){
			if (bodyMatchedREs.find(&*it2) != bodyMatchedREs.end())
				continue;
			TRACE("Matching %s against body", it2->pattern, body);
			if (!match(*it2, body, bodylen)){
				TRACE("not matched");
				matched = false;
			}
			else{
				TRACE("matched");
				bodyMatchedREs.insert(&*it2);
			}
		}
		if (matched){
			TRACE("inserting %d", rule->fid);
			result.push_back(rule);	
			
			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}

	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}
}

list<const Rule*>
ActiveRuleSet::checkURL(const char *url_arg, int urllen_arg)
{
	list<const Rule *> result;
	this->url = strdup(url_arg);
	this->urllen = urllen_arg;
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking url - rules invalidated");
		return result;
	}

	searchAPI->search_find(URL_CONTENT, (unsigned char *)url, urllen,
			       urlContentMatched, this);

	for(int i = 0; i < NACTIONS; i++){
		checkUrlSet(noContent[i], result);
		checkUrlSet(urlContentCorrect[i], result);
		for(set<Rule *>::iterator it = urlContentCorrect[i].begin();
		    it != urlContentCorrect[i].end(); ++it)
			if ((*it)->headers_content.size() == 0 && 
			    (*it)->body_content.size() == 0)
				noContent[i].insert(*it);	
		
	}
	return result;	
}


list<const Rule*>
ActiveRuleSet::checkHeaders(const char *headers_arg, int headerslen_arg)
{
	list<const Rule *> result;
	this->headers = strdup(headers_arg);
	this->headerslen = headerslen_arg;
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking headers - rules invalidated");
		return result;
	}
	
	searchAPI->search_find(HEADERS_CONTENT, (unsigned char *)headers, headerslen, 
			       headersContentMatched, this);
	
	for(int i = 0; i < NACTIONS; i++){
		checkUrlSet(headersContentCorrect[i], result);
		
		checkHeadersSet(noContent[i], result);
		checkHeadersSet(headersContentCorrect[i], result);

		for(set<Rule *>::iterator it = headersContentCorrect[i].begin();
		    it != headersContentCorrect[i].end(); ++it)
			if ((*it)->body_content.size() == 0)
				noContent[i].insert(*it);	
	}
	return result;

}

list<const Rule*>
ActiveRuleSet::checkBody(const char *body, int bodylen)
{
	list<const Rule *> result;
	TRACE("called");
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking body - rules invalidated");
		return result;
	}
	
	TRACE("search API calling %d %d", BODY_CONTENT, bodylen);
	searchAPI->search_find(BODY_CONTENT, (unsigned char *)body, bodylen, 
			       bodyContentMatched, this);
	TRACE("ok");
	for(int i = 0; i < NACTIONS; i++){
		checkBodySet(noContent[i], body, bodylen, result);
		TRACE("%d", bodyContentCorrect[i].size());
		checkBodySet(bodyContentCorrect[i], body, bodylen, result);

	}
	return result;

}

