/* The functions which do most of the real work in the calculator */


#include <ctype.h>
#include <math.h>
#include <string.h>

#include "hp67.h"
#include "calcfloat.h"
#include "stack.h"

extern Stack<CalcFloat> fltstack;

#include "hpglobals.h"
#include "flags.h"

Flags flag;

#define SET_LASTX fltstack.lastx = *fltstack.elem_x



/* Beware of calling hp_* functions from within other hp_*
 * functions. The *scratchspace[] variables are global, so there is
 * the possibility of interaction unless you're careful. */


void hp_convert_ang_to_rad(Arithdatatype &angle)
{
  switch(angmode) {
  case RAD:
    return;
    break;
  case DEG:
    *angscratch = M_PI / 180.0;
    angle.times(*angscratch);
    break;
  case GRD:
    *angscratch = M_PI / 200.0;
    angle.times(*angscratch);
    break;
  }
  return;  
}


void hp_convert_ang_from_rad(Arithdatatype &angle)
{
  switch(angmode) {
  case RAD:
    return;
    break;
  case DEG:
    *angscratch = 180.0 / M_PI;
    angle.times(*angscratch);
    break;
  case GRD:
    *angscratch = 200.0 / M_PI;
    angle.times(*angscratch);
    break;
  }
  return;  
}



sp_acts push_scalar(CalcProgMem *pmem)
{
  fltstack.push(*pmem->value);
  return NONE;
}

sp_acts hp_add(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().plus(fltstack.pop()));
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_subtract(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().minus(fltstack.pop()));
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_multiply(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().times(fltstack.pop()));
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_divide(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().div_by(fltstack.pop()));
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_sin(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  hp_convert_ang_to_rad(*scratchspace[0]);
  scratchspace[0]->sin();
  fltstack.push(*scratchspace[0]);
  return NONE;
}

sp_acts hp_cos(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  hp_convert_ang_to_rad(*scratchspace[0]);
  scratchspace[0]->cos();
  fltstack.push(*scratchspace[0]);
  return NONE;
}

sp_acts hp_tan(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  hp_convert_ang_to_rad(*scratchspace[0]);
  scratchspace[0]->tan();
  fltstack.push(*scratchspace[0]);
  return NONE;
}

sp_acts hp_arcsin(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  scratchspace[0]->arcsin();
  hp_convert_ang_from_rad(*scratchspace[0]);
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_arccos(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  scratchspace[0]->arccos();
  hp_convert_ang_from_rad(*scratchspace[0]);
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_arctan(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  scratchspace[0]->arctan();
  hp_convert_ang_from_rad(*scratchspace[0]);
  fltstack.push(*scratchspace[0]);
  return NONE;
}

sp_acts hp_sqrt(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().sqrt());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_square(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().square());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_log_e(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().log_e());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_exp_e(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().exp_e());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_log_10(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().log_10());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_exp_10(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().exp_10());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_recip(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().recip());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_power(CalcProgMem *)
{
  SET_LASTX;

  *scratchspace[0] = fltstack.pop();

  fltstack.push(*scratchspace[1] = fltstack.pop().power(*scratchspace[0]));
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}

sp_acts hp_abs(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(fltstack.pop().abs());
  return NONE;
}

sp_acts hp_factorial(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().factorial());
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}


// Other actions


// A label in the program. Nothing happens.
sp_acts hp_label(CalcProgMem *)
{
  return NONE;
}


// A goto statement
sp_acts hp_goto(CalcProgMem *pmem)
{
  if (pmem->perform_goto() == -1) {
    runmode = IMMED;
    return ERROR;
  }
  return NONE;
}


// A gosub statement
sp_acts hp_gosub(CalcProgMem *pmem)
{
  ProgramSpace *pspace;

  pspace = pmem->myspace;
  if (pspace == NULL)
    pspace = &program;


  pspace->push_retaddr();

  if (pmem->perform_goto() == -1) {
    pspace->pop_retaddr();
    runmode = IMMED;
    return ERROR;
  }
  return NONE;
}


sp_acts hp_run(CalcProgMem *pmem)
{
  if (pmem->argument == NULL ||  
      *pmem->argument == 0) {     // No argument, run from where we are.
    runmode = RUN_PROG;
    return NONE;
  }

  if (pmem->perform_goto() == -1) {
    runmode = IMMED;
    return ERROR;
  }

  program.clear_return_stack();

  runmode = RUN_PROG;
  return NONE;
}



// A return statement
sp_acts hp_return(CalcProgMem *pmem)
{
  if (pmem->myspace->pop_retaddr() == -1) {   /* No return address. Done. */
    runmode = IMMED;
    return NONE;
  }

  return NONE;
}


// Change the number of digits to display
sp_acts hp_dispn(CalcProgMem *pmem)
{
  int n;

  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    n = I_REGISTER.int_value();
  } else {
    n = atoi(pmem->argument);
  }


  if (n < 0 || n >= MAXDIGS) {
    runmode = IMMED;
    return ERROR;
  }

  dispdigs = n;
  return NONE;
}


// Change the display mode
sp_acts hp_fix(CalcProgMem *)
{
  dispmode = FIX;
  return NONE;
}


sp_acts hp_sci(CalcProgMem *)
{
  dispmode = SCI;
  return NONE;
}


sp_acts hp_eng(CalcProgMem *)
{
  dispmode = ENG;
  return NONE;
}


sp_acts hp_hexmode(CalcProgMem *)
{
  dispmode = HEX;
  return NONE;
}


// Exchange x and i registers
sp_acts hp_x_int_i(CalcProgMem *)
{
  *scratchspace[0] = I_REGISTER;
  I_REGISTER = fltstack.pop();
  fltstack.push(*scratchspace[0]);
  return NONE;
}


// Not done yet
sp_acts hp_round(CalcProgMem *)
{
  char obuffer[1000];
  
  *scratchspace[0] = fltstack.pop();
  scratchspace[0]->printvalue(dispdigs, dispmode, obuffer, 1000, NULL, NULL);
  *scratchspace[0] = atof(obuffer);
  fltstack.push(*scratchspace[0]);
  return NONE;
}


// Store in memory
sp_acts hp_store(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    *scratchspace[0] = fltstack.reveal_value(0);
    memories[obuf] = *scratchspace[0];
    return NONE;
  } else {
    *scratchspace[0] = fltstack.reveal_value(0);
    memories[pmem->argument] = *scratchspace[0];
    return NONE;
  }
}


sp_acts hp_store_plus(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[obuf].plus(*scratchspace[0]);
  } else {
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[pmem->argument].plus(*scratchspace[0]);
  }
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_store_minus(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[obuf].minus(*scratchspace[0]);
  } else {
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[pmem->argument].minus(*scratchspace[0]);
  }
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_store_times(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[obuf].times(*scratchspace[0]);
  } else {
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[pmem->argument].times(*scratchspace[0]);
  }
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_store_divide(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[obuf].div_by(*scratchspace[0]);
  } else {
    *scratchspace[0] = fltstack.reveal_value(0);
    *scratchspace[1] = memories[pmem->argument].div_by(*scratchspace[0]);
  }
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_store_i(CalcProgMem *)
{
  I_REGISTER = fltstack.pop();
  fltstack.push(I_REGISTER);
  return NONE;
}


sp_acts hp_dsz(CalcProgMem *)
{
  *scratchspace[0] = 1;
  I_REGISTER.minus(*scratchspace[0]);
  if (I_REGISTER.int_value() == 0)
    return SKIP_NEXT;
  return NONE;
}


sp_acts hp_dsz_indirect(CalcProgMem *)
{
  intstring obuf;

  *scratchspace[0] = 1;
  sprintf(obuf, "%d", I_REGISTER.int_value());
  memories[obuf].minus(*scratchspace[0]);
  if (memories[obuf].int_value() == 0)
    return SKIP_NEXT;

  return NONE;
}



sp_acts hp_recall(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {  /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    fltstack.push(memories[obuf]);
    return NONE;
  } else {
    fltstack.push(memories[pmem->argument]);
    return NONE;
  }
}


sp_acts hp_recall_i(CalcProgMem *)
{
  fltstack.push(I_REGISTER);
  return NONE;
}


sp_acts hp_isz(CalcProgMem *)
{
  *scratchspace[0] = 1;
  I_REGISTER.plus(*scratchspace[0]);
  if (I_REGISTER.int_value() == 0)
    return SKIP_NEXT;
  return NONE;
}


sp_acts hp_isz_indirect(CalcProgMem *)
{
  intstring obuf;

  *scratchspace[0] = 1;
  sprintf(obuf, "%d", I_REGISTER.int_value());
  memories[obuf].plus(*scratchspace[0]);
  if (memories[obuf].int_value() == 0)
    return SKIP_NEXT;

  return NONE;
}



// Change the angle units
sp_acts hp_deg(CalcProgMem *)
{
  angmode = DEG;
  return NONE;
}


sp_acts hp_rad(CalcProgMem *)
{
  angmode = RAD;
  return NONE;
}


sp_acts hp_grad(CalcProgMem *)
{
  angmode = GRD;
  return NONE;
}


// Change the sign
sp_acts hp_chgsgn(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(fltstack.pop().times(*scratchspace[0] = -1));
  return NONE;
}


sp_acts hp_clearx(CalcProgMem *)
{
  (void) fltstack.pop();
  return NONE;
}


// Some logical operators
sp_acts hp_x_not_zero(CalcProgMem *)
{
  if (*fltstack.elem_x != 0)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_is_zero(CalcProgMem *)
{
  if (*fltstack.elem_x == 0)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_lt_zero(CalcProgMem *)
{
  if (*fltstack.elem_x < 0)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_gt_zero(CalcProgMem *)
{
  if (*fltstack.elem_x > 0)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_eq_y(CalcProgMem *)
{
  if (*fltstack.elem_x == *fltstack.elem_y)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_neq_y(CalcProgMem *)
{
  if (*fltstack.elem_x != *fltstack.elem_y)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_leq_y(CalcProgMem *)
{
  if (*fltstack.elem_x <= *fltstack.elem_y)
    return NONE;
  return SKIP_NEXT;
}


sp_acts hp_x_gt_y(CalcProgMem *)
{
  if (*fltstack.elem_x > *fltstack.elem_y)
    return NONE;
  return SKIP_NEXT;
}



// More stack operations
sp_acts hp_x_int_y(CalcProgMem *)
{
  fltstack.exchange_xy();
  return NONE;
}


// Some conversions
sp_acts hp_rec2polar(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  *scratchspace[1] = fltstack.pop();

  scratchspace[0]->rec2polar(scratchspace[0], scratchspace[1]);
  hp_convert_ang_from_rad(*scratchspace[1]);
  fltstack.push(*scratchspace[1]);
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_polar2rec(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  *scratchspace[1] = fltstack.pop();
  hp_convert_ang_to_rad(*scratchspace[1]);

  scratchspace[0]->polar2rec(scratchspace[0], scratchspace[1]);
  fltstack.push(*scratchspace[1]);
  fltstack.push(*scratchspace[0]);
  return NONE;
}


sp_acts hp_deg2rad(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  *scratchspace[1] = M_PI / 180;
  scratchspace[0]->times(*scratchspace[1]);
  fltstack.push(*scratchspace[0]);
  return NONE;
}


sp_acts hp_rad2deg(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  *scratchspace[1] = 180 / M_PI;
  scratchspace[0]->times(*scratchspace[1]);
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_h2hms(CalcProgMem *)
{
  int negative;
  int hours, minutes, seconds;

  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  negative = (*scratchspace[0] < 0 ? 1 : 0);
  if (negative) scratchspace[0]->times(*scratchspace[3] = -1);

  hours = scratchspace[0]->int_value();

  scratchspace[0]->minus(*scratchspace[1] = hours);
  scratchspace[0]->times(*scratchspace[1] = 60);
  minutes = scratchspace[0]->int_value();

  scratchspace[0]->minus(*scratchspace[1] = minutes);
  scratchspace[0]->times(*scratchspace[1] = 60);
  seconds = scratchspace[0]->int_value();

  *scratchspace[0] = hours + minutes / 100.0 + seconds / 10000.0;
  if (negative) scratchspace[0]->times(*scratchspace[1] = -1);
  fltstack.push(*scratchspace[0]);
  return NONE;
}



sp_acts hp_hms2h(CalcProgMem *)
{
  int negative;
  int hours, minutes, seconds;

  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  negative = (*scratchspace[0] < 0 ? 1 : 0);
  if (negative) scratchspace[0]->times(*scratchspace[1] = -1);

  hours = scratchspace[0]->int_value();
 
  scratchspace[0]->minus(*scratchspace[1] = hours);
  scratchspace[0]->times(*scratchspace[1] = 100);
  minutes = scratchspace[0]->int_value();

  scratchspace[0]->minus(*scratchspace[1] = minutes);
  scratchspace[0]->times(*scratchspace[1] = 100);
  seconds = scratchspace[0]->int_value();

  *scratchspace[0] = hours + minutes / 60.0 + seconds / 3600.0;

  if (negative) scratchspace[0]->times(*scratchspace[1] = -1);
  fltstack.push(*scratchspace[0]);
  if (scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_pival(CalcProgMem *)
{
  fltstack.push(*scratchspace[0] = M_PI);
  return NONE;
}



// Add two numbers written as hhh.mmss
sp_acts hp_hmsplus(CalcProgMem *)
{
  SET_LASTX;
  (void) hp_hms2h(NULL);
  (void) hp_x_int_y(NULL);
  (void) hp_hms2h(NULL);
  (void) hp_add(NULL);
  return hp_h2hms(NULL);
}


sp_acts hp_intpart(CalcProgMem *)
{
  SET_LASTX;
  fltstack.push(*scratchspace[0] = fltstack.pop().int_value());
  return NONE;
}


sp_acts hp_fracpart(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = fltstack.pop();
  scratchspace[0]->minus(*scratchspace[1] = scratchspace[0]->int_value());
  fltstack.push(*scratchspace[0]);
  return NONE;
}


sp_acts hp_percent(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = *fltstack.elem_x;
  *scratchspace[1] = *fltstack.elem_y;
  scratchspace[0]->times(*scratchspace[1]);
  scratchspace[0]->div_by(*scratchspace[2] = 100);
  (void) fltstack.pop();
  fltstack.push(*scratchspace[0]);
  return NONE;
}


sp_acts hp_pctchange(CalcProgMem *)
{
  SET_LASTX;
  *scratchspace[0] = *fltstack.elem_x;
  *scratchspace[1] = *fltstack.elem_y;
  scratchspace[0]->minus(*scratchspace[1]);
  scratchspace[0]->div_by(*scratchspace[1]);
  (void) fltstack.pop();
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_setflag(CalcProgMem *pmem)
{
  char *holdme;
  bool tclr = FALSE;

  holdme = strdup(pmem->argument);
  if (strcmp(&holdme[strlen(holdme) - strlen(CLRONTESTSUFFIX)], CLRONTESTSUFFIX) == 0) {   // Make this a test-cleared flag
    tclr = TRUE;
    holdme[strlen(holdme) - strlen(CLRONTESTSUFFIX)] = 0;
  }

  if (strcmp(holdme, INDIRECT) == 0) {   /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    flag.set_flag(obuf, tclr);
    free(holdme);
    return NONE;
  } else {
    flag.set_flag(holdme, tclr);
    free(holdme);
    return NONE;
  }
}


sp_acts hp_clrflag(CalcProgMem *pmem)
{
  if (strcmp(pmem->argument, INDIRECT) == 0) {   /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    flag.clr_flag(obuf);
  } else {
    flag.clr_flag(pmem->argument);
  }
  return NONE;
}


sp_acts hp_testflag(CalcProgMem *pmem)
{
  bool retval;

  if (strcmp(pmem->argument, INDIRECT) == 0) {   /* Indirect */
    intstring obuf;

    sprintf(obuf, "%d", I_REGISTER.int_value());
    retval = flag.test_flag(obuf);
  } else {
    retval = flag.test_flag(pmem->argument);
  }
  if (retval) return NONE;
  return SKIP_NEXT;
}


// Statistical functions
sp_acts hp_addstats(CalcProgMem *)
{
  SET_LASTX;
  memories[STATS_SUMX].plus(*fltstack.elem_x);
  *scratchspace[0] = *fltstack.elem_x;
  scratchspace[0]->times(*fltstack.elem_x);
  memories[STATS_SUMX2].plus(*scratchspace[0]);
  memories[STATS_SUMY].plus(*fltstack.elem_y);
  *scratchspace[0] = *fltstack.elem_y;
  scratchspace[0]->times(*fltstack.elem_y);
  memories[STATS_SUMY2].plus(*scratchspace[0]);
  *scratchspace[0] = *fltstack.elem_x;
  scratchspace[0]->times(*fltstack.elem_y);
  memories[STATS_SUMXY].plus(*scratchspace[0]);
  memories[STATS_NPTS].plus(*scratchspace[0] = 1);
  (void) fltstack.pop();
  fltstack.push(memories[STATS_NPTS]);
  return NONE;
}


sp_acts hp_subtractstats(CalcProgMem *)
{
  SET_LASTX;
  memories[STATS_SUMX].minus(*fltstack.elem_x);
  *scratchspace[0] = *fltstack.elem_x;
  scratchspace[0]->times(*fltstack.elem_x);
  memories[STATS_SUMX2].minus(*scratchspace[0]);
  memories[STATS_SUMY].minus(*fltstack.elem_y);
  *scratchspace[0] = *fltstack.elem_y;
  scratchspace[0]->times(*fltstack.elem_y);
  memories[STATS_SUMY2].minus(*scratchspace[0]);
  *scratchspace[0] = *fltstack.elem_x;
  scratchspace[0]->times(*fltstack.elem_y);
  memories[STATS_SUMXY].minus(*scratchspace[0]);
  memories[STATS_NPTS].minus(*scratchspace[0] = 1);
  (void) fltstack.pop();
  fltstack.push(memories[STATS_NPTS]);
  return NONE;
}


sp_acts hp_average(CalcProgMem *)
{
  SET_LASTX;
  (void) fltstack.pop();
  (void) fltstack.pop();

  *scratchspace[0] = memories[STATS_SUMY];
  scratchspace[0]->div_by(memories[STATS_NPTS]);
  fltstack.push(*scratchspace[0]);
  *scratchspace[1] = memories[STATS_SUMX];
  scratchspace[1]->div_by(memories[STATS_NPTS]);
  fltstack.push(*scratchspace[1]);
  if (scratchspace[0]->value_ok() &&
      scratchspace[1]->value_ok())
    return NONE;
  else
    return ERROR;
}


sp_acts hp_sdev(CalcProgMem *)
{
  SET_LASTX;
  (void) fltstack.pop();
  (void) fltstack.pop();

  *scratchspace[0] = memories[STATS_SUMY2];
  *scratchspace[1] = memories[STATS_SUMY];
  scratchspace[1]->times(*scratchspace[1]);
  scratchspace[1]->div_by(memories[STATS_NPTS]);
  scratchspace[0]->minus(*scratchspace[1]);
  *scratchspace[1] = memories[STATS_NPTS];
  scratchspace[1]->minus(*scratchspace[2] = 1);
  scratchspace[0]->div_by(*scratchspace[1]);
  scratchspace[0]->sqrt();
  fltstack.push(*scratchspace[2] = *scratchspace[0]);
  *scratchspace[0] = memories[STATS_SUMX2];
  *scratchspace[1] = memories[STATS_SUMX];
  scratchspace[1]->times(*scratchspace[1]);
  scratchspace[1]->div_by(memories[STATS_NPTS]);
  scratchspace[0]->minus(*scratchspace[1]);
  *scratchspace[1] = memories[STATS_NPTS];
  scratchspace[1]->minus(*scratchspace[2] = 1);
  scratchspace[0]->div_by(*scratchspace[1]);
  scratchspace[0]->sqrt();
  fltstack.push(*scratchspace[0]);
  if (scratchspace[0]->value_ok() &&
      scratchspace[2]->value_ok())
    return NONE;
  else
    return ERROR;
}




#ifdef STRICT_STATS
// This is like the p<>s operation on the HP-67. It exchanges
// the values of memory registers 0 and 10, 1 and 11, 2 and 12, and
// so on up to 9 and 19.
sp_acts hp_exchg_p_s(CalcProgMem *)
{
  int i;
  intstring preg, sreg;

  for (i = 0; i < 10; i++) {
    sprintf(preg, "%d", i);
    sprintf(sreg, "%d", i + 10);
    *scratchspace[0] = memories[preg];
    memories[preg] = memories[sreg];
    memories[sreg] = *scratchspace[0];
  }
  return NONE;
}
#endif  /* STRICT_STATS */


// This function clears whatever registers are in labels "0" to "9"
// and "20" to "24", or all registers if there is no argument of "primary"
#ifdef STRICT_STATS
sp_acts hp_clrreg(CalcProgMem *pmem)
#else
sp_acts hp_clrreg(CalcProgMem *)
#endif
{
  int clr_everything = TRUE;

#ifdef STRICT_STATS
  if (pmem->argument != NULL &&
      pmem->argument[0] != 0 &&
      strcmp(pmem->argument, CLRONLYPRIMARY) == 0)
    clr_everything = FALSE;
#endif  /* STRICT_STATS */  

  if (clr_everything) {
    memories.erase_memory();
  } else {

    int i;
    intstring regname;

    for (i = 0; i < 10; i++) {
      sprintf(regname, "%d", i);
      memories[regname] = 0;
    }

    for (i = 20; i < 25; i++) {
      sprintf(regname, "%d", i);
      memories[regname] = 0;
    }
  }
  return NONE;
}


sp_acts hp_clrstack(CalcProgMem *)
{
  fltstack.erase_stack();
  return NONE;
}


sp_acts hp_clrpmem(CalcProgMem *)
{
  program.erase_program();
  flag.clr_all_flags();
  return NONE;
}



sp_acts hp_enter(CalcProgMem *)
{
  *scratchspace[0] = fltstack.pop();
  fltstack.push(*scratchspace[0]);
  fltstack.push(*scratchspace[0]);
  return NONE;
}


sp_acts hp_prog_mode(CalcProgMem *)
{
  runmode = ENTER_PROG;
  return NONE;
}


sp_acts hp_immed_mode(CalcProgMem *)
{
  runmode = IMMED;
  return NONE;
}


sp_acts hp_step(CalcProgMem *)
{
  runmode = STEPPING;
  return TAKE_A_STEP;
}


sp_acts hp_runstop(CalcProgMem *)
{
  if (runmode == IMMED)
    runmode = RUN_PROG;
  else
    runmode = IMMED;

  return NONE;
}

sp_acts hp_lastx(CalcProgMem *)
{
  fltstack.push(fltstack.lastx);
  return NONE;
}


sp_acts hp_diskinput(CalcProgMem *pmem)
{
  inputstream = fopen(pmem->argument, "r");
  if (inputstream == NULL) return ERROR;
  return GET_DISK_INPUT;
}


sp_acts hp_diskoutput(CalcProgMem *pmem)
{
  outputstream = fopen(pmem->argument, "w");
  if (outputstream == NULL) return ERROR;
  return WRITE_PROGRAM;
}


sp_acts hp_dataoutput(CalcProgMem *pmem)
{
  outputstream = fopen(pmem->argument, "w");
  if (outputstream == NULL) return ERROR;
  return WRITE_MEMORY;
}


sp_acts hp_rolldown(CalcProgMem *)
{
  fltstack.roll_down();
  return NONE;
}


sp_acts hp_rollup(CalcProgMem *)
{
  fltstack.roll_up();
  return NONE;
}

