/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "realfunctions.h"
#include "kalc.h"


void f_exp(void)
{
  /*
   * This function calls the _f_exp function through the wrapper.
   */

  run1_1_Function (_f_exp, "exp");
}


Object _f_exp(Object n, int *err)
{
  /*
   * This function returns e raised to its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_exp(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_exp(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_ln(void)
{
  /*
   * This function calls the _f_ln function through the wrapper.
   */

  run1_1_Function(_f_ln, "ln");
}


Object _f_ln(Object n, int *err)
{
  /*
   * This function returns the natural logarithm of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the number is greater than or equal to zero, the result is a
       real number */
    if (n.value.real >= 0) {
      *err = ERR_NOERR;
      n.value.real = re_ln(n.value.real);
      break;
    }
    /* Otherwise, it is a complex number */
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_ln(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_pow(void)
{
  /*
   * This function calls the _f_pow function through the wrapper.
   */

  run2_1_Function(_f_pow, "^");
}


Object _f_pow(Object n, Object p, int *err)
{
  /*
   * This command raises n to the pth power.
   */

  switch (type(n)) {
  case TYPE_REAL:
    switch (type(p)) {
    case TYPE_REAL:
      /* If the base is positive, the result is real. If the base is
         negative, the result is real if the absolute value of the
         exponent is <= 1. */
      if ((n.value.real < 0 && fabs(p.value.real) >= 1)
	  || n.value.real >= 0) {
	*err = ERR_NOERR;
        n.value.real = re_pow(n.value.real, p.value.real);
	break;
      }
      __f_reTOc(&p);
      
    case TYPE_CMP:
      *err = ERR_NOERR;
      __f_reTOc(&n);
      n.value.cmp = cmp_pow(n.value.cmp, p.value.cmp);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;

  case TYPE_CMP:
    switch (type(p)) {
    case TYPE_REAL:
      __f_reTOc(&p);
    case TYPE_CMP:
      *err = ERR_NOERR;
      n.value.cmp = cmp_pow(n.value.cmp, p.value.cmp);
      break;
    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_xroot(void)
{
  /*
   * This function runs the _f_xroot function through the wrapper.
   */

  run2_1_Function(_f_xroot, "xroot");
}


Object _f_xroot(Object n, Object p, int *err)
{
  /*
   * This function returns the pth root of n.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;

    n.value.real = re_xroot(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_sq(void)
{
  /*
   * This function calls the _f_sq function through the wrapper.
   */

  run1_1_Function(_f_sq, "sq");
}


Object _f_sq(Object n, int *err)
{
  /*
   * This function returns the square of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real *= n.value.real;
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sq(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_sqrt(void)
{
  /*
   * This function calls the _f_sqrt function through the wrapper.
   */

  run1_1_Function(_f_sqrt, "sqrt");
}


Object _f_sqrt(Object n, int *err)
{
  /*
   * This function returns the square root of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the number is positive, the result is a real number */
    if (n.value.real >= 0) {
      *err = ERR_NOERR;
      n.value.real = re_sqrt(n.value.real);
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sqrt(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_lnp1(void)
{
  /*
   * This function calls the _f_lnp1 function through the wrapper.
   */

  run1_1_Function(_f_lnp1, "lnp1");
}


Object _f_lnp1(Object n, int *err)
{
  /*
   * This function returns ln(1 + n)
   */

  if (type(n) == TYPE_REAL) {
    /* If the number is less than -1, error */
    *err = (n.value.real >= -1) ? ERR_NOERR : ERR_NONREALRESULT;
    n.value.real = re_lnp1(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_expm1(void)
{
  /*
   * This function calls the _f_expm1 function through the wrapper.
   */

  run1_1_Function (_f_expm1, "expm1");
}


Object _f_expm1(Object n, int *err)
{
  /*
   * This function returns exp(n - 1)
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_expm1(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_log(void)
{
  /*
   * This function calls the _f_log function through the wrapper.
   */
  
  run1_1_Function (_f_log, "log");
}
  

Object _f_log(Object n, int *err)
{
  /*
   * This function returns the decimal logarithm of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the number is greater than or equal to zero, the result is a
       real number */
    if (n.value.real >= 0) {
      *err = ERR_NOERR;
      n.value.real = re_log(n.value.real);
      break;
    }
    /* Otherwise, it is a complex number */
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_log(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_alog(void)
{
  /*
   * This function calls the _f_alog function through the wrapper.
   */

  run1_1_Function (_f_alog, "alog");
}


Object _f_alog(Object n, int *err)
{
  /*
   * This function returns 10 raised to its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_alog(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_alog(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_cis(void) 
{
  /*
   * This function calls the _f_cis function through the wrapper.
   */

  run1_1_Function(_f_cis, "cis");
}


Object _f_cis(Object n, int *err) 
{
  if (type(n) != TYPE_REAL)
    *err = ERR_BADARGUMENTTYPE;
  else {
    *err = ERR_NOERR;
    n.type = TYPE_CMP;
    n.value.cmp = cmp_cis(n.value.real);
  }

  return n;
}
