/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>

#include "realfunctions.h"


double re_gcd(register double u, register double v)
{
  /*
   * This function returns the greatest common divisor of u and v, using
   * Euclid's algorithm.
   *
   * Altough the arguments and return value are doubles, this is just to
   * allow very big numbers to be used. There is no sense in taking the
   * gcd of non-integers, and this function will give meaningless results
   * if non-integer arguments are given to it.
   */

  register double r;

  while (v != 0) {
    r = fmod(u, v);
    u = v;
    v = r;
  }

  return u;
}


double re_xroot(double b, double e)
{
  /*
   * This function returns the eth root of b.
   *
   * If e == 2 or e == 3, uses built-in functions, otherwise uses the
   * Newton-Raphson method.
   */
  if (e == 2)
    return sqrt(b);
  else if (e == 3)
    return cbrt(b);
  else {
    register double x = b / 10;

    while (fabs(pow(x, e) - b) > DBL_EPSILON)
      x += (b - pow(x, e)) / (e * pow(x, e - 1));

    return x;
  }
}


double re_ipORfp(double n, int integer)
{
  /*
   * Returns either the integer part of the fractional part of the argument.
   * If integer != 0, returns integer part. Else, returns fractional part.
   */

  double ip, fp;

  fp = modf(n, &ip);
  if (integer)
    return ip;
  else
    return fp;
}


double re_xpon(double n)
{
  /*
   * This function returns the exponent of its argument.
   * For 0, returns 0.
   */

  double xp = log10(fabs(n));
  if (n == 0)
    return 0;

  if (fabs(n) < 0)
    xp = -xp;

  return floor(xp);
}
