/*
	New Gadget eXperiment
	
	- edit dialog (outside)
  - connecting figures
  - scrolling & no scrolling
  - make TFigure::INSIDE a negative value with special handling because
    currently we're able to select objects _behind_ filled objects
*/

#include <toad/toad.hh>
#include <toad/io/serializable.hh>
#include <toad/form.hh>
#include <toad/menubar.hh>
#include <toad/pushbutton.hh>
#include <toad/fatradiobutton.hh>

#include <iostream>
#include <fstream>
#include <vector>
#include <list>
#include <set>

#include <cmath>

#include "figure.hh"
#include "pencil.hh"

class TPaintArea:
	public TWindow, public TFigureEditor
{
		typedef TWindow super;
		static double lx, ly;				// last mouse down position
		static int handle;					// last selected handle
		static double gridx, gridy;
		static TFigure* gadget;
		static TFigure* ftemplate;	// gadget template for creation of figures

	public:
		void Save();
		void Load();
	
		void SetMode(EMode);
		void SetCreate(TFigure*);	// MODE_CREATE
		
		void SetLineColor(const TRGB&);
		void SetFillColor(const TRGB&);
		void SetFilled(bool);
		
		void InvalidateFigure(TFigure*);
		
		void DeleteGadget(TFigure*);

		//. Unselect all selected objects.
		void ClearSelection();
		
		//. Delete Selected Objects
		void DeleteSelection();
		
		void Selection2Top();
		void Selection2Bottom();
		void SelectionUp();
		void SelectionDown();

		void Group();
		void Ungroup();

	private:
	
		bool filled;
		TRGB line_color;
		TRGB fill_color;
	
		void StopMode();					// stop the current mode
		
	public:
		TPaintArea(TWindow *, const string &t);
		~TPaintArea();
		void paint();

		void mouseLDown(int,int,unsigned);
		void mouseMove(int,int,unsigned);
		void mouseLUp(int,int,unsigned);
		void keyDown(TKey, char*, unsigned);
		
		typedef list<TFigure*> TGadgetList;
		TGadgetList figures;
		
		typedef set<TFigure*> TFigureSet;
		TFigureSet selection;
};

void TPaintArea::Save()
{
	ofstream fs("picture.tvx");
	TOutBinStream out(&fs);
	
	out.WriteDWord(figures.size());
	TGadgetList::iterator p,e;
	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		TFigure::Store(out, *p);
		p++;
	}
}

void TPaintArea::Load()
{
	ClearSelection();
	gadget=NULL;
	figures.erase(figures.begin(), figures.end());

	ifstream fs("picture.tvx");
	TInBinStream in(&fs);
	
	unsigned count = in.ReadDWord();
	for(unsigned i=0; i<count; i++) {
		figures.push_back(TFigure::Restore(in));
	}
	
	Invalidate();
}

void TPaintArea::DeleteGadget(TFigure *g)
{
	if (g==gadget)
		gadget=NULL;
	if (g==ftemplate)
		ftemplate=NULL;
	
	TGadgetList::iterator p,e;
	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		if (g==*p) {
			figures.erase(p);
			break;
		}
		p++;
	}

	TFigureSet::iterator s;
	s = selection.find(g);
	if (s!=selection.end())
		selection.erase(s);	

	delete g;
}

class TMainWindow:
	public TForm
{
		typedef TForm super;
	public:
		TMainWindow(TWindow* p, const string &t);
};

class TColorWindow:
	public TWindow
{
	public:
		TColorWindow(TWindow *p, const string &t, TPaintArea *);
		void paint();
		void mouseLDown(int,int,unsigned);
		
		static const int s = 8;
		int fy;
		TPaintArea *pa;
		
		TRGB line_color;
		TRGB fill_color;
		bool filled;
};

TColorWindow::TColorWindow(TWindow *p, const string &t, TPaintArea *pa):
	TWindow(p,t)
{
	this->pa = pa;
	line_color.Set(0,0,0);
	fill_color.Set(255,255,255);
	filled = false;
}

void TColorWindow::paint()
{
	TPen pen(this);
	
	TColor::EColor16 c = TColor::BLACK;
	for(unsigned y=0; ; y+=s) {
		for(unsigned x=0; x<Width(); x+=s) {
			pen.SetColor(c);
			pen.FillRectangle(x,y,s,s);
			c=c+1;
			if (c == TColor::COLOR16_MAX) {
				fy=y+s;
				goto end_of_loop;
			}
		}
	}

end_of_loop:
	pen.SetColor(0,0,0);
	pen.SetFillColor(line_color);
	pen.FillRectangle(5,fy+5,Width()-10,Width()-10);
	if (filled)
		pen.SetFillColor(fill_color);
	else
		pen.SetFillColor(255,255,255);
	pen.FillRectangle(10,fy+10,Width()-20,Width()-20);
	if (!filled) {
		pen.DrawLine(10,fy+10,Width()-10,fy+Width()-10);
		pen.DrawLine(10,fy+Width()-10,Width()-10,fy+10);
	}
}

void TColorWindow::mouseLDown(int mx,int my,unsigned m)
{
	TColor::EColor16 c = TColor::BLACK;
	for(unsigned y=0; ; y+=s) {
		for(unsigned x=0; x<Width(); x+=s) {
			if (x<=mx && mx<x+s &&
					y<=my && my<y+s) {
				TColor color(c);
				fill_color = color;
				pa->SetFillColor(color);
				Invalidate();
				return;
			}
			c=c+1;
			if (c == TColor::COLOR16_MAX)
				goto end_of_loop;
		}
	}
end_of_loop:

	TRect r(5,fy+5,Width()-10,Width()-10);
	if (r.IsInside(mx,my)) {
		filled=!filled;
		pa->SetFilled(filled);
	} else {
		TRGB a;
		a = line_color;
		line_color = fill_color;
		fill_color = a;
		pa->SetFillColor(fill_color);
		pa->SetLineColor(line_color);
	}
	Invalidate();
}

int ToadMain()
{
	TFigure::InitStorage();
	TMainWindow(NULL,"Gadget v2").Run();
}

TMainWindow::TMainWindow(TWindow* p, const string &t):
	super(p, t)
{
	static TFigRectangle grect;
	static TFigCircle gcirc;
	static TFigText gtext;
	
	SetSize(500,370);

	TPaintArea *pa = new TPaintArea(this, "paintarea");

	TMenuBar *mb = new TMenuBar(this, "menubar");
	mb->BgnPulldown("File");
		mb->AddItem("Save")->sigActivate.Add(pa,&pa->Save);
		mb->AddItem("Load")->sigActivate.Add(pa,&pa->Load);
		mb->AddItem("Exit")->sigActivate.Add(this, &closeRequest);
	mb->EndPulldown();

	Attach(mb, TOP|LEFT|RIGHT);

	TRadioState *state = new TRadioState();
	TFatRadioButton *rb, *orb;
	TPushButton *pb, *opb;

	const int w = 64;
	
	rb = new TFatRadioButton(this, "Select", state, 1);
		rb->SetSize(w,24);
		rb->sigActivate.Add(pa, &pa->SetMode, TPaintArea::MODE_SELECT);
		rb->SetDown(true);
	Attach(rb, TOP, mb);
	Attach(rb, LEFT);
	orb = rb;
	
	rb = new TFatRadioButton(this, "Rect", state, 2);
		rb->SetSize(w,24);
		rb->sigActivate.Add(pa, &pa->SetCreate, &grect);
	Attach(rb, LEFT);
	Attach(rb, TOP, orb);
	orb = rb;

	rb = new TFatRadioButton(this, "Circle", state, 3);
		rb->SetSize(w,24);
		rb->sigActivate.Add(pa, &pa->SetCreate, &gcirc);
	Attach(rb, LEFT);
	Attach(rb, TOP, orb);
	orb = rb;
	
	rb = new TFatRadioButton(this, "Text", state, 4);
		rb->SetSize(w,24);
		rb->sigActivate.Add(pa, &pa->SetCreate, &gtext);
	Attach(rb, LEFT);
	Attach(rb, TOP, orb);
	orb = rb;

	pb = new TPushButton(this, "Top");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->Selection2Top);
	Attach(pb, LEFT);
	Attach(pb, TOP, orb);
	opb = pb;
	Distance(pb, 5, TOP);

	pb = new TPushButton(this, "Up");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->SelectionUp);
	Attach(pb, LEFT);
	Attach(pb, TOP, opb);
	opb = pb;
	
	pb = new TPushButton(this, "Down");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->SelectionDown);
	Attach(pb, LEFT);
	Attach(pb, TOP, opb);
	opb = pb;
	
	pb = new TPushButton(this, "Bottom");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->Selection2Bottom);
	Attach(pb, LEFT);
	Attach(pb, TOP, opb);
	opb = pb;
	
	pb = new TPushButton(this, "Group");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->Group);
	Attach(pb, LEFT);
	Attach(pb, TOP, opb);
	opb = pb;
	
	pb = new TPushButton(this, "Ungroup");
		pb->SetSize(w,24);
		pb->sigActivate.Add(pa, &pa->Ungroup);
	Attach(pb, LEFT);
	Attach(pb, TOP, opb);
	opb = pb;

	TColorWindow *cw = new TColorWindow(this, "colorwindow", pa);
		cw->SetSize(w,1);
	Attach(cw, LEFT | BOTTOM);
	Attach(cw, TOP, opb);

	Attach(pa, TOP, mb);
	Attach(pa, RIGHT | BOTTOM);
	Attach(pa, LEFT, rb);
}

double TPaintArea::lx, TPaintArea::ly;
int TPaintArea::handle=-1;
double TPaintArea::gridx=8, TPaintArea::gridy=8;
TFigure* TPaintArea::gadget = NULL;
TFigure* TPaintArea::ftemplate = NULL;

TPaintArea::TPaintArea(TWindow* p, const string &t):
	super(p, t)
{
	line_color.Set(0,0,0);
	fill_color.Set(255,255,255);
	filled = false;
	TFigureEditor::window = this;
	mode = MODE_SELECT;

//	SetCursor(TCursor::CROSSHAIR);
	SetMouseMoveMessages(TMMM_LBUTTON);

/*
	TBitmap *bitmap = new TBitmap(8,8, TBITMAP_SERVER);
	TPen pen(bitmap);
	pen.SetColor(TColor::WHITE);
	pen.FillRectangle(0,0,8,8);
	pen.SetColor(TColor::LIGHTGRAY);
	pen.DrawPoint(0,0);
	SetBackground(bitmap);
*/
	bNoBackground = true;
}

TPaintArea::~TPaintArea()
{
	SetMode(MODE_SELECT);
	cout << "figures total: " << figures.size() << endl;
}

void TPaintArea::paint()
{
	TBitmap bmp(Width(), Height(), TBITMAP_SERVER);
	TScreenPencil pen(&bmp);

	pen.SetColor(255,255,255);
	pen.FillRectangle(0,0,Width(),Height());
	pen.SetColor(0,0,0);
	
	TGadgetList::iterator p,e;
	
	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		(*p)->paint(pen, *p == gadget && (mode==MODE_EDIT_IN_PLACE ||
																			mode==MODE_CREATING ));
		p++;
	}

	pen.SetColor(0,0,0);
	TFigureSet::iterator sp,se;
	sp = selection.begin();
	se = selection.end();
	while(sp!=se) {
		(*sp)->paintHandles(pen);
		sp++;
	}

	TPen scr(this);
	scr.DrawBitmap(0,0, &bmp);
}

void TPaintArea::SetLineColor(const TRGB &rgb)
{
	line_color = rgb;
	TFigureSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->line_color = line_color;
		p++;
	}
	if (ftemplate)
		ftemplate->line_color = line_color;
	Invalidate();
}

void TPaintArea::SetFillColor(const TRGB &rgb)
{
	fill_color = rgb;
	TFigureSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->fill_color = fill_color;
		p++;
	}
	if (ftemplate)
		ftemplate->fill_color = fill_color;
	Invalidate();
}

void TPaintArea::SetFilled(bool b)
{
	filled = b;
	TFigureSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->filled = b;
		p++;
	}
	if (ftemplate)
		ftemplate->filled = b;
	Invalidate();
}

void TPaintArea::ClearSelection()
{
	selection.erase(selection.begin(), selection.end());
	Invalidate();
}

void TPaintArea::DeleteSelection()
{
	TGadgetList::iterator p,e,del;
	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			del = p;
			p++;
			if (gadget==*del)
				gadget=NULL;
			delete *del;
			figures.erase(del);
		} else {
			p++;
		}
	}
	ClearSelection();
}

void TPaintArea::Selection2Top()
{
	TGadgetList::iterator p,b,np;
	p = figures.end();
	b = figures.begin();

	if (p==b)
		return;

	p--; np=p;											// p, np @ last element

	if (p==b)
		return;

	while(true) {
		if (selection.find(*p)!=selection.end()) {
			if (p!=np) {
				TFigure *akku = *p;
				TGadgetList::iterator mp = p;
				while(mp!=np) {
					TGadgetList::iterator op = mp;
					mp++;
					*op = *mp;
				}
				*np = akku;
			}
			np--;
		}
		if (p==b)
			break;
		p--;
	}
	
	Invalidate();
}

void TPaintArea::Selection2Bottom()
{
	TGadgetList::iterator p,e,np;
	p = np = figures.begin();
	e = figures.end();
	if (p==e)
		return;
	if (selection.find(*p)!=selection.end())
		np++;
	p++;
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			TFigure *akku = *p;
			TGadgetList::iterator mp = p;
			while(mp!=np) {
				TGadgetList::iterator op = mp;
				mp--;
				*op = *mp;
			}
			*np = akku;
			np++;
		}
		p++;
	}
	Invalidate();
}

void TPaintArea::SelectionUp()
{
	TGadgetList::iterator p,e,b,prev;
	p = e = prev = figures.end();
	b = figures.begin();
	if (p==b)
		return;
	while(true) {
		if (selection.find(*p)!=selection.end()) {
			if (prev!=e) {
				TFigure* a = *p;
				*p = *prev;
				*prev = a;
			}
			prev = e;
		} else {
			prev = p;
		}
		if (p==b)
			break;
		p--;
	}
	Invalidate();
}

void TPaintArea::SelectionDown()
{
	TGadgetList::iterator p,e,prev;
	p = figures.begin();
	e = prev = figures.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			if (prev!=e) {
				TFigure* a = *p;
				*p = *prev;
				*prev = a;
			}
			prev=e;
		} else {
			prev=p;
		}
		p++;
	}
	Invalidate();
}

void TPaintArea::Group()
{
	if (selection.size()<2)
		return;
	TFigGroup *group = new TFigGroup();
	TGadgetList::iterator p,e;

	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			group->figures.push_back(*p);
			TGadgetList::iterator del = p;
			p++;
			figures.erase(del);
		} else {
			p++;
		}
	}

	ClearSelection();
	group->CalcSize();
	figures.insert(p, group);
	selection.insert(group);
}

void TPaintArea::Ungroup()
{
	TGadgetList::iterator p,e;
	p = figures.begin();
	e = figures.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			TFigGroup *group = dynamic_cast<TFigGroup*>(*p);
			if (group) {
				TFigGroup::TFigureVector::iterator vp,ve;
				vp = group->figures.begin();
				ve = group->figures.end();
				figures.insert(p, vp,ve);
				while(vp!=ve) {
					(*vp)->translate(group->dx, group->dy);
					vp++;
				}
				group->figures.erase(group->figures.begin(),group->figures.end());
				delete group;
				TGadgetList::iterator del = p;
				p++;
				figures.erase(del);
				continue;
			}
		}
		p++;
	}
	ClearSelection();
}

void TPaintArea::SetMode(EMode m)
{
	SetFocus();
	if (gadget &&
				(mode==MODE_CREATE ||
				mode==MODE_CREATING ||
				mode==MODE_EDIT_IN_PLACE) ) {
		unsigned r = gadget->stop();
		if (r & TFigure::DELETE) {
			DeleteGadget(gadget);
		}
		ClearSelection();
		Invalidate();
	}
	mode = m;
}

void TPaintArea::SetCreate(TFigure *t)
{
	if (ftemplate) {
		delete ftemplate;
	}
	TCloneable *clone = t->clone();
	ftemplate = dynamic_cast<TFigure*>(clone);
	if (!ftemplate) {
		cerr << "TFigure::clone() didn't delivered a TFigure" << endl;
		delete clone;
		return;
	}
	ftemplate->line_color = line_color;
	ftemplate->fill_color = fill_color;
	ftemplate->filled     = filled;
	ClearSelection();
	SetMode(MODE_CREATE);
}

void TPaintArea::StopMode()
{
	switch(mode) {
		case MODE_CREATING:
			mode = MODE_CREATE;
			ClearSelection();
			// selection.insert(gadget);
			// Invalidate(gadget);
			break;
		case MODE_EDIT_IN_PLACE:
			mode = MODE_SELECT;
			break;
		default:
			cerr << __PRETTY_FUNCTION__ << ": can't stop this mode" << endl;
			break;
	}
}

void TPaintArea::keyDown(TKey key, char *s, unsigned m)
{
redo:

	if (gadget && (mode==MODE_CREATING || mode==MODE_EDIT_IN_PLACE) ) {
		unsigned r = gadget->keyDown(this,key,s,m);
		if (r & TFigure::STOP)
			StopMode();
		if (r & TFigure::DELETE)
			DeleteGadget(gadget);
		if (r & TFigure::REPEAT)
			goto redo;
		return;
	} else {
		switch(key) {
			case TK_DELETE:
				DeleteSelection();
				break;
		}
	}
}

static bool ready_to_move;
static int  bx, by;

void TPaintArea::mouseLDown(int mx,int my, unsigned m)
{
	ready_to_move = false;
	bx = -1;

	double x = floor( ((double)mx + gridx/2.0 ) / gridx ) * gridx;
	double y = floor( ((double)my + gridy/2.0 ) / gridy ) * gridy;

	lx = x; ly=y;
	SetFocus();

	// handle special operation modes
	//--------------------------------
redo:

	if (mode==MODE_CREATE) {
		ClearSelection();
		gadget = static_cast<TFigure*>(ftemplate->clone());
		figures.push_back(gadget);
		gadget->startCreate();
		unsigned r = gadget->mouseLDown(this,x,y,m);
		mode = MODE_CREATING;
		if (r & TFigure::STOP)
			StopMode();
		if (r & TFigure::DELETE)
			DeleteGadget(gadget);
		if (r & TFigure::REPEAT)
			goto redo;
		return;
	}

	if (mode==MODE_CREATING || mode==MODE_EDIT_IN_PLACE) {
		if (!gadget) {
			cerr << "uups, no gadget in " << __FILE__ << " at line " << __LINE__ << endl;
			return;
		}
		unsigned r = gadget->mouseLDown(this,x,y,m);
		if (r & TFigure::STOP)
			StopMode();
		if (r & TFigure::DELETE)
			DeleteGadget(gadget);
		if (r & TFigure::REPEAT)
			goto redo;
		return;
	}

	// handle the handles
	//--------------------
	if ( !selection.empty() && !(m&MK_DOUBLE) ) {
		TFigureSet::iterator p,e;
		p = selection.begin();
		e = selection.end();
		while(p!=e) {
			handle = (*p)->getHandle(x, y);
			if (handle!=-1) {
				if (selection.size()>1) {
					TFigure *g = *p;
					ClearSelection();
					selection.insert(g);
				}
				return;
			}
			p++;
		}
	}

	// find a gadget
	//--------------------
	double distance = TFigure::OUT_OF_RANGE;
	TGadgetList::iterator p,b,found;
	p = found = figures.end();
	b = figures.begin();
	while(p!=b) {
		p--;
		double d = (*p)->distance(x, y);
		if (d<distance) {
			distance = d;
			found = p;
		}
	}

	if (distance > TFigure::RANGE) {
		if (!(m & MK_SHIFT)) {
			ClearSelection();
			Invalidate();
		}
		return;
	}
	
	TFigure *g = *found;
		
	if (m&MK_DOUBLE) {
		if (g->startInPlace()) {
			ClearSelection();
			gadget = g;
			mode = MODE_EDIT_IN_PLACE;
			#if 0
			g->mouseLDown(this,x,y,m);
			return;
			#else
			goto redo;
			#endif
		}
	}

	Invalidate();
	if (m&MK_SHIFT) {
		TFigureSet::iterator sp = selection.find(g);
		if (sp!=selection.end()) {
			selection.erase(sp);
			return;
		}
	} else {
		ready_to_move = true;
		TFigureSet::iterator sp = selection.find(g);
		if (sp!=selection.end()) {
			return;
		}
		ClearSelection();
	}
	selection.insert(g);
}


void TPaintArea::mouseMove(int mx, int my, unsigned m)
{
	double x = floor( ((double)mx + gridx/2.0 ) / gridx ) * gridx;
	double y = floor( ((double)my + gridy/2.0 ) / gridy ) * gridy;

redo:

	if (gadget && (mode==MODE_CREATING || mode==MODE_EDIT_IN_PLACE) ) {
		unsigned r = gadget->mouseMove(this,x,y,m);
		if (r & TFigure::STOP)
			StopMode();
		if (r & TFigure::DELETE)
			DeleteGadget(gadget);
		if (r & TFigure::REPEAT)
			goto redo;
		return;
	}


	// handles
	//--------------------------
	TRect r;

	if (handle!=-1) {
		InvalidateFigure(*selection.begin());
		handle = (*selection.begin())->translateHandle(handle, x, y);
		InvalidateFigure(*selection.begin());
		return;
	}

	if (ready_to_move) {
		int dx = x-lx; lx=x;
		int dy = y-ly; ly=y;

		TFigureSet::iterator p,e;
		p = selection.begin();
		e = selection.end();
		while(p!=e) {
			InvalidateFigure(*p);
			(*p)->translate(dx, dy);
			InvalidateFigure(*p);
			p++;
		}
		return;
	}

	Invalidate();
	PaintNow();
	
	bx = x;
	by = y;
	
	TPen pen(this);
	pen.SetLineStyle(TPen::DOT);
	r.Set(lx,ly,x-lx,y-ly);
	r.Adjust();
	pen.DrawRectangle(r);
}

void TPaintArea::mouseLUp(int mx, int my, unsigned m)
{
	double x = floor( ((double)mx + gridx/2.0 ) / gridx ) * gridx;
	double y = floor( ((double)my + gridy/2.0 ) / gridy ) * gridy;

redo:
	if (gadget && (mode==MODE_CREATING || mode==MODE_EDIT_IN_PLACE) ) {
		unsigned r = gadget->mouseLUp(this,x,y,m);
		if (r & TFigure::STOP)
			StopMode();
		if (r & TFigure::DELETE)
			DeleteGadget(gadget);
		if (r & TFigure::REPEAT)
			goto redo;
		return;
	}

	// select rectangular area
	//-------------------------	
	if (mode==MODE_SELECT && !ready_to_move) {
		TGadgetList::iterator p, e;
		p = figures.begin();
		e = figures.end();
		while(p!=e) {
			TRect r(lx,ly,x-lx,y-ly);
			r.Adjust();
			if (r.IsInside( (*p)->x, (*p)->y ) &&
					r.IsInside( (*p)->x+(*p)->w, (*p)->y+(*p)->h ) ) {
				selection.insert(*p);
			}
			p++;
		}
		Invalidate();
	}
}

void TPaintArea::InvalidateFigure(TFigure* g)
{
#if 0
	TRect r;
	r.Set(g->x, g->y, g->w, g->h);
	r.x-=2;
	r.y-=2;
	r.w+=4;
	r.h+=4;
	Invalidate(r);
#else
	Invalidate();
#endif
}