package util;

import java.io.*;
import java.util.*;
import java.awt.image.*;
import java.net.*;

/**
 * Base for different tests.  It contains a few utility functions.
 */

public abstract class Util {
    // ===========================================================
    // A few utilities for testing...

    // check that two streams have the same content.
    private boolean sameContent(InputStream i1, InputStream i2) throws IOException {
	boolean ok = true;
	int c1;
	int c2;
	try {
	    while ((c1 = i1.read()) != -1) {
		c2 = i2.read();
		if (c1 != c2) {
		    debug("SameContent: false");
		    i1.close();
		    i2.close();
		    return false;
		}
	    }
	    ok = (i2.read() == -1);
	} catch (Exception ex) {
	    ok = false;
	}
	i1.close();
	i2.close();
	debug("SameContent: "+ok);
	return (ok);
    }

    // check that a stream has at least one byte...
    private boolean readable(InputStream s) throws IOException {
	try {
	    int c = s.read();
	    s.close();
	    return (c!=-1);
	} catch (Exception ex) {
	    s.close();
	    return false;
	}
    }

    //=============================================================

    /**
     * These are protected methods to be used in specific tests
     */

    // can the resource be found?
    protected boolean canFind(Class klass, String name) throws TestException {
	debug("Class: "+klass);
	debug("ClassLoader: "+klass.getClassLoader());
	debug("Resource Name: "+name);
	URL url = klass.getResource(name);
	debug("URL: "+url);
	return (url != null);
    }

    // is the resource not found?
    protected boolean canNotFind(Class klass, String name) throws TestException {
	InputStream is = klass.getResourceAsStream(name);
	URL url = klass.getResource(name);
	if (is == null && url == null) {
	    return true;
	}
	if (is == null || url == null) {
	    throw new TestException(name+" shows inconsistency between getResource and getResourceAsStream");
	}
	return false;
    }

    // is the resource consistent as a Stream through both interfaces?
    // assumes resource can be found and can be read
    protected boolean isConsistent(Class klass, String name) throws TestException {
	debug("Class: "+klass);
	debug("ClassLoader: "+klass.getClassLoader());
	debug("Resource Name: "+name);
	InputStream is = klass.getResourceAsStream(name);
	debug("InputStream: "+is);
	URL url = klass.getResource(name);
	debug("URL: "+url);
	if (is != null && url != null) {
	    try {
		return sameContent(is, url.openStream());
	    } catch (Exception ex) {
		ex.printStackTrace();
		throw new TestException(name+" throws exception in consistency check");
	    }
	}
	return false;
    }

    // is the resource readable, by both interfaces!
    // asumes resource is visible..
    protected boolean isReadable(Class klass, String name) throws TestException {
	InputStream is = klass.getResourceAsStream(name);
	URL url = klass.getResource(name);
	if (is == null || url == null) {
	    throw new TestException(name+" cannot be found");
	}
	try {
	    if (!readable(is)) {
		return false;
	    }
	    InputStream s = url.openStream();
	    if (!readable(s)) {
		return false;
	    }
	} catch (Exception ex) {
	    return false;
	}
	return true;
    }


    // is the resource Not readable, by either interface!
    // assumes resource can be found.
    protected boolean isNotReadable(Class klass, String name) throws TestException {
	InputStream is = klass.getResourceAsStream(name);
	URL url = klass.getResource(name);
	if (is == null || url == null) {
	    throw new TestException(name+" cannot be found");
	}
	try {
	    if (readable(is)) {
		return false;
	    }
	    InputStream s = url.openStream();
	    if (readable(s)) {
		return false;
	    }
	} catch (Exception ex) {
	}
	return true;
    }

    private boolean compatible(Class k1, Class k2) {
	if (InputStream.class.isAssignableFrom(k1) &&
	    InputStream.class.isAssignableFrom(k2)) {
	    print1(" assuming "+k1+" equivalent to "+k2);
	    return true;
	}
	if (ImageProducer.class.isAssignableFrom(k1) &&
	    ImageProducer.class.isAssignableFrom(k2)) {
	    print1(" assuming "+k1+" equivalent to "+k2);
	    return true;
	}
	return false;
    }

    // is the resource content-equal to a verify URL
    // assumes resource can be found.
    // returns null if OK, otherwise a description of the problem.
    protected String contentEquals(Class klass, String name, URL verify,
				   boolean strictCheck)
	    throws TestException {
	URL url = klass.getResource(name);
	if (verify == null || url == null) {
	    throw new TestException(name+" cannot be found");
	}
	Object o1;
	Object o2;
	try {
	    o1 = url.getContent();
	} catch (Exception ex) {
	    return "could not get content of "+url;
	}
	try {
	    o2 = verify.getContent();
	} catch (Exception ex) {
	    return "could not get content of "+verify;
	}
	if (o1 == null) {
	    return name+" has null content";
	}
	if (o2 == null) {
	    return verify+" has null content";
	}
	Class k1 = o1.getClass();
	Class k2 = o2.getClass();

	if (k1 != k2 &&
	    (strictCheck || !compatible(k1, k2))) {
	    return "incompatible classes;\n"+
		" "+url+" content class: "+k1+";\n"+
		" "+verify+" content class: "+k2;
	}
	if (o1 instanceof InputStream) {
	    try {
		boolean ok;
		ok = sameContent((InputStream) o1, (InputStream) o2);
		if (ok) {
		    return null;
		} else {
		    return url+" has different content to "+verify;
		}
	    } catch (Exception ex) {
		return "trouble getting stream content of "+url+" and "+verify;
	    }
	}
	return null;
    }

    // ==========================================================
    // debug & output stuff

    protected static void setOutput(PrintWriter p) {
	printWriter = p;
    }

    protected static void print(String msg) {
	printWriter.println(msg);
    }
    protected static void print(String msg, Object arr[]) {
	printWriter.print(msg);
	for (int i=0; i<arr.length; i++) {
	    printWriter.print(arr[i]);
	    if (i+1<arr.length) {
		printWriter.print(" ");
	    }
	}
	printWriter.println();
    }
    protected static void flush() {
	printWriter.flush();
    }

    protected static String listBaseOptions() {
	return "[-012] [-k] [-d] {";
    }
    protected static void explainBaseOptions() {
	print("  -012 are the level of verbosity; 0 is lowest and default");
	print("  -k means exit forcefully after last test.");
	print("     Its need is a bug; it is due to some threads hanging around");
	print("  -d set debug on");
    }
    protected static String[] doBaseOptions(String args[]) {
	while (args[0].startsWith("-")) {
	    if (args[0].equals("-0")) {
		setVerbose(0);
	    } else if (args[0].equals("-1")) {
		setVerbose(1);
	    } else if (args[0].equals("-2")) {
		setVerbose(2);
	    } else if (args[0].equals("-k")) {
		setKill(true);
	    } else if (args[0].equals("-d")) {
		setDebug(true);
	    }
	    args = shiftArgs(args);
	    if (args.length == 0) {
		break;
	    }
	}
	return args;
    }

    private static boolean kill = false;
    protected static boolean getKill() { return kill; }
    protected static void setKill(boolean what) {
	if (what) {
	    print1("WARNING: Will forcefully exit after all tests");
	}
	kill = what;
    }

    private static int verbose = 1;
    protected static int getVerbose() { return verbose; }
    protected static void setVerbose(int what) { verbose = what; }
    protected static void print1(String msg) {
	if (verbose > 0) {
	    print(msg);
	}
    }
    protected static void print1(String msg, Object a[]) {
	if (verbose > 0) {
	    print(msg, a);
	}
    }
    protected static void print2(String msg) {
	if (verbose > 1) {
	    print(msg);
	}
    }
    protected static void print2(String msg, Object a[]) {
	if (verbose > 1) {
	    print(msg, a);
	}
    }

    protected static void error(String err) {
	System.err.println(err);
    }

    private static boolean debug = false;
    protected static void setDebug(boolean what) { debug = what; }
    protected static boolean getDebug() { return debug; }
    protected static void debug(String msg) {
	if (debug) {
	    System.err.println(msg);
	}
    }
    protected static void debug(Object a[]) {
	if (debug) {
	    for (int i=0; i<a.length; i++) {
		System.err.print(a[i]);
		if (i+1<a.length) {
		    System.err.print(" ");
		}
	    }
	    System.err.println();
	}
    }

    // ===========================================================
    // Pass/Failure stuff

    protected void resetCounters() {
	failCount = 0;
	passCount = 0;
    }

    protected int getFailCount() {
	return failCount;
    }

    protected void pass(String msg) {
	passCount += 1;
	print2("    PASS: "+msg);
    }

    protected void fail(String msg) {
	failCount += 1;
	error("    FAIL: "+msg);
    }

    protected void reportTest() {
	print(this.getClass().getName()+":: "+
		passCount+" passed; "+failCount+" failed");
	print("");
    }

    private int failCount = 0;
    private int passCount = 0;

    // ===========================================================
    // Misc

    protected static String[] shiftArgs(String arr[]) {
	return shiftArgs(arr, 1);
    }

    protected static String[] shiftArgs(String arr[], int count) {
	String back[] = new String[arr.length-count];
	for (int i=0; i<arr.length-count; i++) {
	    back[i] = arr[i+count];
	}
	return back;
    }

    private static PrintWriter printWriter = new PrintWriter(System.err, true);


    // =============================================================
    // Random Stuff that might be useful
    /*
     * Automatic determination of the ClassLoader to be used to load
     * resources on behalf of the client.  N.B. The client is getLoader's
     * caller's caller.
     */
    static class StackLookup extends java.lang.SecurityManager {
	ClassLoader getLoader() {
	    Class[] stack = getClassContext();
	    /* Magic number 2 identifies our caller's caller */
	    Class c = stack[2];
	    ClassLoader cl = (c == null) ? null : c.getClassLoader();
	    return cl;
	}
	
	// access to protected method
	ClassLoader getCurrentClassLoader() {
	    return currentClassLoader();
	}

	void printClassContext() {
	    Class arr[] = getClassContext();
	    for (int i = 0; i<arr.length; i++) {
		System.err.println("@"+i+"; Class: "+arr[i]+
				   "; ClassLoader: "+arr[i].getClassLoader());
	    }
	}
    }
    static StackLookup stackLookup = new StackLookup();

    /**
     * the most recent class loader executing on the stack
     */
    protected ClassLoader myCurrentClassLoader() {
	return stackLookup.getCurrentClassLoader();
    }

    /**
     * print classes in the stack and their classloaders
     */
    protected void printClassContext() {
	stackLookup.printClassContext();
    }

    // ============================================================
    // Indirection stuff
    public void check() throws TestException {
	System.err.println("NYI");
	System.exit(1);
    }

    protected void indirect(Class klass) throws TestException {
	try {
	    if (Check.class.isAssignableFrom(klass)) {
		// the test class passed on supports indirect invocation
		debug(klass+" supports Check interface; invoking test through it");
		Check x = (Check) klass.newInstance();
		x.check((Test) this);
	    } else {
		// direct invocation
		debug("simple, direct invocation of test");
		check();
	    }
	} catch (TestException ex) {
	    throw ex;
	} catch (Exception ex) {
	    fail("failed: caught exception; ex: "+ex);
	}
    }

    

}
