import java.net.*;
import java.io.*;
import java.util.Random;

/* This test forks a simple Http client and server, and tests that the appropriate
 * actions happen on the client side (under the covers, in sun.net.* classes).  In 
 * particular the client should:
 * - get a file uncorrupted over http.
 * - get/set http headers correctly
 * - follow redirects appropriately
 * - implement Basic Http authentication
 * - recover gracefully from a "flaky" server that closes connections abruptly
 */
 
public class HttpTest extends NetTest {
 
    static int SERVER_PORT = 45987;
    static final String CGI = "/cgi-bin/random";
    static final long SEED = 1L;
    static final int POST_LEN = 1000;

    String [] args;

    /* operations the client and server do in tandem */
    static final int GET = 0;
    static final int REDIR = 1;
    static final int AUTH = 2;
    static final int FLAKE = 3;
    static final int POST = 4;
    static final int DONE = 5;
    
    /* the current operation */
    int status = GET;
    static String[][] headers = {
	{"a", "b"}, {"c", "d"}, {"e", "f"}
    };

    static String[] files = {
	"HttpTest.java",
	"HttpTest.class",
	"HttpClient.class",
	"HttpServer.class"
    };

    HttpTest(String[] a) {
	super(a);
	args = a;
    }
    
    public void run() {
	HttpServer server = new HttpServer(args);
	HttpClient client = new HttpClient(args);
	if (verbose) {
	    server.verbose = true;
	    client.verbose = true;
	}
	(new Thread(server, "HttpServer")).start();
	(new Thread(client, "HttpClient")).start();
	exit();
    }

    public static void main (String[] a) {
	(new HttpTest(a)).run();
    }
}

		
class HttpClient extends HttpTest implements sun.net.www.protocol.http.HttpAuthenticator {

    HttpClient(String[] a) {
	super(a);
    }

    public boolean schemeSupported(String s) {
	verbose("Client: returning true for scheme " +s);
	return true;
    }

    public String authString(URL u, String s, String r) {
	verbose("Client: get authentication for URL " + u + ", scheme="+s+
		"\n realm="+r);
	return "myName+myPasswd";
    }

    public void run() {

	sun.net.www.protocol.http.HttpURLConnection.setDefaultAuthenticator(this);
    
	for (status = 0; status < POST; status++) {
	    get();
	}
	post();
	exit();
    }

    private void post() {
	try {
	    URL u = new URL("http", InetAddress.getLocalHost().getHostName(), 
			    SERVER_PORT, CGI);
	    URLConnection uc = u.openConnection();
	    uc.setDoOutput(true);
	    for (int i = 0; i < headers.length; i++) {
		uc.setRequestProperty(headers[i][0], headers[i][1]);
		assert(headers[i][1].equals(uc.getRequestProperty(headers[i][0])),
		       "URLConn.getRequestProperty() doesn't work");
	    }
	
	    OutputStream os = uc.getOutputStream();
	    Random rand = new Random(SEED);
	    for (int i = 0; i < POST_LEN; i++) {
		os.write(rand.nextInt());
	    }
	    os.close();
	    InputStream is = uc.getInputStream();
	    // check expected response headers
	    checkHeaders(uc);
	    
	    int clen = uc.getContentLength();
	    assert(clen == POST_LEN, "content-length != POST length: "+clen);
	    
	    for (int i = 0; i < POST_LEN; i++) {
		int b = is.read();
		byte r = (byte) rand.nextInt();
		assert(b != -1, "unexpected EOF in post response, byte " + i);
		assert(r == (byte)b, "post response differs at byte " + i);
	    }

	    assert(is.read() == -1, "expected EOF after reading content-length from post");
	} catch (Exception e) {
	    pe(e);
	}
    }
    
    private void get() {
	try {
	    verbose("client: getting file: " + files[status]);
	    File file = new File(files[status]);
	    URL u = new URL("http", InetAddress.getLocalHost().getHostName(), 
			    SERVER_PORT, "/"+ file.getName());
	    URLConnection uc = u.openConnection();
	    for (int i = 0; i < headers.length; i++) {
		uc.setRequestProperty(headers[i][0], headers[i][1]);
		assert(headers[i][1].equals(uc.getRequestProperty(headers[i][0])),
		       "URLConn.getRequestProperty() doesn't work");
	    }
	    
	    // get responses by getting input stream
	    DataInputStream dis = new DataInputStream(uc.getInputStream());
	    
	    URL u2 = uc.getURL();

	    if (status == REDIR) {
		assert(u2 != u, "expected different URL's after redirect");
		file = new File(u2.getFile().substring(1));
	    } else {
		assert (u2 == u, "expected same URL, not a redirect");
	    }

	    // check expected response headers
	    checkHeaders(uc);
	    
	    int clen = uc.getContentLength();
	    int len = (int) file.length();
	    assert(clen == len, "content-length != file length");

	    DataInputStream fis = new DataInputStream(new BufferedInputStream(new FileInputStream(file)));
	    
	    byte[] buf1 = new byte[clen];
	    byte[] buf2 = new byte[len];

	    dis.readFully(buf1);
	    fis.readFully(buf2);	    
	    
	    assert(uc.getInputStream().read() == -1, "expected EOF after reading content-length, 1");
	    assert(dis.read() == -1, "expected EOF after reading content-length, 2");

	    for (int i = 0; i < clen; i++) {
		assert(buf1[i] == buf2[i], "http output differs at byte " + i+": http="+
		       (char)buf1[i]+ ", file="+(char)buf2[i]);
	    }
	    
	    dis.close();
	    fis.close();

	} catch (Exception e) {
	    pe(e);
	    exit();
	}
    }

    private void checkHeaders(URLConnection uc) {
	String k;
	String v = uc.getHeaderField(0);
	assert ("HTTP/1.0 200 OK".equals(v), "unexpected 1st header: " + v);
	
	for (int i = 1; i <= headers.length; i++) {
	    k = uc.getHeaderFieldKey(i);
	    v = uc.getHeaderField(i);
	    verbose("Client: header "+i+") " + k + ": " + v);
	    assert(k.equals(headers[i-1][0]), "key " + i +" not correct in response");
	    assert(v.equals(headers[i-1][1]), "value " + i+ " not correct in response");
	    v = uc.getHeaderField(headers[i-1][0]);
	    assert(v.equals(headers[i-1][1]), "value not correct for key "+headers[i-1][0]+" in response");
	}
    }
}

class HttpServer extends HttpTest {

    static final String EOL = "\r\n";
    ServerSocket ss;
    Socket s;
    PrintStream ps;
    DataInputStream is;
    HttpServer(String[] a) {
	super(a);
    }
    
    public void run() {
	try {
	    ss  = new ServerSocket(SERVER_PORT);
	    
	    boolean failOnce = true;
	    while (status < DONE) {
		s = ss.accept();
		ps = new PrintStream(s.getOutputStream());
		is = new DataInputStream(new BufferedInputStream(s.getInputStream()));
		switch (status) {
		  case GET:
		      vrfyHeaders();
		      sendFile();
		      status++;
		      break;
		case REDIR:
		    if (failOnce) {
			vrfyHeaders();
			redir();
			failOnce = false;
		    } else {
			vrfyHeaders();
			sendFile();
			failOnce = true;
			status++;
		    }
		    break;
		case AUTH:
		    if (failOnce) {
			vrfyHeaders();
			sendAuth();
			failOnce = false;
		    } else {
			vrfyHeaders();
			vrfyAuth();
			sendFile();
			failOnce = true;
			status++;
		    }
		    break;
		case FLAKE:
		    if (failOnce) {
			vrfyHeaders();
			s.close();
			failOnce = false;
		    } else {
			vrfyHeaders();
			sendFile();
			failOnce = true;
			status++;
		    }
		    break;
		case POST:
		    vrfyHeaders();
		    post();
		    status++;
		    break;
		}
	    }
	} catch (Exception e) {
	    pe(e);
	}
	exit();
    }

    void vrfyHeaders() throws IOException {
	is.mark(1000);
	String line = is.readLine();
	verbose("Server: vrfyHeaders, rqust line: \""+line+"\"");
	java.util.Hashtable h = new java.util.Hashtable();
	while ((line = is.readLine()) != null && line.length() > 0) {
	    int ind;
	    if ((ind = line.indexOf(':')) < 0) {
		continue;
	    }
	    String k = line.substring(0, ind);
	    String v = line.substring(ind+2);
	    verbose("Server: vrfyHeaders: putting in k:v: \""+k+"\":\""+v+"\"");
	    h.put(k, v);
	}
	is.reset();
	verbose("Server: read all headers: check them:");
	for (int i = 0; i < headers.length; i++) {
	    String v = (String) h.get(headers[i][0]);
	    assert( v != null && headers[i][1].equals(v), 
		    "expected client headers for key " + headers[i][0]);
	}
    }

    void post() throws IOException {
	String req = is.readLine(); // should be "POST /cgi-bin/random HTTP/1.0"
	assert(req.startsWith("POST"), "expected POST, got: " + req);
	assert(req.indexOf(CGI) == 5, "wrong index for cgi POST: " +
	       req.indexOf(CGI));
	String line;
	int cl = -1;
	while ((line = is.readLine()) != null && line.length() > 0) {
	    line = line.toLowerCase();
	    if (line.startsWith("content-length")) {
		try {
		    int ind = line.indexOf(' ');
		    cl = Integer.parseInt(line.substring(ind+1));
		    verbose("Server: post: got cl="+cl+" from line " + line);
		} catch (Exception e) {
		    pe(e);
		}
	    }
	}
	assert(cl > 0, "server got no content-length from HTTP post");
	assert(cl == POST_LEN, "server got wrong content-length: " + cl);
	Random rand = new Random(SEED);

	for (int i = 0; i < POST_LEN; i++) {
	    byte b = (byte) rand.nextInt();
	    int r = is.read();
	    assert(r != -1, "unexpected EOF from post at byte: " + i);
	    assert( (byte)r == b, "post data incorrect at byte " + i);
	}

	// this should be end of post data..
	s.setSoTimeout(2000);
	
	verbose("Server: read to stated end of post data, check it:");
	try {
	    int r = is.read();
	    pe("server: post: shouldn't have read a byte after cl: " + r);
	} catch (InterruptedIOException OK) {}
	ps.print("HTTP/1.0 200 OK"+EOL);
	for (int i = 0; i < headers.length; i++) {
	    ps.print(headers[i][0] + ": " + headers[i][1]+EOL);
	}
	ps.print("Content-length: " + POST_LEN+EOL);
	ps.write('\r');
	ps.write('\n');

	for (int i = 0; i < POST_LEN; i++) {
	    ps.write(rand.nextInt());
	}
	ps.flush();
	if (ps.checkError()) {
	    throw new IOException("bad printstream");
	}
	s.close();
	return;
    }
	
    void sendFile() throws IOException {
	String req = is.readLine(); // should look like "GET /file HTTP/1.0"
	assert(req.indexOf(files[status]) == 5, "wrong index for rqested file: " + 
	       req.indexOf(files[status]));

	File f = new File(files[status]);
	int clen = (int) f.length();
	verbose("server: sending file: " + files[status]);
	
	ps.print("HTTP/1.0 200 OK"+EOL);
	for (int i = 0; i < headers.length; i++) {
	    ps.print(headers[i][0] + ": " + headers[i][1]+EOL);
	}
	ps.print("Content-length: " + clen+EOL);
	ps.write('\r');
	ps.write('\n');
	ps.flush();
	if (ps.checkError()) {
	    throw new IOException("bad printstream");
	}
	byte[] buf = new byte[1024];
	InputStream fis = new FileInputStream(f);
	int r;
	while ((r = fis.read(buf)) != -1)
	    ps.write(buf, 0, r);
	ps.flush();
	if (ps.checkError()) {
	    throw new IOException("bad printstream");
	}
	fis.close();
	s.close();
    }

    void redir() throws IOException {
	String req = is.readLine(); // should look like "GET /file HTTP/1.0"
	assert(req.indexOf(files[status]) == 5, "wrong index for rqested file: " + 
	       req.indexOf(files[status]));
	ps.print("HTTP/1.0 302 Moved"+EOL);
	ps.print("Location: /"+files[status]+EOL);
	ps.print("\r\n");
	ps.flush();
	if (ps.checkError()) {
	    throw new IOException("bad printstream");
	}
	s.close();
    }

    void sendAuth() throws IOException {
	String req = is.readLine(); // should look like "GET /file HTTP/1.0"
	assert(req.indexOf(files[status]) == 5, "wrong index for rqested file: " + 
	       req.indexOf(files[status]));
    
	ps.print("HTTP/1.0 401 Unauthorized"+EOL);
	ps.print("WWW-Authenticate: Basic realm=\"Dave's House\""+EOL);
	ps.print("\r\n");
	ps.flush();
	if (ps.checkError()) {
	    throw new IOException("bad printstream");
	}
	s.close();
    }

    void vrfyAuth() throws IOException {
	is.mark(1000);
	String line;
	while ((line = is.readLine()) != null && line.length() > 1) {
	    verbose("server: vrfyAuth: " + line);
	    line = line.toLowerCase();
	    if (line.startsWith("authorization") && line.indexOf("basic") > 0 &&
		line.indexOf("myname+mypasswd") > 0) {
		is.reset();
		return;
	    }
	}
	pe("no authentication from client");
	is.reset();
    }
}
