/*
 * @(#)SocketTimedTest.java	1.5 97/01/27
 * 
 * Copyright (c) 1995, 1996 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * CopyrightVersion 1.1_beta
 * 
 */
import java.net.*;
import java.io.*;
import java.util.Random;

/*
 * - Test SoTimeout on TCP sockets/serversockets
 * - a client and server repeatedly connect to each other and ensure
 *   that connections timeout, and don't timeout, when expected
 *
 * - Also, we make sure to explicitly close sockets here, in hopes of
 *   catching the Solaris "close-an-fd-twice bug" if it still exists
 */


public class SocketTimedTest extends NetTest {
    // number of cycles
    int iter = 1;
    // spacing of timeouts - should be > 100
    int spacing = 500;
    // base for timeout mods - timeouts are 0 < timeout < tmod
    int tmod = 500;
    // our server
    ServerSocketTimedTest server = null;
    // are we waiting for server? 
    boolean waiter = false;
    // random # generator
    Random rand = new Random(System.currentTimeMillis());

    static byte[] msg = {
	'A', 'n' , 'd', ' ', 't', 'h', 'e', ' ', 'w', 'i', 'n', 'd',
	' ', 'c', 'r', 'i', 'e', 'd', ' ', 'M', 'a', 'r', 'y' 
    };

    public static void main(String[] a) {
	(new SocketTimedTest(a)).run();
    }

    synchronized boolean shouldWait() {
	if (waiter) {
	    notify();
	}
	return waiter = !waiter;
    }

    protected void usage() {
	p("java SocketTimedTest [\"mod\" <mod-time>] [\"spacing\" <spacing-time>] " +
	  "[iter <loop>] [verbose]");
	p("tests timeout socket option for TCP Socket, ServerSocket");
	p("\tmod: range in ms for socket timeouts - 0 < $(randomTimeout) < mod");
	p("\tspacing: time in ms added to a timeout btw. socket peers.");
	p("\titer: #iterations (default 3)");
	p("\t         (should be > 100 ms to avoid spurious errors).");
	p("\tverbose: verbose output");
	exit();
    }
    
    public SocketTimedTest(String[] a) {
	super(a);
	for (int i = 0; i < a.length; i++) {
	    if (i == 0 && a[i].length() > 0 && Character.isDigit(a[i].charAt(0))) 
		iter = Integer.parseInt(a[0]);
	    if (a[i].equals("mod")) {
		tmod = Integer.parseInt(a[++i]);
	    } else if (a[i].equals("iter")) {
		iter = Integer.parseInt(a[++i]);
	    } else if (a[i].equals("spacing")) {
		spacing = Integer.parseInt(a[++i]);
	    }
	}
    }

    public synchronized void run() {
	ServerSocket ss = null;
	Thread.currentThread().setPriority(Thread.MIN_PRIORITY + 2);
	try {
	    InetAddress in = InetAddress.getLocalHost();
	    for (int i = 0; i < iter; i++) {

		/********* initialize server thread */
		p("make ServerSocket bound to " + in + " loop #" + (i+1) + " of " + iter);
		ss = new ServerSocket(0, 10, in);
		assert(ss.getInetAddress().equals(InetAddress.getLocalHost()), "server socket bind addr != local addr");
		server = new ServerSocketTimedTest(this, ss, argv);
		server.verbose = verbose;
		Thread t = new Thread(server, "acceptor#" + i);
		t.start();
		verbose("started acceptor thread");

		/********* tell it to go, do first go-round: no-one times out, no wait*/
		verbose("wait...");
		
		if (shouldWait()) {
		    wait();
		}
		server.updateState(0, 0, false, false, false);

		Socket s = new Socket(in, ss.getLocalPort(), in, 0);

		int r = getRand();
		s.setSoTimeout(r);
		assert(r == s.getSoTimeout(), "Requested timeout " + r +  " != " + s.getSoTimeout());

		verbose("process, no timeout expected: " + s.getSoTimeout());
		process(s.getInputStream(), s.getOutputStream(), false, s);
		verbose("first process succeeded");
		s.close();


		/******** 2nd go round - server has a wait, no one times out ****/
		r = getRand();
		if (shouldWait()) {
		    verbose("wait...");
		    wait();
		}
		server.updateState(r, 0, false, false, true);
		s = new Socket(in, ss.getLocalPort(), in, 0);
		s.setSoTimeout(r + spacing);
		assert((r + spacing) == s.getSoTimeout(),
		       "Requested timeout " + (r+spacing) +  " != " + s.getSoTimeout());

		verbose("process, no timeout expected: " + s.getSoTimeout());
		process(s.getInputStream(), s.getOutputStream(), false, s);
		verbose("2nd process succeeded");
		s.close();

		/******** 3rd go round - client should timeout *****/
		r = getRand();
		if (shouldWait()) {
		    verbose("wait...");
		    wait();
		}
		server.updateState(r, 0, false, false, true);
		s = new Socket(in, ss.getLocalPort(), in, 0);
		s.setSoTimeout(r - spacing);
		assert((r-spacing) == s.getSoTimeout(),
		       "Requested timeout " + (r-spacing) +  " != " + s.getSoTimeout());

		verbose("process, timeout expected: " + s.getSoTimeout());
		process(s.getInputStream(), s.getOutputStream(), true, s);
		verbose("3rd process succeeded");
		s.close();

		/******** 4th go-round: server should timeout on accept, and die 
		 * before we connect
		 */
		if (shouldWait()) {
		    verbose("wait...");
		    wait();
		}
		server.updateState(0, r, true, true, false);
		Thread.sleep(r + spacing);
		verbose("make socket that'll be unaccepted:");
		try {
		    s = new Socket(in, ss.getLocalPort(), in, 0);
		    pe("connecting unaccepted socket should have failed");
		} catch (ConnectException e) {
		    verbose("unconnected socket did the right thing");
		}
		
	    }
	} catch (Exception e) {
	    pe(e);
	    server.die();
	    try {
		ss.close();
	    } catch (Throwable t) {}
	} finally {
	    verbose("Client side done!");
	    exit();
	}
	return;
    }

    int getRand() {
	int r = rand.nextInt() % tmod;
	if (r < 0)
	    r = -r;
	return r+spacing;
    }

    synchronized void process(InputStream is, OutputStream os, 
			      boolean expectTimeout, Socket s) throws Exception {
	os.write(msg);
	int r, nread = 0;
	long start = 0;
	try {
	    while (nread < msg.length) {
		start = System.currentTimeMillis();
		r = is.read();
		assert(!expectTimeout, "Client:process: expected to timeout");
		assert((byte)r == msg[nread++], "Client:process: wrong byte " + nread);
	    }
	    start = System.currentTimeMillis();
	    assert(is.read() == -1, "Client:process: expected EOF"); 
	} catch (InterruptedIOException e) {
	    assert(expectTimeout, "Client:process: didn't expect timeout: byte "+nread+
		   " of " +msg.length+", avail="+is.available()+", delta="+
		   (System.currentTimeMillis()-start)+", timeout="+s.getSoTimeout());
	}
    }
}

class ServerSocketTimedTest extends NetTest {
    ServerSocket ss;


    int sleep = 0;
    int timeout = 0;
    boolean expectTimeout = false;
    boolean expectDeath = false;
    boolean expectClientDeath = false;

    /* kill w/ extreme prejudice */
    boolean die = false;

    /* notify him - two way handshake */

    SocketTimedTest him;

    synchronized void updateState(int sleep, int timeout, boolean exTi, 
				  boolean exD, boolean exCl) {
	this.sleep = sleep;
	this.timeout = timeout;
	expectTimeout = exTi;
	expectDeath = exD;
	expectClientDeath = exCl;
	verbose("updating server state!");
	notify();
    }

    synchronized void die() {
	die = true;
	pe("Premature death");
	notify();
    }

    static byte[] msg = {
	'A', 'n', 'd', ' ', 't', 'h', 'e', ' ', 'w', 'i', 'n', 'd',
	' ', 'c', 'r', 'i', 'e', 'd', ' ', 'M', 'a', 'r', 'y'
    };

    public ServerSocketTimedTest(SocketTimedTest him, ServerSocket s, String[] a) {
	super(a);
	this.ss = s;
	this.him = him;
    }

    public synchronized void run() {
	while (true) {
	    try {
		if (!him.shouldWait()) {
		    verbose("server> wait...");
		    wait();
		}
		if (die) {
		    pe("server> premature death");
		    exit();
		    return;
		}
		verbose("server> do accept, timeout = "+ timeout);
		ss.setSoTimeout(timeout);
		if (ss.getSoTimeout() != timeout) {
		    pe("requested timeout " + timeout + " != " + ss.getSoTimeout());
		}
		Socket s = ss.accept();
		verbose("server> returned from accept");
		int nread = 0;
		InputStream is = s.getInputStream();
		OutputStream os = s.getOutputStream();
		while(nread < msg.length) {
		    int r;
		    if((r = is.read()) < 0) {
			pe("server> unexpected EOF");
			exit();
		    }

		    nread++;
		    verbose("server: sleep for " + sleep);
		    Thread.sleep(sleep);
		    os.write(r);
		}
		s.close();
	    } catch (InterruptedIOException e) {
		if (expectTimeout) {
		    verbose("server> got expected timeout");
		} else {
		    pe("server> didn't expect timeout");
		}
		if (expectDeath) {
		    verbose("server> expect death, exiting...");
		    try {
			ss.close();
		    } catch (Throwable t) {}
		    exit();
		    return;
		}
	    } catch (IOException e) {
		if (!expectClientDeath) {
		    pe(e);
		    exit();
		    return;
		}
	    } catch (Exception e) {
		pe(e);
		exit();
		return;
	    }
	}
    }
}
