/*
 * @(#)LoggingOutputStream.java	1.3 96/11/23
 * 
 * Copyright (c) 1995, 1996 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * CopyrightVersion 1.1_beta
 * 
 */


import java.io.*;

/**
 * This class lets you write primitive Java data types
 * to a stream in a portable way. Primitive data types are well
 * understood types with associated operations.  For example, an 
 * Integer is considered to be a good primitive data type.
 *
 * The data can be converted back using a DataInputStream.
 */

public
class LoggingOutputStream extends FilterOutputStream implements DataOutput {
    /**
     * Creates a new LoggingOutputStream.
     * @param out	the output stream
     */
    public LoggingOutputStream(OutputStream out) {
	super(out);
    }

    /**
     * Writes a byte. Will block until the byte is actually
     * written.
     * @param b the byte to be written
     * @exception IOException If an I/O error has occurred.l
     */
    public synchronized void write(int b) throws IOException {
	out.write(b);
	System.out.print("B:" + Integer.toString(b&0xff, 16) + " ");
    }

    /**
     * Writes a sub array of bytes.  
     * @param b	the data to be written
     * @param off	the start offset in the data
     * @param len	the number of bytes that are written
     * @exception IOException If an I/O error has occurred.
     */
    public synchronized void write(byte b[], int off, int len)
	throws IOException
    {
	out.write(b, off, len);
	System.out.print("[B:");
	for (int i = 0; i < len; i++) {
	    System.out.print(" " + Integer.toString(b[off+i] & 0xff, 16));
	}
	System.out.print("] ");
    }

    /**
     * Flushes the stream. This will write any buffered
     * output bytes.
     * @exception IOException If an I/O error has occurred.
     */
    public void flush() throws IOException {
	out.flush();
	System.out.println();
	System.out.flush();
    }

    /**
     * Writes a boolean.
     * @param v the boolean to be written
     */
    public final void writeBoolean(boolean v) throws IOException {
	out.write(v ? 1 : 0);
	if (v)
	    System.out.print("Z:1 ");
	else
	    System.out.print("Z:0 ");
    }

    /**
     * Writes an 8 bit byte.
     * @param v the byte value to be written
     */
    public final void writeByte(int v) throws IOException {
	out.write(v);
	System.out.print("B:" + Integer.toString(v&0xff, 16) + " ");
    }

    /**
     * Writes a 16 bit short.
     * @param v the short value to be written
     */
    public final void writeShort(int v) throws IOException {
	OutputStream out = this.out;
	out.write((v >>> 8) & 0xFF);
	out.write((v >>> 0) & 0xFF);
	System.out.print("S:" + Integer.toString(v&0xffff, 16) + " ");
    }

    /**
     * Writes a 16 bit char.
     * @param v the char value to be written
     */
    public final void writeChar(int v) throws IOException {
	OutputStream out = this.out;
	out.write((v >>> 8) & 0xFF);
	out.write((v >>> 0) & 0xFF);
	System.out.print("C:" + Integer.toString(v&0xffff, 16) + " ");
    }

    /**
     * Writes a 32 bit int.
     * @param v the integer value to be written
     */
    public final void writeInt(int v) throws IOException {
	OutputStream out = this.out;
	out.write((v >>> 24) & 0xFF);
	out.write((v >>> 16) & 0xFF);
	out.write((v >>>  8) & 0xFF);
	out.write((v >>>  0) & 0xFF);
	System.out.print("I:" + Integer.toString(v, 16) + " ");
    }

    /**
     * Writes a 64 bit long.
     * @param v the long value to be written
     */
    public final void writeLong(long v) throws IOException {
	OutputStream out = this.out;
	out.write((int)(v >>> 56) & 0xFF);
	out.write((int)(v >>> 48) & 0xFF);
	out.write((int)(v >>> 40) & 0xFF);
	out.write((int)(v >>> 32) & 0xFF);
	out.write((int)(v >>> 24) & 0xFF);
	out.write((int)(v >>> 16) & 0xFF);
	out.write((int)(v >>>  8) & 0xFF);
	out.write((int)(v >>>  0) & 0xFF);
	System.out.print("J:" + Integer.toString((int)(v& 0xffffffff), 16) +
		"," + Integer.toString((int)((v>>>32)& 0xffffffff), 16) + " ");
    }

    /**
     * Writes a 32 bit float.
     * @param v the float value to be written
     */
    public final void writeFloat(float v) throws IOException {
	System.out.print("F:" + Float.toString(v) + "= ");
	writeInt(Float.floatToIntBits(v));
	System.out.print("; ");
    }

    /**
     * Writes a 64 bit double.
     * @param v the double value to be written
     */
    public final void writeDouble(double v) throws IOException {
	System.out.print("D:" + new Double(v).toString() + "= ");
	writeLong(Double.doubleToLongBits(v));
	System.out.print("; ");
    }

    /**
     * Writes a String as a sequence of bytes.
     * @param s the String of bytes to be written
     */
    public final void writeBytes(String s) throws IOException {
	OutputStream out = this.out;
	int len = s.length();
	System.out.print("B's:");
	for (int i = 0 ; i < len ; i++) {
	    out.write((byte)s.charAt(i));
	    System.out.print(" " + (byte)s.charAt(i));
	}
	System.out.print("; ");
    }

    /**
     * Writes a String as a sequence of chars.
     * @param s the String of chars to be written
     */
    public final void writeChars(String s) throws IOException {
	OutputStream out = this.out;
	System.out.print("C's:");
	int len = s.length();
	for (int i = 0 ; i < len ; i++) {
	    int v = s.charAt(i);
	    out.write((v >>> 8) & 0xFF);
	    out.write((v >>> 0) & 0xFF);
	    System.out.print(" " + Integer.toString((v >>> 8) & 0xff) +
				Integer.toString((v >>> 0) & 0xff));
	}
	System.out.print(". ");
    }

    /**
     * Writes a String in UTF format.
     * @param str the String in UTF format
     */
    public final void writeUTF(String str) throws IOException {
	OutputStream out = this.out;
	int strlen = str.length();
	int utflen = 0;

	for (int i = 0 ; i < strlen ; i++) {
	    int c = str.charAt(i);
	    if ((c >= 0x0001) && (c <= 0x007F)) {
		utflen++;
	    } else if (c > 0x07FF) {
		utflen += 3;
	    } else {
		utflen += 2;
	    }
	}

	out.write((utflen >>> 8) & 0xFF);
	out.write((utflen >>> 0) & 0xFF);
	for (int i = 0 ; i < strlen ; i++) {
	    int c = str.charAt(i);
	    if ((c >= 0x0001) && (c <= 0x007F)) {
		out.write(c);
	    } else if (c > 0x07FF) {
		out.write(0xE0 | ((c >> 12) & 0x0F));
		out.write(0x80 | ((c >>  6) & 0x3F));
		out.write(0x80 | ((c >>  0) & 0x3F));
	    } else {
		out.write(0xC0 | ((c >>  6) & 0x1F));
		out.write(0x80 | ((c >>  0) & 0x3F));
	    }
	}

	int len = str.length();
	System.out.print("UTF:" + str + "; ");
    }

    /**
     * Returns the number of bytes written.
     * @return	the number of bytes written thus far.
     */
    public final int size() {
	return 0;
    }
}
